/*
 * $Id: cap_extint.c,v 1.2 1997/04/28 00:57:11 morgan Exp $
 *
 * Copyright (c) 1997 Andrew G Morgan <morgan@parc.power.net>
 *
 * See end of file for Log.
 *
 * This file deals with exchanging internal and external
 * representations of capability sets.
 */

#include "libcap.h"

/*
 * External representation for capabilities. (exported as a fixed
 * length (void *))
 */
#define CAP_EXT_MAGIC "\220\302\001\121"
#define CAP_EXT_MAGIC_SIZE 4
static __u8 external_magic[CAP_EXT_MAGIC_SIZE+1] = CAP_EXT_MAGIC;

struct cap_ext_struct {
    __u8 magic[CAP_EXT_MAGIC_SIZE];
    __u8 length_of_capset;
    __u8 bytes[sizeof(struct __cap_s)][3];
};

/*
 * return size of external capability set
 */

ssize_t cap_size(cap_t caps)
{
    return sizeof(struct cap_ext_struct);
}

/*
 * Copy the internal (cap_d) capability set into an external
 * representation.  The external representation is portable to other
 * Linux architectures.
 */

ssize_t cap_copy_ext(void *cap_ext, cap_t cap_d, ssize_t length)
{
    struct cap_ext_struct *result = (struct cap_ext_struct *) cap_ext;
    int i;

    /* valid arguments? */
    if (!good_cap_t(cap_d) || length < sizeof(struct cap_ext_struct)
	|| cap_ext == NULL) {
	errno = EINVAL;
	return -1;
    }

    /* fill external capability set */
    memcpy(&result->magic,external_magic,CAP_EXT_MAGIC_SIZE);
    result->length_of_capset = sizeof(struct __cap_s);

    for (i=CAP_EFFECTIVE; i<=CAP_PERMITTED; ++i) {
	int j;
	for (j=0; j<__CAP_BLKS; ++j) {
	    __u32 val;
	    int k = j << 2;

	    val = cap_d->set[i]._blk[j];
	    result->bytes[k++][i] =  val        & 0xFF;
	    result->bytes[k++][i] = (val >>= 8) & 0xFF;
	    result->bytes[k++][i] = (val >>= 8) & 0xFF;
	    result->bytes[k][i]   = (val >> 8)  & 0xFF;
	}
    }

    /* All done: return length of external representation */
    return (sizeof(struct cap_ext_struct));
}

/*
 * Import an external representation to produce an internal rep.
 * the internal rep should be liberated with cap_free().
 */

/*
 * XXX - need to take a little more care when importing small
 * capability sets.
 */

cap_t cap_copy_int(const void *cap_ext)
{
    const struct cap_ext_struct *export =
	(const struct cap_ext_struct *) cap_ext;
    cap_t cap_d;
    int set, blen;

    /* Does the external representation make sense? */
    if (export == NULL || !memcmp(export->magic, external_magic
				  , CAP_EXT_MAGIC_SIZE)) {
	errno = EINVAL;
	return NULL;
    }

    /* Obtain a new internal capability set */
    if (!(cap_d = cap_init()))
       return NULL;

    blen = export->length_of_capset;
    for (set=CAP_EFFECTIVE; set<=CAP_PERMITTED; ++set) {
	int blk;
	int bno = 0;
	for (blk=0; blk<__CAP_BLKS; ++blk) {
	    __u32 val = 0;

	    if (bno != blen)
		val  = export->bytes[bno++][set];
	    if (bno != blen)
		val |= export->bytes[bno++][set] << 8;
	    if (bno != blen)
		val |= export->bytes[bno++][set] << 16;
	    if (bno != blen)
		val |= export->bytes[bno++][set] << 24;

	    cap_d->set[set]._blk[blk] = val;
	}
    }

    /* all done */
    return cap_d;
}

/*
 * $Log: cap_extint.c,v $
 * Revision 1.2  1997/04/28 00:57:11  morgan
 * fixes and zefram's patches
 *
 * Revision 1.1  1997/04/21 04:32:52  morgan
 * Initial revision
 *
 */
