/*
 * Handle module symbols
 *
 * Copyright 1994, 1995, 1996, 1997 Jacques Gelinas <jack@solucorp.qc.ca>
 *
 * This file is part of the Linux modutils.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include "depmod.h"
extern "C" {
#include "util.h"
};

#define ALLOC_SYM	10000	/* Number of symbols allocated per chunk */
/*
 *	List of symbols
 *	For each symbol, save the name or the originating module
 */
SYMBOLS::SYMBOLS()
{
	sym.cur = NULL;
	memset(hash, 0, sizeof(hash));
	allocsym();
}

/*
 *	Allocate a new buffer for the collection of symblos
 */
void SYMBOLS::allocsym()
{
	LIST_SYMBOL *list = (LIST_SYMBOL*)xmalloc(sizeof(LIST_SYMBOL) +
						  ALLOC_SYM * sizeof(SYMBOL));
	list->next = sym.cur;
	sym.cur = list;
	sym.nb  = 0;
	sym.ptacc = list->alloc;
	sym.lastacc = list->alloc+ALLOC_SYM;
}

/*
 *	Find (or add) a symbol.
 *	Ce symbole est soit publique ou un external (need).
 *
 *	Return the found (or added) symbol
 */
SYMBOL *SYMBOLS::add (
	const char *name,
	MODULE *module,		/* Module that declares this symbol (or NULL) */
	SYM_STATUS status,
	int &module_need,	/* Value != 0 if one has established that the
				 * module being processed is required by a
				 * previously seen module.
				 * Assume that it already has a value.
				 */
	int is_common)
{
	/* Search for the name */
	unsigned hashval = 0;
	const char *pt = name;

	for (pt = name; *pt; ++pt)
		hashval = (hashval << 1) + *pt;
	hashval %= 2048;

	SYMBOL **ffind = hash + hashval;
	SYMBOL *find;

	for (find = *ffind; find; find = find->next) {
		if (strcmp(find->name, name) == 0)
			break;
	}
	if (!find) {
		/* Add to the list */
		if (sym.ptacc == sym.lastacc)
			allocsym();
		find = sym.ptacc++;
		sym.nb++;

		find->name = xstrdup(name);
		find->module = NULL;
		find->next = *ffind;
		*ffind = find;
		find->force = 0;
		find->is_common = 0;
		find->is_dup = 0;
		if (status == SYM_NEED) {
			find->need = 1;
			find->defined = 0;
			find->seen = 0;
		} else if (status == SYM_NOTUSED) {
			find->need = 0;
			find->defined = 0;
			find->seen = 0;
		} else {
			find->need = 0;
			find->defined = 1;
			find->seen = 1;
			find->is_common = is_common;
			find->module = module;
		}
	} else { /* Already there */
		if (status == SYM_DEFINED) {
			if (find->defined) {
				/*
				 * The symbol is already defined and thus not
				 * inserted in the hash. This is a duplicate.
				 * If a module is never "charged" and has a
				 * duplicate symbol, one can signal the problem.
				 */
				if (sym.ptacc == sym.lastacc)
					allocsym();
				find = sym.ptacc++;
				sym.nb++;
				find->name = xstrdup(name);
				find->next = NULL;
				find->force = 0;
				find->is_common = 0;
				find->is_dup = 1;
				find->need = 0;
				find->seen = 0;
			}
			find->module = module;
			find->defined = 1;
			find->is_common = is_common;
			if (find->need)
				module_need = 1;
		} else if (status == SYM_NEED) {
			find->need = 1;
		}
	}
	return find;
}
