/* Copyright (C) 2002, 2003 Thorsten Kukuk
   Author: Thorsten Kukuk <kukuk@suse.de>

   This code based havy on the NSS files module of glibc 2.3.2.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _GNU_SOURCE

#include <pwd.h>
#include <shadow.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <nss.h>
#include <bits/libc-lock.h>
#define __libc_lock_t pthread_mutex_t

#include "read-files.h"

static enum { none, getent, getby }last_use;

#define ISCOLON(c) ((c) == ':')

#define STRING_FIELD(variable, terminator_p, swallow)                        \
  {                                                                           \
    variable = line;                                                          \
    while (*line != '\0' && !terminator_p (*line))                            \
      ++line;                                                                 \
    if (*line != '\0')                                                        \
      {                                                                       \
        *line = '\0';                                                         \
        do                                                                    \
          ++line;                                                             \
        while (swallow && terminator_p (*line));                              \
      }                                                                       \
  }

#define INT_FIELD(variable, terminator_p, swallow, base, convert)            \
  {                                                                           \
    char *endp;                                                               \
    variable = convert (strtoul (line, &endp, base));                         \
    if (endp == line)                                                         \
      return 0;                                                               \
    else if (terminator_p (*endp))                                            \
      do                                                                      \
        ++endp;                                                               \
      while (swallow && terminator_p (*endp));                                \
    else if (*endp != '\0')                                                   \
      return 0;                                                               \
    line = endp;                                                              \
  }

#define INT_FIELD_MAYBE_NULL(variable, terminator_p, swallow, base, convert, default)        \
  {                                                                           \
    char *endp;                                                               \
    if (*line == '\0')                                                        \
      /* We expect some more input, so don't allow the string to end here. */ \
      return 0;                                                               \
    variable = convert (strtoul (line, &endp, base));                         \
    if (endp == line)                                                         \
      variable = default;                                                     \
    if (terminator_p (*endp))                                                 \
      do                                                                      \
        ++endp;                                                               \
      while (swallow && terminator_p (*endp));                                \
    else if (*endp != '\0')                                                   \
      return 0;                                                               \
    line = endp;                                                              \
  }

static char **
parse_list (char *line, char *data, size_t datalen, int *errnop)
{
  void *eol;
  char **list, **p;

  if (line >= data && line < data + datalen)
    /* Find the end of the line buffer, we will use the space in DATA after
       it for storing the vector of pointers.  */
    eol = strchr (line, '\0') + 1;
  else
    /* LINE does not point within DATA->linebuffer, so that space is
       not being used for scratch space right now.  We can use all of
       it for the pointer vector storage.  */
    eol = data;
  /* Adjust the pointer so it is aligned for storing pointers.  */
  eol = (char *) eol + __alignof__ (char *) - 1;
  eol = (char *) eol - ((char *) eol - (char *) 0) % __alignof__ (char *);
  /* We will start the storage here for the vector of pointers.  */
  list = (char **) eol;

  p = list;
  while (1)
    {
      char *elt;

      if ((size_t) ((char *) &p[1] - (char *) data) > datalen)
        {
          /* We cannot fit another pointer in the buffer.  */
          *errnop = ERANGE;
          return NULL;
        }
      if (*line == '\0')
        break;

      /* Skip leading white space.  This might not be portable but useful.  */
      while (isspace (*line))
        ++line;

      elt = line;
      while (1)
        {
          if (*line == '\0' || *line == ',')
            {
              /* End of the next entry.  */
              if (line > elt)
                /* We really found some data.  */
                *p++ = elt;

              /* Terminate string if necessary.  */
              if (*line != '\0')
                *line++ = '\0';
              break;
            }
          ++line;
        }
    }
  *p = NULL;

  return list;
}


static int
parse_grent (char *line, struct group *result,
	     char *buffer, size_t buflen, int *errnop)
{
  char *p = strpbrk (line, "\n");
  if (p != NULL)
    *p = '\0';
  STRING_FIELD (result->gr_name, ISCOLON, 0);
  if (line[0] == '\0'
      && (result->gr_name[0] == '+' || result->gr_name[0] == '-'))
    {
      result->gr_passwd = NULL;
      result->gr_gid = 0;
    }
  else
    {
      STRING_FIELD (result->gr_passwd, ISCOLON, 0);
      if (result->gr_name[0] == '+' || result->gr_name[0] == '-')
	INT_FIELD_MAYBE_NULL (result->gr_gid, ISCOLON, 0, 10, , 0)
      else
	INT_FIELD (result->gr_gid, ISCOLON, 0, 10,)
    }
  {
    char **list = parse_list (line, buffer, buflen, errnop);
    if (list)
      result->gr_mem = list;
    else
      return -1;          /* -1 indicates we ran out of space.  */
  }

  return 1;
}

/* XXX merge with read-files.c  */
static enum nss_status
internal_setent (FILE **stream, const char *file)
{
  enum nss_status status = NSS_STATUS_SUCCESS;

  if (*stream == NULL)
    {
      char *filename = alloca (strlen (files_etc_dir) + 8);
      strcpy (filename, files_etc_dir);
      strcat (filename, file);

      *stream = fopen (filename, "r");

      if (*stream == NULL)
	status = errno == EAGAIN ? NSS_STATUS_TRYAGAIN : NSS_STATUS_UNAVAIL;
      else
	{
	  int result, flags;

	  result = flags = fcntl (fileno (*stream), F_GETFD, 0);
	  if (result >= 0)
	    {
	      flags |= 1;
	      result = fcntl (fileno (*stream), F_SETFD, flags);
	    }

	  if (result < 0)
	    {
	      fclose (*stream);
	      stream = NULL;
	      status = NSS_STATUS_UNAVAIL;
	    }
	}
    }
  else
    rewind (*stream);

  return status;
}

/* XXX merge with read-files.c  */
static void
internal_endent (FILE **stream)
{
  if (*stream != NULL)
    {
      fclose (*stream);
      stream = NULL;
    }
}

static enum nss_status
internal_getgrent (FILE *stream, struct group *result,
		   char *buffer, size_t buflen, int *errnop)
{
  char *p;
  char *data = (void *) buffer;
  int linebuflen = buffer + buflen - data;
  int parse_result;

  if (buflen < sizeof *data + 2)
    {
      *errnop = ERANGE;
      return NSS_STATUS_TRYAGAIN;
    }

  do {
    ((unsigned char *) data)[linebuflen - 1] = '\xff';

    p = fgets (data, linebuflen, stream);
    if (p == NULL)
      {
	*errnop = ENOENT;
	return NSS_STATUS_NOTFOUND;
      }
    else if (((unsigned char *) data)[linebuflen - 1] != 0xff)
      {
	*errnop = ERANGE;
	return NSS_STATUS_TRYAGAIN;
      }

    /* Skip leading blanks.  */
    while (isspace (*p))
      ++p;
  }
  while (*p == '\0' || *p == '#'
	 || !(parse_result = parse_grent (p, result, buffer,
					  buflen, errnop)));


  return parse_result == -1 ? NSS_STATUS_TRYAGAIN : NSS_STATUS_SUCCESS;
}

enum nss_status
files_getgrnam_r (const char *name, struct group *result,
		  char *buffer, size_t buflen, int *errnop)
{
  /* Locks the static variables in this file.  */
  __libc_lock_define_initialized (static, lock)
  enum nss_status status;
  FILE *stream = NULL;

  __libc_lock_lock (lock);

  status = internal_setent (&stream, "/group");
  if (status == NSS_STATUS_SUCCESS)
    {
      last_use = getby;
      while ((status = internal_getgrent (stream, result, buffer, buflen,
					  errnop)) == NSS_STATUS_SUCCESS)
	{
	  if (name[0] != '+' && name[0] != '-'
	      && ! strcmp (name, result->gr_name))
	    break;
	}
      internal_endent (&stream);
    }

  __libc_lock_unlock (lock);

  return status;
}

enum nss_status
files_getgrgid_r (gid_t gid, struct group *result, char *buffer,
		  size_t buflen, int *errnop)
{
  /* Locks the static variables in this file.  */
  __libc_lock_define_initialized (static, lock)
  enum nss_status status;
  FILE *stream = NULL;

  __libc_lock_lock (lock);

  status = internal_setent (&stream, "/group");
  if (status == NSS_STATUS_SUCCESS)
    {
      last_use = getby;
      while ((status = internal_getgrent (stream, result, buffer, buflen,
					  errnop)) == NSS_STATUS_SUCCESS)
	{
	  if (gid == result->gr_gid && result->gr_name[0] != '+' &&
	      result->gr_name[0] != '-')
	    break;
	}
      internal_endent (&stream);
    }

  __libc_lock_unlock (lock);

  return status;
}

__libc_lock_define_initialized (static, lock);
static FILE *stream;
static fpos_t position;

/* Return the next entry from the database file, doing locking.  */
enum nss_status
files_getgrent_r (struct group *result, char *buffer,
		  size_t buflen, int *errnop)
{
  /* Return next entry in host file.  */
  enum nss_status status = NSS_STATUS_SUCCESS;

  __libc_lock_lock (lock);

  /* Be prepared that the set*ent function was not called before.  */
  if (stream == NULL)
    {
      status = internal_setent (&stream, "/group");

      if (status == NSS_STATUS_SUCCESS && fgetpos (stream, &position) < 0)
        {
          fclose (stream);
          stream = NULL;
          status = NSS_STATUS_UNAVAIL;
        }
    }

  if (status == NSS_STATUS_SUCCESS)
    {
      /* If the last use was not by the getent function we need the
         position the stream.  */
      if (last_use != getent)
        {
          if (fsetpos (stream, &position) < 0)
            status = NSS_STATUS_UNAVAIL;
          else
            last_use = getent;
        }

      if (status == NSS_STATUS_SUCCESS)
        {
          status = internal_getgrent (stream, result, buffer, buflen, errnop);

          /* Remember this position if we were successful.  If the
             operation failed we give the user a chance to repeat the
             operation (perhaps the buffer was too small).  */
          if (status == NSS_STATUS_SUCCESS)
            fgetpos (stream, &position);
          else
            /* We must make sure we reposition the stream the next call.  */
            last_use = none;
        }
    }

  __libc_lock_unlock (lock);

  return status;
}
