static char qacct_c_rcsid[]="qacct.c,v 1.3 1994/06/25 16:25:18 kerce Exp";

/*-----------------------------------------------------------------------------
 * Kingsley Kerce
 *
 * Copyright (c) 1994
 *
 * Supercomputer Computations Research Institute (SCRI)
 * Florida State University
 *
 * SCRI representatives make no claims about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 *
 * qacct.c,v
 * Revision 1.3  1994/06/25  16:25:18  kerce
 * changed version numbers from 3.05 to 3.1.0
 *
 * Revision 1.2  1994/06/22  15:48:14  kerce
 * For "CPU Usage Options" window, allow insertion/removal of defaults
 *   with the mere click of a button
 *
 * Better sanity checking for qusage/qacct's options
 *
 * Revision 1.1.1.1  1994/06/18  19:43:28  kerce
 * DQS X Distribution
 *
 *---------------------------------------------------------------------------*/

#define MAINPROGRAM
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include "error.h"
#include "strdup.h"
#include "stdstuff.h"
#include "cpyright.h"
#include "acctfile.h"
#include "statfile.h"
#include "request.h"
#include "util.h"

char *ProgramName = NULL;
void (*FatalErrorHook)();
char *Title = "DQS Accounting";

#ifndef __DATE__
char *Version = " v3.1.0 beta";
#else
char *Version = " v3.1.0 beta (compiled " __DATE__ " " __TIME__ ")";
#endif

char *EnvAcctFileName = NULL;

static char *AboutTextString = COPYRIGHT;

REQUEST *
RequestInit ()
{
  REQUEST *Req;

  DENTER ((DQS_EVENT, "RequestInit"));

  Req = (REQUEST *) calloc (1, sizeof (REQUEST));
  if (Req == NULL)
    {
      Error (__FILE__, __LINE__, 0, "FATAL ERROR: out of memory");
      DEXIT;
      exit (EXIT_FAILURE);
    }

  Req->QName = "";
  Req->HostName = "";
  Req->QComplexStr = "";
  Req->Group = "";
  Req->Owner = "";
  Req->JobName = "";
  Req->QComplex = NULL;

  DEXIT;
  return ((REQUEST *) Req);
}

static void
ShowUsage ()
{
  DENTER ((DQS_EVENT, "ShowUsage"));

  fprintf (stderr, "Usage: %s ", ProgramName);
  fprintf (stderr, "[-f AcctFileName] ");
  fprintf (stderr, "[-s StartDaysAgo] ");
  fprintf (stderr, "[-d Days] ");
  fprintf (stderr, "[-c QComplex] ");
  fprintf (stderr, "\n");
  fprintf (stderr, "             ");
  fprintf (stderr, "[-q QName] ");
  fprintf (stderr, "[-h HostName] ");
  fprintf (stderr, "[-g Group] ");
  fprintf (stderr, "[-o Owner] ");
  fprintf (stderr, "[-j JobName] ");
  fprintf (stderr, "\n");
  fprintf (stderr, "             ");
  fprintf (stderr, "[-B BinSize] ");
  fprintf (stderr, "[-r] ");
  fprintf (stderr, "[-2] ");
  fprintf (stderr, "[-p] ");
  fprintf (stderr, "[-b BeginTime] ");
  fprintf (stderr, "[-e EndTime] ");

  fprintf (stderr, "\n");
  
  fprintf (stderr, "   or: %s -help\n", ProgramName);
  fprintf (stderr, "   or: %s -version\n", ProgramName);

  DEXIT;
}

static void
ShowHelp ()
{
  char *AcctFileName;

  DENTER ((DQS_EVENT, "ShowHelp"));

  AcctFileName= BuildFileName (ACCTF);
  if (AcctFileName == (char *) NULL)
    {
      Error (__FILE__, __LINE__, 0, "FATAL ERROR: out of memory");
      DEXIT;
      exit (EXIT_FAILURE);
    }

  ShowUsage ();
  fprintf (stderr, "\n");
  fprintf (stderr, "where: AcctFileName is the name of an accounting file\n");
  fprintf (stderr, "         (default: `%s')\n", AcctFileName);
  fprintf (stderr, "       StartDaysAgo is the number of days ago to start\n");
  fprintf (stderr, "         (default: %d)\n", DEFAULTSTARTDAYSAGO);
  fprintf (stderr, "       Days is the number of days for which to account\n");
  fprintf (stderr, "         (default: %d)\n", DEFAULTDAYS);
  fprintf (stderr, "       QComplex is a queue complex specification\n");
  fprintf (stderr, "       QName is the name of a queue\n");
  fprintf (stderr, "       HostName is the name of a host\n");
  fprintf (stderr, "       Group is the name of an accounting group\n");
  fprintf (stderr, "       Owner is the name of an owner\n");
  fprintf (stderr, "       JobName is the name of a job\n");
  fprintf (stderr, "       BinSize is the size of each bin in seconds\n");
  fprintf (stderr, "         (default: %d)\n", DEFAULTBINSIZE);
  fprintf (stderr, "       BeginTime is the interval start time\n");
  fprintf (stderr, "         (overrides StartDaysAgo)\n");
  fprintf (stderr, "       EndTime is the interval stop time\n");
  fprintf (stderr, "         (overrides Days)\n");
  fprintf (stderr, "\n");
  fprintf (stderr, "       p (pretty) means format output for human consumption\n");
  fprintf (stderr, "       r (running) means use information gathered from");
  fprintf (stderr, " periodic polling of\n");
  fprintf (stderr, "         queue states\n");
  fprintf (stderr, "       2 (2.1) means poll file is in DQS v2.1 format\n");
  fprintf (stderr, "         (only valid in combination with `r' switch)\n");
  fprintf (stderr, "\n");
  fprintf (stderr, "       (Note: QComplex, QName, HostName, Group, Owner,");
  fprintf (stderr, " and JobName may\n");
  fprintf (stderr, "        contain wildcarded expressions i.e. shell-style");
  fprintf (stderr, " pattern matching for\n");
  fprintf (stderr, "        ?, \\, [], and * characters)\n");

  DEXIT;
}

static void
ShowVersion ()
{
  DENTER ((DQS_EVENT, "ShowVersion"));

  fprintf (stderr, BuildAboutTextString (Title, Version, AboutTextString));

  DEXIT;
}

static void
ProcessCmdLine (argc, argv, AcctFileName, Req, StartDaysAgo, Days, BeginTime,
		GotBeginTime, EndTime, GotEndTime, BinSize, Running,
		PrettyPrint, Ver2_1)
     int *argc;
     char **argv;
     char **AcctFileName;
     REQUEST *Req;
     unsigned int *StartDaysAgo;
     unsigned int *Days;
     unsigned long *BeginTime;
     BOOLEAN *GotBeginTime;
     unsigned long *EndTime;
     BOOLEAN *GotEndTime;
     unsigned long *BinSize;
     BOOLEAN *Running;
     BOOLEAN *PrettyPrint;
     BOOLEAN *Ver2_1;
{
  char c;
  BOOLEAN NextOption;
  int sscanfResult;

  DENTER ((DQS_EVENT, "ProcessCmdLine"));

  if (DEBUG_ON == 1)
    DisplayArgs (*argc, argv);
  
  while ((--*argc > 0) && ((*++argv)[0] == '-'))
    {
      NextOption = FALSE;
      while ((c = *++argv[0]) && (NextOption == FALSE))
	switch (c)
	  {
	  case 'f':
	    --*argc;
	    if (*argc > 0)
	      {
		argv++;
		*AcctFileName = argv[0];
		NextOption = TRUE;
	      }
	    else
	      {
		Error (NULL, 0, 0,
		       "-f option must be followed by a file name");
		ShowUsage ();
		DEXIT;
		exit (EXIT_FAILURE);
	      }

	    break;

	  case 's':
	    --*argc;
	    if (*argc > 0)
	      {
		argv++;
		sscanfResult = sscanf (argv[0], "%u", StartDaysAgo);
		if (sscanfResult != 1)
		  {
		    Error (NULL, 0, errno,
			   "FATAL ERROR: can't parse `%s' for StartDaysAgo",
			   argv[0]);
		    Error (NULL, 0, 0,
			   "FATAL ERROR: StartDaysAgo should be a positive integer");
		    DEXIT;
		    exit (EXIT_FAILURE);
		  }
		NextOption = TRUE;
	      }
	    else
	      {
		Error (NULL, 0, 0,
		       "-s option must be followed by a number of days");
		ShowUsage ();
		DEXIT;
		exit (EXIT_FAILURE);
	      }

	    break;
	    
	  case 'd':
	    --*argc;
	    if (*argc > 0)
	      {
		argv++;
		sscanfResult = sscanf (argv[0], "%u", Days);
		if (sscanfResult != 1)
		  {
		    Error (NULL, 0, errno,
			   "FATAL ERROR: can't parse `%s' for Days",
			   argv[0]);
		    Error (NULL, 0, 0,
			   "FATAL ERROR: Days should be a positive integer");
		    DEXIT;
		    exit (EXIT_FAILURE);
		  }
		NextOption = TRUE;
	      }
	    else
	      {
		Error (NULL, 0, 0,
		       "-d option must be followed by a number of days");
		ShowUsage ();
		DEXIT;
		exit (EXIT_FAILURE);
	      }

	    break;
	    
	  case 'e':
	    --*argc;
	    if (*argc > 0)
	      {
		argv++;
		sscanfResult = sscanf (argv[0], "%lu", EndTime);
		if (sscanfResult != 1)
		  {
		    Error (NULL, 0, errno,
			   "FATAL ERROR: can't parse `%s' for EndTime",
			   argv[0]);
		    Error (NULL, 0, 0,
			   "FATAL ERROR: EndTime should be a positive integer");
		    DEXIT;
		    exit (EXIT_FAILURE);
		  }

		*GotEndTime = TRUE;
		NextOption = TRUE;
	      }
	    else
	      {
		Error (NULL, 0, 0,
		       "-e option must be followed by an end time");
		ShowUsage ();
		DEXIT;
		exit (EXIT_FAILURE);
	      }

	    break;
	    
	  case 'c':
	    --*argc;
	    if (*argc > 0)
	      {
		argv++;
		Req->QComplexStr = argv[0];
		NextOption = TRUE;
	      }
	    else
	      {
		Error (NULL, 0, 0,
		       "-c option must be followed by a queue complex");
		ShowUsage ();
		DEXIT;
		exit (EXIT_FAILURE);
	      }

	    break;

	  case 'q':
	    --*argc;
	    if (*argc > 0)
	      {
		argv++;
		Req->QName = argv[0];
		NextOption = TRUE;
	      }
	    else
	      {
		Error (NULL, 0, 0,
		       "-q option must be followed by a queue name");
		ShowUsage ();
		DEXIT;
		exit (EXIT_FAILURE);
	      }

	    break;

	  case 'r':
	    *Running = TRUE;

	    break;

	  case 'p':
	    *PrettyPrint = TRUE;

	    break;

	  case '2':
	    *Ver2_1 = TRUE;

	    break;

	  case 'h':
	    /* See if 2nd char is 'e'. */
	    if ((strlen (argv[0]) >= 2) && (argv[0][1] == 'e'))
	      {
		ShowHelp ();
		DEXIT;
		exit (EXIT_SUCCESS);
		break;
	      }
	    
	    --*argc;
	    if (*argc > 0)
	      {
		argv++;
		Req->HostName = argv[0];
		NextOption = TRUE;
	      }
	    else
	      {
		Error (NULL, 0, 0,
		       "-h option must be followed by a host name");
		ShowUsage ();
		DEXIT;
		exit (EXIT_FAILURE);
	      }

	    break;

	  case 'g':
	    --*argc;
	    if (*argc > 0)
	      {
		argv++;
		Req->Group = argv[0];
		NextOption = TRUE;
	      }
	    else
	      {
		Error (NULL, 0, 0,
		       "-g option must be followed by a group name");
		ShowUsage ();
		DEXIT;
		exit (EXIT_FAILURE);
	      }

	    break;

	  case 'o':
	    --*argc;
	    if (*argc > 0)
	      {
		argv++;
		Req->Owner = argv[0];
		NextOption = TRUE;
	      }
	    else
	      {
		Error (NULL, 0, 0,
		       "-o option must be followed by an owner name");
		ShowUsage ();
		DEXIT;
		exit (EXIT_FAILURE);
	      }

	    break;

	  case 'j':
	    --*argc;
	    if (*argc > 0)
	      {
		argv++;
		Req->JobName = argv[0];
		NextOption = TRUE;
	      }
	    else
	      {
		Error (NULL, 0, 0,
		       "-j option must be followed by a job name");
		ShowUsage ();
		DEXIT;
		exit (EXIT_FAILURE);
	      }

	    break;

	  case 'b':
	    --*argc;
	    if (*argc > 0)
	      {
		argv++;
		sscanfResult = sscanf (argv[0], "%lu", BeginTime);
		if (sscanfResult != 1)
		  {
		    Error (NULL, 0, errno,
			   "FATAL ERROR: can't parse `%s' for BeginTime",
			   argv[0]);
		    Error (NULL, 0, 0,
			   "FATAL ERROR: BeginTime should be a positive integer");
		    DEXIT;
		    exit (EXIT_FAILURE);
		  }
		
		*GotBeginTime = TRUE;
		NextOption = TRUE;
	      }
	    else
	      {
		Error (NULL, 0, 0,
		       "-b option must be followed by a begin time");
		ShowUsage ();
		DEXIT;
		exit (EXIT_FAILURE);
	      }
	    
	    break;
	    
	  case 'B':
	    --*argc;
	    if (*argc > 0)
	      {
		argv++;
		sscanfResult = sscanf (argv[0], "%u", BinSize);
		if (sscanfResult != 1)
		  {
		    Error (NULL, 0, errno,
			   "FATAL ERROR: can't parse `%s' for BinSize",
			   argv[0]);
		    Error (NULL, 0, 0,
			   "FATAL ERROR: BinSize should be a positive integer");
		    DEXIT;
		    exit (EXIT_FAILURE);
		  }
		NextOption = TRUE;
	      }
	    else
	      {
		Error (NULL, 0, 0,
		       "-B option must be followed by a bin size");
		ShowUsage ();
		DEXIT;
		exit (EXIT_FAILURE);
	      }
	    
	    break;

	  case '?':
	    ShowHelp ();
	    DEXIT;
	    exit (EXIT_SUCCESS);
	    break;
	    
	  case 'v':
	    ShowVersion ();
	    DEXIT;
	    exit (EXIT_SUCCESS);
	    break;
	    
	  default:
	    Error (NULL, 0, 0, "unknown option `%c'", c);
	    break;
	  }
    }

  if (DEBUG_ON == 1)
    DisplayArgs (*argc, argv);

  if (*argc > 0)
    {
      Error (NULL, 0, 0,
	     "WARNING: the following command line arguments were unused:");

      while (*argc > 0)
	{
	  fprintf (stderr, "%s ", argv[0]);
	  *argc -= 1;
	  argv++;
	}

      fputc ('\n', stderr);
    }

  fflush (stderr);

  DEXIT;
}

static void
AccountingAcctFile (AcctFileName, IStart, IStop, BinSize, Req, PrettyPrint)
     char *AcctFileName;
     unsigned long IStart;
     unsigned long IStop;
     unsigned int BinSize;
     REQUEST *Req;
     BOOLEAN PrettyPrint;
{
  ACCTFILE *AcctFile;
  int AcctFileCreateResult;
  
  DENTER ((DQS_EVENT, "AccountingAcctFile"));

  EnvAcctFileName = getenv (ACCTFILEENVVARNAME);
  if (EnvAcctFileName != NULL)
    {
      if (AcctFileName != NULL)
	Error (NULL, 0, 0,
	       "WARNING: overriding environment variable %s=%s",
	       ACCTFILEENVVARNAME, EnvAcctFileName);
      else
	AcctFileName = EnvAcctFileName;
    }
  
  if (AcctFileName == (char *) NULL)
    {
      AcctFileName = BuildFileName (ACCTF);
      if (AcctFileName == (char *) NULL)
	{
	  Error (__FILE__, __LINE__, 0, "FATAL ERROR: out of memory");
	  DEXIT;
	  exit (EXIT_FAILURE);
	}
    }
  
  AcctFile = AcctFileCreate (AcctFileName, IStart, IStop, BinSize,
			     &AcctFileCreateResult);
  
  if (AcctFileCreateResult != ACCTFILE_SUCCESS)
    {
      switch (AcctFileCreateResult)
	{
	case ACCTFILE_OUTOFMEM:
	  Error (__FILE__, __LINE__, 0, "FATAL ERROR: out of memory");
	  Error (NULL, 0, 0,
		 "FATAL ERROR: This is probably due to a unreasonable");
	  Error (NULL, 0, 0,
		 "FATAL ERROR: combination of StartDaysAgo (or BeginTime), Days (or EndTime), and BinSize");
	  break;
	case ACCTFILE_NOFOPEN:
	  Error (NULL, 0, errno, "FATAL ERROR: can't read file named `%s'",
		 AcctFileName);
	  break;
	case ACCTFILE_NOSTAT:
	  Error (NULL, 0, errno, "FATAL ERROR: can't access file named `%s'",
		 AcctFileName);
	  break;
	case ACCTFILE_SMALLBINSIZE:
	  Error (NULL, 0, 0, "BinSize must be > 0");
	  break;
	case ACCTFILE_LARGEBINSIZE:
	  Error (NULL, 0, 0, "BinSize must be < (EndTime - BeginTime)");
	  break;
	case ACCTFILE_SMALLISTOP:
	  Error (NULL, 0, 0, "EndTime must be >= BeginTime");
	  break;
	case ACCTFILE_SMALLISTART:
	  Error (NULL, 0, 0, "BeginTime must be >= 0");
	  break;
	default:
	  Error (__FILE__, __LINE__, 0, "FATAL INTERNAL ERROR");
	  break;
	}
      
      DEXIT;
      exit (EXIT_FAILURE);
    }

  if ((Req->QComplexStr != (char *) NULL)
      && (strlen (Req->QComplexStr) != 0))
    dqs_parse_resource_list (&Req->QComplex, Req->QComplexStr);
  else
    Req->QComplex = (dqs_list_type *) NULL;

  {
    int i;

    do
      {
	i = AcctFileRead (AcctFile, Req, 0);
	if (i == 0)
	  printf ("%u\t", AcctFile->PercentageRead);
	else if (i < 0)
	  {
	    Error (NULL, 0, 0,
		   "FATAL ERROR: can't parse contents of file named `%s'",
		   AcctFile->AcctFileName);
	    DEXIT;
	    exit (EXIT_FAILURE);
	  }
      }
    while (i != 1);
  }

  if (AcctFile->Matches == 0)
    {
      Error (NULL, 0, 0, "No matches found.");
      DEXIT;
      exit (EXIT_FAILURE);
    }

  {
    unsigned long i;
    
    if (PrettyPrint == TRUE)
      {
        printf ("------------+---------------+--------------\n");
        printf ("|  Queue    |          CPU Usage          |\n");
        printf ("|  Usage    |     User      |    System   |\n");
        printf ("------------+---------------+--------------\n");
      }

    for (i = 0; i < AcctFile->CPU.NumPoints; i++)
      printf ("%lf\t%lf\t%lf\n",
	      AcctFile->Wallclock.DataPoints[i],
	      AcctFile->CPU.UserDataPoints[i],
	      AcctFile->CPU.SysDataPoints[i]);
  }
  
  free (AcctFile->AcctFileName);
  AcctFileDestroy (AcctFile);

  DEXIT;
}

static void
AccountingStatFile (AcctFileName, IStart, IStop, BinSize, Req, PrettyPrint,
		    Ver2_1)
     char *AcctFileName;
     unsigned long IStart;
     unsigned long IStop;
     unsigned int BinSize;
     REQUEST *Req;
     BOOLEAN PrettyPrint;
     BOOLEAN Ver2_1;
{
  STATFILE *StatFile;
  int StatFileResult;
  char qcomplex_str[255];
  
  DENTER ((DQS_EVENT, "AccountingStatFile"));

  EnvAcctFileName = getenv (STATFILEENVVARNAME);
  if (EnvAcctFileName != NULL)
    {
      if (AcctFileName != NULL)
	Error (NULL, 0, 0,
	       "WARNING: overriding environment variable %s=%s",
	       STATFILEENVVARNAME, EnvAcctFileName);
      else
	AcctFileName = EnvAcctFileName;
    }
  
  if (AcctFileName == (char *) NULL)
    {
      AcctFileName = BuildFileName (STATF);
      if (AcctFileName == (char *) NULL)
	{
	  Error (__FILE__, __LINE__, 0, "FATAL ERROR: out of memory");
	  DEXIT;
	  exit (EXIT_FAILURE);
	}
    }
  
  StatFile = StatFileCreate (AcctFileName, &StatFileResult);
  
  if (StatFileResult != STATFILE_SUCCESS)
    {
      Error (NULL, 0, 0, "cannot open stat file named `%s'", AcctFileName);
      DEXIT;
      exit (EXIT_FAILURE);
    }
  
  StatFileResult = StatFileSearch (StatFile, IStart, IStop, Ver2_1);
  if (StatFileResult == STATFSRCH_SUCCESS)
    StatFileResult = StatFileCalc (StatFile, IStop, Req, Ver2_1);

  switch (StatFileResult)
    {
    case STATFSRCH_NOT_FOUND:
      Error (NULL, 0, 0, "stat file named `%s' has no stats in the interval",
	     StatFile->StatFileName);
      break;

    case STATFSRCH_STAT_FAIL:
      Error (NULL, 0, 0, "cannot access stat file named `%s'",
	     StatFile->StatFileName);
      break;

    case STATFSRCH_FSEEK_FAIL:
      Error (NULL, 0, 0, "fseek() failed on stat file named `%s'",
	     StatFile->StatFileName);
      break;

    case STATFSRCH_EMPTY:
      Error (NULL, 0, 0, "stat file named `%s' is empty",
	     StatFile->StatFileName);
      break;

    case STATFSRCH_NULLFILE:
      Error (__FILE__, __LINE__, 0, "INTERNAL ERROR");
      break;

    case STATFSRCH_UNEXP_EOF:
    case STATFSRCH_FSCAN_FAIL:
      Error (NULL, 0, 0, "stat file named `%s' cannot be parsed",
	     StatFile->StatFileName);
      break;

    case STATFSRCH_ARITH_OVERFLOW:
    case STATFSRCH_SUCCESS:
      break;

    default:
      Error (__FILE__, __LINE__, 0, "FATAL INTERNAL ERROR");
      DEXIT;
      exit (EXIT_FAILURE);
      break;
    }

  StatFileDestroy (StatFile);

  DEXIT;
}

int
main (argc, argv)
     int argc;
     char **argv;
{
  unsigned long IStart, IStop, BeginTime, EndTime;
  unsigned int StartDaysAgo, Days, BinSize;
  char *AcctFileName;
  int sscanfResult;
  REQUEST *Req;
  BOOLEAN Running, PrettyPrint, GotBeginTime, GotEndTime, Ver2_1;

  DENTER_MAIN ((DQS_EVENT, "qacct"));
  dqs_setup (QSUB, argv[0]);

  FatalErrorHook = NULL;
  ProgramName = argv[0];

  if (DEBUG_ON == 1)
    DisplayArgs (argc, argv);

  Req = RequestInit ();

  /* Defaults. */
  StartDaysAgo = DEFAULTSTARTDAYSAGO;
  Days = DEFAULTDAYS;
  BinSize = DEFAULTBINSIZE;
  Running = FALSE;
  PrettyPrint = FALSE;
  Ver2_1 = FALSE;
  
  GotBeginTime = FALSE;
  GotEndTime = FALSE;

  AcctFileName = NULL;
  ProcessCmdLine (&argc, argv, &AcctFileName, Req, &StartDaysAgo, &Days,
		  &BeginTime, &GotBeginTime, &EndTime, &GotEndTime, &BinSize,
		  &Running, &PrettyPrint, &Ver2_1);

  if ((Ver2_1 == TRUE) && (Running == FALSE))
    {
      Error (NULL, 0, 0,
	     "`running' option must be chosen to use `2.1' option");
      DEXIT;
      exit (EXIT_FAILURE);
    }

  if (Days > StartDaysAgo)
    {
      Error (NULL, 0, 0,
	     "Days (%u) must be less than or equal to StartDaysAgo (%u).",
	     Days, StartDaysAgo);
      DEXIT;
      exit (EXIT_FAILURE);
    }

  IStartStopCalc (StartDaysAgo, Days, &IStart, &IStop, TimeOfDayInSecs ());

  if (GotBeginTime == TRUE)
    IStart = BeginTime;

  if (GotEndTime == TRUE)
    IStop = EndTime;

  if ((strcmp (Req->QName, "") == 0)
      && (strcmp (Req->HostName, "") == 0)
      && (strcmp (Req->QComplexStr, "") == 0)
      && (strcmp (Req->Group, "") == 0)
      && (strcmp (Req->Owner, "") == 0)
      && (strcmp (Req->JobName, "") == 0))
    {
      Error (NULL, 0, 0,
	     "at least one of QName, HostName, QComplex, Group, Owner, or JobName must be given; otherwise, no jobs will be matched.");

      DEXIT;
      exit (EXIT_FAILURE);
    }

  if (Running == FALSE)
    AccountingAcctFile (AcctFileName, IStart, IStop, BinSize, Req,
			PrettyPrint);
  else
    AccountingStatFile (AcctFileName, IStart, IStop, BinSize, Req,
			PrettyPrint, Ver2_1);

  DEXIT;
  exit (EXIT_SUCCESS);
}


/* Set Emacs C-mode style.
   Local Variables:
   c-style: GNU
   End:
 */
