/*
    AGInit.cpp
    OpenAG, libOpenAG, OpenAG X
 
    Created by Eric Seidel on Tue Nov 20 2001.
    
    Copyright (c) 2001-2002 Eric Seidel. All rights reserved.
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.h"

#include <netinet/in.h>	/* for sockaddr_in */
#include <arpa/inet.h>  /* for ntohs & ntohl */
#include <fcntl.h>
#include <glob.h>	/* for glob() */
#include <errno.h>	/* for errno */
#include <netdb.h>	/* for gethostbyname() */

#include "AGInit.h"
#include "AGMessage.h"
#include "ParseMessage.h"
#include "error.h"

char** 	RedirectionServers = NULL;
int 	RedirectionServersLength = 0;


void addAllSharesInDirectory(//server_status* STATSptr, 
    const char* DirPath, FileList* theList, int recurseDepth);

/* INITIALIZATION FUNCTIONS */


void clientInit(server_status &STATS)
{
    
    if (STATS.setDisplayedServerStatus != NULL)
            STATS.setDisplayedServerStatus("Initializing client...");
    
    /* Clear out old connections */
    /* and hard-close all sockets... */
    deleteAllConnectionRecords(STATS);
    
    /* Initialize to start values */
    
    STATS.current = 0;
    FD_ZERO(&(STATS.rset));
    FD_ZERO(&(STATS.wset));
    STATS.nConnections = 0; // initialize
    STATS.maxfd = 0;
    STATS.sendKeepAlive = false;
    STATS.LastKeepAlive = 0;
    if (STATS.Prefs != NULL)
    {
        if (STATS.Prefs->IsGoldAccount)
            STATS.redirectionDNS = GOLD_REDIRECTION_DNS;
        else
            STATS.redirectionDNS = NORMAL_REDIRECTION_DNS;
    }
    else
    {
        err_print("STATS.Prefs is NULL, that can't be a good thing!\n");
        STATS.redirectionDNS = NULL;  // will be handled later.
    }
    
    bzero(STATS.connections, sizeof(STATS.connections));  // this will work, but only since it's static.
    
    #if AGDEBUGLEVEL > 3
    printf("sizeof(STATS.connections) : %li\n", sizeof(STATS.connections));
    printf("Init\n");
    #endif
    
    /* CONNECT TO AN INFORMATION SERVER */

    if (STATS.setDisplayedServerStatus != NULL)
            STATS.setDisplayedServerStatus("Connecting to server...");
    
    int s = getMeAnInfoServer(STATS.redirectionDNS);
    
    if (STATS.setDisplayedServerStatus != NULL)
            STATS.setDisplayedServerStatus("Connected!");

    /* INITIALIZE INFO SERVER CONNECTION RECORD */
    STATS.connections[INFOSERVER].socket = s;
    STATS.connections[INFOSERVER].flags = INFOSERVER_FLAG;
    #if AGDEBUGLEVEL > 5
    if (STATS.connections[INFOSERVER].Ibuf == NULL) err_exit("STATS.connections[i].Ibuf not initialized\n");
    else printf("got Ibuffer of size: %li\n", sizeof(STATS.connections[INFOSERVER].Ibuf));
    
    if (STATS.connections[INFOSERVER].Obuf == NULL) err_exit("STATS.connections[i].Obuf not initialized\n");
    else printf("got Obuffer of size: %li\n", sizeof(STATS.connections[INFOSERVER].Obuf));
    #endif
    STATS.connections[INFOSERVER].IwPtr = STATS.connections[INFOSERVER].IrPtr =  STATS.connections[INFOSERVER].Ibuf;
    STATS.connections[INFOSERVER].OwPtr = STATS.connections[INFOSERVER].OrPtr =  STATS.connections[INFOSERVER].Obuf;
    STATS.connections[INFOSERVER].hasOpenFile = false; //not needed...
    STATS.connections[INFOSERVER].TimeOut = 120; // timeout after 2 minutes of no activity.
        
    /*INITIALIZE fd_set vars */
    FD_ZERO(&(STATS.rset));
    FD_ZERO(&(STATS.wset));
    FD_SET(s, &(STATS.rset));
    
    /*set STATS.maxfd and nConnections for InfoServer.*/
    STATS.maxfd = s;
    STATS.nConnections = 1;
    
    /* set up timers */
    STATS.old_time = time(NULL);
    
    STATS.select_timer.tv_sec = 22;
    STATS.select_timer.tv_usec = 0;
    
    
    /* Get my own IP address / Handle router issues */
    
    if (STATS.myIP != NULL)
    {
        delete(STATS.myIP);
        STATS.myIP = NULL;
    }
    
    if (STATS.Prefs->SpecifiedIP == NULL)
    {
        sockaddr_in	sAddr; // we know this already, no need to check if AF_INET
        socklen_t	sLen = sizeof(sAddr);
        if (getsockname(s, (struct sockaddr*) &sAddr, &sLen) < 0)
        {
            err_exit("Error getting local connection info\nProbably means the connection dropped suddenly.\nThis is a bug, just restart OpenAG.");
            /* FIX - if the server drops the connection before I call this, this will error.. I think */
        }
        //printf("sockaddr: %s\n", sAddr.sin_addr);
        STATS.myIP = strdup(inet_ntoa(sAddr.sin_addr));
        //#if AGDEBUGLEVEL > 3
        printf("Got local IP Address of: %s\n", STATS.myIP);
        //#endif
    }
    else
    {
        // FIX -- Should I test what they entered as an IP?
        printf("Was given a specified IP of: %s\n", STATS.Prefs->SpecifiedIP);
        STATS.myIP = strdup(STATS.Prefs->SpecifiedIP);
    }

    /*Make and send the Login information. */
    sendLoginMessage(STATS);
}


int getMeAnInfoServer(char* redirectionDNS)
{    
    int maxfd = 0;
    fd_set masterReadSet;
    FD_ZERO(&masterReadSet);
    redirectionConnection* rConnections = NULL;
    /* initialize redirection connection array */
    int nRedirectionConnections = getRedirectionServers(rConnections, redirectionDNS);
    
    /* Initialize Timeout Stuff: */
    timeval timer;
    time_t new_time;
    
    /* No Wait on first time through */
    timer.tv_sec = 0;
    timer.tv_usec = 0;
    
    int x;
    bool notConnected = true;
    
    /* enter loop */
    while(notConnected)
    {
        #if AGDEBUGLEVEL > 4
        printf("top of redirect while\n");
        #endif
        fd_set tempReadSet = masterReadSet;
        
        int n = select(maxfd+1,&tempReadSet, NULL, NULL, &timer);
        /* reset timer to defaults */
        timer.tv_sec = 10;
        timer.tv_usec = 0;
        #if AGDEBUGLEVEL > 4
        printf("select returned: %i\n", n);
        #endif
        
        for(x = 0; x < nRedirectionConnections; x++)
        {
            /* Update timeout stuff */
            new_time = time(NULL);
            /* shoule execute for everyone first time through */
            if ( (new_time - rConnections[x].lastTime) > rConnections[x].TimeOut)
            {
                /* Handle Connection Reset/Timeout */
                #if AGDEBUGLEVEL > 3
                printf("Connection %i, recieved TIMEOUT.\n", x);
                #endif
                struct sockaddr_in PeerAddress;
                bzero(&PeerAddress,sizeof(PeerAddress));
                PeerAddress.sin_addr.s_addr = inet_addr(rConnections[x].RedirectServerIP);
                if (PeerAddress.sin_addr.s_addr < 0)
                {
                    #if AGDEBUGLEVEL > 4
                    err_print("Error while converting Redirection server address on conneciton: %i, trying again.\n", x);
                    #endif
                    resetRConnection(rConnections[x], masterReadSet);
                    continue;
                }
                #if AGDEBUGLEVEL > 4
                printf("conn: %i, converted\n",x);
                #endif
                PeerAddress.sin_family	= AF_INET;
                PeerAddress.sin_port	= htons((u_short)REDIRECTION_SERVER_PORT);
                
                if ( (rConnections[x].socket = socket(AF_INET,SOCK_STREAM,0)) < 0)
                {
                    #if AGDEBUGLEVEL > 4
                    err_print("conn: %i, err getting socket\n",x);
                    #endif
                    resetRConnection(rConnections[x], masterReadSet);
                    continue;
                }
                #if AGDEBUGLEVEL > 3
                printf("conn: %i, got socket: %i\n",x ,rConnections[x].socket);
                #endif
                /* get socket */
                rConnections[x].socketOpen = true;
                
                if (rConnections[x].socket > maxfd)
                    maxfd = rConnections[x].socket;
                    
                FD_SET(rConnections[x].socket, &masterReadSet);
                
                /*SET NONBLOCKING */
                /*
                int f = fcntl(rConnections[x].socket, F_GETFL,0);
                printf("f before : %i, but ored: %i\n", f,  f | O_NONBLOCK);
                fcntl(rConnections[x].socket, F_SETFL, f | O_NONBLOCK);
                */
                
                //printf("ServerIP before connect: %s\n", rConnections[x].RedirectServerIP);
                #if AGDEBUGLEVEL > 4
                printf("PeerAddress before connect: %s\n", inet_ntoa(PeerAddress.sin_addr) );
                #endif
                #if AGDEBUGLEVEL > 2
                printf("attempting to connect to redirection server: %s at port %i\n", 
                        rConnections[x].RedirectServerIP, REDIRECTION_SERVER_PORT);
                #endif
                
                rConnections[x].flags = R_CONNECTING;
                
                if (connect(rConnections[x].socket,
                            (struct sockaddr *)&PeerAddress,
                            sizeof(PeerAddress)) < 0)
                
                {
                    
                    //printf("ServerIP after connect: %s\n", rConnections[x].RedirectServerIP);
                    #if AGDEBUGLEVEL > 4
                    printf("PeerAddress after connect: %s\n", inet_ntoa(PeerAddress.sin_addr) );
                    
                    printf("conn: %i, got connect error (maybe ok...)\n",x);
                    #endif
                    /* GOT CONNECTION ERROR */
                    if (errno == EINPROGRESS)
                    {
                        #if AGDEBUGLEVEL > 4
                        printf("conn: %i, yeap, connection just still in progress\n", x);
                        #endif
                        rConnections[x].flags = R_CONNECTING;
                    }
                    else if (errno == ECONNREFUSED)
                    {
                        resetRConnection(rConnections[x], masterReadSet);
                        continue;
                    }
                    else
                    {
                        err_print("I recieved an unhandled error while attempting a connection to the redirection server: %s, I am reseting connection and trying again.\n", rConnections[x].RedirectServerIP);
                        resetRConnection(rConnections[x], masterReadSet);
                        continue;
                    }
                }
                else
                {
                    /* CONNECTED, HANDLE */
                    //printf("ServerIP after connect: %s\n", rConnections[x].RedirectServerIP);
                    #if AGDEBUGLEVEL > 4
                    printf("PeerAddress after connect: %s\n", inet_ntoa(PeerAddress.sin_addr) );
                    printf("conn: %i, connected, going to handle\n", x);
                    #endif
                    FD_SET(rConnections[x].socket,&tempReadSet); // fake it.  handle below.
                    //hum... what if it errors?
                }
                rConnections[x].lastTime = time(NULL);
            }
            
            /* test if readable and read */
            if( FD_ISSET(rConnections[x].socket,&tempReadSet) )
            {
                int error;
                socklen_t errorsize = sizeof(error);

                #if AGDEBUGLEVEL > 4
                printf("connection: %i readable... checking\n",x);
                #endif
                
                if (getsockopt(rConnections[x].socket, SOL_SOCKET, SO_ERROR, &error, &errorsize) < 0
                    || error != 0)
                {
                    #if AGDEBUGLEVEL > 4
                    printf("hum... got %s on %i\n",strerror(error),x);
                    #endif
                    resetRConnection(rConnections[x], masterReadSet);
                    continue;
                }
                /* connection was successful */
                if (rConnections[x].flags & R_CONNECTING)
                {
                    #if AGDEBUGLEVEL > 4
                    printf("conn: %i, handling connection\n",x);
                    #endif
                    char	buf[100];
                    char *	readHere = buf;
                    int n;
                    /* Handle Redirect Info */
                    if (n = read(rConnections[x].socket,&buf,100) <= 0)
                    {
                        #if AGDEBUGLEVEL > 4
                        err_print("got error on connection: %i \n", x);
                        #endif
                        resetRConnection(rConnections[x], masterReadSet);
                        continue;
                    }
                    else
                    {
                        /* Don't need to check how much it read, it is blocking, so it will error if it didn't read enough */
                        #if AGDEBUGLEVEL > 2
                        printf("Closing connection\n\n");
                        #endif
                        close(rConnections[x].socket);
                        rConnections[x].socketOpen = false;
                        
                        /* FIX -- this probably should use the AGMessage constructor */
                        
                        int StartSignal	= ntohl(*((int*)readHere));			readHere += 4;
                        int MessageSize	= ntohl(*((int*)readHere));			readHere += 4;
                        int MessageID	= ntohl(*((int*)readHere));			readHere += 4;
                        rConnections[x].InfoServerPort	= ntohl(*((int*)readHere));	readHere += 4;
                        short unsigned int IPLength	= ntohs(*((short*)readHere));	readHere += 2;
                        
                        //shouldn't need...
                        bzero(rConnections[x].InfoServerIP, sizeof(rConnections[x].InfoServerIP)); 
                        
                        memcpy(&(rConnections[x].InfoServerIP), readHere, IPLength);
                        
                        #if AGDEBUGLEVEL > 4
                        printf("Start Signal: %i\n", StartSignal);
                        printf("Message Size: %i\n", MessageSize);
                        printf("Message ID: %i\n", MessageID);
                        printf("Port Number: %i\n",rConnections[x].InfoServerPort);
                        printf("IP Length: %i\n",IPLength);
                        printf("Redirected Server Address: %s\n",rConnections[x].InfoServerIP);
                        #endif
                        
                        /* Attempt to Connect to InfoServer */
                                
                        struct sockaddr_in PeerAddress;
                        
                        bzero(&PeerAddress,sizeof(PeerAddress));
                        #if AGDEBUGLEVEL > 4
                        printf("conn: %i, about to convert name\n", x);
                        #endif
                        PeerAddress.sin_addr.s_addr = inet_addr(rConnections[x].InfoServerIP);
                        if (PeerAddress.sin_addr.s_addr < 0)
                        {
                            err_print("unabled to convert InfoServer Address. on connection: %i\n", x);
                            resetRConnection(rConnections[x], masterReadSet);
                            continue;
                        }
                        #if AGDEBUGLEVEL > 4
                        printf("conn: %i, name converted\n",x);
                        #endif
                        PeerAddress.sin_family	= AF_INET;
                        PeerAddress.sin_port	= htons((u_short)rConnections[x].InfoServerPort);
                        
                        if ( (rConnections[x].socket = socket(AF_INET,SOCK_STREAM,0)) < 0)
                        {
                            err_print("error getting socket\n");
                            resetRConnection(rConnections[x], masterReadSet);
                            continue;
                        }
                        #if AGDEBUGLEVEL > 4
                        printf("conn: %i, got socket: %i\n",x, rConnections[x].socket);
                        #endif
                        /* get socket */
                        rConnections[x].socketOpen = true;
                        if (rConnections[x].socket > maxfd) maxfd = rConnections[x].socket;
                        FD_SET(rConnections[x].socket, &masterReadSet);
                        /*SET NONBLOCKING */
                        //int f = fcntl(rConnections[x].socket, F_GETFL,0);
                        //fcntl(rConnections[x].socket, F_SETFL, f | O_NONBLOCK);
                        #if AGDEBUGLEVEL > 2
                        printf("conn: %i, attempting to connect to %s at port %i\n", x,
                                rConnections[x].InfoServerIP, rConnections[x].InfoServerPort);
                        #endif
                        if (connect(rConnections[x].socket,
                                    (struct sockaddr *)&PeerAddress,
                                    sizeof(PeerAddress)) < 0)
                        {
                            printf("I recieved an error while connecting on connection #%i, description below.\n", x);
                            /* GOT CONNECTION ERROR */
                            if (errno == EINPROGRESS)
                                rConnections[x].flags = I_CONNECTING;
                            else if ( (errno == ECONNREFUSED) || (errno == ECONNRESET) )
                            {
                                resetRConnection(rConnections[x], masterReadSet);
                                continue;
                            }
                            else
                            {
                                err_print("unhandled error connecting\n");
                                resetRConnection(rConnections[x], masterReadSet);
                                continue;
                            }
                        }
                        else
                        {
                            printf("Connection to Information Server succesfully established!\n");
                            /* CONNECTED, HANDLE */
                            notConnected = false;
                            break;
                        }
                    }
                }
                else if (rConnections[x].flags & I_CONNECTING)
                {
                    /*We successfully got an InfoServer Connection */
                    notConnected = false;
                    break;
                }
                else
                {
                    printf("hum... flags = %i\n", rConnections[x].flags);
                    //resetRConnection(rConnections[x], masterReadSet);
                }
                rConnections[x].lastTime = time(NULL);
            } // close if readable
        } // close for
    } //close while
    
    /* Set infoServerSocket as the socket selected when we finshed */
    int infoServerSocket = rConnections[x].socket;
    
    /* Close all other connections */
    for (int y = 0; y < nRedirectionConnections; y++)
        if ( (y != x) && (rConnections[y].socketOpen) )
            close(rConnections[y].socket);
    
    /* Clean up */
    delete(rConnections);
    
    
    printf("got an infoserver on socket %i\n", infoServerSocket);
    
    /*
    int f = fcntl(infoServerSocket, F_GETFL,0);
    printf(" f = %i \n", f);
    fcntl(infoServerSocket, F_SETFL, f ^ O_NONBLOCK);
    
    f = fcntl(infoServerSocket, F_GETFL,0);
    printf(" f = %i \n", f);
    if ((f & O_NONBLOCK) == f)
        printf("hum... some how it didn't change\n");
    */
    
    return infoServerSocket;
}

void resetRConnection(redirectionConnection &c, fd_set &rs)
{        
    #if AGDEBUGLEVEL > 3
    printf("RESETTING RCONNECTION\n");
    #endif
    if (c.socketOpen && (c.socket > -1))
    {
        close(c.socket);
        FD_CLR(c.socket, &rs);
    }
    // ignoring - if (c.socket == maxfd) maxfd--;
    
    c.socketOpen = false;
    bzero(c.InfoServerIP, sizeof(c.InfoServerIP));
    c.socket	= -1; //should be ok.
    c.flags = RESETTING; //never checked... just not 0 or any other checked constant.
    c.lastTime = 0;
    c.TimeOut = 0; // should be ok.
    
    /* Should now timeout -- therefore reconnecting.*/
}

int getRedirectionServers(redirectionConnection* &s, char* redirectionDNS)
{    
    /* GET THE CURRENT LIST FROM A DNS LOOKUP */
    struct hostent*     hp;
    char* HardCodedAddresses[REDIRECTION_SERVER_HARDCODED_NUMBER] = REDIRECTION_SERVER_HARDCODED_ADDRESSES;
    
    if (redirectionDNS == NULL)
    {
        err_print("redirection DNS was NULL, using DEFAULT (NON-GOLD) DNS SERVER.  PLEASE REPORT THIS ERROR.\n");
        redirectionDNS = NORMAL_REDIRECTION_DNS;
    }
    
    // FIX - Got SIGPIPE here... what can I do about that?
    if ((hp = gethostbyname(redirectionDNS)) != NULL)
    {
        #if AGDEBUGLEVEL > 4
        printf("DNS lookup successful, found redirect host list\n");
        #endif
        RedirectionServers = hp->h_addr_list;
        if (hp->h_length == 6) err_exit("ugh.  got back ipv6 addresses which I don't handle, sorry\n");
    }
    else
    {	/* do we know the host's */
        /* IF WE FAIL GETTING THIS INFORMATION USE THE HARDCODED LIST */
        RedirectionServers = HardCodedAddresses;
        RedirectionServersLength = REDIRECTION_SERVER_HARDCODED_NUMBER;
        //delete(hp);// is this right?
        err_exit("The DNS name garlic.audiogalaxy.com was not found.\nPlease check to make sure that you are currently connected to the internet.\n\n[For the time being I have removed hardcoded address support.  If you have an internet connection with NO DNS server access, I would first suggest that you get a new connection.  If you actually need hardcoded address support (because of a flaky DNS server, or some other reason), please submit a bugreport/feature request.]\n");
    }
    /*could be IPv6... but we hope not.. we don't handle v6. */
    
    /* FIX -- this doesn't work if it used the hard coded addresses!!!! */
    #if AGDEBUGLEVEL > 3
    printf("got redirection list:\n");
    #endif
    int x;
    for (x = 0; ; x++)
    {
        if (RedirectionServers[x] == NULL)
        {
            #if AGDEBUGLEVEL > 3
            printf("%i servers, total\n",x); 
            #endif
            break;
        }
        #if AGDEBUGLEVEL > 3
        printf("%s\n",inet_ntoa(*(struct in_addr*)(RedirectionServers[x])));
        #endif
    }
    int numServers = x;
    
    /* Allocate */
    if (s != NULL && s != 0)
        delete(s);
    s = new redirectionConnection[numServers];
    /* and clear out */
    bzero(s, numServers * sizeof(redirectionConnection));
    #if AGDEBUGLEVEL > 3
    printf("size of s: %li\n", numServers * sizeof(redirectionConnection));
    #endif
    
    for (x = 0; x < numServers; x++)
    {
        char * temp = inet_ntoa(*(struct in_addr*)(RedirectionServers[x]));
        bzero(&(s[x].RedirectServerIP), sizeof(s[x].RedirectServerIP)); //see if this helps... 
        memcpy(s[x].RedirectServerIP,temp, strlen(temp));
        s[x].socketOpen = false;
        s[x].lastTime = time(NULL) - 100; // should timeout first time then.
        s[x].TimeOut = 10;
        //delete(temp); // i believe that it is static... 
    }
    return numServers;
}

void setBaseDirectory(server_status &STATS, const char* basePath)
{
    if (STATS.BaseDirectory != NULL && STATS.BaseDirectory != 0)
        delete(STATS.BaseDirectory);
    
    #if AGDEBUGLEVEL > 4
    printf("about to handle basepath stuff\n");
    #endif
    if (basePath == NULL)
    {
        STATS.BaseDirectory = new char[PATH_MAX];
        if (getcwd(STATS.BaseDirectory, PATH_MAX) == NULL)
            err_exit("Error accessing current directory.  Sorry, I can't continue.  Please report this error.");
        
        #if AGDEBUGLEVEL > 4
        printf("got BaseDirectory: %s\n", STATS.BaseDirectory);
        #endif
    }
    else
        STATS.BaseDirectory = strdup(basePath);
}


void readPrefs(server_status &STATS)
{
    if (STATS.Prefs != NULL && STATS.Prefs != 0)
    {	
        delete(STATS.Prefs);
        #if AGDEBUGLEVEL > 4
        printf("Deleted old prefs\n");
   	#endif
    }
    STATS.Prefs = new AGPrefs(); 	//allocate new one.
    
    /* Try to read from launched directory */
    //if (STATS.Prefs->readFromDisk(STATS.BaseDirectory) == 0)
    //    return;
    
    /* Try to read from Home , normally ~/.OpenAG/ */
    
    char* temp = strdup(DATABASES_DIRECTORY);
    makePathAbsolute(temp, STATS.BaseDirectory); // FIX -- probably shouldn't return void!
    int result = STATS.Prefs->readFromDisk(temp);
    delete(temp);
    
    if (result == 0)
        return;
    if (errno == ENOENT)
    {
        err_exit("I searched: %s but I did not find a user.txt file.\nPlease create a user.txt file with your email and pass first.\n  For more information on the format of user.txt files, please consult the README.\n", DATABASES_DIRECTORY);
    }
    err_exit("Sorry, an unexpected and unhandled error occoured when trying to open your prefrences (user.txt) file.\n");
}

void readResumes(server_status &STATS)
{
    if (STATS.setDisplayedServerStatus != NULL)
            STATS.setDisplayedServerStatus("Reading resumes database...");
    #if AGDEBUGLEVEL > 2
    printf("\n\nReading resumes database...");
    #endif
    
    if (STATS.Resumes != NULL && STATS.Resumes != 0)
    {
        delete(STATS.Resumes);
        STATS.Resumes = NULL; // hopefully this will be OK if we return early... if we get interupted.
    }
    
    // just make sure:
    
    char* temp = strdup(RESUME_DATABASE_NAME);
    char* temp2 = strdup(DATABASES_DIRECTORY);
    makePathAbsolute(temp2, STATS.BaseDirectory);
    makePathAbsolute(temp, temp2);
    int size = strlen(temp2) + 25;
    char* temp3 = new char[size];
    memset(temp3, '\0', size);
    snprintf(temp3, size, "mkdir -p \"%s\"", temp2);
    printf("About to execute: %s\n",temp3);
    if (system(temp3) != 0)
        err_print("There was an error trying to create the databases directory\n");
    delete(temp3);
    delete(temp2);
    
    //STATS.Resumes = new FileList(10, temp); // just temporary, incase we return early.
    
    FileList* CurResumes = new FileList(10, temp); // This one is for all the files in current temp.
    
    FileList* OldResumes = new FileList(10, temp); // this one is for the database file itself
    delete temp;
    
    #if AGDEBUGLEVEL > 4
    printf("initialized Resumes\n");
    printf("reading all files in temp directory\n");
    #endif
    
    if (STATS.Prefs->IncompleteDownloadsDirectory != NULL)
    {
        makePathAbsolute(STATS.Prefs->IncompleteDownloadsDirectory, STATS.BaseDirectory);
        addAllSharesInDirectory(//&STATS, 
            STATS.Prefs->IncompleteDownloadsDirectory, CurResumes, 0);
        // only want things in this temp directory
    }
    
    //#if AGDEBUGLEVEL > 3
    CurResumes->printList();
    printf("length of FileList: %i\n", CurResumes->howManyFiles());
    //#endif
    #if AGDEBUGLEVEL > 4
    printf("about to sort list\n");
    #endif
    CurResumes->sortListByFileName(); // sort all the files in temp directory, not really needed.
    #if AGDEBUGLEVEL > 3
    printf("about to read in resumes database\n");
    #endif
    if (OldResumes->readFromDisk() == 0) // assuming sorted.  Is that ok?
    {
        #if AGDEBUGLEVEL > 3
        printf("Found %i resumes in resumes database\n", OldResumes->howManyFiles());
        printf("About to merge old and new databases\n");
        #endif
        
        char* temp = strdup(DEFAULT_TEMP_DIRECTORY);
        makePathAbsolute(temp, STATS.BaseDirectory);
        if (strcmp(temp, STATS.Prefs->IncompleteDownloadsDirectory) == 0)
            STATS.Resumes = MergeResumesDatabase(OldResumes, CurResumes, true);
        else
            STATS.Resumes = MergeResumesDatabase(OldResumes, CurResumes, false);
    }
    else
    {
        if (errno == ENOENT)
            printf("No %s database file found.  Assumign then no previous files to resume.\n", OldResumes->DatabasePath);
        else
            err_print("Couldn't read %s (resumes.data) file. See error below.  This may also prevent writing of the database \nIf you don't know how to solve this issue, PLEASE REPORT.\n", OldResumes->DatabasePath);
        delete(OldResumes);
        OldResumes = NULL;

        char* temp = strdup(RESUME_DATABASE_NAME);
        char* temp2 = strdup(DATABASES_DIRECTORY);
        makePathAbsolute(temp2, STATS.BaseDirectory);
        makePathAbsolute(temp, temp2);
        int size = strlen(temp2) + 25;
        char* temp3 = new char[size];
        memset(temp3, '\0', size);
        snprintf(temp3, size, "mkdir -p \"%s\"", temp2);
        printf("about to execute: %s\n",temp3);
        if (system(temp3) != 0)
            err_print("there was an error trying to create the databases directory\n");
        delete(temp3);
        delete(temp2);
        STATS.Resumes = new FileList(10, temp); // We didn't have an old one, so we create an empty new one.
        delete temp;
    }
    
    #if AGDEBUGLEVEL > 2
    STATS.Resumes->printList();
    printf("length of FileList: %i\n", STATS.Resumes->howManyFiles());
    #endif
    
    #if AGDEBUGLEVEL > 4
    printf("about to return from readResumes().\n");
    #endif
}


void readShares(server_status &STATS)
{    
    if (STATS.setDisplayedServerStatus != NULL)
            STATS.setDisplayedServerStatus("Reading shares database...");
    #if AGDEBUGLEVEL > 2
    printf("\n\nReading shares database...");
    #endif
    if (STATS.Shares != NULL && STATS.Shares != 0)
    {
        delete(STATS.Shares); 			// clear out old FileList
        printf("Shares Database already exists in memory, clearing...\n");
        STATS.Shares = NULL;
    }
    
    // just make sure:
    
    char* thePath = strdup(SHARE_DATABASE_NAME);
    char* temp2 = strdup(DATABASES_DIRECTORY);
    makePathAbsolute(temp2, STATS.BaseDirectory);
    makePathAbsolute(thePath, temp2);
    delete(temp2);
    
    STATS.Shares = new FileList(10, thePath); 	//allocate new one.
    
    FileList* OldShares = new FileList(10, thePath); // allocate for the Old shares aswell.
    delete thePath;
    #if AGDEBUGLEVEL > 4
    printf("initialized Shares\n");
    #endif
    
    if (STATS.setDisplayedServerStatus != NULL)
            STATS.setDisplayedServerStatus("Compiling shares...");
    
    #if AGDEBUGLEVEL > 2
    printf("READING SHARES\n");
    #endif
    
    /*Loop through share directories*/
    
    for (int x = 0; STATS.Prefs->ShareDirectories[x] != NULL; x++)
    {
        #if AGDEBUGLEVEL > 5
        printf("top of readShares while loop, x=%i\n", x);
        #endif
        // fix the path up.(already done while reading prefs ???)
        // makePathAbsolute(STATS.Prefs->ShareDirectories[x]->FolderPath, STATS.BaseDirectory);
             
        addAllSharesInDirectory(//&STATS,
            STATS.Prefs->ShareDirectories[x]->FolderPath, STATS.Shares,
            STATS.Prefs->ShareDirectories[x]->ScanDepth);
    }//close while
    
    #if AGDEBUGLEVEL > 3
    STATS.Shares->printList();
    printf("length of FileList: %i\n", STATS.Shares->howManyFiles());
    #endif
    
    STATS.Shares->sortListByFileName();
    #if AGDEBUGLEVEL > 2
    printf("reading shares database...\n");
    #endif
    if (OldShares->readFromDisk() == 0) // assuming sorted.  Is that ok?
    {
        #if AGDEBUGLEVEL > 3
        OldShares->printList();
        #endif
        printf("Found %i shares in shares database\n", OldShares->howManyFiles());
        printf("Found %i shares in share directories\n", STATS.Shares->howManyFiles());
        printf("About to merge old and new databases\n");
        STATS.Shares = MergeSharesDatabase(OldShares, STATS.Shares);
    }
    else
    {
        if (errno == ENOENT)
            printf("No %s file found.  All shares will be sent as \"new\"\n", OldShares->DatabasePath);
        else
            err_print("Couldn't read %s (shares.data) file. See error below.  Likely this error is just occuring because you are upgrading, and your databases are being regenerated.  If this isn't the case, and you don't know how to solve this issue, PLEASE REPORT.\n", OldShares->DatabasePath);
        delete(OldShares);
    }
    
    /* now that we have a merged copy, we need to verify the mp3 info */
    printf("reading in mp3 info for new files\n");
    STATS.Shares->updateMP3InfoForUnvalidatedRecords();
    
    #if AGDEBUGLEVEL > 2
    STATS.Shares->printList();
    printf("length of FileList: %i\n", STATS.Shares->howManyFiles());
    #endif
    
    #if AGDEBUGLEVEL > 4
    printf("about to return from readshares().\n");
    #endif
}

int globError(const char* epath, int errnum)
{
    printf("received errnum: %i, on path: %s\n",errnum, epath);
    printf("that means: %s\n", strerror(errnum));
    return 0;
}


void sendLoginMessage(server_status &STATS)
{
    if (STATS.setDisplayedServerStatus != NULL)
        STATS.setDisplayedServerStatus("Sending login information...");
    
    /* SEND LOGIN INFORMATION */
    #if AGDEBUGLEVEL > 5
    printf("About to initialize login message\n");
    #endif
    
    AGMessage* LoginMessage = new AGMessage(STATS.connections[INFOSERVER].socket);
    
    LoginMessage->setType(LOGIN);
    LoginMessage->write_zeros(4);  //FIX -- write 4 zeros... unknown?
    printf("user check: %s\n", STATS.Prefs->EmailAddress);
    LoginMessage->write_string(STATS.Prefs->EmailAddress); // shouldn't need to be converted!
    
    unsigned char* newBuff = NULL;
    // Convert the string to the server format before sending
    
    string_localToServer(newBuff, (unsigned char*)STATS.Prefs->Password, sizeof(STATS.Prefs->Password));
    
    LoginMessage->write_string((char*)newBuff); // FIX -- need to be converted?
    delete (newBuff);
    
    LoginMessage->write_string(VERSION_STRING);
    printf("just checking IP: %s\n", STATS.myIP);
    LoginMessage->write_string(STATS.myIP);
    LoginMessage->write_zeros(6);  //FIX -- write 6 zero's for the unknown section...
    LoginMessage->queueMessage(INFOSERVER, STATS);
    
    //LoginMessage->dumpHex();

    delete(LoginMessage);
}


void sendLogoffMessage(server_status &STATS)
{
    if (STATS.setDisplayedServerStatus != NULL)
        STATS.setDisplayedServerStatus("Sending logoff...");
    
    int s = STATS.connections[INFOSERVER].socket;
    
    /* SEND LOGIN INFORMATION */
    #if AGDEBUGLEVEL > 5
    printf("about to initialize logoff message\n");
    #endif

    AGMessage* LogoffMessage = new AGMessage(s);
    
    LogoffMessage->setType(LOGOFF);
    LogoffMessage->sendMessage(); // ignore return, we don't care if it succeedes.
    delete(LogoffMessage);
    
    printf("about to return from sending logoff message\n");
}

void sendNewUserMessage(server_status &STATS)
{
    /*
        THIS IS ONLY A PROTOTYPE.  THIS WILL NOT WORK WITHOUT OTHER CODE
        CHANGES, INCLUDING CHANGING WHAT START SIGNAL IS SENT WITH THIS
        MESSAGE!
    */
    
    if (STATS.setDisplayedServerStatus != NULL)
        STATS.setDisplayedServerStatus("Sending new user request...");
    
    /* SEND NEW USER INFORMATION */
    #if AGDEBUGLEVEL > 5
    printf("About to initialize new user message\n");
    #endif
    
    AGMessage* NewUserMessage = new AGMessage(STATS.connections[INFOSERVER].socket);
    
    NewUserMessage->setType(NEW_USER);
    NewUserMessage->write_string("0.606W");
    NewUserMessage->write_string("username");
    NewUserMessage->write_string("email");
    NewUserMessage->write_string("county");
    NewUserMessage->write_string("age");
    NewUserMessage->write_string("sex");
    NewUserMessage->write_string("city");
    NewUserMessage->write_string("state");
    NewUserMessage->write_string("zip");
    NewUserMessage->write_string("county");
    NewUserMessage->write_zeros(4);  //FIX -- write 4 zero's for the unknown section...
    NewUserMessage->queueMessage(INFOSERVER, STATS);
    
    //NewUserMessage->dumpHex();

    delete(NewUserMessage);
}



void addAllSharesInDirectory(//server_status* STATSptr, 
    const char* DirPath, FileList* theList, int recurseDepth)
{
    /* Change to the given directory */
    if ((DirPath == NULL) || (chdir( DirPath ) != 0) )
    {
        err_print("Skipping directory: %s not valid.  The error below describes why I couldn't read the directory.\nIf this is your temporary files/cache directory, you can ignore this message, it will be created after the first login.  If this is one of your shared directories, then you should correct this directory in the preferences to a valid directory.\n", DirPath);
	return;
    }
    /* glob for the file names  --  FREE THE GLOB POINTER AT THE END*/
    /* WHAT IF THIS IS A REPEATED CALL? */
    glob_t pglob;
    int globerr;
    if (globerr = glob("*.mp3", 0, globError, &pglob) != 0)
    {
        printf("Error reading directory: %s\n", DirPath);
        printf("%i Files were succesfully found prior to error and will be shared.\n", pglob.gl_pathc);
    }
    #if AGDEBUGLEVEL > 0
    else
    {
        if (pglob.gl_pathc > 0)
            printf("%i files found in: %s\n",pglob.gl_pathc, DirPath);
    }
    #endif
    
    #if AGDEBUGLEVEL > 4
    printf("globbed\n");
    #endif
    /* Copy the path data out of pglob */
    for(int i = 0; i < pglob.gl_pathc; i++)
    {
        #if AGDEBUGLEVEL > 4
        printf("%s\n", pglob.gl_pathv[i]);		// print out all shares.
        #endif
        theList->addNewFileRecordFromPath(pglob.gl_pathv[i], DirPath); // add share to list.
    }
    
    #if AGDEBUGLEVEL > 4
    printf("sizeof(pglob->gl_pathv): %li\n", sizeof(pglob.gl_pathv));
    
    printf("about to free glob.\n");
    #endif
    /* then free glob's memory */
    globfree(&pglob);
    
    if (recurseDepth > 0)
    {
        glob_t pglob2;
        if (globerr = glob("*", GLOB_MARK , globError, &pglob2) != 0)
        {
            printf("Error reading sub-directories.\n");
            printf("%i entries were succesfully found prior to error and will be scanned for subdirectories.\n",
                    pglob2.gl_pathc);
        }
        #if AGDEBUGLEVEL > 1
        else
            printf("Scan of %s completed sucessfully, %i possible subdirectories found to scan for files.\n",
                    DirPath, pglob2.gl_pathc);
        #endif
        
        recurseDepth--;
        
        for(int i = 0; i < pglob2.gl_pathc; i++)
        {
            int len = strlen(pglob2.gl_pathv[i]);
            if ( (len > 0) && ( pglob2.gl_pathv[i][len - 1] == '/' ) )
            {
                //add all shares in subdirectories.
                makePathAbsolute(pglob2.gl_pathv[i], DirPath);
                
                /*
                if (!(STATS.FinishedDownloadsAreShared) ) // will this work with aliases too?  I think so...
                {
                    if (strcmp(pglob2.gl_pathv[i], STATS.Prefs->FinishedDownloadsDirectory) == 0)
                        STATS.FinishedDownloadsAreShared = true;
                }
                */

                addAllSharesInDirectory(//STATSptr, 
                    pglob2.gl_pathv[i], theList, recurseDepth );
                    // should always be true.
            }
        }
        
        globfree(&pglob2);
    }
    else
        printf("Not recursing, already at maximum depth\n");
}
