/*
    FileList.cpp
    OpenAG, libOpenAG, OpenAG X
 
    Created by Eric Seidel on Tue Nov 20 2001.
 
    Copyright (c) 2001-2002 Eric Seidel. All rights reserved.
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
 
#include "common.h"

#include <fcntl.h>
#include <errno.h>	/* for errno */

#include "FileList.h"
#include "FileInfo.h"
#include "AGMain.h"
#include "AGMessage.h"
#include "error.h"
#include "socket.h"

int FileRecordCompare(const void* left, const void* right);

/*
void quicksort( FileRecord *a, int low, int high );
int partition( FileRecord *a, int low, int high );
void SWAP( FileRecord *a, int left, int right );
*/
 
FileList::FileList(char* path)
{
    DatabasePath = NULL;
    FileList(100, path);
}
 
// creates an emply FileList of atleast size
FileList::FileList(int size, char* path)
{
    DatabasePath = strdup(path);
    
    #if AGDEBUGLEVEL > 5
	printf("New FileList constructed:\n");
	printf("    Path = %s\n", path);
	printf("    Size = %d\n", size);
    #endif
    
    FilesArraySize = size + 10;
    FilesArray = new FileRecord[FilesArraySize];
    bzero(FilesArray, FilesArraySize * sizeof(FileRecord));
    lastRecordPlus1 = 0;
}


// cleans up.
FileList::~FileList()
{
    #if AGDEBUGLEVEL > 4
    printf("****WOH.  A FILE LIST IS BEING DELETED. EITHER YOU ARE EXITING, OR SOMETHIGN IS WRONG.\n");
    #endif
    if ((DatabasePath != NULL) && (DatabasePath != 0))
        delete DatabasePath;
    
    while( lastRecordPlus1 > 0)
    {
        deleteFileRecordWithLocalID(lastRecordPlus1 - 1);
        #ifndef AUTOCONDENSEDATABASESONDELETE
        lastRecordPlus1--;
        #endif
    }
        
    if (FilesArray != NULL && FilesArray != 0)
        delete(FilesArray);
}

FileRecord* FileList::getNewFileRecord()
{    
    if (lastRecordPlus1 >= FilesArraySize)
        expand();
    
    int x = 0;
    
    for(; x < lastRecordPlus1; x++)
        if (FilesArray[x].flags == NOT_IN_USE) break;
        
    if (x == lastRecordPlus1)
        lastRecordPlus1++;
    
    deleteFileRecordWithLocalID(x);

    return (&FilesArray[x]);
}

void FileList::deleteFileRecordWithLocalID(int x)
{
    if (x >= lastRecordPlus1 || x < 0) // fixed can't delete beyond the end.
    {
        #if AGDEBUGLEVEL > 2
        printf("********* TRIED TO DELETE FILE RECORD WITH BAD ID: %i******\n", x);
        #endif
        return;
    }
    if ((FilesArray[x].SuggestedFileName != NULL) && (FilesArray[x].SuggestedFileName != 0))
        delete(FilesArray[x].SuggestedFileName);
    if ((FilesArray[x].DiskFileName != NULL) && (FilesArray[x].DiskFileName != 0))
        delete(FilesArray[x].DiskFileName);
    if ((FilesArray[x].DirectoryName != NULL) && (FilesArray[x].DirectoryName != 0))
        delete(FilesArray[x].DirectoryName);
    if ((FilesArray[x].FileSet != NULL) && (FilesArray[x].FileSet != 0))
        delete(FilesArray[x].FileSet);
    /*if (FilesArray[x].FileID != NULL && FilesArray[x].FileID != 0)
        delete(FilesArray[x].FileID);*/
        
    if ((FilesArray[x].ID3v1_Title != NULL) && (FilesArray[x].ID3v1_Title != 0))
        delete(FilesArray[x].ID3v1_Title);
    if ((FilesArray[x].ID3v1_Artist != NULL) && (FilesArray[x].ID3v1_Artist != 0))
        delete(FilesArray[x].ID3v1_Artist);
    if ((FilesArray[x].ID3v1_Album != NULL) && (FilesArray[x].ID3v1_Album != 0))
        delete(FilesArray[x].ID3v1_Album);
    
    if ((FilesArray[x].ID3v2_Title != NULL) && (FilesArray[x].ID3v2_Title != 0))
        delete(FilesArray[x].ID3v2_Title);
    if ((FilesArray[x].ID3v2_Artist != NULL) && (FilesArray[x].ID3v2_Artist != 0))
        delete(FilesArray[x].ID3v2_Artist);
    if ((FilesArray[x].ID3v2_Album != NULL) && (FilesArray[x].ID3v2_Album != 0))
        delete(FilesArray[x].ID3v2_Album);
    if ((FilesArray[x].ID3v2_Tag != NULL) && (FilesArray[x].ID3v2_Tag != 0))
        delete(FilesArray[x].ID3v2_Tag);
    
    bzero(&(FilesArray[x]), sizeof(FileRecord));
    
    FilesArray[x].SuggestedFileName = NULL;
    FilesArray[x].DiskFileName = NULL;
    FilesArray[x].DirectoryName = NULL;
    FilesArray[x].FileSet = NULL;
    FilesArray[x].LocalID = x;
    FilesArray[x].SongID = -1;
    FilesArray[x].ConnectionPtr = NULL;
    
    
#ifdef AUTOCONDENSEDATABASESONDELETE
    if ( x == (lastRecordPlus1 - 1) )
    {
        if (lastRecordPlus1 > 0)
            lastRecordPlus1--;
    }
    else if ( x < (lastRecordPlus1) )
    {
        // we didn't just remove one off the end, so we have to adjust...
        memmove(&FilesArray[x-1], &FilesArray[x], sizeof(FileRecord)*( (lastRecordPlus1) - x) );
        lastRecordPlus1--;
        
        reassignLocalIDs();
    }
#endif
}

int FileList::addNewFileRecordFromPath(const char* FileName, const char* DirectoryName)
{
    #if AGDEBUGLEVEL > 4
    printf("adding share\n");
    #endif
    
    FileRecord* newRecord = getNewFileRecord();
    
    newRecord->DiskFileName = strdup(FileName); // reliable?
    newRecord->DirectoryName = strdup(DirectoryName); // reliable?
    
    struct stat stat_buf;
    char* file_to_check = new char[PATH_MAX];
    snprintf(file_to_check, PATH_MAX, "%s/%s", newRecord->DirectoryName, newRecord->DiskFileName);
    printf("file: %s\n",file_to_check);
    
    if (stat(file_to_check , &stat_buf) != 0)
    {
        err_print("Error getting file Info, not adding share: %s of directory:%s\n", FileName, DirectoryName);
        delete(file_to_check);
        deleteFileRecordWithLocalID(newRecord->LocalID);
        return (-1);
    }
    delete(file_to_check);
    newRecord->FileSize		= stat_buf.st_size;
    newRecord->FileLastModified = stat_buf.st_mtime;
    
    /* don't need to do anythign for the MP3 info, will be provided later */
    
    newRecord->flags = UNVALIDATED_SHARE;
    
    #if AGDEBUGLEVEL > 4
    printf("finished adding share\n");
    #endif
    return (newRecord->LocalID);
}

int FileList::addNewFileRecordFromFileRecord(FileRecord* newFile)
{
    // FIX.  Should test and return correctly.
    if ( (newFile == NULL) || (newFile == 0) )
        return (-1);
    FileRecord* newRecord = getNewFileRecord();
    int theLocalID = newRecord->LocalID;
    
    memcpy(newRecord, newFile, sizeof(FileRecord));
    bzero(newFile, sizeof(FileRecord));
    
    FilesArray[theLocalID].LocalID = theLocalID;
    return 0;
}


void FileList::updateMP3InfoForUnvalidatedRecords()
{
    for(int x = 0; x < lastRecordPlus1; x++)
    {
        
        FileRecord* theRecord = getFileRecordByLocalID(x);
        
        if (theRecord == NULL)
            printf("encountered a NULL for localID: %i, skipping\n", x);
        
        if (theRecord->flags == VALIDATED_SHARE)
            continue;
        
        if (readMP3info(theRecord) < 0)
        {
            err_print("Error reading MP3 Info for file: %s of directory: %s, removing from database\n", theRecord->DiskFileName, theRecord->DirectoryName);
            deleteFileRecordWithLocalID(x);
        }
    }
    
    removeBlankRecords();
}

FileRecord* FileList::getFileRecordByLocalID(int LocalID)
{
    #if AGDEBUGLEVEL > 5
    printf("asking for file record with local ID: %i with lastRecordPlus1: %i\n", LocalID, lastRecordPlus1);
    #endif
    
    if ( (LocalID >= lastRecordPlus1) || (LocalID < 0) )
        return NULL;
    //what if LocalID is bad?
    if (FilesArray[LocalID].flags == NOT_IN_USE)
        return NULL;
    
    return &(FilesArray[LocalID]);
}


// returns the FileRecord variable for the first file in the DB with that ID.

/*
FileRecord* FileList::getFileRecordByFileID(char* File_ID, int FileIDLength)
{
    int x;
    for (x = 0; x < lastRecordPlus1; x++)
    {
        if( (FileIDLength == FilesArray[x].FileIDLength) &&
            (memcmp(File_ID, FilesArray[x].FileID, FileIDLength) == 0) ) break;
    }
    
    if (x >= lastRecordPlus1)
    {
        printf("Record Not Found!\n");
        return NULL;
    }
    
    return &(FilesArray[x]);
}
*/

FileRecord* FileList::getFileRecordBySuggestedNameAndSize(const char* FileName, unsigned int size)
{
    return getFileRecordByGivenNameAndSize(FileName, size, USE_SUGGESTED_NAME);
}

FileRecord* FileList::getFileRecordByDiskNameAndSize(const char* FileName, unsigned int size)
{
    return getFileRecordByGivenNameAndSize(FileName, size, USE_DISK_NAME);
}



FileRecord* FileList::getFileRecordByGivenNameAndSize(const char* FileName, unsigned int size, bool useDiskName)
{    
    if (FileName == NULL)
    {
        printf("WOH.  TRIED TO LOOKUP A NULL FILENAME!\n");
        return NULL;
    }

    #if AGDEBUGLEVEL > 2
    if (lastRecordPlus1 < 1) // I'm getting EXC_BAD_ACCESS here, why?
    {
        printf("No (%i) records in Database\n", lastRecordPlus1);
    }

    printf("comparing %s, size: %i, with below:\n",FileName,size);
    #endif
    
    char* theFileName = NULL;
    
    int x;
    for (x = 0; x < lastRecordPlus1; x++)
    {
        if (useDiskName)
            theFileName = FilesArray[x].DiskFileName;
        else 
            theFileName = FilesArray[x].SuggestedFileName;
        
        if (FilesArray[x].flags == NOT_IN_USE)
        {
            printf("skipping file record: %i, not in use\n", x);
            continue;
        }
        
        if ( theFileName == NULL)
        {
            err_print("Oops.  Encountered a NULL filename on a FileRecord (#%i) in use.  Removing it.  PLEASE REPORT.\n", x);
            deleteFileRecordWithLocalID(x);
            continue;
        }
        
        if( (strcmp(FileName, theFileName) == 0) )
        {
            printf("%s matches %s at record: %i\n",FileName, theFileName, x);
            if ((size == FilesArray[x].FileSize))
                break;
            //else if (size == -1) break; // size not set, so it was just found in the resume directory.
        }
        #if AGDEBUGLEVEL > 2
        printf("with %s, size: %li, decided not equal\n", theFileName, FilesArray[x].FileSize);
        #endif
    }
    
    if (x >= lastRecordPlus1)
    {
        #if AGDEBUGLEVEL > 2
        printf("Record Not Found! lastRecordPlus1:%i  x:%i\n", lastRecordPlus1, x);
        #endif
        return NULL;
    }
    else
        printf("found record #%i of %i satisfies, returning pointer: %p\n", x, lastRecordPlus1, &(FilesArray[x]) );
    
    return (&(FilesArray[x]));
}

/*

int FileList::setFileID(int LocalID, char* FileID, int FileIDLength)
{
    if ((LocalID >= lastRecordPlus1) || (LocalID < 0) || (FileIDLength < 0) || (FileID == NULL) || (FileID == 0))
        return -1;
    else
    {
        if (FilesArray[LocalID].FileID != NULL || FilesArray[LocalID].FileID == 0)
            delete (FilesArray[LocalID].FileID);
        FilesArray[LocalID].FileIDLength = FileIDLength;
        FilesArray[LocalID].FileID = new char[FileIDLength];
        memcpy(FilesArray[LocalID].FileID, FileID, FileIDLength);
    }
    return 0;
}



int FileList::setFileID(char* FileName, int FileSize, char* FileID, int FileIDLength)
{
    if ( (FileName == NULL) || (FileName == 0) || (FileSize < 0) 
        || (FileIDLength < 0) || (FileID == NULL) || (FileID == 0) )
        return (-1);
    else
    {
        int LocalID = getLocalID(FileName, FileSize);
        if ( LocalID < 0)
            return LocalID;
        
        return setFileID(LocalID, FileID, FileIDLength);
    }
}

*/
/*
int FileList::getLocalID(char* FileName, int FileSize)
{
    for(int x = 0; x < lastRecordPlus1; x++)
    {
        if ( (FileSize == FilesArray[x].FileSize) 
              && ( strcmp((char*)FileName, FilesArray[x].FileName) == 0) )
              return x;
    }
    return (-1);
}
*/

/*
int FileList::setServerID(int LocalID, int ServerID)
{
    if ((LocalID >= lastRecordPlus1) || (LocalID < 0))
        return -1;
    else
        FilesArray[LocalID].ServerID = ServerID;
    return 0;
}
*/

int FileList::setSongID(int LocalID, int SongID)
{
    if ((LocalID >= lastRecordPlus1) || (LocalID < 0))
        return -1;
    else
    {
        FilesArray[LocalID].SongID = SongID;
        FilesArray[LocalID].flags = VALIDATED_SHARE;
    }
    return 0;
}


int FileList::setFileSet(int LocalID, const char* FileSet)
{
    if ((LocalID >= lastRecordPlus1) || (LocalID < 0))
    {
        printf("CAN'T set fileset for record #%i = %s!  LRP1: %i\n", LocalID, FileSet, lastRecordPlus1);
        return -1;
    }
    else
    {
        if ( (FilesArray[LocalID].FileSet != NULL) || (FilesArray[LocalID].FileSet != 0) )
            delete (FilesArray[LocalID].FileSet);
        #if AGDEBUGLEVEL > 4
        printf("setting fileset for record #%i = %s\n", LocalID, FileSet);
        #endif
        FilesArray[LocalID].FileSet = strdup(FileSet);
    }
    return 0;
}


void FileList::printList()
{
    printf("FileList Contains:\n");
    
    int x;
    for (x = 0; x < lastRecordPlus1; x++)
    {
        if (FilesArray[x].flags != 0)
            printf("%s\n", FilesArray[x].DiskFileName);
        else
            printf("skipping %i, not in use, but filename: %s\n",x, FilesArray[x].DiskFileName);
    }
}

int FileList::readFromDisk()
{
    if (DatabasePath != NULL && DatabasePath != 0)
        return readFromDisk(DatabasePath);
    else
        return(-1);
}

int FileList::readFromDisk(char* Path)
{
    int resultfd;
    int O_flags;
    
    O_flags = (O_RDONLY);
    if ( ( resultfd = open(Path, O_flags) ) < 0 )
        return (-1); // they will check errno
        
    int tempSize = 2000;
    char* temp = new char[tempSize];
    char* wptr = temp;
    char* rptr = temp;
    bool gotEOF = false;
    int re = -1;
    
    int revisionCode;
    
    if (read_data(resultfd, (char*)&revisionCode, sizeof(int)) < 0)
    {
        delete(temp);
        errno = 0;
        return (-1);
    }
    
    if (revisionCode != FILELIST_REVISION_CODE)
    {
        err_print("read revision code: %i from database, which is not the expected: %i, returning error, database will be removed.\n",
            revisionCode, FILELIST_REVISION_CODE);
        delete(temp);
        errno = 0;
        return (-1);
    }
    //printf("Reading database of revision: %i as expected\n", revisionCode);
    
    /* FIX -- This whole logic below needs to be cleaned! */
    
    while(!gotEOF) /* FIX!!! don't think this whoel thing is right */
    {
        #if AGDEBUGLEVEL > 6
        printf("about to read, asking for: %i\n", tempSize - (wptr - temp) );
        #endif
        re = read(resultfd, wptr, tempSize - (wptr - temp) );
        #if AGDEBUGLEVEL > 6
        printf("read %i bytes\n", re);
        #endif
        if( re < 0 )
        {
            close(resultfd);
            delete(temp);
            return(-1);
        }
        else if (re == 0)
        {
            #if AGDEBUGLEVEL > 2
            printf("got EOF\n");
            #endif
            gotEOF = true;
        }
        else
            wptr += re;
        
        while ( (wptr-rptr) > 0 )  // there is still stuff to read.
        {
            
            #if AGDEBUGLEVEL > 3
            printf("parsing FileRecord from file\n");
            #endif
            int completeSize;
            
            if (wptr-rptr < 12)
                break; // just avoid continuing, we don't have enough even for the message headers.
            
            AGMessage* recordRead =
                new AGMessage(rptr,wptr-rptr,PEER_MESSAGE); //read the share.
            completeSize = recordRead->getCompleteMessageSize();
            if (recordRead->getMessageStatus() == AG_MESSAGE_COMPLETE) //crude... but works.
                rptr += completeSize;
            else if (recordRead->getMessageStatus() == AG_MESSAGE_INCOMPLETE)
            {
                #if AGDEBUGLEVEL > 2
                printf("Reading message in database: Message not complete yet, must read more.\n");
                #endif
                if(recordRead != NULL) // shouldn't be necessary.
                {
                    delete(recordRead);
                    recordRead = NULL;
                }
                break;
            }
            else
            {
                err_print("Error encountered while reading database, database will be discarded and regenerated.\n");
                delete(recordRead);
                return (-1);
            }
            /* get a record */
            FileRecord* newFile = getNewFileRecord();
            
            #if AGDEBUGLEVEL > 5
            printf("found valid FileRecord, about to parse, this is number: %li\n", newFile->LocalID);
            #endif
            
            unsigned short SuggestedFileNameLength = 
                recordRead->read_string(newFile->SuggestedFileName, DO_TERMINATE); // FIX
            unsigned short DiskFileNameLength = 
                recordRead->read_string(newFile->DiskFileName, DO_TERMINATE); // FIX
            
            #if AGDEBUGLEVEL > 5
            printf("read filename: %s of length: %i\n", newFile->DiskFileName, DiskFileNameLength);
            #endif
            
            unsigned short DirectoryNameLength = 
                recordRead->read_string(newFile->DirectoryName, DO_TERMINATE); // FIX
            unsigned short FileSetLength = 
                recordRead->read_string(newFile->FileSet, DO_TERMINATE); // FIX
            /*
            unsigned short FileIDLength = recordRead->read_short();
            recordRead->read_string(newFile->FileID, FileIDLength); // FIX
            */
            
            newFile->SongID 		= recordRead->read_int();
            newFile->Bitrate		= recordRead->read_short();
            newFile->SongLength 	= recordRead->read_int();
            newFile->FileResumePosition = recordRead->read_int();
            newFile->FileSize		= recordRead->read_int();
            newFile->FileLastModified	= recordRead->read_int();
            
            newFile->HasID3v1		= recordRead->read_byte();
            if (newFile->HasID3v1)
            {
                recordRead->read_string(newFile->ID3v1_Title, DO_TERMINATE);
                recordRead->read_string(newFile->ID3v1_Artist, DO_TERMINATE);
                recordRead->read_string(newFile->ID3v1_Album, DO_TERMINATE);
            }
            newFile->HasID3v2 		= recordRead->read_byte();
            if (newFile->HasID3v2)
            {
                recordRead->read_string(newFile->ID3v2_Title, DO_TERMINATE);
                recordRead->read_string(newFile->ID3v2_Artist, DO_TERMINATE);
                recordRead->read_string(newFile->ID3v2_Album, DO_TERMINATE);
                newFile->ID3v2_Track = recordRead->read_short();
                recordRead->read_string(newFile->ID3v2_Tag, DO_TERMINATE);
            }
            
            newFile->flags 		= VALIDATED_SHARE; // That should be right... 
            
            #if AGDEBUGLEVEL > 5
            printf("the FileResumePostion read was: %li\n", newFile->FileResumePosition);
            printf("the FileSize read was: %li\n", newFile->FileSize);
            printf("the songID read was: %li\n", newFile->SongID);
            #endif
            if (recordRead != NULL)
            {
                delete(recordRead);
                recordRead = NULL;
            }
        }
        
        if (gotEOF)
        {
            if ((wptr - rptr) > 0) printf("SOMETHING FISHY WITH DATABASE FILE, NOT EVERYTHING READ\n");
            break;
        }
        else
            doAdjustBuffer(temp, rptr, wptr, true); 
        
        if (wptr == (temp + tempSize)) // buffer still full after adjust.
        {
            printf("Woh.  Error reading from FilesArray database.  Read full buffer without finding a single share message\n");
            close(resultfd);
            delete(temp);
            return(-1);
        }
    }
    close(resultfd);
    delete(temp);
    return 0; //success
}


int FileList::writeToDisk(void (*setDisplayedFileSystemChange) (char*))
{
    int resultfd;
    int O_flags;
    
    O_flags = (O_CREAT | O_TRUNC | O_WRONLY);
    if ( ( resultfd = open((char*)DatabasePath, O_flags) ) < 0 )
    {
        err_print("Couldn't write database file: %s\n", DatabasePath);
        return (-1); // they will check errno
    }
    
    if (setDisplayedFileSystemChange != NULL)
        setDisplayedFileSystemChange(DatabasePath);
    
    /* changing permissions */
    fchmod(resultfd, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
    
    #if AGDEBUGLEVEL > 1
    printf("Got database: %s , %i FileRecords to write.\n", DatabasePath, lastRecordPlus1 );
    #endif
    
    printf("Removing blank records from database...\n");
    removeBlankRecords();
    
    printf("Sorting database by FileName...\n");
    sortListByFileName();
    
    printf("Writing database, version number %i\n", FILELIST_REVISION_CODE);
    int revisionCode = FILELIST_REVISION_CODE;
    if (write_data(resultfd, (char*)&revisionCode, sizeof(int) ) < 0)
        return (-1);
            
    for (int x = 0; x < lastRecordPlus1; x++)
    {
        if (FilesArray[x].flags == NOT_IN_USE) continue;
        
        #if AGDEBUGLEVEL > 4
        printf("writting FileRecord %i\n", x);
        #endif
        
        AGMessage* shareToWrite = new AGMessage(resultfd);
        
        // automatically handles writing of nulls
        shareToWrite->write_string(FilesArray[x].SuggestedFileName);
        shareToWrite->write_string(FilesArray[x].DiskFileName);
        shareToWrite->write_string(FilesArray[x].DirectoryName);
        shareToWrite->write_string(FilesArray[x].FileSet);
        //shareToWrite->write_string(FilesArray[x].FileID);
            
        shareToWrite->write_int(FilesArray[x].SongID);
        shareToWrite->write_short(FilesArray[x].Bitrate);
        shareToWrite->write_int(FilesArray[x].SongLength);
        shareToWrite->write_int(FilesArray[x].FileResumePosition);
        shareToWrite->write_int(FilesArray[x].FileSize);
        shareToWrite->write_int(FilesArray[x].FileLastModified);
        
        shareToWrite->write_byte(FilesArray[x].HasID3v1);
        if (FilesArray[x].HasID3v1)
        {
            shareToWrite->write_string(FilesArray[x].ID3v1_Title);
            shareToWrite->write_string(FilesArray[x].ID3v1_Artist);
            shareToWrite->write_string(FilesArray[x].ID3v1_Album);
        }
        shareToWrite->write_byte(FilesArray[x].HasID3v2);
        if (FilesArray[x].HasID3v2)
        {
            shareToWrite->write_string(FilesArray[x].ID3v2_Title);
            shareToWrite->write_string(FilesArray[x].ID3v2_Artist);
            shareToWrite->write_string(FilesArray[x].ID3v2_Album);
            shareToWrite->write_short(FilesArray[x].ID3v2_Track);
            shareToWrite->write_string(FilesArray[x].ID3v2_Tag);
        }
        
        int size = shareToWrite->getCompleteMessageSize();
        #if AGDEBUGLEVEL > 5
        printf("message is of size: %i\n", size);
        #endif
        char* temp = new char[size+20]; // FIX -- Just in case, shouldn't cause trouble.
                
        #if AGDEBUGLEVEL > 6
        shareToWrite->dumpHex();
        #endif
        
        int size2 = shareToWrite->getCompleteMessageData(temp);
        delete(shareToWrite);
        #if AGDEBUGLEVEL > 5
        printf("data returned size of: %i\n", size2);
        #endif
        
        int written = 0;
        int wr;
        
        while ( (size - written) > 0)
        {
            #if AGDEBUGLEVEL > 5
            printf(" size - written, or %i bytes left to write.\n", size - written);
            printf(" temp is: %p\n", temp);
            #endif
            
            wr = write(resultfd, temp + written, size - written);
            #if AGDEBUGLEVEL > 5
            printf("wrote %i bytes\n", wr);
            #endif
            if (wr == 0)
                err_print("wrote 0 bytes!\n");
            if(wr < 0)
            {
                err_print("error writting to database\n");
                delete(temp);
                close(resultfd);
                return(-1);
            }
            else
                written += wr;
        }
        delete(temp);
    }
    
    close(resultfd);
    return 0; //success
}

void FileList::expand()
{
    #if AGDEBUGLEVEL > 3
    printf("expanding, current: %i\n", FilesArraySize);
    #endif
    FileRecord* NewList = new FileRecord[FilesArraySize * 2]; //allocate new space
    bzero(NewList, FilesArraySize * 2 * sizeof(FileRecord));
    //printf("zero check: %i\n", NewList[lastRecordPlus1].FileName);
    memcpy(NewList, FilesArray, FilesArraySize*sizeof(FileRecord)); // copy in data
    //Clear old space.
    delete (FilesArray);
    //set pointer
    FilesArray = NewList;
    FilesArraySize *=2;
}


void FileList::sortListByFileName()
{
    #if AGDEBUGLEVEL > 2
    printf("about to sort\n");
    #endif
    if (lastRecordPlus1 > 1)
        qsort((char*)(FilesArray), lastRecordPlus1, sizeof(FileRecord), FileRecordCompare);
    #if AGDEBUGLEVEL > 2
    printf("readjusting lastRecordPlus1\n");
    #endif
    while ( (lastRecordPlus1 > 0) && (FilesArray[lastRecordPlus1-1].flags == NOT_IN_USE) )
        lastRecordPlus1--;
    #if AGDEBUGLEVEL > 2
    printf("about to reassign\n");
    #endif
    reassignLocalIDs();
    #if AGDEBUGLEVEL > 2
    printf("done sorting.\n");
    #endif
}

void FileList::removeDuplicateRecords()
{
    // FIX -- probalby shoudl do this from the other end first...
    for(int y = 1; y < lastRecordPlus1; y++)
    {
        if( FileRecordCompare( &FilesArray[ y - 1 ], &FilesArray[ y ] ) == 0 )
            deleteFileRecordWithLocalID( y - 1 );
    }
    
    //removeBlankRecords();
}

void FileList::removeBlankRecords()
{
    
    int moveCount = 0;
    for(int y = 0; y < lastRecordPlus1; y++)
    {
        if( FilesArray[y].flags == NOT_IN_USE)
            moveCount++; // one more in a row.
        else if (moveCount > 0)
        {
            // ok, we've hit ones in use again, move the rest of the bunch.
            memmove(&FilesArray[y-moveCount], &FilesArray[y], sizeof(FileRecord)*( (lastRecordPlus1) - y) );
            lastRecordPlus1 -= moveCount;
            moveCount = 0;
        }
    }
    
    if (moveCount > 0) // just a bunch of empty records at the end.
        lastRecordPlus1 -= moveCount;
    
    reassignLocalIDs();
    // should we check to see if we can shrink the array?
}

int FileRecordCompare(const void* left, const void* right) // always uses disk name.
{
    FileRecord* FLeft = (FileRecord*) left;
    FileRecord* FRight = (FileRecord*) right;
    
    if (FLeft->flags == NOT_IN_USE)
        return 1;  // push it to the end.
    if (FRight->flags == NOT_IN_USE)
        return 1; // put it to the end.
    
    
    // This has to be defined... doesn't always do good things though...
    if (   ( (left == NULL) || (left == 0) || (FLeft->DiskFileName == NULL) || (FLeft->DiskFileName == 0) )
        && ( (right != NULL) && (right != 0) && (FRight->DiskFileName != NULL) && (FRight->DiskFileName != 0) )  )
        return 1;
    if (   ( (left != NULL) && (left != 0) && (FLeft->DiskFileName != NULL) && (FLeft->DiskFileName != 0) )
        && ( (right == NULL) || (right == 0) || (FRight->DiskFileName == NULL) || (FRight->DiskFileName == 0) )  )
        return -1;
    if (   ( (left == NULL) || (left == 0) || (FLeft->DiskFileName == NULL) || (FLeft->DiskFileName == 0) )
        && ( (right == NULL) || (right == 0) || (FRight->DiskFileName == NULL) || (FRight->DiskFileName == 0) )  )
        return 0;
        
    // FIX -- IS ALL THIS REALLY NECESSARY?
    if (   ( (left == NULL) || (left == 0) || (FLeft->DirectoryName == NULL) || (FLeft->DirectoryName == 0) )
        && ( (right != NULL) && (right != 0) && (FRight->DirectoryName != NULL) && (FRight->DirectoryName != 0) )  )
        return 1;
    if (   ( (left != NULL) && (left != 0) && (FLeft->DirectoryName != NULL) && (FLeft->DirectoryName != 0) )
        && ( (right == NULL) || (right == 0) || (FRight->DirectoryName == NULL) || (FRight->DirectoryName == 0) )  )
        return -1;
    if (   ( (left == NULL) || (left == 0) || (FLeft->DirectoryName == NULL) || (FLeft->DirectoryName == 0) )
        && ( (right == NULL) || (right == 0) || (FRight->DirectoryName == NULL) || (FRight->DirectoryName == 0) )  )
        return 0;
    
    int result1 = strcmp(FLeft->DirectoryName, FRight->DirectoryName);
    
    if (result1 == 0)
    {
        int result2 = strcmp(FLeft->DiskFileName, FRight->DiskFileName);
        
        if (result2 == 0)
        {
            if ( FLeft->FileResumePosition == FRight->FileSize )
                return 0; // this is a resume...
            
            if (FLeft->FileSize < FRight->FileSize)
                return -1;
            else if (FLeft->FileSize > FRight->FileSize)
                return 1;
            else
                return 0;
        }
        else
            return result2;
    }
    else
        return result1;
}


void FileList::reassignLocalIDs()
{
    for(int x = 0; x < lastRecordPlus1; x++)
    {
        FilesArray[x].LocalID = x;
        // if it's got an active connection, update the LocalID...
        if (FilesArray[x].ConnectionPtr != NULL)
            FilesArray[x].ConnectionPtr->LocalID = x;
    }
    
}


FileList* MergeSharesDatabase(FileList* &Old, FileList* &Current)
{
    FileList* Combined = new FileList(100, Old->DatabasePath);
    
    int OldNum = 0;
    int CurNum = 0;
    int BadIDs = 0;
    int inBoth = 0;
    
    int OldMax = Old->lastRecordPlus1;
    int CurMax = Current->lastRecordPlus1;
    
    FileRecord* OldRecs = Old->FilesArray;
    FileRecord* CurRecs = Current->FilesArray;
    
    #if AGDEBUGLEVEL > 3
    printf("comparing the two share lists:\n");
    Old->printList();
    printf("\nAnd:\n");
    Current->printList();
    #endif
    
    while ((OldNum < OldMax) && (CurNum < CurMax))
    {
        while ( (OldNum < OldMax) && (OldRecs[OldNum].SongID < 2) )
        {
            #if AGDEBUGLEVEL > 3
            printf("songID: %li for: #%i name: %s less than 2, ignoring record\n", 
                OldRecs[OldNum].SongID, OldNum, OldRecs[OldNum].DiskFileName);
            #endif
            OldNum++; // ignore all Olds which have a "bad" or no serverID.
            BadIDs++;
        }
        
        if (OldNum >= OldMax)
            break;  // all of the remaining files in DB were "bad"
        
        int compareResult = FileRecordCompare(&OldRecs[OldNum],&CurRecs[CurNum]);
        if(compareResult == 0)
        {
            inBoth++;
            Combined->addNewFileRecordFromFileRecord(&OldRecs[OldNum]); // found one which we already know about.
            OldNum++;
            CurNum++;
        }
        else if (compareResult < 0)
            OldNum++;	// didn't find one that we already know about.
        else
        {
            Combined->addNewFileRecordFromFileRecord(&CurRecs[CurNum]); // found one we don't know about.
            CurNum++;
        
        }
    }
    
    printf("Removing remaining %i unfound database records\n", OldMax - OldNum);
    while (OldNum < OldMax)
    {
        if ((OldRecs[OldNum].SongID < 2))
        {
            printf("Removing record: %i, %s -- Bad ID: %i\n", OldNum, OldRecs[OldNum].DiskFileName, OldRecs[OldNum].SongID);
            BadIDs++;
        }
        else
            printf("File: %s, found in DB but not on disk.\n", OldRecs[OldNum].DiskFileName);
        OldNum++;
    }
    
    printf("Adding remaining %i new shares...\n", CurMax - CurNum);
    while (CurNum < CurMax)
    {
        Combined->addNewFileRecordFromFileRecord(&CurRecs[CurNum]);
        CurNum++;
    }
    
    printf("%i shares were found to both exist in the database and on disk\n", inBoth);
    printf("%i old shares were ignored because of bad IDs\n", BadIDs);
    printf("%i old shares were not found on disk and removed from the database\n", (OldNum - inBoth) - BadIDs);
    printf("In Total: %i old shares were removed from the database\n", OldNum - inBoth);
    printf("%i new shares were found not in the database and were added\n", CurNum - inBoth);
    
    if (OldNum < OldMax && CurNum < CurMax)
        printf("*********SOMETHING REALLY STRANGE WITH FILELISTS*******\n");
    
    delete(Old);
    delete(Current);
    
    return(Combined);
}


FileList* MergeResumesDatabase(FileList* &Old, FileList* &Current, bool deleteOK)
{
    if (deleteOK)
        printf("Merging resumes, and DELETING extras!\n");
    
    FileList* Combined = new FileList(100, Old->DatabasePath);
    
    int OldNum = 0;
    int CurNum = 0;
    int inBoth = 0;
    
    int OldMax = Old->lastRecordPlus1;
    int CurMax = Current->lastRecordPlus1;
    
    FileRecord* OldRecs = Old->FilesArray;
    FileRecord* CurRecs = Current->FilesArray;
    
    while ((OldNum < OldMax) && (CurNum < CurMax))
    {
        //while ( (OldNum < OldMax) && (OldRecs[OldNum].SongID < 2) )
        //    OldNum++; // ignore all Olds which have a "bad" or no serverID.
        // other things that should be removed from the database?
        
        //if (OldNum >= OldMax)
        //    break;  // all of the remaining files in DB were "bad"
        
        int compareResult = FileRecordCompare(&OldRecs[OldNum],&CurRecs[CurNum]);
        if(compareResult == 0)
        {
            inBoth++;
            Combined->addNewFileRecordFromFileRecord(&OldRecs[OldNum]); // found one which we already know about.
            OldNum++;
            CurNum++;
        }
        else if (compareResult < 0)
        {
            printf("File: %s, found in DB but not on disk.\n", OldRecs[OldNum].DiskFileName);
            OldNum++;	// didn't find one that we already know about.
        }
        else
        {
            if (deleteOK)
            {
                char* temp = strdup(CurRecs[CurNum].DiskFileName);
                //makePathAbsolute(temp, CurRecs[CurNum].FolderName);
                printf("Deleting tempfile: %s, not in resumes database.\n", temp);
                unlink(temp); // could delete other files.. but probably bad.
                delete(temp);
            }
            CurNum++;  // found one we don't know about, not adding to list.
        }
    }
    
    printf("Removing remaining %i database records\n", OldMax - OldNum);
    while (OldNum < OldMax)
    {
        printf("File: %s, found in DB but not on disk.\n", OldRecs[OldNum].DiskFileName);
        OldNum++;
    }
    
    if (deleteOK)
    {
        printf("Removing remaining %i temporary files\n", CurMax - CurNum);
        while (CurNum < CurMax)
        {
            char* temp = strdup(CurRecs[CurNum].DiskFileName);
            //makePathAbsolute(temp, CurRecs[CurNum].FolderName);
            printf("Deleting tempfile: %s, not in resumes database.\n", temp);
            unlink(temp); // could delete other files.. but probably bad.
            delete(temp);
            CurNum++;
        }
    }
    
    printf("%i resumes were found to exist already in the database\n", inBoth);
    printf("%i resumes were removed from the database\n", OldNum - inBoth);
    printf("%i resumes were found not in the database\n", CurNum - inBoth);
    
    if (OldNum < OldMax && CurNum < CurMax)
        printf("*********SOMETHING REALLY STRANGE WITH FILELISTS*******\n");
    
    delete(Old);
    delete(Current);
    
    return(Combined);
}



