/*
    ParseMessage.cpp
    OpenAG, libOpenAG, OpenAG X
 
    Created by Eric Seidel on Fri Nov 02 2001.
 
    Copyright (c) 2001-2002 Eric Seidel. All rights reserved.
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.h"

#include <errno.h>
#include <signal.h>
#include <sys/wait.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "error.h"
#include <fcntl.h>	/* for File control Functions */

#include "ParseMessage.h"
#include "AGInit.h"

void printConnection(connection* c);

void parseMessage(server_status &STATS, AGMessage* &TheMessage)
{
    int MessageType = TheMessage->getType();
    
    switch( MessageType )
    {
     //case LOGIN:			//never received
     //case LOGOFF:			//never received
     //case READY_FOR_TRANSFER:		//never received
     //case FILE_TRANSFER_STATE:	//never received
     //case READY_TO_SEND_SHARES:	//never received
     case SEND_SHARES_STAND_BY:
        {
            if (STATS.setDisplayedServerStatus != NULL)
                STATS.setDisplayedServerStatus("Connected");
            
            
            #if AGDEBUGLEVEL > 2
            printf("****RECEIVED SEND_SHARES_STAND_BY****\n sending READY_TO_SEND_SHARES\n\n");
            #endif
            //simply reply with #7
            AGMessage* ReadyToSendShares = new AGMessage(STATS.connections[INFOSERVER].socket);
            ReadyToSendShares->setType(READY_TO_SEND_SHARES);
            ReadyToSendShares->queueMessage(INFOSERVER, STATS);
            delete(ReadyToSendShares);
            break;
        }
     case SEND_SHARES:
        {
            #if AGDEBUGLEVEL > 3
            printf("****RECEIVED SEND_SHARES****\n\n");
            #endif
            
            printf("Sending share list to server...\n");
            if (STATS.setDisplayedServerStatus != NULL)
                STATS.setDisplayedServerStatus("Sending shares...");
        #ifdef SEND_NO_SHARES
            AGMessage* HaveNoShares = new AGMessage(STATS.connections[INFOSERVER].socket);
            HaveNoShares->setType(READY_TO_SEND_SHARES);
            HaveNoShares->queueMessage();
            delete(HaveNoShares);
            
        #else

            if ( (STATS.Shares == NULL) || (STATS.Shares->howManyFiles() == 0) )
            {
                #if AGDEBUGLEVEL > 3
                printf("sending NO shares\n");
                #endif
                
                AGMessage* HaveNoShares = new AGMessage(STATS.connections[INFOSERVER].socket);
                HaveNoShares->setType(READY_TO_SEND_SHARES);
                HaveNoShares->queueMessage(INFOSERVER, STATS);
                delete(HaveNoShares);
            }
            else
            {
                /* Handle old shares first */
                
                int oldOnes = 0;
                int newOnes = 0;
                
                #if AGDEBUGLEVEL > 3
                printf("about to send Old shares\n");
                #endif
                
                int howMany = STATS.Shares->howManyFiles();
                for(int x = 0; x < howMany; x++)
                {
                    FileRecord* sh = STATS.Shares->getFileRecordByLocalID(x);
                    
                    if (sh == NULL)
                    {
                        err_print("not sending share for id #%i, database lookup returned null\n", x);
                        continue;
                    }
                    
                    if (sh->SongID > 1)
                    {
                        #if AGDEBUGLEVEL > 3
                        printf("sending old share #%i, filename: %s strlen: %li\n", x, sh->DiskFileName, strlen(sh->DiskFileName));
                        #endif
                        /*
                            Local ID?? = (int) 0 (4 bytes) 00 00 00 00
                            FileSet? = (string) "0" (2 bytes + 1 ) 00 01 30
                            FileID? = (int) 0 (4 bytes) 00 00 00 00
                            SongID? = (int) 8731 (4 bytes) 00 00 22 1b
                            FileSize = (int) 4974507 (4 bytes)
                            KB/S = (short) 173 (4 bytes)
                        */
                        
                        // FIX -- not really quite correct...
                        
                        AGMessage* OldShare = new AGMessage(STATS.connections[INFOSERVER].socket);
                        OldShare->setType(OLD_SHARE);
                        OldShare->write_int(sh->LocalID);
                        
                        OldShare->write_string(sh->FileSet); // need to convert?  nah.
                        //OldShare->write_string("0");
                        
                        //OldShare->write_int(sh.ServerID);  -- writing zero's instead.
                        OldShare->write_zeros(4);
                        
                        OldShare->write_int(sh->SongID);
                        
                        OldShare->write_int(sh->FileSize);
                        
                        OldShare->write_short(sh->Bitrate);
                        
                        //OldShare->write_int(sh last mod date? );
                        if (OldShare->sendMessage() < 0)
                        {
                            err_print("Failed to send an old share.  Likely the connection dropped... resetting...\n");
                            delete(OldShare);
                            //delete(TheMessage); //will be done outside.
                            clientInit(STATS); // hope this works.
                            return;
                        }
                        delete(OldShare);
                        
                        oldOnes++;
                    }
                }
                
                /* Then handle new shares */
                #if AGDEBUGLEVEL > 3
                printf("about to send New shares\n");
                #endif
                
                howMany = STATS.Shares->howManyFiles();
                for(int x = 0; x < howMany; x++)
                {
                    FileRecord* sh = STATS.Shares->getFileRecordByLocalID(x);
                    
                    if (sh == NULL) continue; // skip if that one was empty
                    
                    // When adding old shares, remove this. -- done.
                    if (sh->SongID < 2)
                    {
                        #if AGDEBUGLEVEL > 3
                        printf("sending new share #%i, diskfilename: %s strlen: %li\n", x,
                            sh->DiskFileName, strlen(sh->DiskFileName));
                        #endif
                        
                        AGMessage* NewShare = new AGMessage(STATS.connections[INFOSERVER].socket);
                        NewShare->setType(NEW_SHARE);
                        NewShare->write_int(sh->LocalID);
                        
                        unsigned char* newBuff = NULL;
                        string_localToServer(newBuff, (unsigned char*)sh->DiskFileName, sizeof(sh->DiskFileName));
                        NewShare->write_string((char*)newBuff); // do I need to call basename?
                        delete(newBuff);
                        
                        NewShare->write_string("shares");//sh->DirectoryName);
                        NewShare->write_int(sh->FileSize);
                        NewShare->write_short(sh->Bitrate);
                        NewShare->write_int(sh->SongLength);
                        //NewShare->write_int(sh->FileLastModified);
                        NewShare->write_byte(sh->HasID3v1);
                        if (sh->HasID3v1)
                        {
                            NewShare->write_string(sh->ID3v1_Title);
                            NewShare->write_string(sh->ID3v1_Artist);
                            NewShare->write_string(sh->ID3v1_Album);
                        }
                        NewShare->write_byte(sh->HasID3v2);
                        if (sh->HasID3v2)
                        {
                            NewShare->write_string(sh->ID3v2_Title);
                            NewShare->write_string(sh->ID3v2_Artist);
                            NewShare->write_string(sh->ID3v2_Album);
                            NewShare->write_short(sh->ID3v2_Track);
                            NewShare->write_string(sh->ID3v2_Tag);
                        }
                        
                        //NewShare->dumpHex();
                        
                        if (NewShare->sendMessage() < 0)
                        {
                            err_print("Failed to send a new share.  Likely the connection dropped... resetting...\n");
                            delete(NewShare);
                            //delete(TheMessage); //will be done outside.
                            clientInit(STATS); // hope this works.
                            return;
                        }
                        delete(NewShare);
                        
                        newOnes++;
                    }
                }
                
                printf("Sent %i total shares, %i old, %i new.\n", oldOnes + newOnes, oldOnes, newOnes);
            }
        #endif
        
        printf("Share list sent, waiting for share responses...\n");
        if (STATS.setDisplayedServerStatus != NULL)
                STATS.setDisplayedServerStatus("Shares sent, receiving replies...");
        
            /* DONE SENDING SHARES, SEND KEEP ALIVE MESSAGES NOW */
            STATS.sendKeepAlive = true;
            
            break;
        }
     case FILE_TRANSFER:
        { //open FILE_TRANSFER:
            /************ PARSE MESSAGE *************/
            #if AGDEBUGLEVEL > 1
            printf("\n******** RECEIVED FILE_TRANSFER MESSAGE ******\n");
            #endif
            
            int newConnection = getNewConnectionRecord(STATS);
            if (newConnection == -1)
            {
                /* FIX - HANDLE THIS */
                sendFileTransferState(newConnection,STATS,FTS_INTERNAL_ERROR);
                deleteConnectionRecord(newConnection,STATS);
                err_print("I've reached the maximum number of connections allowed.  PLEASE REPORT SO I CAN FIX THIS.\n");
                return;
            }
            
            STATS.connections[newConnection].FileIDLength	= TheMessage->read_short();
            TheMessage->read_string(STATS.connections[newConnection].FileID,
                                    STATS.connections[newConnection].FileIDLength);
            STATS.connections[newConnection].ConnectFlag	= TheMessage->read_byte();
            STATS.connections[newConnection].DirectionFlag	= TheMessage->read_byte();
            int ServerIPLength = 
                TheMessage->read_string(STATS.connections[newConnection].ServerIP, DO_TERMINATE);
            STATS.connections[newConnection].ServerPort		= TheMessage->read_int();
            STATS.connections[newConnection].LocalID		= TheMessage->read_int();
            
            int FileNameLength = 
                TheMessage->read_string(STATS.connections[newConnection].FileName, DO_TERMINATE);
            
            unsigned char* newBuff = NULL;
            // Convert the string to our local format (for printing and file system operations)
            FileNameLength = string_serverToLocal(newBuff, (unsigned char*)STATS.connections[newConnection].FileName, FileNameLength);
            delete(STATS.connections[newConnection].FileName);
            STATS.connections[newConnection].FileName = (char*)newBuff;
                                    
            STATS.connections[newConnection].FileSize		= TheMessage->read_int();
            
            #if AGDEBUGLEVEL > 1
            if (STATS.connections[newConnection].DirectionFlag == DIRECTION_SEND)
                printf("requesting new outgoing on: %i, filename: %s\n", newConnection,
                    STATS.connections[newConnection].FileName);
            else
                printf("requesting new incoming on: %i, filename: %s\n", newConnection,
                    STATS.connections[newConnection].FileName);
            #endif
            

            /* // this does not belong here... we set up now, complain later.
            if (STATS.connections[newConnection].DirectionFlag == DIRECTION_SEND)
            {
                FileRecord* temp;
                if ( ( temp = STATS.Shares->getFileRecordByLocalID(STATS.connections[newConnection].LocalID)
                {
                    err_print("We received a request for a share:%s which we don't have in the database.  Reporting \"Internal Error\" to the server.\n", STATS.connections[newConnection].FileName);
                    sendFileTransferState(newConnection, STATS, FTS_INTERNAL_ERROR);
                    deleteConnectionRecord(newConnection, STATS);
                    return;
                }
                #if AGDEBUGLEVEL > 3
                else
                    printf("found share, setting up to send.\n");
                #endif
            }
            */
            
            // handle resume stuff when we open the file.
            
            #if AGDEBUGLEVEL > 2
            printConnection(&(STATS.connections[newConnection]));
            #endif
            
            /************ HANDLE CONNECTION MAKING ... *************/
            
            if (STATS.connections[newConnection].ConnectFlag == CONNECT_CONNECT)  /*****WE'RE CONNECTING****/
            {
                #if AGDEBUGLEVEL > 3
                printf("\nAttempting Connection to %s port: %i on connection: %i\n",
                    STATS.connections[newConnection].ServerIP, STATS.connections[newConnection].ServerPort,
                    newConnection);
                #endif
                struct sockaddr_in PeerAddress;
                                
                bzero(&PeerAddress,sizeof(PeerAddress));
                
                PeerAddress.sin_addr.s_addr = inet_addr(STATS.connections[newConnection].ServerIP);
                if (PeerAddress.sin_addr.s_addr < 0)
                {
                    sendFileTransferState(newConnection,STATS,FTS_COULD_NOT_CONNECT); // FIX, is this correct?
                    deleteConnectionRecord(newConnection,STATS);
                    err_print("Error converting peer server IP Address, I must have received bad packet data.\n");
                    return;
                }
                
                PeerAddress.sin_family	= AF_INET;
                PeerAddress.sin_port	= htons((u_short)STATS.connections[newConnection].ServerPort);
                
                if ( (STATS.connections[newConnection].socket = socket(AF_INET,SOCK_STREAM,0)) < 0)     /* get socket */
                {
                    sendFileTransferState(newConnection,STATS, FTS_INTERNAL_ERROR); // correct error?
                    deleteConnectionRecord(newConnection,STATS);
                    err_print("I couldn't get a socket for network connection.\n");
                }
                #if AGDEBUGLEVEL > 2
                else printf("got socket: %i\n", STATS.connections[newConnection].socket);
                #endif
                
                /*SET CONNECTION NONBLOCKING */
                int f = fcntl(STATS.connections[newConnection].socket, F_GETFL,0);
                fcntl(STATS.connections[newConnection].socket, F_SETFL, f | O_NONBLOCK);
                
                /* HANDLE UPDATING MAXFD to include the new socket.*/
                if (STATS.connections[newConnection].socket > STATS.maxfd)
                    STATS.maxfd = STATS.connections[newConnection].socket;
                    
                #if AGDEBUGLEVEL > 5
                printf("maxfd: %li\n", STATS.maxfd);
                #endif
                
                
                if (connect(STATS.connections[newConnection].socket,
                            (struct sockaddr *)&PeerAddress,
                            sizeof(PeerAddress)) < 0)
                {
                    // errno is set.
                    handlePeerConnectingError(newConnection,STATS);
                    break;
                }
                else
                {
                    #if AGDEBUGLEVEL > 3
                    printf("Immeadiate Connection Completed.  Telling server -- We're ready\n");
                    #endif
                    
                    sendFileTransferState(newConnection, STATS, FTS_SETTING_UP);
                    
                    /* CONNECTED, HANDLE */
                    handleNewConnection(newConnection, STATS);
                }
            }
            
            else /*****WE'RE LISTENING****/
            {
                struct sockaddr_in ListenAddress;
                int ListeningSocket;
                #if AGDEBUGLEVEL > 2
                printf("opening listening port %i\n",STATS.connections[newConnection].ServerPort);
                #endif
                bzero(&ListenAddress, sizeof(ListenAddress));
                ListenAddress.sin_addr.s_addr = htonl(INADDR_ANY);  /* set server to any address */
                ListenAddress.sin_family = AF_INET;
                ListenAddress.sin_port = htons(STATS.connections[newConnection].ServerPort); //short?
                
                if ( ( ListeningSocket = socket(AF_INET,SOCK_STREAM,0)) < 0)     /* get socket */
                {
                    sendFileTransferState(newConnection,STATS, FTS_INTERNAL_ERROR);
                    deleteConnectionRecord(newConnection,STATS);
                    err_print("I was not able to get a socket!!!\n");
                }
                
                if (bind(ListeningSocket, (sockaddr *)&ListenAddress, sizeof(ListenAddress)) != 0)
                {
                    #if AGDEBUGLEVEL > 1
                    err_print("failed to bind to port: %i, notifying server\n", STATS.connections[newConnection].ServerPort);
                    #endif
                    sendFileTransferState(newConnection, STATS, FTS_PORT_NOT_AVAILABLE);
                    deleteConnectionRecord(newConnection, STATS);
                    break;
                }
                
                if (listen(ListeningSocket, 5) != 0) /* queue is 5, but doesn't really need to. */
                {
                    err_print("failed to listen on port: %i, notifying server\n",
                                STATS.connections[newConnection].ServerPort);
                    // FIX -- Do we send this after a FTS_SETTING_UP?
                    sendFileTransferState(newConnection, STATS, FTS_PORT_NOT_AVAILABLE);
                    deleteConnectionRecord(newConnection, STATS);
                    break;
                }
                
                STATS.connections[newConnection].socket = ListeningSocket;
                STATS.connections[newConnection].flags = ACCEPTING;
                
                /* HANDLE UPDATING MAXFD to include the new socket.*/
                if (STATS.connections[newConnection].socket > STATS.maxfd)
                    STATS.maxfd = STATS.connections[newConnection].socket;
                
                #if AGDEBUGLEVEL > 5
                printf("maxfd: %li\n", STATS.maxfd);
                #endif
                
                /* ADD THE SOCKET TO THE LISTENERS... */
                FD_SET(STATS.connections[newConnection].socket, &(STATS.rset));
                
                sendFileTransferState(newConnection, STATS, FTS_SETTING_UP); //ready and listening
            }
            
            
            /* FIX, REDUDNANT? */
            /* Correct STATS.maxfd with new socket */
            if (STATS.connections[newConnection].socket > STATS.maxfd)
                STATS.maxfd = STATS.connections[newConnection].socket;
            
            break;
        }
    case DELETE_REQUEST: /* Fix when we have resumable downloads! */
    {
        /* Stop the transfer corresponding to the given FileID and remove the file.*/
        
        unsigned short FileIDLength		= TheMessage->read_short();
        char* FileID = new char[FileIDLength];
        TheMessage->read_string(FileID,FileIDLength);
        char* FileName = NULL;
        short FileNameSize = TheMessage->read_string(FileName, DO_TERMINATE);
        
        unsigned char* newBuff = NULL;
        // Convert the string to our local format (for printing and file system operations)
        FileNameSize = string_serverToLocal(newBuff, (unsigned char*)FileName, FileNameSize);
        delete(FileName);
        FileName = (char*)newBuff;
        
        #if AGDEBUGLEVEL > 2
        printf("****RECEIVED MESSAGE TO STOP and DELETE FILE: %s*****\n", FileName);
        #endif
        
        #if AGDEBUGLEVEL > 4
        TheMessage->dumpHex();
        printf("FileIDLength: %i\n", FileIDLength);
        printf("FileName: %s\n", FileName);
        printf("currently %i connections.\n", STATS.nConnections);
        #endif
        
        int i;
        if ( (i = findConnectionRecordWithFileID(FileID, FileIDLength, STATS) ) != -1)
        {
            #if AGDEBUGLEVEL > 1
            printf("Closing Connection and deleting: %i, Filename: %s\n", i,
                    STATS.connections[i].FileName);
            #endif
            
            sendDeleteResponse(i, STATS);
            //FIX - I would actually have to delete the file!!! if we start doing resumes!
            deleteConnectionRecord(i,STATS);
        }
        #if AGDEBUGLEVEL > 1
        else
            printf("**CONNECTION AND THAT FILE NOT FOUND, bad.\n");
        #endif
        
        /* FIX  */
        delete(FileID);
        delete(FileName);
        break;
    }
    
    case STOP_FILE_TRANSFER:
        {
            /* Stop the transfer corresponding to the given FileID */
            
            unsigned short FileIDLength		= TheMessage->read_short();
            char* FileID = new char[FileIDLength];
            TheMessage->read_string(FileID,FileIDLength);
            
            #if AGDEBUGLEVEL > 2
            printf("****RECEIVED MESSAGE TO STOP TRANSFER*****\n");
            #endif
            
            #if AGDEBUGLEVEL > 4
            TheMessage->dumpHex();
            printf("FileIDLength: %i\n", FileIDLength);
            printf("currently %i connections.\n", STATS.nConnections);
            #endif
            
            int i;
            if ( (i = findConnectionRecordWithFileID(FileID, FileIDLength, STATS) ) != -1)
            {
                #if AGDEBUGLEVEL > 1
                printf("Closing Connection: %i, Filename: %s\n", i,
                        STATS.connections[i].FileName);
                #endif
                
                sendFileTransferState(i, STATS, FTS_STOPPED);
                // takes care of all deleting...
                deleteConnectionRecord(i,STATS);
            }
            else
            {
                #if AGDEBUGLEVEL > 0
                printf("CONNECTION FOR THAT FILE NOT FOUND, bad, telling them we stopped it anyway.\n");
                TheMessage->dumpHex();
                #endif
                AGMessage* weStopped = new AGMessage(STATS.connections[INFOSERVER].socket);
                weStopped->setType(FILE_TRANSFER_STATE);
                weStopped->write_string(FileID, FileIDLength, true); /// include length.
                weStopped->write_int(FTS_STOPPED);
                weStopped->queueMessage(INFOSERVER,STATS);
                delete(weStopped);
            }
            
            /* FIX  */
            delete(FileID);
            break;
        }
     case NEW_SHARE_RESPONSE:
        {
            #if AGDEBUGLEVEL > 2
            printf("****RECEIVED NEW_SHARE_RESPONSE *****\n\n");
            #endif
            
            int LocalID			= TheMessage->read_int();
            char* FileSet = NULL; //allocate for me
            int FileSetLength = TheMessage->read_string(FileSet, DO_TERMINATE); //need to read the length first?
            int ServerID		= TheMessage->read_int();
            int SongID			= TheMessage->read_int();
            int FileSize		= TheMessage->read_int();
            
            #if AGDEBUGLEVEL > 3
            printf("received share response with songID: %i for local ID: %i\n",SongID, LocalID);
            #endif
            
            #if AGDEBUGLEVEL > 5
            TheMessage->dumpHex();
            printf("dumped.\n");
            #endif
            
            #if AGDEBUGLEVEL > 4
            //debug:
            printf("LocalID: %i\n", LocalID);
            printf("FileSet: %s\n", FileSet);
            printf("ServerID: %i\n", ServerID);
            printf("SongID: %i\n", SongID);
            printf("FileSize: %i\n", FileSize);
            
            printf("recording SongID in FileList\n");
            #endif
            
            #if AGDEBUGLEVEL > 3
            printf("server returned ID of %i, songID: %i for share #%i\n", ServerID, SongID, LocalID);
            #endif
            
            /*
            #if AGDEBUGLEVEL > 0
            if (SongID == 1)
                printf("Server returned SongID of 1 for Local share # %i.  That is an error message, meaning the server did not like our new_share message.  This is a problem with the current known protocol specification, and is being addressed by the  Open Audiogalaxy Protocol Developer community.  You may see this error several times.\n", LocalID);
            #endif
            */
            
            if ( STATS.Shares->setSongID(LocalID, SongID) < 0 )
                err_print("Error setting ServerID: %i, for LocalID: %i.  Share will not be marked as validated.", ServerID, FileSet);
                
            if (ServerID != 0)
                printf("Wow, I got a ServerID, not 0, but rather: %i\n", ServerID);
            
            if (STATS.Shares->setFileSet(LocalID, FileSet) < 0)
                err_print("Error setting FileSet: %s, for LocalID: %i.  Share will not be marked as validated.", FileSet, LocalID);
            
            delete(FileSet);
            break;
        }
     case WRONG_LOGIN_MESSAGE:
        {
            
            int ErrorType		= TheMessage->read_int();
            
            #if AGDEBUGLEVEL > 3
            TheMessage->dumpHex();
            #endif
            switch(ErrorType)
            {
            case 0:
                delete(TheMessage);
                err_exit("Oops, server responded: \"Incorrect Password.\" I sent Username: %s Password: %s Please check that these are right and correct them in your preferences (or user.txt file).\n\n", STATS.Prefs->EmailAddress, STATS.Prefs->Password);
                break;
            case 1:
                delete(TheMessage);
                err_exit("Oops, server responded: \"Unknown username.\" I sent Username: %s Password: %s  Did you remember to register first at audioglaxy.com?  Otherwise please check that the above are right and correct them in your preferences (or user.txt file).\nYou might also try entering your Email instead of your username, or your username instead of your email.  Sometimes Audiogalaxy servers just get \"moody.\"\n", STATS.Prefs->EmailAddress, STATS.Prefs->Password);
                break;
            case 2:
                delete(TheMessage);
                err_exit("Oops, server responded: \"Bad client version\"\nI sent client version: %s\n
                        Please check %s for updates.", VERSION_STRING, AG_OPENAG_DOWNLOADPAGE);
                break;
            case 3:
                delete(TheMessage);
                err_exit("Oops, looks like you just logged in with the same name somewhere else.  You have been disconnected here.\n");
                break;
            case 28187388:
                delete(TheMessage);
                err_exit("Oops, server responded with error number: %i which is believed to be a \"Non-valid gold acount\" error.\nI sent client version: %s with Username: %s Password: %s\n Please check to make sure that \"This is a Gold account\" is checked in the preferences ONLY IF you actually have a gold acocun. ", VERSION_STRING, STATS.Prefs->EmailAddress, STATS.Prefs->Password);
                break;
                
            default:
                delete(TheMessage);
                err_exit("Got unknown WRONG LOGIN ERROR of type: %i\nI sent Username/Email: %s, Password: %s, Protocol Version: %s\n", ErrorType, STATS.Prefs->EmailAddress, STATS.Prefs->Password, VERSION_STRING);
                break;
            }
            break; // not needed.
        }
     //case EX_ACCOUNT_INFORMATION:	//DON'T KNOW
     //case OLD_SHARES:			//Never received
     //case DATA_TO_UPDATE_SHARES:	//never received
     //case NEW_ACCOUNT_INFORMATION:	//never received
     case NEW_ACCOUNT_INFORMATION_RESPONSE:
     case EX_ACCOUNT_INFORMATION_RESPONSE:
     //case DB_UPDATE_WITH_FILE_ID:
     case PLAY_A_FILE_IN_LOCAL_DB:
     case LOGIN_HASH_PREFIX:
     default:
        {
            err_print("*****RECEIVED UNHANDLED MESSAGE OF TYPE: %i **********\n\n", MessageType);
            //notDone = false;
            
            TheMessage->dumpHex();
        }
    }
    #if AGDEBUGLEVEL > 4
    printf("about to return from parse message\n");
    #endif
}

bool lookupLocalFileID(char* FileID, short FileIDLength, int FileSize)
{
    return false;
}   

void printConnection(connection* c)
{
    //printf("FileID: %s\n", c->FileID);
    printf("FileIDLength: %i\n", c->FileIDLength);
    printf("ConnectFlag: %i\n", c->ConnectFlag);
    printf("DirectionFlag: %i\n", c->DirectionFlag);
    printf("ServerIP: %s\n", c->ServerIP);
    printf("ServerPort: %i\n", c->ServerPort);
    printf("Song Length?: %i\n",c->SongLength); // not certian.
    printf("FileName: %s\n", c->FileName);
    printf("FileSize: %i\n", c->FileSize);
    /* there is more... */
}


void handlePeerConnectingError(int newConnection, server_status &STATS)
{
    if (errno == EINPROGRESS)
    {
        #if AGDEBUGLEVEL > 3
        printf("CONNECTION STILL IN PROGRESS.\n");
        #endif
        STATS.connections[newConnection].flags = CONNECTING;
        FD_SET(STATS.connections[newConnection].socket, &(STATS.rset));
        FD_SET(STATS.connections[newConnection].socket, &(STATS.wset));
    }
    else if (errno == EWOULDBLOCK)
    {
        #if AGDEBUGLEVEL > 3
        printf("CONNECTION STILL IN PROGRESS.\n");
        #endif
        STATS.connections[newConnection].flags = CONNECTING;
        FD_SET(STATS.connections[newConnection].socket, &(STATS.rset));
        FD_SET(STATS.connections[newConnection].socket, &(STATS.wset));
    }
    else
    {
        #if AGDEBUGLEVEL > 3
        err_print("CLOSING CONNECTION, ERROR -- notifying server\n");
        #endif
        if (errno == ECONNRESET) // connection reset by peer
            sendFileTransferState(newConnection, STATS, FTS_CONNECTION_CLOSED); //reset by peer
        else if (errno == ECONNREFUSED) // connection refused.
            sendFileTransferState(newConnection, STATS, FTS_CONNECTION_CLOSED); //FIX -- connection refused
        else if (errno == EHOSTUNREACH)
            sendFileTransferState(newConnection, STATS, FTS_CONNECTION_CLOSED); //FIX - host unreachable...
        else if (errno == ETIMEDOUT)
            sendFileTransferState(newConnection, STATS, FTS_CONNECTION_CLOSED); //FIX - operation timed out.
        else
        {
            err_print("I'm sorry, I received a strange error while connecting to a peer.\n");
            sendFileTransferState(newConnection, STATS, FTS_CONNECTION_CLOSED); // fix.
        }
        deleteConnectionRecord(newConnection, STATS); // closing handled within.
    }
}

int findConnectionRecordWithFileID(char* FileID, int FileIDLength, server_status &STATS)
{
    int cmp;
    for(int i = 1; i < STATS.nConnections; i++)
    {
        #if AGDEBUGLEVEL > 5
        printf("top of for in stop connection: %i\n", i);
        #endif
        if (STATS.connections[i].flags & TRANSFER_SENDING || STATS.connections[i].flags & TRANSFER_RECEIVING)
        {
            #if AGDEBUGLEVEL > 3
            if (STATS.connections[i].FileID == NULL) printf("it's NULL\n");
            if (STATS.connections[i].FileID == 0) printf("it's 0\n");
            #endif
            
            if ( (FileIDLength == STATS.connections[i].FileIDLength) &&
                (cmp = memcmp((char*)STATS.connections[i].FileID,(char*)FileID, FileIDLength) == 0) )
                return i;
                
            #if AGDEBUGLEVEL > 4
            else
                printf("the FileID's differ by %i\n", cmp);
            #endif
        }

    }
    return (-1);
}

void sendDeleteResponse(int i, server_status &STATS)
{
    #if AGDEBUGLEVEL > 2
    printf("SENDING FILE DELETE RESPONSE: %s\n",STATS.connections[i].FileName);
    #endif
    
    /* Send FileTransferState Message */
    AGMessage* DeleteResponse = new AGMessage(STATS.connections[INFOSERVER].socket);
    DeleteResponse->setType(DELETE_RESPONSE);
    DeleteResponse->write_string((char *)STATS.connections[i].FileID,
        STATS.connections[i].FileIDLength, true);
    DeleteResponse->write_int(1); // FIX - allways successfull.
    DeleteResponse->queueMessage(INFOSERVER, STATS);//send it.
    delete(DeleteResponse);
    
    #if AGDEBUGLEVEL > 4
    printf("done sendingDeleteResponse\n");
    #endif
}


