#include "WormApp.hh"
#include "XVision/Menu.hh"
#include "PaintInterface/TargaPicture.hh"
#include "PlayerSelector.hh"
#include "Game/Players.hh"
#include "Network/NetConnector.hh"
#include "Network/NetworkTimings.hh"
#include "Compat.hh"
#include "Utils.hh"
#include "PlayerSelectorItem.hh"
#include "ScoreTable.hh"
#include "PathConfig.hh"

#define WormzName "Wormz"
#define WormzVersion "1.0 beta 4"

const char * text_connected = "Connected";
const char * text_disconnected = "Not connected";
const Pixel_t color_connected = Pixel_t_Green;
const Pixel_t color_disconnected = Pixel_t_Red;

const int max_hostname_len = 30;
const char * default_server_hostname = "localhost"; // must be <= max_hostname_len
const char * server_path = ServerBinInstallPath;

const int min_rounds = 1;
const int max_rounds = 999;
const int default_rounds = 10;

class WormMenu : public Menu {
  protected:
    StaticText * title;
    const char * text;
  public:
    WormMenu(StaticText * _title, const char * _text, const Align _align=VAVertical);
    void Show();
};

WormMenu::WormMenu(StaticText * _title, const char * _text, const Align _align=VAVertical)
    : Menu(_align), title(_title), text(_text)
{
}

void WormMenu::Show()
{
  if (title)
    title->SetText(text);
  Menu::Show();
}

WormApp::WormApp(FrontBuffer * _front, FrameBuffer * _fb, FrameBuffer * _scorepanel,
		 EventManager * _evman, Players * _players, NetConnector * _network)
    : Application(_front, _fb, _evman, VAHorizontal), network(_network),
  players(_players),
  gamemenu(0), startmenu(0), mainmenu(0),
  JustStartedRound(FALSE),
  rounds(default_rounds), current_round(0),
  scorepanel(_scorepanel), LastFPSW(1), LastFPSH(1), ScorePanelVisible(TRUE),
  AlarmCounter(0), MenuVisible(TRUE), LastMsgW(1), LastMsgH(1),
  ShowFPS(FALSE), PlayingGame(FALSE), ReallyPlaying(FALSE),
  currentMaxFPS(init_FPS), WantQuit(FALSE)
{
}

void WormApp::Initialize()
{
  Label * label = new Label(" ", cmRootMenu, TRUE, tdWest, fsSmallRegular);
  label->SetColor(RGB(0, 0x80, 0), ciBackground);
  label->SetColor(RGB(0, 0xc0, 0), ciHighlightedBG);

  connected_label = new StaticText(text_disconnected, tdEast, fsSmallRegular);
  connected_label->SetColor(color_disconnected, ciBackground);
  
  Group * gamesetup_sm =
    CreateGroup(new WormMenu(label, "Game setup menu"),
      CreateItem(new Title("Players"),
      CreateItem((player_selector_group = new PlayerSelector(players)),
      0)));
  
  Group * options_sm =
    CreateGroup(new WormMenu(label, "Options menu"),
      CreateItem((option_surround_label = new CheckBox("Surround", TRUE)),
      CreateItem((option_show_score_label = new CheckBox("Show score panel", TRUE)),
      CreateItem((option_show_fps_label = new CheckBox("Show FPS", FALSE)),
      CreateItem(
        CreateGroup(new Group(VAHorizontal),
          CreateItem(new StaticText("Rounds:"),
          CreateItem(new Padding(VAHorizontal),		     
	  CreateItem((rounds_label = new InputNumber(default_rounds, min_rounds, max_rounds)),
          0)))),
      CreateItem(
        CreateGroup(new Group(VAHorizontal),
          CreateItem(new StaticText("Speed:"),
          CreateItem(new Padding(VAHorizontal),		     
	  CreateItem((speed_label = new InputNumber(init_FPS, min_FPS, max_FPS)),
          0)))),
      CreateItem(
        CreateGroup(new Group(VAHorizontal),
          CreateItem(new StaticText("Worm step size:"),
          CreateItem(new Padding(VAHorizontal),		     
	  CreateItem((worm_step_size_label = new InputNumber(Worm::StepSize, Worm::MinStepSize, Worm::MaxStepSize)),
          0)))),
      CreateItem(
        CreateGroup(new Group(VAHorizontal),
          CreateItem(new StaticText("Worm turn size:"),
          CreateItem(new Padding(VAHorizontal),		     
	  CreateItem((worm_turn_size_label = new InputNumber(Worm::TurnSize, Worm::MinTurnSize, Worm::MaxTurnSize)),
          0)))),		 
      0))))))));
  
  Group * gamemain_sm =
    CreateGroup(new WormMenu(label, "Game menu"),
      CreateItem(new MenuItem("Continue", cmContinueGame),
      CreateItem(new Submenu("Re-setup game", gamesetup_sm),
      CreateItem(new MenuItem("End game", cmEndGame),
      CreateItem(new Submenu("Game options", options_sm),
      0)))));
  gamemain_sm->SetHidden(TRUE);
  gamemenu = (Menu *) gamemain_sm;
  
  Group * start_sm =
    CreateGroup(new WormMenu(label, "Start menu"),
      CreateItem(new StaticText("Waiting for other players to setup game"),
      CreateItem(new Label("Hang up", cmEndGame),
      0)));
  start_sm->SetHidden(TRUE);
  startmenu = (Menu *) start_sm;
  
  Group * network_sm =
    CreateGroup(new WormMenu(label, "Network setup"),
      CreateItem(
        CreateGroup(new Group(VAHorizontal),
          CreateItem(new StaticText("Server hostname:"),
          CreateItem(new Padding(VAHorizontal),		     
	  CreateItem((server_hostname_label = new InputLine(default_server_hostname, max_hostname_len)),
          0)))),
      CreateItem(
        CreateGroup(new Group(VAHorizontal),
          CreateItem(new StaticText("Server port:"),
          CreateItem(new Padding(VAHorizontal),		     
	  CreateItem((server_port_label = new InputNumber(ServerDefaultPort, 3000, 65535)),
          0)))),
      CreateItem(
	  CreateGroup(new Group(VAHorizontal),
          CreateItem(new Label("Create local server", cmCreateServer),		      
          CreateItem(new Padding(VAHorizontal),
          CreateItem(new Label("Connect", cmConnect),
          CreateItem(new Padding(VAHorizontal),		     
          CreateItem(new Label("Disconnect", cmDisconnect),
      0)))))),
  0))));
  CreateServer(TRUE);
  
  Group * about_sm =
    CreateGroup(new WormMenu(label, "About Wormz"),
       CreateItem(new Label(WormzName " version " WormzVersion, cmAboutClick, TRUE, tdNorth, fsOblique),
       CreateItem(new StaticText("Copyright 1997 Jaromir Koutek", tdNorth, fsSmallRegular),
       CreateItem(new StaticText(WormzName " comes with ABSOLUTELY NO WARANTY", tdNorth, fsSmallRegular),
       CreateItem(new StaticText("This is free software, you can redistribute it and/or modify", tdNorth, fsSmallRegular),
       CreateItem(new StaticText("it under terms of the GNU General Public License as published by", tdNorth, fsSmallRegular),
       CreateItem(new StaticText("the Free Software Foundation; either version 2 of the License, or", tdNorth, fsSmallRegular),		     
       CreateItem(new StaticText("(at your opinion) any later version.", tdNorth, fsSmallRegular),		     		     
       CreateItem(new StaticText("See file COPYING for details.", tdNorth, fsSmallOblique),		     
      0)))))))));
  about_sm->SetHidden(TRUE);
  aboutmenu = (Menu *) about_sm;

  Group * score_sm =
    CreateGroup(new WormMenu(label, "High scores"),
       CreateItem(new StaticText("Game over!"),
       CreateItem((scoretable = new ScoreTable),	
    0)));
  score_sm->SetHidden(TRUE);
  scoremenu = (Menu *) score_sm;
  
  Group * m =
    CreateGroup(new WormMenu(label, "Main menu"),
      CreateItem(new Submenu("New game", gamesetup_sm),
      CreateItem(new Submenu("Setup network", network_sm),
      CreateItem(new Submenu("Game options", options_sm),
      CreateItem(new MenuItem("Quit", cmQuit),
    0)))));
  mainmenu = (Menu *) m;
  
  View * title = new PictureView(new TargaPicture(LibInstallPath "Title.tga"));
  title->SetAutoCenter();
  Group * g =
    CreateGroup(new Group,
      CreateItem(new Padding(VAVertical, 5),
      CreateItem(title,
      CreateItem(new Padding(VAVertical),
      CreateItem(m,
      CreateItem(gamesetup_sm,
      CreateItem(options_sm,
      CreateItem(gamemain_sm,
      CreateItem(network_sm,
      CreateItem(start_sm,
      CreateItem(about_sm,	
      CreateItem(score_sm,			 
      CreateItem(new Padding(VAVertical),
      CreateItem((chat_line_recv = new StaticText("", tdNorth, fsSmallRegular)),		 
      CreateItem((chat_line_send = new InputLine("", nmChat::MsgSize, cmSendChatLine, TRUE, tdNorth, fsSmallRegular)),
      CreateItem(new Padding(VAVertical),		 
    0))))))))))))))));
  chat_line_send->SetEnabled(FALSE);
   
  Group * g3 =
    CreateGroup(new Group(VAVertical),
      CreateItem(new Padding(VAVertical, 70),
      CreateItem(
	CreateGroup(new Group(VAHorizontal),
          CreateItem(label,
   	  CreateItem(new Padding(VAHorizontal, 30),
	  CreateItem(g,
	  CreateItem(new Label("About", cmShowAbout, FALSE, tdEast, fsSmallRegular),
	  CreateItem(new Padding(VAHorizontal, 30),
          CreateItem(connected_label,		     
        0))))))),
      CreateItem(new Padding(VAVertical, 70),
   0))));
  g3->SetColor(RGB(20, 20, 20), ciBackground);
  g3->SetColor(RGB(20, 20, 20), ciHighlightedBG);
  SetColor(RGB(20, 20, 20), ciBackground);
  SetColor(RGB(20, 20, 20), ciHighlightedBG);
  Insert(new Padding(VAHorizontal, 40));
  Insert(g3);
  Insert(new Padding(VAHorizontal, 40));
  Application::Initialize();
}

void WormApp::SwitchMenu(bool Flip=TRUE, bool DoShow=TRUE)
{
  if (!Flip && ((Visible && DoShow) || (!Visible && !DoShow)))
    return;
  if (Visible) {
    if (!PlayingGame)
      return;
    Hide();
    MenuVisible = FALSE;
    fb_save->Fill(0, 0, fb_save->WindowSizeX, fb_save->WindowSizeY, RGB(0, 0, 0));
    front->RemoveFrameBuffer(fb);
  } else {
    Show();
    MenuVisible = TRUE;
    gamemenu->ShowMenu();
    front->AddFrameBuffer(fb);
  }
}

bool WormApp::HandleEvent(Event& event)
{
  if (!Application::HandleEvent(event))
    return FALSE;
 if (event.type == evnCommand) {
   switch (event.command) {
   case cmQuit:
     Quit();
     return FALSE;
   case cmConnect:
     Connect();
     return FALSE;
   case cmDisconnect:
     Disconnect();
     return FALSE;
   case cmCreateServer:
     CreateServer(FALSE);
     return FALSE;
   case cmContinueGame:
     SwitchMenu(FALSE, FALSE);
     return FALSE;
   case cmHideMainMenu:
     SwitchMenu(FALSE, FALSE);
     return FALSE;
   case cmStartGame:
     StartGame(TRUE);
     return FALSE;
   case cmEndGame:
     EndGame(PlayingGame);
     return FALSE;
   case cmRootMenu:
     if (PlayingGame)
       gamemenu->ShowMenu();
     else
       mainmenu->ShowMenu();
     return FALSE;
   case cmShowAbout:
     aboutmenu->ShowMenu();
   case cmSlotCaptured:
     if (event.to == this) {
       SendPlayerCaptured(event.data, (PlayerSelectorItem *) event.creator);
       return FALSE;
     }
     break;
   case cmSendChatLine:
     SendChatLine();
     return FALSE;
   case cmDataChanged:
     if ((event.creator == worm_step_size_label) || (event.creator == worm_turn_size_label)
	 || (event.creator == option_surround_label) || (event.creator == rounds_label)
	 || (event.creator == speed_label)) {
       OptionsChanged(TRUE, 0);
       return FALSE;
     }
     if (event.creator == option_show_fps_label) {
       ShowFPS = option_show_fps_label->IsChecked();
       PaintFPS();
       return FALSE;
     }
     if (event.creator == option_show_score_label) {
       RepaintScorePanel(option_show_score_label->IsChecked());
       return FALSE;
     }
//     break;
   }
 }
 if ((event.type == evKeypress) && (event.key == ktEsc)) {
   SwitchMenu(TRUE);
   return FALSE;
 }
 return TRUE;
}

void WormApp::StartGame(bool OnlyRequest)
{
  if (PlayingGame)
    return;
  if (OnlyRequest && (network->NetworkGame())) {
    // request for starting game
    nmStartGame m(rounds);
    network->SendMsg(&m);
    SwitchMenu(FALSE, TRUE);
    startmenu->ShowMenu();
    return;
  }
  ReallyPlaying = FALSE;
  players->NewGame();
  current_round = 1;
  PlayingGame = TRUE;
  if (network->NetworkGame()) {
    nmStartRound m;
    network->SendMsg(&m);
  } else
    StartRound(TRUE);
  SwitchMenu(FALSE, FALSE);
}

void WormApp::EndGame(bool OnlyRequest)
{
  if (!PlayingGame)
    return;
  if (OnlyRequest && (network->NetworkGame())) {
    nmEndGame m;
    network->SendMsg(&m);
    return;
  }
  PlayingGame = FALSE;
  ReallyPlaying = FALSE;
  ShowScore();
}

void WormApp::EndRound(bool OnlyRequest)
{
  if (!PlayingGame || !ReallyPlaying)
    return;
  if (OnlyRequest && network->NetworkGame()) {
    nmEndRound nm;
    network->SendMsg(&nm);
    return;
  }
  char s[100];
  const char * sp = s;
  Worm * w = players->GetPlayer(players->winner_num);
  if (w)
    sprintf(s, "Player %s wins round %i!", w->GetName(), current_round);
  else
    sp = "No winner";
  TypeText(sp, Pixel_t_HiBlue, usec2sec, &EndRoundCallback1);
}

void WormApp::EndRoundCallback1()
{
  ReallyPlaying = FALSE;
  if ((++current_round) > rounds) {
    EndGame(TRUE);
    return;
  }
  StartRound(TRUE);  
}

void WormApp::StartRound(bool OnlyRequest)
{
  if (!PlayingGame || ReallyPlaying)
    return;
  if (OnlyRequest && network->NetworkGame()) {
    nmStartRound nm2;
    network->SendMsg(&nm2);
    return;
  }
  ReallyPlaying = TRUE;
  JustStartedRound = TRUE;
  players->NewRound();
  char s[40];
  sprintf(s, "Round %i of %i", current_round, rounds);
  TypeText(s, Pixel_t_Blue, usec2sec, &StartRoundCallback1);
}

void WormApp::StartRoundCallback1()
{
  players->DrawAll();
  TypeText("GET READY!", Pixel_t_Red, usec2sec, &StartRoundCallback2);
}

void WormApp::StartRoundCallback2()
{
  TypeText("GET READY!", Pixel_t_Green, usec2sec/10, 0);
}

void WormApp::Quit()
{
  //   delete this;
  Disconnect();
  microsleep(500000); // wait 0.5 s for sending Quit game message
  WantQuit = TRUE;
}

void WormApp::Disconnect()
{
  if (network->NetworkGame()) {
    network->Disconnect();
    for (int i=players->MAXPlayers-1; i>=0; i--) {
      Worm * w = players->GetPlayer(i);
      if (w && (w->GetType() == ptNetwork))
	players->AddPlayer(0, i); // delete all network players
    }
    chat_line_send->SetEnabled(FALSE);
    connected_label->SetText(text_disconnected);
    connected_label->SetColor(color_disconnected, ciBackground);
  }
  RecvChatLine(network->status);
}

void WormApp::Connect()
{
  network->connection->SetHostname(server_hostname_label->GetText());
  network->connection->SetPeerPort(server_port_label->GetNumber());
  if (network->Connect()) {
    for (int i=players->MAXPlayers-1; i>=0; i--)
	players->AddPlayer(0, i); // delete all players
    player_selector_group->DeleteAllCapturedKeys();
    connected_label->SetText(text_connected);
    connected_label->SetColor(color_connected, ciBackground);
    chat_line_send->SetEnabled(TRUE);
    mainmenu->ShowMenu();
  }
  RecvChatLine(network->status);
}

void WormApp::CreateServer(bool OnlySetHostname=FALSE)
{
  char s[max_hostname_len];
  strcpy(s, default_server_hostname);
  gethostname(s, max_hostname_len);
  server_hostname_label->SetText(s);
  if (OnlySetHostname)
    return;
  int port = server_port_label->GetNumber();
  pid_t ret = fork();
  if (ret == -1) {
    StdError("can't fork");
    return;
  }
  if (!ret) {
    // we need to close all i/o except 0, 1 and 2 (stdin, stdout, stderr)
    // 255 should be enough, I don't know where is limit for fd
    for (int i=3; i<255; i++)
      close(i);
    // now, exec the server
    char port_s[100];
    sprintf(port_s, "%i", port);
    execl(server_path, server_path, "-p", port_s, 0);
    StdError("can't exec");
    exit(0);
  }
  sleep(2); // wait for server to initialize
  Connect();
}

void WormApp::OptionsChanged(bool OnlyRequest, NetMessage * options)
{
  if (OnlyRequest && network->NetworkGame()) {
    nmOptions nm(
		 option_surround_label->IsChecked(),
		 speed_label->GetNumber(),
		 worm_step_size_label->GetNumber(),
		 worm_turn_size_label->GetNumber(),
		 rounds_label->GetNumber());
    network->SendMsg(&nm);
    return;
  }
  nmOptions * opt = (nmOptions *) options;
  if (network->NetworkGame()) {
    if (!opt)
      return; // ugh!
    option_surround_label->SetChecked(opt->Surround);
    rounds_label->SetNumber(opt->Rounds);
    speed_label->SetNumber(opt->Speed);
    worm_step_size_label->SetNumber(opt->WormStepSize);
    worm_turn_size_label->SetNumber(opt->WormTurnSize);
  }
  rounds = rounds_label->GetNumber();
  currentMaxFPS = speed_label->GetNumber();
  Worm::StepSize = worm_step_size_label->GetNumber();
  Worm::TurnSize = worm_turn_size_label->GetNumber();
  players->Surround(option_surround_label->IsChecked());
}

void WormApp::UpdateGame2()
{
  if (ReallyPlaying && (!MenuVisible || network->NetworkGame())) {
    players->UpdateRemote();
    if (!AlarmCounter)
      players->StepRemote();
  }
  UpdateAlarm();
  if (!network->NetworkGame())
    return;
  NetMessage * nm = network->GetMsg(mtStartGame);
  if (nm) {
    rounds = ((nmStartGame *) nm)->rounds;
    StartGame(FALSE);
    delete nm;
  }
  nm = network->GetMsg(mtEndGame);
  if (nm) {
    EndGame(FALSE);
    delete nm;
  }
  nm = network->GetMsg(mtStartRound);
  if (nm) {
    StartRound(FALSE);
    delete nm;
  }
  nm = network->GetMsg(mtEndRound);
  if (nm) {
    EndRound(FALSE);
    delete nm;
  }
  nm = network->GetMsg(mtOptions);
  if (nm) {
    OptionsChanged(FALSE, nm);
    delete nm;
  }
  while ((nm = network->GetMsg(mtPlayerSetup))) {
    RecvPlayerCaptured(nm);
    delete nm;
  }
  while ((nm = network->GetMsg(mtChat))) {
    RecvChatLine(((nmChat *) nm)->Text);
    delete nm;
  }
}

void WormApp::UpdateGame1()
{
  if (AlarmCounter) {
     if (ReallyPlaying && network->NetworkGame())
       players->SendLocal();
    return;
  }
   
  if (ReallyPlaying) {
    if (JustStartedRound)
      JustStartedRound = FALSE;
    else
      if (players->AllDead()) {
        EndRound(TRUE);
	return;
    }
    if  (!MenuVisible || network->NetworkGame()) {
      players->StepLocal();
      players->SendLocal();
    }
  }
}

void WormApp::SendPlayerCaptured(int PlayerNum, PlayerSelectorItem * item)
{
  if (!network->NetworkGame())
    return;
  PlayerType_t type;
  FillerType_t filler;
  int opt1, opt2;
  const char * name;
  item->GetPlayer(type, filler, opt1, opt2, &name);
  if (type == ptNetwork)
    return;
  if (type != ptNone)
    type = ptNetwork;
  if (!name)
    name = "";
  nmPlayerSetup nm(PlayerNum, type, filler, opt1, opt2, name);
  network->SendMsg(&nm);
}

void WormApp::RecvPlayerCaptured(NetMessage * msg)
{
  nmPlayerSetup * nm = (nmPlayerSetup *) msg;
  player_selector_group->ChangeRemoteItem(nm->PlayerNumber,
	PlayerType_t(nm->Type), FillerType_t(nm->FillerMainType),
        nm->FillerFirstOption,
        nm->FillerSecondOption, nm->Name);
}

void WormApp::SendChatLine()
{
  const char * t = chat_line_send->GetText();
  if (!t)
    t = "";
  nmChat nm(t);
  network->SendMsg(&nm);
  chat_line_send->SetText("");
}

void WormApp::RecvChatLine(const char * Text)
{
  if (Text)
    chat_line_recv->SetText(Text);
}

void WormApp::PaintFPS()
{
  scorepanel->Fill_sd(scorepanel->WindowSizeX-LastFPSW,
		      scorepanel->WindowSizeY-LastFPSH,
		      LastFPSW, LastFPSH, Pixel_t_Zero);  
  if (!ShowFPS)
    return;
  char s[10];
  sprintf(s, "%i", currentFPS);
  int W, H;
  Pixel_t * im = fontcreator->CreateFontImage(s, Pixel_t_White, tdNorth, fsSmallRegular, W, H);
  scorepanel->PutMaskImage_sd(scorepanel->WindowSizeX-W,
			      scorepanel->WindowSizeY-H, W, H, im, W);
  delete im;
  LastFPSW = W;
  LastFPSH = H;      
}

void WormApp::RepaintScorePanel(bool show)
{
  if ((show && ScorePanelVisible) || (!show && !ScorePanelVisible))
    return;
  ScorePanelVisible = show;
  if (show) {
    if (MenuVisible)
      front->InsertFrameBuffer(scorepanel, fb_save);
    else // menu/fb_save isn't in framebuffer
      front->AddFrameBuffer(scorepanel);
  } else
    front->RemoveFrameBuffer(scorepanel);
  front->Clear();
}

void WormApp::TypeText(const char * text, Pixel_t bg_color, int WaitTime, WormAppCallback_t callback)
{
  AlarmCallback = callback;
  RepaintScorePanel(TRUE);
  AlarmCounter = (WaitTime*currentFPS/usec2sec)+1;
  int W, H;
  Pixel_t * img = fontcreator->CreateFontImage(text, Pixel_t_White, tdNorth, fsOblique, W, H);
  // center text in screen
  int X = (scorepanel->WindowSizeX-W)/2;
  int Y = (scorepanel->WindowSizeY-H)/2;
  const int border_size = 25;
  LastMsgW = W+border_size*2;
  LastMsgH = H+border_size*2;
  scorepanel->Fill_sd(X-border_size, Y-border_size, LastMsgW, LastMsgH, bg_color);
  scorepanel->PutMaskImage_sd(X, Y, W, H, img, W);
  delete img;
}

void WormApp::UpdateAlarm()
{
  if (!AlarmCounter)
    return;
  if (--AlarmCounter)
    return;
  // Alarm!!!
  RepaintScorePanel(option_show_score_label->IsChecked());
  // delete text in screen
  int W = LastMsgW;
  int H = LastMsgH;
  int X = (scorepanel->WindowSizeX-W)/2;
  int Y = (scorepanel->WindowSizeY-H)/2;
  scorepanel->Fill_sd(X, Y, W, H, Pixel_t_Zero);
  if (AlarmCallback)
    (this->*AlarmCallback)();
}

void WormApp::ShowScore()
{
  // show highscores and setup upmenu to mainmenu
  SwitchMenu(FALSE, TRUE);
  scoretable->UpdateScores(players);
  mainmenu->ShowMenu();
  scoremenu->ShowMenu();
}
