#include "Players.hh"
#include "Worm.hh"
#include "FrameBuffer.hh"
#include "Picture.hh"
#include "Network/NetMessage.hh"
#include "Network/NetConnector.hh"
#include "Remote.hh"
#include "Utils.hh"
#include "FontCreator.hh"
#include "Filler.hh"
#include "Compat.hh"

CrashBuffer_t * crashBuffer;

const int PanelItemSizeX = 80; // not including filler!
const int PanelItemSizeY = 20; 
const int PanelItemFillerSize = 20;
const Pixel_t PanelColor = RGB(0, 0, 0x80);

const int BorderWallSize = 5; // size of border wall

const int killbonus_playernum = -2;
const int invalid_playernum = -1;

Players::Players(FrameBuffer * _arena, FrameBuffer * _scorepanel,
		 Picture * _background, Picture * _wallborder, NetConnector * _network)
    : worms(new (Worm *)[MAXPlayers]), arena(_arena), scorepanel(_scorepanel),
  background(_background), wallborder(_wallborder),
  currentMaxPlayerNum(0), network(_network),
  score(0), IsSurround(TRUE), winner_num(invalid_playernum)
{
  for (int i=0; i<MAXPlayers; i++)
    worms[i] = 0;
  crashBuffer = (CrashBuffer_t *) new CrashBuffer_t; // ???
}

Players::~Players()
{
  for (int i=0; i<MAXPlayers; i++)
    DeletePlayer(i);
  delete worms;
  delete crashBuffer;
}

int Players::AddPlayer(Worm * player, int number=-1)
{
  bool IsSame = FALSE;
  if (between(number, 0, MAXPlayers-1)) {
    if (worms[number] != player) {
      DeletePlayer(number);
      worms[number] = player;
    } else
      IsSame = TRUE;
  } else {
    number = -1;
    for (int i=0; i<MAXPlayers; i++)
      if (!worms[i]) {
      number = i;
      worms[i] = player;
      break;
    }
  }
  if (number>=0) {
    if (player) {
      player->SetNumber(number+1);
      player->SetFrameBuffer(arena);
      if (!IsSame) {
	player->NewGame(); // if the game is already started
	NewRoundPlayer(number);
      }
      if (number >= currentMaxPlayerNum)
	currentMaxPlayerNum = number+1;
      UpdateScorePanel(number);
    }
  } else
    delete player;
  return number;
}

void Players::DeletePlayer(int number)
{
  delete worms[number];
  worms[number] = 0;
  if (number == currentMaxPlayerNum) {
    currentMaxPlayerNum = 0;
    for (int i=MAXPlayers-1; i>=0; i--)
      if (worms[i]) {
	currentMaxPlayerNum = i+1;
	break;
    }
  }
}

void Players::NewGame()
{
  for (int i=0; i<currentMaxPlayerNum; i++)
    if (worms[i])
      worms[i]->NewGame();
  winner_num = invalid_playernum;
}

void Players::DrawAll()
{
  for (int i=0; i<currentMaxPlayerNum; i++)
    if (worms[i])
      worms[i]->Draw();
}

void Players::NewRoundPlayer(int number)
{
  Worm * w = worms[number];
  if (!w->IsRemote()) {
    w->NewRound();
    int X, Y, A;
    w->GetPos(X, Y, A);
    nmPlayerSetPos nm(number, X, Y, A);
    network->SendMsg(&nm);
  }
}

void Players::NewRound()
{
  InitializeCrashBuffer();
  if (background)
    arena->TileImage(background->Width(), background->Height(),
		     background->Data(), background->Width());
  else
    arena->Clear();
  Surround(IsSurround);
//  Print("new round");
  score = 0;
  for (int i=0; i<currentMaxPlayerNum; i++) {
    Worm * w = worms[i];
    if (!w)
      continue;
    NewRoundPlayer(i);
  }
  winner_num = invalid_playernum;
}

void Players::StepLocal()
{
//  Print("Step local");
  for (int i=0; i<currentMaxPlayerNum; i++) {
    Worm * w = worms[i];
    if (w && !w->IsRemote())
      w->Step();
  }
  if (!network->NetworkGame())
    UpdateLocalDeath();
}

void Players::StepRemote()
{
//  Print("Step remote");
  for (int i=0; i<currentMaxPlayerNum; i++) {
    Worm * w = worms[i];
    if (w && w->IsRemote())
      w->Step();
  }
}

bool Players::AllDead()
{
  int num_lives = 0;
  int num_sum = 0;
  int last_player = 0;
  for (int i=0; i<currentMaxPlayerNum; i++) {
    Worm * w = worms[i];
    if (w) {
      num_sum++;
      if (w->IsLive()) {
	last_player = i;
	if ((++num_lives)>1)
	  return FALSE;
      }
    }
  }
  if ((num_sum == 1) && (num_lives == 1))
    return FALSE; // only 1 player and lives
  if (num_lives == 1) // last player => kill him
    if (worms[last_player]->GetType() != ptNetwork) {
      KillAndUpdate(last_player, killbonus_playernum);
      if (network->NetworkGame()) {
	nmPlayerDead nm(last_player, killbonus_playernum, score);
	network->SendMsg(&nm);
      }
    }
  return TRUE;
}

Worm * Players::GetPlayer(int playernum)
{
  if (between(playernum, 0, MAXPlayers-1))
    return worms[playernum];
  return 0;
}

void Players::SendLocal()
{
  if (!network->NetworkGame())
    return;
  nmStep s(nmStep::MAXArraySize);
  int stepp = 0;
  for (int i=0; i<currentMaxPlayerNum; i++) {
    Worm * w = worms[i];
    if (w && !w->IsRemote() && w->GetDirChanged())  {
      s.array[stepp++].Set(i, w->GetDir());
    }
  }
  if (!stepp)
    return;
  s.Resize(stepp);
  network->SendMsg(&s);
  UpdateLocalDeath();
}

static void updateremoteprinterror(int i)
{
  PrintErr << "player " << i << " isn't remote, ignoring remote control" << nl;
}

void Players::UpdateRemote()
{
  if (!network->NetworkGame())
    return;
  nmPlayerSetPos * p;
  while ((p = (nmPlayerSetPos *) network->GetMsg(mtPlayerSetPos))) {
    int r = p->PlayerNum;
    Worm * w = worms[r];
//    PrintErr << "update remote: " << r << nl;
    if (w && (w->GetType() == ptNetwork))
      w->NewRound(p->X, p->Y, p->Angle);
    else
      updateremoteprinterror(r);
  }
  nmStep * s = 0;
  while ((s = (nmStep *) network->GetMsg(mtStep))) {
    int count = s->GetCount();
    nmStepItem * si = &s->array[0];
    for (int i=0; i<count; i++, si++) {
      Remote * r = (Remote *) worms[si->PlayerNumber];
      if (r && (r->GetType() == ptNetwork)) { // can't use IsRemote, it may not be Remote
	bool Left, Press;
	switch (si->Dir) {
	case si->DirLeft:
	  Left = TRUE; Press = TRUE; break;
	case si->DirRight:
	  Left = FALSE; Press = TRUE; break;
	default:
	case si->DirUnpress:
	  Left = TRUE; Press = FALSE;
	}	
	r->Tune(Left, Press);
      } else
	updateremoteprinterror(i);
    }
    delete s;
  }
  UpdateRemoteDeath();
}

void Players::KillAndUpdate(int Who, int ByWhom)
{
  //  PrintErr << "kau " << Who << " by " << ByWhom << nl;
  bool killbonus = (ByWhom == killbonus_playernum);
  Worm * w = worms[Who];
  if (w) {
    w->Death();
    w->AddPoints(killbonus ? score*2 : score);
    UpdateScorePanel(Who);
  }
  if (killbonus) {
    winner_num = Who;
    return;
  }
  Worm * k = worms[ByWhom];
  if (!k || (w == k))
      return; // self-kill didn't give score/2 points!
  k->AddPoints(score/2);
  UpdateScorePanel(ByWhom);
}

void Players::UpdateLocalDeath()
{
  // update death :-)
  bool ScoreUsed = FALSE;
  for (int i=0; i<currentMaxPlayerNum; i++) {
    Worm * w = worms[i];
    if (!w || (w->IsRemote()))
      continue;
    int KilledBy = w->GetKilledBy();
    if (KilledBy <= 0)
      continue;
    ScoreUsed = TRUE;
    KilledBy--;
    if (network->NetworkGame()) {
      nmPlayerDead nm(i, KilledBy, score);
      network->SendMsg(&nm);
    }
    KillAndUpdate(i, KilledBy);
  }
  if (ScoreUsed)
    score++;
}

void Players::UpdateRemoteDeath()
{
  nmPlayerDead * nm;
  while ((nm = (nmPlayerDead *) network->GetMsg(mtPlayerDead))) {
    score = nm->AddPoints;
    KillAndUpdate(nm->PlayerNumber, nm->KillerNumber);
    score++;
  }
}

void Players::UpdateScorePanel(int PlayerNumber)
{
  Worm * w = worms[PlayerNumber];
  int p = (scorepanel->WindowSizeX)/(PanelItemSizeX+PanelItemFillerSize);
  int X = (PlayerNumber%p)*(PanelItemSizeX+PanelItemFillerSize);
  int Y  = (PlayerNumber/p)*PanelItemSizeY;
  if (!w) {
    scorepanel->Fill_sd(X, Y, PanelItemSizeX, PanelItemSizeY, Pixel_t_Zero);
    return;
  }
  Filler * filler = w->GetFiller();
  scorepanel->TileImageWindow_sd(X, Y, PanelItemFillerSize, PanelItemSizeY,
			      filler->GetImage(0, 0), filler->ImageWidth, filler->ImageHS);
  scorepanel->Fill_sd(X+PanelItemFillerSize, Y, PanelItemSizeX,
		      PanelItemSizeY, PanelColor);
  char text[40];
  sprintf(text, "%i %s", w->GetScore(), w->GetName());
  int W, H;
  Pixel_t * img = fontcreator->CreateFontImage(text, Pixel_t_White,
					       tdNorth, fsSmallRegular, W, H);
  int W2 = Min(W, PanelItemSizeX);
  H = Min(H, PanelItemSizeY);
  scorepanel->PutMaskImage_sd(X+PanelItemFillerSize, Y, W2, H, img, W);
  delete img;
}

void Players::InitializeCrashBuffer()
{
  bzero(crashBuffer, sizeof(CrashBuffer_t));
}

void Players::Surround(bool Set)
{
  IsSurround = Set;
  int WX = arena->WindowSizeX;
  int WY = arena->WindowSizeY;
  int BS = BorderWallSize;
  int EX = WX-BS;
  int EY = WY-BS;
  int LY = EY-BS;
  Picture * p = (Set ? background : wallborder);
  const Pixel_t * data = p->Data();
  int PW = p->Width();
  int PH = p->Height();
  arena->TileImageWindow_sd(0, 0, WX, BS, data, PW, PH);
  arena->TileImageWindow_sd(0, EY, WX, BS, data, PW, PH);
  arena->TileImageWindow_sd(0, BS, BS, LY, data, PW, PH);
  arena->TileImageWindow_sd(EX, BS, BS, LY, data, PW, PH);
  // For faster operation, fill both items in CrashBufferItem_t to same value
  char c = (Set ? VoidCrashB_playernum : WallBorderCrashB_playernum);
  CrashBufferItem_t bi;
  bi.PlayerNumber = c;
  bi.Counter = 0;
  // horizontal first
  int hor_size = WX*BS*sizeof(CrashBufferItem_t);
  int ver_size2 = BS*sizeof(CrashBufferItem_t)*2;
  memset(*crashBuffer, c, hor_size);
  memset((*crashBuffer)+EY*WX, c, hor_size);
  // now vertical - warning!!! it's using 'surround' feature!
  CrashBufferItem_t * start = (*crashBuffer)+WX*BS-BS;
  for(int i=0; i<=LY; i++, start += WX)
    memset(start, c, ver_size2);
}
