#include "UDPConnection.hh"
#include "Debug.hh"
#include "NetworkTimings.hh"
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>   
#include <netdb.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include "Compat.hh"
#include "Utils.hh"

UDPConnection::UDPConnection(bool _server)
    : Connection(_server)
{
  SetTimeout(UDPConnectionTimeout);
}

UDPConnection::~UDPConnection()
{
  Disconnect();
}
      
void UDPConnection::SetHostname(const char * _name)
{
  Connection::SetHostname(_name);
  bzero(&addr, sizeof (addr));
  if (!_name)
    return;
  addr.sin_family = AF_INET;
//  addr.sin_port = htons(port);
#ifdef Linux
  if (!inet_aton(hostname, &addr.sin_addr)) {
#else
  if ((int)(addr.sin_addr.s_addr = inet_addr(hostname)) == (int)-1) {
#endif
    hostent * he = gethostbyname(hostname);
    if (!he) {
      StdError("can't get address for hostname");
      return;
    }
    addr.sin_addr.s_addr = ((in_addr *)((void *) (he->h_addr)))->s_addr;
  }
}

bool UDPConnection::Connect()
{
  Disconnect();
  DEBUGP("Creating client socket");
  socket_fd = socket(AF_INET, SOCK_DGRAM, 0);
  if (socket_fd < 0) {
    StdError("can't create client socket");
    return FALSE;
  }
  addr.sin_port = 0; // any
  sockaddr_in a = addr;
  a.sin_addr.s_addr = INADDR_ANY;
     
  if (bind (socket_fd, (sockaddr *) &a, sizeof (addr)) < 0) {
    close(socket_fd);
    StdError("can't bind socket");
    return FALSE;
  }
   
  DEBUGP("Connecting to server" << hostname << ", port " << peer_port);
  int addr_len = sizeof(addr);
  if (getsockname (socket_fd, (sockaddr *) &a, &addr_len)) {
    close(socket_fd);
    StdError("can't get sockname");
    return FALSE;
  }
  addr.sin_port = a.sin_port;

  our_port = ntohs(addr.sin_port);  
  addr.sin_port = htons(peer_port);
 
  if (connect(socket_fd, (sockaddr *) &addr, sizeof(addr)) < 0) {
    StdError("can't connect");
    Disconnect();
    return FALSE;
  }
  DEBUGP("Connected from " << our_port);
  connected = TRUE;
  return TRUE;
}

void UDPConnection::Disconnect()
{
  if (!connected)
    return;
  DEBUGP("Disconnecting from " << hostname);
  connected = FALSE;
  if (shutdown(socket_fd, 2) < 0)
    StdError("error while shutdown of socket");
  if (close(socket_fd) < 0)
    StdError("error while closing socket");
}

bool UDPConnection::Listen()
{
  Disconnect();
  DEBUGP("Creating server socket at port " << our_port);
  socket_fd = socket(AF_INET, SOCK_DGRAM, 0);
  if (socket_fd < 0) {
    StdError("can't create server socket");
    return FALSE;
  }
  addr.sin_port = htons(our_port);
  if (any_addr)
    addr.sin_addr.s_addr = INADDR_ANY;
  if (bind(socket_fd, (sockaddr *) &addr, sizeof (addr)) < 0) {
    close(socket_fd);
    StdError("can't bind socket");
    return FALSE;
  }
  int addr_len = sizeof(addr);
  if (getsockname (socket_fd, (sockaddr *) &addr, &addr_len)) {
    close(socket_fd);
    StdError("can't get sockname");
    return FALSE;
  }
  our_port = ntohs(addr.sin_port);
  DEBUGP("Awaiting clients at port " << our_port);
  DEBUGP("Expecting clients from port " << peer_port << ", hostname " << hostname);
  connected = TRUE;
  return TRUE;
}

bool UDPConnection::Send(char * buffer, int bufsize)
{
  if (!connected)
    return FALSE;
  int ret = 0;
  addr.sin_port = htons(peer_port);
  ret = sendto(socket_fd, buffer, bufsize, 0, (sockaddr *) &addr, sizeof (addr));  
  if (ret < 0) {
    StdError("can't send message");
    return FALSE;
  }
  if (ret != (int) bufsize) {
    Error("can't send whole buffer!");
    return FALSE;
  }
  DEBUGP("sent " << bufsize << " bytes to " << hostname);
  return TRUE;
}

int UDPConnection::Receive(char * buffer, int maxsize, bool Wait=TRUE)
{
  fd_set set;
  timeval timeout;

  timeout.tv_sec = 0;
  timeout.tv_usec = (Wait ? timeout_usec : 0);

  FD_SET(socket_fd, &set);

  if (select (socket_fd+1, &set, 0, 0, &timeout) < 0)
    return 0;
//    return ((errno == EINTR) ? 0 : -1);
  
  if (!FD_ISSET(socket_fd, &set))
    return 0;

  // some data!
  int addr_len = sizeof(addr);
  sockaddr_in in_addr;
  bzero(&in_addr, addr_len);
  int ret = recvfrom (socket_fd, buffer, maxsize, 0, (sockaddr *) &in_addr, &addr_len);
  if (ret <= 0) {
    StdError("can't receive message");
    return 0;
  }
  if (!any_addr) {
    if ((addr.sin_addr.s_addr != in_addr.sin_addr.s_addr) ||
      (in_addr.sin_port != htons(peer_port))) {
      PrintErr << "got message from unexpected host ";
      hostent * he = gethostbyaddr((const char *) &in_addr.sin_addr, sizeof(in_addr.sin_addr), AF_INET);
      if (he && (he->h_name))
	PrintErr << he->h_name;
      else
	PrintErr << "(unknown address)";
      PrintErr << ", port " << ntohs(in_addr.sin_port) << nl;
      // accepting packets only from hostname
      return 0;
    }
    // save server port into peer_port
    peer_port = ntohs(in_addr.sin_port);
  } else { // any_addr
    if (in_addr.sin_addr.s_addr != addr.sin_addr.s_addr) {
      // address is different -> create new last_hostname
      hostent * he = gethostbyaddr((const char *) &in_addr.sin_addr, sizeof(in_addr.sin_addr), AF_INET);   
      if ((!he) || (!(he->h_name))) // ignore
	return 0;
      delete client_hostname;
      client_hostname = new char[strlen(he->h_name)+1];
      strcpy(client_hostname, he->h_name);
    }
    // save client addr
    peer_port = ntohs(in_addr.sin_port);
    addr = in_addr;
  }
  DEBUGP("received " << ret << " bytes from " << hostname);
  return ret;
}

int UDPConnection::SendReceive(char * sendbuffer, int bufsize, char * recvbuffer, int maxsize)
{
  int ret;
  for (int i=0; i<UDPConnectionSendRecTryCount; i++) {
    if (Send(sendbuffer, bufsize)) {
      ret = Receive(recvbuffer, maxsize, TRUE);
      if (ret > 0)
	return ret;
    }
  }
  return -1;
}
