#include "Utils.hh"
#include "FrameBuffer.hh"
#include <string.h>
#include "Debug.hh"
#include "Compat.hh"

FrameBuffer::FrameBuffer(const TransparencyType _transparency)
    : transparency(_transparency), buffer(new Pixel_t[WindowSizeX*WindowSizeY])
{
}

FrameBuffer::~FrameBuffer()
{
  delete buffer;
}

void FrameBuffer::Clear()
{
  bzero(buffer, WindowSizeX*WindowSizeY*sizeof(Pixel_t));
  if (!front->FindDuplicate(0, 0, WindowSizeX, WindowSizeY))
    front->AddDirty(0, 0, WindowSizeX, WindowSizeY);
}

void FrameBuffer::Fill(int X, int Y, int Width,
		       int Height, const Pixel_t color)
{
  if (!front->Clip(X, Y, Width, Height))
    return;
  XYCycle(p, X, Y, Width, Height)
    *p = color;
  front->AddDirty(X, Y, Width, Height);
}

void FrameBuffer::TileImage(const int Width, const int Height,
	       const Pixel_t * Img, const int Img_width)
{
  for (int i=0; i<WindowSizeY; i+=Height)
    for (int j=0; j<WindowSizeX; j+=Width)
      PutImage_sd(j, i,
	       (j+Width)>WindowSizeX ? WindowSizeX-j : Width,
	       (i+Height)>WindowSizeY ? WindowSizeY-i : Height,
	       Img, Img_width);
}

void FrameBuffer::Fill_sd(int X, int Y, int Width,
		       int Height, const Pixel_t color)
{
  if (!front->Clip(X, Y, Width, Height))
    return;
  XYCycle(p, X, Y, Width, Height)
    *p = color;
  if (!front->FindDuplicate(X, Y, Width, Height))
    front->AddDirty(X, Y, Width, Height);
}

void FrameBuffer::TileImageWindow_sd(int X, int Y, int Width, int Height,
				  const Pixel_t * Img, const int Img_width, const int Img_height)
{
  if (!front->Clip(X, Y, Width, Height))
    return;
  if (!front->FindDuplicate(X, Y, Width, Height))
    front->AddDirty(X, Y, Width, Height);
  while (Height>0) {
    int IY = Y%Img_height;
    int IH = Img_height-IY;
    int H = Min(Height, IH);
    int W2 = Width;
    int X2 = X;
    while (W2>0) {
      int IX = X2%Img_width;
      int IW = Img_width-IX;
      const Pixel_t * Img_shifted = Img+IX+IY*Img_width;
      int W = Min(W2, IW);
      PutImage_sd(X2, Y, W, H, Img_shifted, Img_width);
      W2 -= W;
      X2 += W;
    }
    Height -= H;
    Y += H;
  }
}

void FrameBuffer::TileMaskImageWindow_sd(int X, int Y, int Width, int Height,
				  const Pixel_t * Img, const int Img_width, const int Img_height)
{ // same as TimeImageWindow_sd, but PutMaskImage_sd instead of PutImage_sd
  if (!front->Clip(X, Y, Width, Height))
    return;
  if (!front->FindDuplicate(X, Y, Width, Height))
    front->AddDirty(X, Y, Width, Height);
  while (Height>0) {
    int IY = Y%Img_height;
    int IH = Img_height-IY;
    int H = Min(Height, IH);
    int W2 = Width;
    int X2 = X;
    while (W2>0) {
      int IX = X2%Img_width;
      int IW = Img_width-IX;
      const Pixel_t * Img_shifted = Img+IX+IY*Img_width;
      int W = Min(W2, IW);
      PutMaskImage_sd(X2, Y, W, H, Img_shifted, Img_width);
      W2 -= W;
      X2 += W;
    }
    Height -= H;
    Y += H;
  }  
}

void FrameBuffer::PutMaskImage_sd(int X, int Y, int Width, int Height,
				  const Pixel_t * Img, const int Img_width)
{
  if (!front->Clip(X, Y, Width, Height))
    return;
  ImgCycle(p, X, Y, Width, Height, s, Img, Img_width)
    if (s->NotZero())
      *p = *s;
  if (!front->FindDuplicate(X, Y, Width, Height))
    front->AddDirty(X, Y, Width, Height);
}

void FrameBuffer::PutImage_sd(int X, int Y, int Width, int Height,
				  const Pixel_t * Img, const int Img_width)
{
  if (!front->Clip(X, Y, Width, Height))
    return;
  ImgCycle(p, X, Y, Width, Height, s, Img, Img_width)
    *p = *s;
  if (!front->FindDuplicate(X, Y, Width, Height))
    front->AddDirty(X, Y, Width, Height);
}

void FrameBuffer::ClearMaskImage_sd(int X, int Y, int Width, int Height,
		      const Pixel_t * Img, const int Img_width)
{
  if (!front->Clip(X, Y, Width, Height))
    return;
  ImgCycle(p, X, Y, Width, Height, s, Img, Img_width)
    if (s->NotZero())
      *p = Pixel_t_Zero;
  if (!front->FindDuplicate(X, Y, Width, Height))
    front->AddDirty(X, Y, Width, Height);  
}
