#include "GameServer.hh"
#include "Network/NetConnector.hh"
#include "Network/NetMessage.hh"
#include "Compat.hh"
#include "Network/UDPConnection.hh"
#include "Utils.hh"
#include "Debug.hh"
#include "Network/NetworkTimings.hh"
#include "Compat.hh"
#include "Game/PlayerType.hh"
#include <string.h>

//#define GSDeb(x)
#define GSDeb(x) x

const int invalid_client = -1;

GameServer::GameServer(int port)
    : num_clients(0), rounds(0), cached_options(0),
  frametime(GameServerInitFrameTime), multip(new Multiplexor)
{

  bzero(clients, sizeof(clients));
  bzero(player_setup, sizeof(player_setup));
  for (int i=0; i<MAXPlayers; i++)
    player_setup_clientnum[i] = invalid_client;
  for (int i=0; i<Multiplexor::MaxFD; i++)
    fd2netconn[i] = invalid_client;
  UDPConnection * c = new UDPConnection(TRUE);
  c->SetOurPort(port);
  c->SetHostname(0);
  init_network = new NetConnector(TRUE, c, 0, 0);
}

GameServer::~GameServer()
{
  delete init_network;
  delete cached_options;
  for (int i=0; i<num_clients; i++)
    delete clients[i];
}

void GameServer::ProcessInit()
{
  init_network->Reset();
  if (!init_network->ReceiveMessages(FALSE))
    return;
  Connection * in_tmp = init_network->connection;
  PrintErr << "request from client at " << in_tmp->GetPeerHostname()
    << ":" << in_tmp->GetPeerPort() << nl;
  nmInitConnection * ic = (nmInitConnection *) init_network->GetMsg(mtInitConnection);
  init_network->DeleteMessages();
  if (!ic) {
    Error("invalid request from client");
    return;
  }
  if (ic->Version != nmInitConnection::CurrentVersion) {
    PrintErr << "error: invalid client version " << ic->Version
      << " (server version " << nmInitConnection::CurrentVersion << ")" << nl;
    nmInitConnection ic2(FALSE, 0, 0, 0);
    init_network->SendMsg(&ic2);
    init_network->SendMessages();
    delete ic;
    return;    
  }
  if (num_clients &&
      ((ic->ScreenSizeX != ScreenSzX) || (ic->ScreenSizeY != ScreenSzY))) {
    // Incompatible screen size
    PrintErr << "error: invalid client's screen size: " << ic->ScreenSizeX << "x"
      << ic->ScreenSizeY << nl;
    nmInitConnection ic2(FALSE, 0, ScreenSzX, ScreenSzY);
    init_network->SendMsg(&ic2);
    init_network->SendMessages();
    delete ic;
    return;
  }
  if (!num_clients) {
    ScreenSzX = ic->ScreenSizeX;
    ScreenSzY = ic->ScreenSizeY;
  }
  delete ic;
  if (num_clients >= MaxClients) {
    // Too many clients
    Error("too many clients - client rejected");
    nmInitConnection ic2(FALSE, 0, 0, 0);
    init_network->SendMsg(&ic2);
    init_network->SendMessages();
    return;
  }
  // Create new connection to client
  int clnum = num_clients;
  NetConnector * clpp = 0;  
  // Create new client connection
  clpp = new NetConnector(TRUE, new UDPConnection(TRUE), 0, 0);
  {
    Connection * c_tmp = clpp->connection;
    c_tmp->SetHostname(0);
    c_tmp->SetOurPort(0); // any
    if (!c_tmp->Listen())
      return;
  }
  nmInitConnection ic2(TRUE, clpp->connection->GetOurPort(), 0, 0);
  init_network->SendMsg(&ic2);
  if (init_network->SendMessages() && clpp->ReceiveMessages(TRUE)) {
    if (cached_options)
      clpp->SendMsg(cached_options); // send options
    for (int a=0; a<MAXPlayers; a++)
      if (player_setup[a])
	clpp->SendMsg(player_setup[a]); // send info about captured slots
    if (clpp->SendMessages()) {
      Connection * c = clpp->connection;
      c->SetHostname(c->GetPeerHostname());
      c->SetPeerPort(c->GetPeerPort());
      num_clients++;
      clients[clnum] = clpp;
      client_state[clnum] = gspsSetup;
      char s[nmChat::MsgSize];
      char t[300];
      sprintf(t, "Connected client from %s:%i",
	       c->GetPeerHostname(), c->GetPeerPort());
      t[nmChat::MsgSize-1] = 0;
      strncpy(s, t, nmChat::MsgSize-1); 
      nmChat cl(s);
      SendAllExcept(&cl, clnum);
      multip->AddConnection(c);
      fd2netconn[c->GetFD()] = clnum;
      Print("client accepted");
      return;
    }
  }
  // Oops! We just lost a new client
  // Kill him ;-)
  Error("client lost");
  delete clpp;
}

void GameServer::CollectMessages()
{
  bool rc[MaxClients];
  bzero(&rc, sizeof(rc));
  int rec_cnt = num_clients;
  int time_out = frametime*GameServerRecvTimeoutC;
  int kill_counter =  GameServerTimeout/time_out;
  multip->SetTimeout(time_out);
  multip->Reset();
  while (rec_cnt) {
    DEBUGP(rec_cnt << "packets left");
    bool AnyRecv = FALSE;
    Connection * c = multip->WaitForConnection();
    if (c) {
      int i = fd2netconn[c->GetFD()];
      if (i == invalid_client)
	Error("message on unused NetConnector!");
      else
	if (clients[i]->ReceiveMessages(FALSE)) {
   	  rec_cnt--;
	  rc[i] = TRUE;
	  AnyRecv = TRUE;
      } else {
	multip->ReWaitConnection(c);
	if (NetworkShowWarnings)
	  Error("unexcepted wakeup");
	AnyRecv = TRUE;
      }
    }
    if (!AnyRecv) {
      if (!(--kill_counter)) {
	// KillCounter loops and no response from clients...
	// kill'em!
	for (int i=0; i<num_clients; i++)
	  if (!rc[i]) {
	  PrintErr << "no response from client " << i << nl;
	  KillPlayer(i);
        }
	return;
      }
      // else resend all remaining messages again
      if (NetworkShowWarnings)
	Error("timeout while collecting messages - resending messages");
      for (int i=0; i<num_clients; i++)
	if (!rc[i])
	  clients[i]->ResendMessages();
    }
  }
  // received info from all clients
}

void GameServer::KillPlayer(int player_num)
{
  if (!clients[player_num])
    return;
  Connection * c = clients[player_num]->connection;
  multip->RemoveConnection(c);
  PrintErr << "killing client " << player_num
    << " from " << c->GetPeerHostname()
      << ":" << c->GetPeerPort() << nl;  
  char s[nmChat::MsgSize];
  char tx[300]; 
  sprintf(tx, "Client from %s:%i killed",
	   c->GetPeerHostname(), c->GetPeerPort());
  tx[nmChat::MsgSize-1] = 0;
  strncpy(s, tx, nmChat::MsgSize-1);
  nmChat cl(s);
  SendAllExcept(&cl, player_num);
  for (int i=0; i<MAXPlayers; i++) {
    nmPlayerSetup * ps = player_setup[i];
    if (ps) {
      int * psc = &player_setup_clientnum[i];
      if (*psc == player_num) {
	nmPlayerSetup nm(ps->PlayerNumber, ptNone, 0, 0, 0, ps->Name);
	SendAllExcept(&nm, player_num); // send uncapture of slots
	delete ps;
	player_setup[i] = 0;
	*psc = invalid_client;
      } else
	if (*psc > player_num)
	  (*psc)--; // bellow is renumbering of clients
    }
  }
  int * fd2 = &fd2netconn[0];
  for (int i=0; i<Multiplexor::MaxFD; i++, fd2++)
    if (*fd2 == player_num)
      *fd2 = invalid_client;
    else
      if (*fd2 > player_num)
	(*fd2)--;
  delete clients[player_num];
  clients[player_num] = 0;
  for (int i=player_num; i<num_clients; i++)
    clients[i] = clients[i+1];
  num_clients--;
  CheckState();
}

void GameServer::SendAll(NetMessage * nm)
{
  for (int j=0; j<num_clients; j++)
    clients[j]->SendMsg(nm);
}

void GameServer::SendAllExcept(NetMessage * nm, int except)
{
  for (int j=0; j<num_clients; j++)
    if (j != except)
      clients[j]->SendMsg(nm);
}

void GameServer::ProcessQuit()
{
  for (int i=0; i<num_clients; i++) {
    NetMessage * nm = clients[i]->GetMsg(mtQuit);
    if (nm) {
      PrintErr << "client " << i << " quits" << nl;
      KillPlayer(i);
      delete nm;
    }
  }
}

void GameServer::ProcessStateType(MsgType_t2 which, GSPlayerState_t type2)
{
  for (int i=0; i<num_clients; i++) {
    NetMessage * nm = clients[i]->GetMsg(which);
    if (nm) {
      client_state[i] = type2;
      if (type2 == mtStartGame) { // EEK! this maybe shouldn't be there
	rounds = Max(rounds, ((nmStartGame *) nm)->rounds);
      }
      delete nm;
    }
  }
}

void GameServer::CheckState()
{
  if (!num_clients)
    return;
  for (int i=0; i<num_clients; i++)
    if (client_state[i] == gspsPlaying)
      return; // can't send anything else when somebody's playing
  for (int i=0; i<num_clients; i++)
    if (client_state[i] == gspsEndRound) {
      for (int j=i; j<num_clients; j++)
	if (client_state[j] == gspsEndRound)
	  client_state[j] = gspsPlaying;
      Print("ending round");
      nmEndRound nm;
      SendAll(&nm);
      return;
      // when someone's waiting for endround and there are no playing players
    }
  for (int i=0; i<num_clients; i++)
    if (client_state[i] == gspsStartRound) {
      for (int j=i; j<num_clients; j++)
	if (client_state[j] == gspsStartRound)
	  client_state[j] = gspsPlaying;
      Print("stating round");
      nmStartRound nm;
      SendAll(&nm);
      return;
      // when there are some players waiting for start round
    }
  for (int i=0; i<num_clients; i++)
    if (client_state[i] == gspsEndGame) {
      for (int j=i; j<num_clients; j++)
	if (client_state[j] == gspsEndGame)
	  client_state[j] = gspsSetup;
      Print("ending game");
      nmEndGame nm;
      SendAll(&nm);
      return;
    }
  for (int i=0; i<num_clients; i++)
    if (client_state[i] == gspsSetup)
      return; // can't start game while someone's changing setup
  for (int i=0; i<num_clients; i++)
    if (client_state[i] != gspsStartGame)
      return;
  // all players are in StartGame state
  PrintErr << "starting game, " << rounds << " rounds" << nl;
  nmStartGame nm(rounds);
  SendAll(&nm);
}

void GameServer::ProcessSetGameFPS()
{
  const int invFPS = 100000;
  int newFPS = invFPS;
  for (int i=0; i<num_clients; i++) {
    nmSetGameFPS * nm = (nmSetGameFPS *) clients[i]->GetMsg(mtSetGameFPS);
    if (nm && (nm->wantFPS < newFPS))
      newFPS = nm->wantFPS;
    delete nm;
  }
  if (newFPS != invFPS) {
    frametime = usec2sec/newFPS;
    nmSetGameFPS m(newFPS);
    SendAll(&m);
  }
}

void GameServer::ProcessResendAllExcept(MsgType_t2 which)
{
  for (int i=0; i<num_clients; i++) {
    NetConnector * c = clients[i];
    NetMessage * nm;
    while ((nm = c->GetMsg(which))) {
      SendAllExcept(nm, i);
      delete nm;
    }
  }
}

void GameServer::ProcessOptions()
{
  bool Changed = FALSE;
  for (int i=0; i<num_clients; i++) {
    NetConnector * c = clients[i];
    NetMessage * nm;
    while ((nm = c->GetMsg(mtOptions))) {
      delete cached_options;
      cached_options = (nmOptions *) nm;
      Changed = TRUE;
    }
  }
  if (Changed)
    SendAll(cached_options);
}

void GameServer::ProcessPlayerSetup()
{
  for (int i=0; i<num_clients; i++) {
    NetConnector * c = clients[i];
    nmPlayerSetup * nm;
    while ((nm = (nmPlayerSetup *) c->GetMsg(mtPlayerSetup))) {
      int n = nm->PlayerNumber;
      if (between(n, 0, MAXPlayers-1)) {
	delete player_setup[n];
	if (nm->Type != ptNone) {
	  player_setup[n] = nm;
	  player_setup_clientnum[n] = i;
	} else {
	  player_setup[n] = 0;
	  player_setup_clientnum[n] = invalid_client;
	}
	SendAllExcept(nm, i);
      } else {
	PrintErr << "attempt to change of no-exist player " << n << " from client " << i << nl;
	delete nm;
      }
    }
  }
}

void GameServer::ProcessMessages()
{
  // mtInitConnection
  ProcessQuit();
  ProcessOptions();
  // mtBreak
  ProcessSetGameFPS();
  ProcessResendAllExcept(mtStep);
  ProcessStateType(mtEndRound, gspsEndRound);
  ProcessStateType(mtStartRound, gspsStartRound);
  ProcessStateType(mtEndGame, gspsEndGame);
  ProcessStateType(mtStartGame, gspsStartGame);
  CheckState();
  ProcessResendAllExcept(mtPlayerSetPos);
  ProcessPlayerSetup();
  ProcessResendAllExcept(mtChat);
  ProcessResendAllExcept(mtPlayerDead);  
  // ok, now send all messagess
}

void GameServer::SendMessages()
{
  bool rc[MaxClients];
  bzero(&rc, sizeof(rc));
  int rec_cnt = num_clients;
  while (rec_cnt)
    for (int i=0; i<num_clients; i++)
      if ((!rc[i]) && clients[i]->SendMessages()) {
        rc[i] = TRUE;
	rec_cnt--;
      }
}

void GameServer::Run()
{
  if (!init_network->connection->Listen())
    return;
  for (int i=0;; i++) {
    if ((!num_clients) || (!(i%GameServerInitCheckCounter)))
      ProcessInit();
    if (num_clients) {
      CollectMessages();
      ProcessMessages();
      SendMessages();
    } else
      microsleep(GameServerIdleSleepTime);
  }
}
