#include "View.hh"
#include "Application.hh"
#include "Debug.hh"

View::View()
    : next(0),
  Visible(FALSE), Hidden(FALSE), Selectable(FALSE), Selected(FALSE),
  Enabled(TRUE), AutoResize(FALSE), AutoCenter(FALSE), IsLargest(TRUE),
  choosecommand(cmInvalid), MinSize(Point(1, 1)), parent(0)
{
  SetColor(RGB(0xe0, 0xe0, 0xe0), ciForeground);
  SetColor(Pixel_t_White, ciHighlightedFG);
  SetColor(RGB(0x50, 0x50, 0x50), ciDisabledFG);  
  SetColor(Pixel_t_Zero, ciBackground);
  SetColor(RGB(0xb0, 0x20, 0x20), ciHighlightedBG);
}

View::~View()
{
  if (app_global)
    Hide();
}

void View::Select()
{
  if (Visible && Selectable && Enabled && (!Hidden)) {
    if (Selected)
      return;
    if (parent) {
      View * cur = parent->current;
      if (cur)
	cur->Deselect();
      parent->current = this;
      Selected = TRUE;
      parent->Select();
      parent->DrawPartBG(this);
      Redraw();
    } else {
      Selected = TRUE;
      Redraw();
    }
  } else
    if (next)
      next->Select();
}

void View::Deselect()
{
  if (!Selected)
    return;
  Selected = FALSE;
  if (parent) {
    parent->current = 0;
    parent->DrawPartBG(this);
  }
  Redraw();
}

void View::SetEnabled(bool _Enabled=TRUE)
{
  if (Enabled == _Enabled)
    return;
  Enabled = _Enabled;
  if (Visible && (!Hidden)) {
    if (parent)
      parent->DrawPartBG(this);
    Redraw();
  }
}

bool View::SelectNext(bool Forward)
{
  if (Forward) {
    if (next)
      next->Select();
    if (!Selected)
      return TRUE;
    // not selected another yet...
    if (parent)
      return parent->SelectNext(TRUE);
    else
      return FALSE;
  }
  // backwards... hope there are not too much views
  // I'm lazy to use 2-way pointer structure for views
  if (!parent)
    return FALSE;
  View * p2 = 0;
  View * prev = parent->first;
  while (prev && (p2 = (((prev->Selectable) && (!prev->Hidden)) ? prev : p2), prev->next != this))
    prev = prev->next;
  if (p2)
    p2->Select();
  else { // find last selectable view
    View * x = parent->first;
    View * s = 0;
    while (x) {
      if (x->Selectable)
	s = x;
      x = x->next;
    }
    if (s)
      s->Select();
  }
  return TRUE;
}

void View::Show()
{
  if ((!Visible) && (!Hidden)) {
    Visible = TRUE;
    if ((!parent) || (Selectable && (!parent->current) && (parent->Selected)))
      Select();
    Redraw();
  }
}

void View::Hide()
{
//  cerr << this << "\n";
  if (Visible) {
    Visible = FALSE;
    if (Selected)
      Deselect();
    if ((!Hidden) && parent)
      parent->DrawPartBG(this);
    else
      if (app_global->fb)
	app_global->fb->Fill_sd(GetX(), GetY(), Area.Size.X, Area.Size.Y, RGB(0, 0, 0));
  }
}

void View::Draw()
{
}

void View::Redraw()
{
  if ((!Visible) || (Hidden) || (!app_global->fb))
    return;
  DrawBackground();
  Draw();
}

void View::DrawBackground()
{
  // if you modify this, change Group::DrawPartBG too!
  Pixel_t color = Colors[Selected ? ciHighlightedBG : ciBackground];
  if (color.NotZero())
    app_global->fb->Fill_sd(GetX(), GetY(), Area.Size.X, Area.Size.Y, color);
}

bool View::IsUsedNow(ColorIndex index) const
{
  switch (index) {
  case ciForeground:
    return !Selected;
  case ciHighlightedFG:
    return Selected;
  case ciDisabledFG:
    return !Enabled;
  case ciBackground:
    return !Selected;
  default: // case ciHighlightedBG:
    return Selected;
  }
}
  
void View::SetColor(const Pixel_t color, const ColorIndex index)
{
  if (Colors[index] == color)
    return;
  Colors[index] = color;
  if (IsUsedNow(index))
    Redraw();
}

Pixel_t View::GetColor(const ColorIndex index)
{
  return Colors[index];
}

bool View::HandleEvent(Event& event)
{
  if (Selected && (event.type == evKeypress) && (event.key == ktEsc) &&
      Visible && (!Hidden)) {
    Event e(this);
    e.type = evnCommand;
    e.command = cmCancel;
    PutEvent(e);
    return FALSE;
  }
  return TRUE;
}

void View::PutEvent(Event& event)
{
  if (parent)
    parent->PutEvent(event);
  else
    app_global->CreateEvent(event);
}

void View::SetOrigin(const Point& neworigin)
{
  if (Area.Origin == neworigin)
    return;
  if (Visible) {
//    Hide();
    Area.Origin = neworigin;
//    Show();
    (parent ? parent : this)->Redraw();
  } else
    Area.Origin = neworigin;
}

void View::SetSize(const Point& newsize, bool redraw=TRUE)
{
  if (Area.Size == newsize)
    return;
  if (!redraw) {
    Area.Size = newsize;
    ReArrange(FALSE);
    Area.Size = newsize;
    return;
  }
  if (Visible) {
    //    Hide();
    Area.Size = newsize;
    //    Show();
    if (IsLargest)
      (parent ? parent : this)->ReArrange();
  } else
    Area.Size = newsize;
}

void View::SetHidden(bool _hidden, bool rearrange=TRUE)
{
  if (_hidden == Hidden)
    return;
  Hidden = _hidden;
  if (rearrange) {
    if (_hidden)
      Hide();
    else
      Show();
    if (parent)
      parent->ReArrange(TRUE);
  }
}

Point View::RelativePoint(const Point& absolute)
{
  Point p(absolute.X-GetX()+Area.Origin.X, absolute.Y-GetY()+Area.Origin.Y);
  return p;
}

void View::Choose()
{
  if (choosecommand == cmInvalid)
    return;
  Select();
  Event e;
  e.type = evnCommand;
  e.command = choosecommand;
  PutEvent(e);
}

#ifdef DEBUGVision
static int aaa = 0;
#endif

void View::ReArrange(bool /* CallParent=TRUE */, View * /* except = 0 */)
{
  DEBUGPVision((aaa++) << "rearr: " << GetX() << ", " << GetY() << " " << Area.Size.X << "x" << Area.Size.Y);
  Area.Size = MinSize;
};

SelectableView::SelectableView()
{
//  SetColor(RGB(0xb0, 0, 0), ciHighlightedBG);
}

bool SelectableView::HandleEvent(Event& event)
{
  if ((!Visible) || Hidden)
    return TRUE;
  if (!View::HandleEvent(event))
    return FALSE;
  if (!Selectable)
    return TRUE;
  if ((event.type == evMousemove) || (event.type == evMouseclick)) {
    Point p = RelativePoint(event.mousepos);
    if (Area.Inside(p)) {
      if (event.type == evMousemove)
	Select();
      else
	Choose();
      return FALSE;
    }
    return TRUE;
  }
  if (event.type == evKeypress) {
    switch (event.key) {
    case ktEnter:
    case ktSpace:
      Choose();
      return FALSE;
    case ktDown:
      SelectNext(TRUE);
      return FALSE;
    case ktUp:
      SelectNext(FALSE);
      return FALSE;
    }
  }
  return TRUE;
}

inline const int View::GetX() const
{
  int X = Area.Origin.X;
  if (parent)
    X += parent->GetX(); // GetX() is virtual from View
  return X;
}

inline const int View::GetY() const
{
  int Y = Area.Origin.Y;
  if (parent)
    Y += parent->GetY();
  return Y;
}

void View::SendDataChanged()
{
  Event e(this);
  e.type = evnCommand;
  e.command = cmDataChanged;
  e.to = parent;
  PutEvent(e);
}
