/*
 * Copyright (c) 1995-2006 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL").
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define __NO_VERSION__

#include <nnpfs/nnpfs_locl.h>
#include <nnpfs/nnpfs_dev.h>
#include <nnpfs/nnpfs_dirent.h>
#include <nnpfs/nnpfs_syscalls.h>
#include <linux/binfmts.h>
#include <linux/file.h>
#include <linux/pagemap.h>

#ifdef LINUX2_5
#include <linux/page-flags.h>
#endif

#ifdef RCSID
RCSID("$Id: nnpfs_inodeops.c,v 1.216 2006/04/04 14:02:57 tol Exp $");
#endif

static int
nnpfs_fsync(struct file *file, struct dentry *dentry, int datasync);

static int
do_fsync(struct nnpfs *nnpfsp, struct nnpfs_node *xn, u_int flag);

static int
nnpfs_open(struct inode *i, struct file *f);


#ifdef LINUX2_5

static int
nnpfs_create(struct inode *dir, struct dentry *dentry,
	     int mode, struct nameidata *nd);

static struct dentry *
nnpfs_lookup (struct inode *dir, struct dentry *dentry, struct nameidata *nd);

static int
nnpfs_permission(struct inode *inode, int mode, struct nameidata *nd);

static int
nnpfs_d_revalidate(struct dentry *dentry, struct nameidata *nd);

#else /* !LINUX2_5 */

static int
nnpfs_create (struct inode * dir, struct dentry *dentry, int mode);

static struct dentry *
nnpfs_lookup (struct inode *dir, struct dentry *dentry);

static int
nnpfs_permission(struct inode *inode, int mode);

static int
nnpfs_d_revalidate(struct dentry *dentry, int flags);

static int
nnpfs_revalidate_inode(struct dentry *dentry);

#endif /* !LINUX2_5 */

static int
nnpfs_mkdir(struct inode * dir, struct dentry *dentry, int mode);

static int
nnpfs_rmdir(struct inode * dir, struct dentry *dentry);

static int
nnpfs_readdir(struct file * file, void * dirent, filldir_t filldir);

static int
nnpfs_rename (struct inode * old_dir, struct dentry *old_dentry,
	     struct inode * new_dir, struct dentry *new_dentry);

static int
nnpfs_unlink (struct inode * dir, struct dentry *dentry);

static int nnpfs_symlink(struct inode *dir, struct dentry *dentry,
		       const char *symname);

static int nnpfs_link(struct dentry *old_dentry,
		    struct inode *dir, struct dentry *dentry);

static int
nnpfs_release_file (struct inode *inode, struct file *file);

static int
nnpfs_flush (struct file *file);

static int nnpfs_file_mmap (struct file *file,
			  struct vm_area_struct *vma);

static int
nnpfs_readlink(struct dentry *dentry, char *buffer, int buflen);

static int
nnpfs_d_delete(struct dentry *dentry);

static void
nnpfs_d_release(struct dentry *dentry);

static ssize_t
nnpfs_write_file(struct file *file, const char *buf, size_t count, loff_t *ppos);

static ssize_t
nnpfs_read_file(struct file *file, char *buf, size_t count, loff_t *ppos);

#ifdef LINUX2_5
static ssize_t
nnpfs_sendfile(struct file *file, loff_t *ppos, size_t count,
	       read_actor_t actor, void *target);
#endif

static int
#ifdef HAVE_GETATTR_THREE_ARGS
nnpfs_getattr(struct vfsmount *mnt, struct dentry *, struct kstat *stat);
#else
nnpfs_getattr (struct dentry *dentry, struct iattr *attr);
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,13)
static void *
#else
static int
#endif
nnpfs_follow_link (struct dentry *dentry, struct nameidata *nd);

/*
 * Return 1 if the `vma' can cause a write to the filesystem, 0 if not.
 */

static int
nnpfs_mightwrite_p (struct vm_area_struct *vma)
{
    if (vma->vm_flags & VM_MAYWRITE && vma->vm_flags & VM_SHARED)
	return 1;
    return 0;
}

/*
 * When we close the mmap:ed memory, flush the data to the fileserver
 */

static void
nnpfs_vma_close (struct vm_area_struct *vma)
{
    int error;
    struct file *file = vma->vm_file;
    struct inode *inode = file->f_dentry->d_inode;
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(inode);
    struct nnpfs_node *xn = VNODE_TO_XNODE(inode);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_vma_close\n"));

    /*
     * I really want to hook the nnpfs_as_writepage, but then everything
     * will be cache:ed in the wrong struct address_space
     */

    if (file->f_mode & FMODE_WRITE && nnpfs_mightwrite_p(vma)) {
	error = do_fsync(nnpfsp, xn, NNPFS_WRITE);
	if (error) {
	    NNPFSDEB(XDEBVNOPS, ("nnpfs_vma_close: do_fsync returned %d\n",
			       error));
	}
    }
}

static struct vm_operations_struct nnpfs_file_vm_ops = {
    nopage:	filemap_nopage,
    close:	nnpfs_vma_close,
};


/*
 * File operations
 *
 * Common to all linux versions
 */

struct file_operations nnpfs_file_operations = {
    read:	nnpfs_read_file,
    write:	nnpfs_write_file,
    mmap:	nnpfs_file_mmap,
    open:	nnpfs_open,
    flush:	nnpfs_flush,
    release:	nnpfs_release_file,
    fsync:	nnpfs_fsync,
#ifdef LINUX2_5
    sendfile:	nnpfs_sendfile,
#endif
};

struct file_operations nnpfs_dead_operations = {
};

struct file_operations nnpfs_dir_operations = {
    readdir:	nnpfs_readdir,
    flush:	nnpfs_flush,
};

struct file_operations nnpfs_link_operations = {
    flush: nnpfs_flush,
};

/*
 * Inode operations
 */

struct inode_operations nnpfs_file_inode_operations = {
    permission:		nnpfs_permission,
    setattr:		nnpfs_setattr,
    getattr:		nnpfs_getattr,
#ifndef LINUX2_5
    revalidate:		nnpfs_revalidate_inode,
#endif
};

struct inode_operations nnpfs_dir_inode_operations = {
    create:		nnpfs_create,
    lookup:		nnpfs_lookup,
    link:		nnpfs_link,
    unlink: 		nnpfs_unlink,
    symlink: 		nnpfs_symlink,
    mkdir:		nnpfs_mkdir,
    rmdir: 		nnpfs_rmdir,
    rename: 		nnpfs_rename,
    permission: 	nnpfs_permission,
    setattr: 		nnpfs_setattr,
    getattr: 		nnpfs_getattr,
#ifndef LINUX2_5
    revalidate:		nnpfs_revalidate_inode,
#endif
};

struct inode_operations nnpfs_dead_inode_operations = {
};

struct inode_operations nnpfs_link_inode_operations = {
    readlink: 		nnpfs_readlink,
    follow_link: 	nnpfs_follow_link,
#ifdef LINUX2_5
    put_link: 		page_put_link,
#else
    revalidate:		nnpfs_revalidate_inode,
#endif
};

struct dentry_operations nnpfs_dentry_operations = {
    d_revalidate: 	nnpfs_d_revalidate,
    d_delete	: 	nnpfs_d_delete,
    d_release	: 	nnpfs_d_release,
};

/*
 *
 */

static void
nnpfs_print_path(struct dentry *dentry)
{
    NNPFSDEB(XDEBVNOPS, ("path: %.*s/%.*s\n",
		       (int)dentry->d_parent->d_name.len,
		       dentry->d_parent->d_name.name,
		       (int)dentry->d_name.len,
		       dentry->d_name.name));
}

/*
 *
 */

void
nnpfs_print_lock(char *s, struct semaphore *sem)
{
    NNPFSDEB(XDEBLOCK, ("lock: %s sem: %p count: %d\n",
		      s, sem, (int)atomic_read(&sem->count)));
}

/*
 *
 */

int
nnpfs_d_init (struct dentry *dentry)
{
    struct nnpfs_dentry_data *dentry_data;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_d_init: dentry: %p\n", dentry));
    dentry_data = nnpfs_alloc(sizeof(*dentry_data), NNPFS_MEM_DENTRY);
    if (dentry_data == NULL)
        return -ENOMEM;
    memset(dentry_data, 0, sizeof(*dentry_data));
    dentry->d_op = &nnpfs_dentry_operations;
    dentry_data->xd_flags = 0;
    dentry->d_fsdata = dentry_data;
    return 0;
}

/*
 *
 */

static void
nnpfs_d_release(struct dentry *dentry)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_d_release: dentry: %p\n", dentry));
    nnpfs_free(dentry->d_fsdata, NNPFS_MEM_DENTRY);
    dentry->d_fsdata = NULL;
}

/*
 * check if we are live
 */

static int
nnpfs_inode_valid(struct inode *inode)
{
    int error = 0;
    if (VNODE_TO_XNODE(inode) == NULL || NNPFS_FROM_VNODE(inode) == NULL)
	error = nnpfs_fetch_xnode(inode);
    if (error == 0
	&& (VNODE_TO_XNODE(inode) == NULL || NNPFS_FROM_VNODE(inode) == NULL))
	return -ENODEV;

    return error;
}

/*
 * nnpfs_lookup now returns a dentry.
 */

static struct dentry *
#ifdef LINUX2_5
nnpfs_lookup (struct inode *dir, struct dentry *dentry, struct nameidata *nd)
#else
nnpfs_lookup (struct inode *dir, struct dentry *dentry)
#endif
{
    struct nnpfs_message_getnode msg;
    struct nnpfs *nnpfsp;
    struct dentry *new_dentry;
    int error = 0;
    
    struct nnpfs_node *d;
    
    NNPFSDEB(XDEBVNOPS, ("nnpfs_lookup: %p name: %.*s dir: %p\n",
		       dentry, (int)dentry->d_name.len, dentry->d_name.name,
		       dir));

    if (dentry->d_name.len >= NNPFS_MAX_NAME)
	return ERR_PTR(-ENAMETOOLONG);

    error = nnpfs_inode_valid(dir);
    if (error != 0)
	return ERR_PTR(error);

    nnpfsp = NNPFS_FROM_VNODE(dir);
    d = VNODE_TO_XNODE(dir);

    do {

	msg.header.opcode = NNPFS_MSG_GETNODE;
        
	msg.cred.uid = current->uid;
	msg.cred.pag = nnpfs_get_pag();
	msg.parent_handle = d->handle;
        
	strlcpy(msg.name, dentry->d_name.name, sizeof(msg.name));

	new_dentry = d_lookup(dentry->d_parent, &dentry->d_name);

	if (new_dentry &&
	    (DENTRY_TO_XDENTRY(new_dentry)->xd_flags & NNPFS_XD_ENTRY_VALID))
	    break;
	if (new_dentry)
	    dput(new_dentry);

	NNPFSDEB(XDEBVNOPS, ("nnpfs_lookup: sending getnode rpc, dentry: %p\n",
			   dentry));
        error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
	NNPFSDEB(XDEBVNOPS, ("nnpfs_lookup: getnode rpc done, dentry: %p\n",
			   dentry));
        
        if (error == 0)
            error = ((struct nnpfs_message_wakeup *) &msg)->error;
    } while (error == 0);

    if (error == -ENOENT) {
        NNPFSDEB(XDEBVNOPS, ("nnpfs_lookup: leaving negative cache\n"));
	
	new_dentry = dentry;
	error = nnpfs_d_init(new_dentry);
	if (error)
	    return ERR_PTR(error);

	d_add(new_dentry, NULL);
	
	DENTRY_TO_XDENTRY(new_dentry)->xd_flags |= 
	    NNPFS_XD_ENTRY_VALID|NNPFS_XD_NAME_VALID;
        return NULL;
    }
    if (error) {
        NNPFSDEB(XDEBVNOPS, ("error %d", error));
	return ERR_PTR(error);
    }
    return new_dentry;
}

/*
 *
 */

static int
nnpfs_open_valid(struct inode *vp, u_int tok)
{
  struct nnpfs *nnpfsp;
  struct nnpfs_node *xn;
  int error = 0;

  error = nnpfs_inode_valid(vp);
  if (error)
      return error;

  nnpfsp = NNPFS_FROM_VNODE(vp);
  xn = VNODE_TO_XNODE(vp);
  
  NNPFSDEB(XDEBVFOPS, ("nnpfs_open_valid: tokens 0x%x\n", xn->tokens));

  do {
    if (!NNPFS_TOKEN_GOT(xn, tok))
      {
	struct nnpfs_message_open msg;
	msg.header.opcode = NNPFS_MSG_OPEN;
	msg.cred.uid = current->uid;
	msg.cred.pag = nnpfs_get_pag();
	msg.handle = xn->handle;
	msg.tokens = tok;
	error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
	if (error == 0)
	  error = ((struct nnpfs_message_wakeup *) &msg)->error;
      }
    else
      {
	goto done;
      }
  } while (error == 0);

done:
  NNPFSDEB(XDEBVFOPS, ("nnpfs_open_valid: exit tokens 0x%x\n", xn->tokens));
  return error;
}

/*
 *
 */

static int
nnpfs_open(struct inode *i, struct file *f)
{
  int ret;
  int done = 0;
  int tries = 0;

  NNPFSDEB(XDEBVNOPS, ("nnpfs_open inode: %p f->f_mode: %d aliases:",
		       i, f->f_mode));
  nnpfs_print_aliases(i);

  while (!done) {
      if (f->f_mode & FMODE_WRITE)
	  ret = nnpfs_open_valid(i, NNPFS_OPEN_NW);
      else
	  ret = nnpfs_open_valid(i, NNPFS_OPEN_NR);
      
      done = 1; /* common case */

#ifdef LINUX2_5
      if (ret == 0) {
	  f->f_mapping = i->i_mapping;
	  
	  if (f->f_mapping->a_ops == &nnpfs_null_aops) {
	      if (++tries > 4) {
		  printk("nnpfs_open: failed, aops was null\n");
		  return -ENODEV;
	      }
	      
	      done = 0; /* bad aops, try again */
	      printk("nnpfs_open: avoided using null aops\n");
	  }
      }
#endif
  }

  return ret;
}

/*
 *
 */

static int
do_fsync(struct nnpfs *nnpfsp, struct nnpfs_node *xn, u_int flag)
{
    int error;
    struct nnpfs_message_putdata msg;

    if (xn == NULL)
	return -ENODEV;

    msg.header.opcode = NNPFS_MSG_PUTDATA;
    msg.cred.uid = current->uid;
    msg.cred.pag = nnpfs_get_pag();
    msg.handle = xn->handle;
    msg.flag = flag;
    XA_CLEAR(&msg.attr);
    XA_SET_MTIME(&msg.attr, NNPFS_GET_TIME_SEC(XNODE_TO_VNODE(xn)->i_mtime));

    error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
    if (error == 0)
	error = ((struct nnpfs_message_wakeup *) & msg)->error;

    if (error == 0)
	xn->flags &= ~NNPFS_DATA_DIRTY;

    NNPFSDEB(XDEBVNOPS, ("do_fsync error:%d\n", error));

    return error;
}

/*
 *
 */

static int
nnpfs_fsync(struct file *file, struct dentry *dentry, int datasync)
{
    struct inode *inode = DENTRY_TO_INODE(dentry);
    struct nnpfs *nnpfsp;
    struct nnpfs_node *xn;
    int error = 0;

    error = nnpfs_inode_valid(inode);
    if (error)
	return error;

    nnpfsp = NNPFS_FROM_VNODE(inode);
    xn = VNODE_TO_XNODE(inode);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_fsync: 0x%p\n", inode));
    NNPFSDEB(XDEBVNOPS, ("nnpfs_fsync: name: %.*s aliases:",
		       (int)dentry->d_name.len, dentry->d_name.name));
    nnpfs_print_aliases(inode);

    if (xn->flags & NNPFS_DATA_DIRTY)
	error = do_fsync(nnpfsp, xn, NNPFS_WRITE | NNPFS_FSYNC);
    return error;
}

/*
 *
 */

static int
nnpfs_attr_valid(struct inode * vp, u_int tok)
{
    struct nnpfs *nnpfsp;
    struct nnpfs_node *xn;
    int error = 0;
    nnpfs_pag_t pag;

    error = nnpfs_inode_valid(vp);
    if (error)
	return error;
    
    nnpfsp = NNPFS_FROM_VNODE(vp);
    xn = VNODE_TO_XNODE(vp);

    pag = nnpfs_get_pag();

    do {
        if (!NNPFS_TOKEN_GOT(xn, tok)) {
            struct nnpfs_message_getattr msg;

            msg.header.opcode = NNPFS_MSG_GETATTR;
            msg.cred.uid = current->uid;
            msg.cred.pag = pag;
            msg.handle = xn->handle;
            error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
            if (error == 0)
                error = ((struct nnpfs_message_wakeup *) & msg)->error;
        } else {
            goto done;
        }
    } while (error == 0);

done:
    return error;
}

/*
 *
 */

static int
nnpfs_rights_valid(struct inode * vp, nnpfs_pag_t pag)
{
    struct nnpfs *nnpfsp;
    struct nnpfs_node *xn;
    int error = 0;
    NNPFSDEB(XDEBVNOPS, ("pag: %d\n", pag));

    error = nnpfs_inode_valid(vp);
    if (error)
	return error;
    
    nnpfsp = NNPFS_FROM_VNODE(vp);
    xn = VNODE_TO_XNODE(vp);

    do {
        if (!nnpfs_has_pag(xn, pag))
        {
            struct nnpfs_message_getattr msg;

            msg.header.opcode = NNPFS_MSG_GETATTR;
            msg.cred.uid = current->uid;
            msg.cred.pag = pag;
            msg.handle = xn->handle;
            error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
            if (error == 0)
                error = ((struct nnpfs_message_wakeup *) & msg)->error;
        }
        else {
            goto done;
        }
    } while (error == 0);

done:
    return error;
}

/*
 *
 */

static int
check_rights (nnpfs_rights rights, int mode)
{
    int error = 0;

    if (mode & MAY_READ)
	if ((rights & NNPFS_RIGHT_R) == 0)
	    error = -EACCES;
    if (mode & MAY_WRITE)
	if ((rights & NNPFS_RIGHT_W) == 0)
	    error = -EACCES;
    if (mode & MAY_EXEC)
	if ((rights & NNPFS_RIGHT_X) == 0)
	    error = -EACCES;
    return error;
}

/*
 *
 */

static int
#ifdef LINUX2_5
nnpfs_permission(struct inode *inode, int mode, struct nameidata *nd)
#else
nnpfs_permission(struct inode *inode, int mode)
#endif
{
    int error = 0;
    nnpfs_pag_t pag = nnpfs_get_pag();
    
    NNPFSDEB(XDEBVNOPS, ("nnpfs_access (%p) mode = 0%o aliases:", inode, mode));
    nnpfs_print_aliases(inode);

    error = nnpfs_attr_valid(inode, NNPFS_ATTR_R);
    if (error == 0) {
	struct nnpfs_node *xn = VNODE_TO_XNODE(inode);
	int i;

	error = check_rights (xn->anonrights, mode);
	
	if (error == 0)
	    goto done;

	NNPFSDEB(XDEBVNOPS, ("nnpfs_access anonaccess failed\n"));

	nnpfs_rights_valid(inode, pag); /* ignore error */
	
	error = -EACCES;
	
	for (i = 0; i < NNPFS_MAXRIGHTS; i++)
	    if (xn->id[i] == pag) {
		error = check_rights (xn->rights[i], mode);
		break;
	    }
    }

done:
    NNPFSDEB(XDEBVNOPS, ("nnpfs_access(0%o) = %d\n", mode, error));
    return error;
}

/*
 *
 */

static int
nnpfs_data_valid(struct inode *vp, u_int tok, loff_t want_offset)
{
    struct nnpfs *nnpfsp;
    struct nnpfs_node *xn;
    int error = 0;
    loff_t offset;
    struct nnpfs_message_getdata msg;

    error = nnpfs_inode_valid(vp);
    if (error != 0)
	return error;
    
    nnpfsp = NNPFS_FROM_VNODE(vp);
    xn = VNODE_TO_XNODE(vp);

    do {
	offset = want_offset;
	if (NNPFS_TOKEN_GOT(xn, tok|NNPFS_ATTR_R) && offset > xn->attr.xa_size)
	    offset = xn->attr.xa_size;

	NNPFSDEB(XDEBVNOPS, ("nnpfs_data_valid: offset: want %lld has %lld, "
			   "tokens: want %x has %x length: %lld\n",
			   (long long) offset, (long long) xn->offset,
			   tok, xn->tokens,
			   (long long) xn->attr.xa_size));

	if (NNPFS_TOKEN_GOT(xn, tok))
	    if(offset <= xn->offset || xn->attr.xa_type == NNPFS_FILE_DIR)
		break;

	msg.header.opcode = NNPFS_MSG_GETDATA;
	msg.cred.uid = current->uid;
	msg.cred.pag = nnpfs_get_pag();
	msg.handle = xn->handle;
	msg.tokens = tok;
	msg.offset = offset;
	error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
	if (error == 0)
	    error = ((struct nnpfs_message_wakeup *) &msg)->error;
	
    } while (error == 0);
    
    return error;
}

/*
 *
 */

#ifndef LINUX2_5
static int
nnpfs_revalidate_inode(struct dentry *dentry)
{
    struct inode *inode = DENTRY_TO_INODE(dentry);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_revalidate_node: dentry: %p inode: %p\n",
		       dentry, inode));

    return nnpfs_attr_valid(inode, NNPFS_ATTR_R);
}
#endif /* !LINUX2_5 */

/*
 * Called when the cached node has been changed, to update the relevant
 * part of the `overlaying' node.
 */

static void
update_from_cache_node (struct inode *inode)
{
    struct inode *t_inode;

    t_inode = DENTRY_TO_INODE(DATA_FROM_VNODE(inode));

    inode->i_size   = t_inode->i_size;
    inode->i_blocks = inode->i_size >> I_BLOCKS_BITS; /* / I_BLOCKS_UNIT */
    VNODE_TO_XNODE(inode)->flags |= NNPFS_DATA_DIRTY;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,16)
    inode->i_mtime  = t_inode->i_mtime;
#endif
}

/*
 *
 */

int nnpfs_file_mmap(struct file * file, struct vm_area_struct * vma)
{
    int flags, error = 0;
    struct inode *inode = file->f_dentry->d_inode;
    struct address_space *mapping = inode->i_mapping;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_mmap inode: %p\n", inode));
    nnpfs_print_path(file->f_dentry);
    NNPFSDEB(XDEBVNOPS, ("nnpfs_mmap aliases:"));
    nnpfs_print_aliases(inode);
    
    if (nnpfs_mightwrite_p(vma))
	flags = NNPFS_DATA_W;
    else
	flags = NNPFS_DATA_R;

    error = nnpfs_data_valid(inode, flags, inode->i_size /* XXX */);
    if (error)
	goto done;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_mmap: data valid\n"));

    /* ~ inlined generic_file_mmap */
    if (nnpfs_mightwrite_p(vma)) {
	if (!mapping->a_ops->writepage)
	    return -EINVAL;
    }
    if (!mapping->a_ops->readpage)
	return -ENOEXEC;

    vma->vm_ops = &nnpfs_file_vm_ops;

 done:
    NNPFSDEB(XDEBVNOPS, ("nnpfs_mmap: done %d\n", error));

    return error;
}

/*
 *
 */

static ssize_t
nnpfs_read_file(struct file *file, char *buf,size_t count, loff_t *ppos)
{
    int error = 0;
    struct inode *inode = file->f_dentry->d_inode;
    struct nnpfs_node *xn = VNODE_TO_XNODE(inode);
    
    if (xn != NULL)
	NNPFSDEB(XDEBVNOPS, ("nnpfs_read_file: tokens: 0x%x\n", xn->tokens));

    error = nnpfs_data_valid(inode, NNPFS_DATA_R, *ppos + count);
    if (error) {
	NNPFSDEB(XDEBVNOPS, ("nnpfs_read_file: data not valid %d\n", error));
	return error;
    }
    
#ifdef LINUX2_5
    if (file->f_mapping->a_ops->readpage == NULL) {
        printk("nnpfs_read_file: bad aops\n");
        return -EINVAL;
    }
#endif

    error = generic_file_read (file, buf, count, ppos);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_read_file: error = %d\n", error));
    return error;
}

/*
 *
 */

static ssize_t
nnpfs_write_file(struct file *file, const char *buf,size_t count, loff_t *ppos)
{
    int error = 0;
    struct inode *inode = file->f_dentry->d_inode;
    struct nnpfs_node *xn = VNODE_TO_XNODE(inode);
    
    if (xn != NULL)
	NNPFSDEB(XDEBVNOPS, ("nnpfs_write_file: tokens: 0x%x\n", xn->tokens));

    error = nnpfs_data_valid(inode, NNPFS_DATA_W, inode->i_size);
    if (error) {
	NNPFSDEB(XDEBVNOPS, ("nnpfs_write_file: data not valid %d\n", error));
	return error;
    }

    error = generic_file_write (file, buf, count, ppos);

    if (error == count)
	update_from_cache_node(inode);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_write_file: error = %d\n", error));
    return error;
}

static int
#ifdef LINUX2_5
nnpfs_create(struct inode *dir, struct dentry *dentry,
	     int mode, struct nameidata *nd)
#else
nnpfs_create (struct inode * dir, struct dentry *dentry, int mode)
#endif
{
    struct nnpfs *nnpfsp;
    struct nnpfs_node *xn;
    int error = 0;
    struct nnpfs_message_create msg;

    if (!dir)
	return -ENOENT;

    error = nnpfs_inode_valid(dir);
    if (error)
	return error;
	
    nnpfsp = NNPFS_FROM_VNODE(dir);
    xn = VNODE_TO_XNODE(dir);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_create: (%s, %d) dir(%p):",
		       dentry->d_name.name, dentry->d_name.len,
		       dir));
    nnpfs_print_aliases(dir);

    msg.header.opcode = NNPFS_MSG_CREATE;
    msg.parent_handle = xn->handle;
    if (strlcpy(msg.name, dentry->d_name.name, sizeof(msg.name)) >= NNPFS_MAX_NAME)
	return -ENAMETOOLONG;

    XA_CLEAR(&msg.attr);
    XA_SET_MODE(&msg.attr, mode);
    XA_SET_TYPE(&msg.attr, NNPFS_FILE_REG);
    XA_SET_GID(&msg.attr, current->fsgid);
    msg.mode = 0;		/* XXX */
    msg.cred.uid = current->uid;
    msg.cred.pag = nnpfs_get_pag();
    error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
    if (error == 0)
	error = ((struct nnpfs_message_wakeup *) & msg)->error;

    /* XXX should this really be here with new style dcache insert */
    if (DENTRY_TO_XDENTRY(dentry)->xd_flags == 0) {
	printk(KERN_EMERG "NNPFS Panic: nnpfs_create: dentry not valid\n");
    }

    return error;
    
}

static int
nnpfs_unlink (struct inode * dir, struct dentry *dentry)
{
    struct nnpfs_message_remove msg;
    struct nnpfs *nnpfsp;
    struct nnpfs_node *xn;
    int error;

    error = nnpfs_inode_valid(dir);
    if (error)
	return error;
	
    nnpfsp = NNPFS_FROM_VNODE(dir);
    xn = VNODE_TO_XNODE(dir);
    
    nnpfs_print_path(dentry);
    NNPFSDEB(XDEBVNOPS, ("nnpfs_remove: dentry: %p aliases:", dentry));
    nnpfs_print_aliases(dentry->d_inode);
    NNPFSDEB(XDEBVNOPS, ("nnpfs_remove: dir: %p aliases:", dir));
    nnpfs_print_aliases(dir);
    
    msg.header.opcode = NNPFS_MSG_REMOVE;
    msg.parent_handle = xn->handle;
    if (strlcpy(msg.name, dentry->d_name.name, sizeof(msg.name)) >= NNPFS_MAX_NAME)
	return -ENAMETOOLONG;

    msg.cred.uid = current->uid;
    msg.cred.pag = nnpfs_get_pag();
    error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
    if (error == 0)
        error = ((struct nnpfs_message_wakeup *) &msg)->error;

#ifndef LINUX2_5   /* 2.6 d_delete:s in vfs_unlink */
    if (error == 0) {
	NNPFSDEB(XDEBVNOPS, ("nnpfs_remove: aliases:"));
	nnpfs_print_aliases(dentry->d_inode);
	d_delete(dentry);	
    }
#endif

    return error;
}

int
nnpfs_rename (struct inode * old_dir, struct dentry *old_dentry,
	     struct inode * new_dir, struct dentry *new_dentry)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(old_dir);
    struct nnpfs_message_rename msg;
    int error;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_rename old"));
    nnpfs_print_path(old_dentry);
    NNPFSDEB(XDEBVNOPS, ("nnpfs_rename: dentry: %p aliases:", old_dentry));
    if (old_dentry->d_inode)
	nnpfs_print_aliases(old_dentry->d_inode);
    else
	NNPFSDEB(XDEBVNOPS, ("\n"));
    NNPFSDEB(XDEBVNOPS, ("nnpfs_rename: dir: %p aliases:", old_dir));
    nnpfs_print_aliases(old_dir);
    NNPFSDEB(XDEBVNOPS, ("nnpfs_rename new"));
    nnpfs_print_path(new_dentry);
    NNPFSDEB(XDEBVNOPS, ("nnpfs_rename: dentry: %p aliases:", new_dentry));
    if (new_dentry->d_inode)
	nnpfs_print_aliases(new_dentry->d_inode);
    else
	NNPFSDEB(XDEBVNOPS, ("\n"));
    NNPFSDEB(XDEBVNOPS, ("nnpfs_rename: dir: %p aliases:", new_dir));
    nnpfs_print_aliases(new_dir);

    msg.header.opcode = NNPFS_MSG_RENAME;
    msg.old_parent_handle = VNODE_TO_XNODE(old_dir)->handle;
    if (strlcpy(msg.old_name, old_dentry->d_name.name, sizeof(msg.old_name)) >= NNPFS_MAX_NAME)
	return -ENAMETOOLONG;

    msg.new_parent_handle = VNODE_TO_XNODE(new_dir)->handle;
    if (strlcpy(msg.new_name, new_dentry->d_name.name, sizeof(msg.new_name)) >= NNPFS_MAX_NAME)
	return -ENAMETOOLONG;
	
    msg.cred.uid = current->uid;
    msg.cred.pag = nnpfs_get_pag();
    error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
    if (error == 0)
	error = ((struct nnpfs_message_wakeup *) &msg)->error;

    return error;
}

static int
nnpfs_mkdir(struct inode * dir, struct dentry *dentry, int mode)
{
    struct nnpfs *nnpfsp;
    struct nnpfs_node *xn;
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_mkdir name:%s\n", dentry->d_name.name));

    if (!dir)
	return -ENOENT;
    if (dentry->d_name.len >= NNPFS_MAX_NAME)
	return -ENAMETOOLONG;

    error = nnpfs_inode_valid(dir);
    if (error)
	return error;

    nnpfsp = NNPFS_FROM_VNODE(dir);
    xn = VNODE_TO_XNODE(dir);

    {
	struct nnpfs_message_mkdir msg;

	msg.header.opcode = NNPFS_MSG_MKDIR;
	msg.parent_handle = xn->handle;
	if (strlcpy(msg.name, dentry->d_name.name, sizeof(msg.name)) >= NNPFS_MAX_NAME)
	    return -ENAMETOOLONG;

	XA_CLEAR(&msg.attr);
	XA_SET_MODE(&msg.attr, mode);
	XA_SET_TYPE(&msg.attr, NNPFS_FILE_DIR);
	XA_SET_GID(&msg.attr, current->fsgid);

	msg.cred.uid = current->uid;
	msg.cred.pag = nnpfs_get_pag();
	error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
	if (error == 0)
	    error = ((struct nnpfs_message_wakeup *) & msg)->error;

	/* XXX should this really be here */
	if (DENTRY_TO_XDENTRY(dentry)->xd_flags == 0) {
	    printk(KERN_EMERG "NNPFS Panic: nnpfs_mkdir: dentry not valid\n");
	}
    }

    return error;
}

static int
nnpfs_rmdir(struct inode * dir, struct dentry *dentry)
{
    struct nnpfs *nnpfsp;
    struct nnpfs_node *xn;
    struct nnpfs_message_rmdir msg;
    int error;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_rmdir: (%.*s)\n",
		       (int)dentry->d_name.len,
		       dentry->d_name.name));

    if (dentry->d_name.len >= NNPFS_MAX_NAME)
	return -ENAMETOOLONG;

    error = nnpfs_inode_valid(dir);
    if (error)
	return error;

    nnpfsp = NNPFS_FROM_VNODE(dir);
    xn = VNODE_TO_XNODE(dir);

    msg.header.opcode = NNPFS_MSG_RMDIR;
    msg.parent_handle = xn->handle;
    if (strlcpy(msg.name, dentry->d_name.name, sizeof(msg.name)) >= NNPFS_MAX_NAME)
	    return -ENAMETOOLONG;
    msg.cred.uid = current->uid;
    msg.cred.pag = nnpfs_get_pag();
    error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
    if (error == 0)
	error = ((struct nnpfs_message_wakeup *) &msg)->error;

    if (error == 0)
	d_delete(dentry);

    return error;
}

static int nnpfs_link(struct dentry *old_dentry,
		    struct inode *dir, struct dentry *dentry)
{
    struct nnpfs *nnpfsp;
    struct nnpfs_node *xn;
    struct nnpfs_node *from_xn;
    int error = 0;
    const char *name = dentry->d_name.name;
    int len = dentry->d_name.len;
    struct inode *oldinode = DENTRY_TO_INODE(old_dentry);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_link name:%.*s\n", len, name));

    error = nnpfs_inode_valid(dir);
    if (error)
	return error;

    nnpfsp = NNPFS_FROM_VNODE(dir);
    xn = VNODE_TO_XNODE(dir);
    from_xn = VNODE_TO_XNODE(oldinode);

    if (from_xn == NULL)
	return -ENODEV;

    {
	struct nnpfs_message_link msg;

	msg.header.opcode = NNPFS_MSG_LINK;
	msg.parent_handle = xn->handle;
	msg.from_handle = from_xn->handle;
	if (strlcpy(msg.name, name, sizeof(msg.name)) >= NNPFS_MAX_NAME)
	    return -ENAMETOOLONG;

	msg.cred.uid = current->uid;
	msg.cred.pag = nnpfs_get_pag();
	error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
	if (error == 0)
	    error = ((struct nnpfs_message_wakeup *) & msg)->error;
    }

    return error;
}

static int nnpfs_symlink(struct inode *dir, struct dentry *dentry,
		       const char *symname)
{
    struct nnpfs *nnpfsp;
    struct nnpfs_node *xn;
    int error = 0;
    const char *name = dentry->d_name.name;
    int len = dentry->d_name.len;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_symlink name:%.*s\n", len, name));

    error = nnpfs_inode_valid(dir);
    if (error)
	return error;

    nnpfsp = NNPFS_FROM_VNODE(dir);
    xn = VNODE_TO_XNODE(dir);

    {
	struct nnpfs_message_symlink msg;

	msg.header.opcode = NNPFS_MSG_SYMLINK;
	msg.parent_handle = xn->handle;
	if (strlcpy(msg.name, name, sizeof(msg.name)) >= NNPFS_MAX_NAME)
	    return -ENAMETOOLONG;
	if (strlcpy(msg.contents, symname, sizeof(msg.contents)) >= NNPFS_MAX_SYMLINK_CONTENT)
	    return -ENAMETOOLONG;
	XA_CLEAR(&msg.attr);
	XA_SET_MODE(&msg.attr, 0777);
	XA_SET_TYPE(&msg.attr, NNPFS_FILE_LNK);
	XA_SET_GID(&msg.attr, current->fsgid);

	msg.cred.uid = current->uid;
	msg.cred.pag = nnpfs_get_pag();
	error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
	if (error == 0)
	    error = ((struct nnpfs_message_wakeup *) & msg)->error;
	/* XXX should this really be here */
	if (DENTRY_TO_XDENTRY(dentry)->xd_flags == 0) {
	    printk(KERN_EMERG "NNPFS Panic: nnpfs_symlink: dentry not valid\n");
	}
    }

    return error;
}

static int
nnpfs_readdir(struct file * file, void * dirent, filldir_t filldir)
{
    int error = 0;
    int filldir_error;
    off_t offset, begin_offset;
    struct inode *inode = file->f_dentry->d_inode;
    char *buf;
    struct dentry *t;
    struct page *page;
    off_t inpage;
    off_t page_num;
    struct address_space *mapping;

    NNPFSDEB(XDEBREADDIR, ("nnpfs_readdir\n"));
    
    error = nnpfs_data_valid(inode, NNPFS_DATA_R, inode->i_size);
    if (error)
	return error;

    t = DATA_FROM_VNODE(inode);
    if (t == NULL) {
	printk(KERN_EMERG "NNPFS Panic: readdir on NULL cache file\n");
	return -EINVAL;
    }
    NNPFSDEB(XDEBREADDIR, ("nnpfs_readdir: inode: %p data:%p\n", inode, t));
    
    while (file->f_pos < DENTRY_TO_INODE(t)->i_size && error >= 0) {
	NNPFSDEB(XDEBREADDIR,
	       ("nnpfs_readdir file->f_pos: %d t->i_size: %d\n",
		(int) file->f_pos, (int) DENTRY_TO_INODE(t)->i_size));
	begin_offset = file->f_pos &~ (NNPFS_DIRENT_BLOCKSIZE - 1);
	offset = file->f_pos & (NNPFS_DIRENT_BLOCKSIZE - 1);
	file->f_pos = begin_offset;
	NNPFSDEB(XDEBREADDIR, ("nnpfs_readdir begin_offset: %d offset: %d\n",
			     (int)begin_offset, (int)offset));
	mapping = t->d_inode->i_mapping;
	inpage = file->f_pos & (PAGE_CACHE_SIZE-1);
	page_num = file->f_pos >> PAGE_CACHE_SHIFT;

	NNPFSDEB(XDEBREADDIR,
	       ("nnpfs_readdir inpage: %d page_num: %d\n",
		(int) inpage,
		(int) page_num));

	page = read_cache_page (mapping, page_num,
				(filler_t *)mapping->a_ops->readpage,
				t);
	if (IS_ERR(page)) {
	    printk(KERN_EMERG "nnpfs_readdir: read_cache_page failed: %ld\n",
		   PTR_ERR(page));
	    return PTR_ERR(page);
	}
#ifdef LINUX2_5
	wait_on_page_locked(page);
	if (!PageUptodate(page)) {
	    printk(KERN_EMERG "nnpfs_readdir: page not uptodate\n");
	    page_cache_release (page);
	    return -EIO;
	}
#else
	wait_on_page(page);
	if (!Page_Uptodate(page)) {
	    printk(KERN_EMERG "nnpfs_readdir: page not uptodate\n");
	    page_cache_release (page);
	    return -EIO;
	}
#endif
	buf = (char *)kmap (page);
	buf += inpage;
	error = NNPFS_DIRENT_BLOCKSIZE;
	file->f_pos += error;

	NNPFSDEB(XDEBREADDIR, ("nnpfs_readdir error: %d\n",error));
	while (offset < NNPFS_DIRENT_BLOCKSIZE) {
	    struct nnpfs_dirent *xdirent = (struct nnpfs_dirent *) (buf + offset);
	    NNPFSDEB(XDEBREADDIR,
		   ("nnpfs_readdir offset: %d namlen: %d offset2: %d\n",
		    (int) offset,
		    (int) xdirent->d_namlen,
		    (int) (offset+begin_offset)));
	    if (xdirent->d_fileno != 0
		&& (filldir_error = filldir (dirent,
					     xdirent->d_name,
					     xdirent->d_namlen,
					     offset+begin_offset,
					     xdirent->d_fileno,
					     DT_UNKNOWN)) < 0) {
		NNPFSDEB(XDEBREADDIR,
		       ("nnpfs_readdir filldir: %d\n", filldir_error));
		file->f_pos = offset + begin_offset;
		kunmap (page);
		page_cache_release (page);
		return 0;
	    }
	    offset += xdirent->d_reclen;

	    if (xdirent->d_reclen == 0) {
		struct inode *cache_inode = DENTRY_TO_INODE(t);
		printk(KERN_EMERG "NNPFS Panic: "
		       "empty dirent at %lld in nnpfs_readdir, pos %d size %d\n",
		       (long long)offset, (int)file->f_pos,
		       (int)cache_inode->i_size);
		NNPFSDEB(XDEBVNOPS, ("inode: %p aliases:", inode));
		nnpfs_print_aliases(inode);

		NNPFSDEB(XDEBVNOPS, ("cache_inode: %p aliases:", cache_inode));
		nnpfs_print_aliases(cache_inode);

		error = -EIO;
		break;
	    }
	}
	kunmap (page);
	page_cache_release (page);
    }
    
    return error;
}
   
/*
 *
 */

static int
nnpfs_readlink (struct dentry *dentry, char *buffer, int buflen)
{
    int error = 0;
    struct inode *inode = DENTRY_TO_INODE(dentry);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_readlink\n"));
    
    error = nnpfs_data_valid(inode, NNPFS_DATA_R, inode->i_size);
    if (error == 0) {
#ifdef LINUX2_5
	error = generic_readlink(dentry, buffer, buflen);
#else
	error = page_readlink(dentry, buffer, buflen);
#endif
    }

    return error;
}

/*
 *
 */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,13)
static void *
#else
static int
#endif
nnpfs_follow_link (struct dentry *dentry,
		 struct nameidata *nd)
{
    int error = 0;
    struct inode *inode = DENTRY_TO_INODE(dentry);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_follow_link\n"));
    
    error = nnpfs_data_valid(inode, NNPFS_DATA_R, inode->i_size);
    if (error)
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,13)
       return ERR_PTR(error);
#else
       return error;
#endif

#ifdef LINUX2_5
    return page_follow_link_light(dentry, nd);
#else
    return page_follow_link(dentry, nd);
#endif
}

/*
 * fetch the attributes of `dentry' and store them in `attr'.
 */

static int
#ifdef HAVE_GETATTR_THREE_ARGS
nnpfs_getattr(struct vfsmount *mnt, struct dentry *dentry, struct kstat *stat)
#else
nnpfs_getattr(struct dentry *dentry, struct iattr *attr)
#endif
{
    struct inode *inode = DENTRY_TO_INODE(dentry);
    int error;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_getattr\n"));

    error = nnpfs_attr_valid(inode, NNPFS_ATTR_R);
    if (error == 0) {
#ifdef HAVE_GETATTR_THREE_ARGS
#ifdef LINUX2_5
	stat->dev = inode->i_sb->s_dev;
	stat->rdev = inode->i_rdev;
#else
        stat->dev = kdev_t_to_nr(inode->i_dev);
	stat->rdev = kdev_t_to_nr(inode->i_rdev);
#endif
	stat->ino = inode->i_ino;
	stat->mode = inode->i_mode;
	stat->nlink = inode->i_nlink;
	stat->uid = inode->i_uid;
	stat->gid = inode->i_gid;
	stat->atime = inode->i_atime;
	stat->mtime = inode->i_mtime;
	stat->ctime = inode->i_ctime;
	stat->size = inode->i_size;
	stat->blocks = inode->i_blocks;
	stat->blksize = inode->i_blksize;
#else /* !HAVE_GETATTR_THREE_ARGS */
	attr->ia_valid = ATTR_MODE | ATTR_UID | ATTR_GID
	    | ATTR_SIZE | ATTR_ATIME | ATTR_MTIME
	    | ATTR_CTIME;
	attr->ia_mode  = inode->i_mode;
	attr->ia_uid   = inode->i_uid;
	attr->ia_gid   = inode->i_gid;
	attr->ia_size  = inode->i_size;
	attr->ia_atime = inode->i_atime;
	attr->ia_mtime = inode->i_mtime;
	attr->ia_ctime = inode->i_ctime;
#endif /* HAVE_GETATTR_THREE_ARGS */
    }

    return error;
}

/*
 * set the attributes of `dentry' to `attr'
 */

int
nnpfs_setattr (struct dentry *dentry, struct iattr *attr)
{
    struct inode *inode = DENTRY_TO_INODE(dentry);
    struct nnpfs_node *xn;
    struct nnpfs *nnpfsp;
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_setattr\n"));

    error = nnpfs_inode_valid(inode);
    if (error)
	return error;

    xn = VNODE_TO_XNODE(inode);
    nnpfsp = NNPFS_FROM_VNODE(inode);


    if (NNPFS_TOKEN_GOT(xn, NNPFS_ATTR_W)) {
        /* Update attributes and mark them dirty. */
        VNODE_TO_XNODE(inode)->flags |= NNPFS_ATTR_DIRTY;
	return -EINVAL;                /* XXX not yet implemented */
    } else {
        struct nnpfs_message_putattr msg;

        msg.header.opcode = NNPFS_MSG_PUTATTR;
	msg.cred.uid = current->uid;
	msg.cred.pag = nnpfs_get_pag();
        msg.handle = xn->handle;
        vattr2nnpfs_attr(attr, &msg.attr);

	if (NNPFS_TOKEN_GOT(xn, NNPFS_DATA_R)) {
	    if (S_ISREG(inode->i_mode)) {
		if (attr->ia_valid & ATTR_SIZE)
		    XA_SET_SIZE(&msg.attr,  attr->ia_size);
		else
		    XA_SET_SIZE(&msg.attr,  inode->i_size);
	    }

	    if (attr->ia_valid & ATTR_MTIME_SET)
		XA_SET_MTIME(&msg.attr, NNPFS_GET_TIME_SEC(attr->ia_mtime));
            else
		XA_SET_MTIME(&msg.attr, NNPFS_GET_TIME_SEC(inode->i_mtime));
	}

        error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
        if (error == 0) {
            error = ((struct nnpfs_message_wakeup *) & msg)->error;
	    
	}
    }
    
    return error;
}

/*
 *
 */

static int
nnpfs_flush (struct file *file)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_flush\n"));

    if (file && file->f_dentry && file->f_dentry->d_inode)
	return nnpfs_release_file(file->f_dentry->d_inode, file);
    else
	return 0;
}

/*
 *
 */

static int
nnpfs_release_file (struct inode *inode, struct file *file)
{
    struct nnpfs *nnpfsp;
    struct nnpfs_node *xn;
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_release_file\n"));

    error = nnpfs_inode_valid(inode);
    if (error)
	return error;

    nnpfsp = NNPFS_FROM_VNODE(inode);
    xn = VNODE_TO_XNODE(inode);

    NNPFSDEB(XDEBVNOPS,
	   ("nnpfs_release_file inode->i_count: %d inode: %p aliases:",
	    nnpfs_icount(inode), inode));
    nnpfs_print_aliases(inode);
    
    if (file->f_mode & FMODE_WRITE
	&& (xn->flags & NNPFS_DATA_DIRTY
	    || (DATA_FROM_VNODE(inode) != NULL
		&& DATA_FROM_VNODE(inode)->d_inode != NULL
		&& DATA_FROM_VNODE(inode)->d_inode->i_state & (I_DIRTY_DATASYNC | I_DIRTY_PAGES))))
	error = do_fsync(nnpfsp, xn, NNPFS_WRITE);
        
    if (error)
	NNPFSDEB(XDEBVNOPS, ("nnpfs_release_file error: %d\n",error));

    return error;
}

/*
 * Return 1 if `dentry' is still valid, otherwise 0.
 */

static int
#ifdef LINUX2_5
nnpfs_d_revalidate(struct dentry *dentry, struct nameidata *nd) 
#else
nnpfs_d_revalidate(struct dentry *dentry, int flags)
#endif
{
    struct inode *inode = DENTRY_TO_INODE(dentry);
    NNPFSDEB(XDEBVNOPS, ("nnpfs_d_revalidate %p \"%.*s\" (inode %p)\n",
		       dentry,
		       (int)dentry->d_name.len,
		       dentry->d_name.name,
		       inode));

    /* If it's the root it's going to be valid. */
    if (IS_ROOT(dentry))
	return 1;

    if ((DENTRY_TO_XDENTRY(dentry)->xd_flags & NNPFS_XD_ENTRY_VALID) == 0) {
	if (nnpfs_dcount(dentry) == 1) /* We are the only one */
	    d_drop(dentry);
	return 0;
    }

    if (DENTRY_TO_XDENTRY(dentry)->xd_flags & NNPFS_XD_NAME_VALID)
	return 1;

    if (inode) {
	int error;

	nnpfs_print_aliases(inode);

	error = nnpfs_attr_valid(inode, NNPFS_ATTR_R);
	if (error) {
	    NNPFSDEB(XDEBVNOPS, ("invalid\n"));
	    return 0;
	} else {
	    NNPFSDEB(XDEBVNOPS, ("valid\n"));
	    return 1;
	}
    } else {
	/*
	 * Negative entries are always valid,
	 * they are cleared in nnpfs_message_invalidnode
	 */
        NNPFSDEB(XDEBVNOPS, ("nnpfs_d_revalidate: negative entry\n"));
	return 1;
    }
    printk(KERN_EMERG "NNPFS Panic: a case in nnpfs_d_revalidate has not "
	   "been taken care of\n");
    return 0;
}

static
int
nnpfs_d_delete(struct dentry *dentry)
{
    struct inode *inode;
    struct nnpfs_node *xn;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_d_delete: dentry %p(%.*s): "
		       "all references dropped\n",
		       dentry,
		       (int)dentry->d_name.len,
		       dentry->d_name.name));

    inode = dentry->d_inode;
    if (inode) {
	xn = VNODE_TO_XNODE(inode);

	if (xn
	    && (xn->flags & NNPFS_STALE) != 0
	    && nnpfs_icount(inode) == 1)
	{
	    NNPFSDEB(XDEBVNOPS, ("nnpfs_d_delete: stale\n"));
	    /* this will cause a iput where d_delete is non void */
	    return 1;
	}
    }
    return 0;
}

#ifdef LINUX2_5
static ssize_t
nnpfs_sendfile(struct file *file, loff_t *ppos, size_t count,
	       read_actor_t actor, void *target)
{
    int error = 0;
    struct inode *inode = file->f_dentry->d_inode;
    struct nnpfs_node *xn = VNODE_TO_XNODE(inode);
    
    if (xn != NULL)
	NNPFSDEB(XDEBVNOPS, ("nnpfs_sendfile: tokens: 0x%x\n", xn->tokens));

    error = nnpfs_data_valid(inode, NNPFS_DATA_R, *ppos + count);
    if (error) {
	NNPFSDEB(XDEBVNOPS, ("nnpfs_sendfile: data not valid %d\n", error));
	return error;
    }

    error = generic_file_sendfile(file, ppos, count, actor, target);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_sendfile: error = %d\n", error));
    return error;
}
#endif
