/*
 * Copyright (c) 1995 - 2005 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL").
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define __NO_VERSION__

#include <nnpfs/nnpfs_locl.h>
#include <nnpfs/nnpfs_common.h>
#include <nnpfs/nnpfs_fs.h>
#include <nnpfs/nnpfs_deb.h>
#include <nnpfs/nnpfs_dev.h>

#ifdef RCSID
RCSID("$Id: nnpfs_node.c,v 1.80 2005/11/15 15:42:34 tol Exp $");
#endif

#define xn_list_entry(ptr, type, member) \
        ((type *)((char *)(ptr)-(unsigned long)(&((type *)0)->member)))

#define xn_hash(node) \
  (((node)->a+(node)->b+(node)->c+(node)->d) % XN_HASHSIZE)

struct address_space_operations nnpfs_null_aops;

void
xn_purge(struct nnpfs *nnpfsp)
{
    struct list_head *next;
    struct list_head *last;
    struct list_head *xh;
    struct nnpfs_nodelist *xf;
    struct nnpfs_node *xn;
    int iter = 0;
    int i;

    NNPFSDEB(XDEBNODE, ("xn_purge: enter\n"));
    down(&nnpfsp->node_sem);
    for (iter = 0; iter < XN_CLEANUP_ITERS; iter++) {
	if (nnpfsp->nnodes <= 1)
	    iter = XN_CLEANUP_ITERS;

	for (i = 0; i < XN_HASHSIZE; i++) {
	    int counter = nnpfsp->nnodes;
	    xh = &nnpfsp->node_head.node_lists[i];
	    last = xh->prev;

	    while (counter-- > 0) {
		if (list_empty(xh))
		    break;
		
		next = xh->next;
		
		list_del(next);
		list_add_tail(next, xh);
		
		xf = list_entry(next, struct nnpfs_nodelist, node_list);
		
		xn = xn_list_entry(xf, struct nnpfs_node, nodes);

		up(&nnpfsp->node_sem);
		nnpfs_force_invalid_xnode(xn);
		down(&nnpfsp->node_sem);
		
		if (next == last)
		    break;
	    }
	}
    }
    
    xn_init_head(&nnpfsp->node_head);

    up(&nnpfsp->node_sem);
    NNPFSDEB(XDEBNODE, ("xn_purge: exit\n"));
}

int
xn_deletedp(struct nnpfs_node *node)
{
    NNPFSDEB(XDEBNODE, ("xn_deletedp: enter\n"));
    return !list_empty(&node->nodes.node_list);
}

void
xn_init_head(struct nnpfs_nodelist_head *head)
{
    int i;
    NNPFSDEB(XDEBNODE, ("xn_init_head: enter\n"));
    for (i=0; i < XN_HASHSIZE; i++) {
	INIT_LIST_HEAD(&head->node_lists[i]);
    }
    NNPFSDEB(XDEBNODE, ("xn_init_head: exit\n"));
}

void 
xn_insert(struct nnpfs_nodelist_head *head, struct nnpfs_node *node)
{
    NNPFSDEB(XDEBNODE, ("xn_insert: enter %d.%d.%d.%d\n", 
			node->handle.a,
			node->handle.b,
			node->handle.c,
			node->handle.d));
    list_add(&node->nodes.node_list, &head->node_lists[xn_hash(&node->handle)]);
    NNPFSDEB(XDEBNODE, ("xn_insert: exit\n"));
}

struct nnpfs_node *
xn_lookup(struct nnpfs_nodelist_head *head, struct nnpfs_handle *handlep)
{
    struct list_head *xh;
    struct nnpfs_nodelist *xf;
    struct nnpfs_node *xn;
    int hashvalue;

    NNPFSDEB(XDEBNODE, ("xn_lookup: enter %d.%d.%d.%d\n",
			handlep->a,
			handlep->b,
			handlep->c,
			handlep->d));
    hashvalue = xn_hash(handlep);
    NNPFSDEB(XDEBNODE, ("xn_lookup: hashvalue = %d\n", hashvalue));
    list_for_each(xh, &head->node_lists[hashvalue]) {
	if (xh == NULL)
	    BUG();
	
	xf = list_entry(xh, struct nnpfs_nodelist, node_list);
	xn = xn_list_entry(xf, struct nnpfs_node, nodes);
	
	if (nnpfs_handle_eq(&xn->handle, handlep)) {
	    NNPFSDEB(XDEBNODE, ("xn_lookup: found node %p\n", xn));
	    return xn;
	}
    }
    NNPFSDEB(XDEBNODE, ("xn_lookup: exit\n"));
    return NULL;
}

void
xn_delete(struct nnpfs_nodelist_head *head, struct nnpfs_node *node)
{
    NNPFSDEB(XDEBNODE, ("xn_delete: enter\n"));
    list_del_init(&node->nodes.node_list);
    NNPFSDEB(XDEBNODE, ("xn_delete:exit\n"));
}


/*
 * Copy the attributes from `attr' into `inode', setting the fields
 * that weren't set in `attr' to reasonable defaults.
 */

void
nnpfs_attr2inode(const struct nnpfs_attr *attr, struct inode *inode,
		 int clear_node)
{
    if (clear_node) {
#ifdef LINUX2_5
	struct timespec notime = {0};
#else
	time_t notime = 0;
#endif

	inode->i_mode   = 0;
	inode->i_uid    = 0;
	inode->i_gid    = 0;
	inode->i_nlink  = 1;
	inode->i_size   = 0;
	inode->i_atime  = notime;
	inode->i_mtime  = notime;
	inode->i_ctime  = notime;
	inode->i_blocks = 0;
    }
    if (XA_VALID_MODE(attr))
	inode->i_mode   = attr->xa_mode;
    if (XA_VALID_UID(attr))
	inode->i_uid    = attr->xa_uid;
    if (XA_VALID_GID(attr))
	inode->i_gid    = attr->xa_gid;
    if (XA_VALID_NLINK(attr))
	inode->i_nlink  = attr->xa_nlink;
    if (XA_VALID_SIZE(attr)) {
	inode->i_size   = attr->xa_size;
	inode->i_blocks = inode->i_size >> I_BLOCKS_BITS; /* / I_BLOCKS_UNIT */
    }
    if (XA_VALID_ATIME(attr))
	NNPFS_SET_TIME(inode->i_atime,attr->xa_atime);
    if (XA_VALID_MTIME(attr))
	NNPFS_SET_TIME(inode->i_mtime,attr->xa_mtime);
    if (XA_VALID_CTIME(attr))
	NNPFS_SET_TIME(inode->i_ctime,attr->xa_ctime);
    if (XA_VALID_FILEID(attr))
	inode->i_ino = attr->xa_fileid;
}

/*
 * Allocate a new inode (of the file system identified by `sb') and
 * return it, associated with `newnode'.  Return the `inode' or NULL.
 * The reference count on `inode' is incremented.
 */

static struct inode *
nnpfs_iget(struct super_block *sb, struct nnpfs_msg_node *node,
	   struct nnpfs_node *newnode)
{
    struct inode *inode;
    struct nnpfs_attr *attr = &node->attr;

    NNPFSDEB(XDEBNODE, ("nnpfs_iget sb: %p node: %p newnode: %p\n", 
			sb, node, newnode));
    if (sb == NULL) {
     	printk(KERN_EMERG "NNPFS Panic: nnpfs_iget: super block is NULL\n");
     	return NULL;
    }
    inode = new_inode (sb);
    if (inode == NULL) {
    	printk(KERN_EMERG "NNPFS Panic: nnpfs_iget: get_empty_inode failed\n");
    	return NULL;
    }
    inode->i_sb  = sb;
#ifndef LINUX2_5
    inode->i_dev = sb->s_dev;
#endif

    if (!XA_VALID_TYPE(attr)) {
	inode->i_op  = &nnpfs_dead_inode_operations;
	inode->i_fop = &nnpfs_dead_operations;
    } else if (attr->xa_type == NNPFS_FILE_REG) {
	inode->i_op  = &nnpfs_file_inode_operations;
	inode->i_fop = &nnpfs_file_operations;
        inode->i_mapping->a_ops = &nnpfs_null_aops;
    } else if (attr->xa_type == NNPFS_FILE_DIR) {
	inode->i_op  = &nnpfs_dir_inode_operations;
	inode->i_fop = &nnpfs_dir_operations;
        inode->i_mapping->a_ops = &nnpfs_null_aops;
    } else if (attr->xa_type == NNPFS_FILE_LNK) {
	inode->i_op  = &nnpfs_link_inode_operations;
	inode->i_fop = &nnpfs_link_operations;
	inode->i_mapping->a_ops = &nnpfs_null_aops;
    } else {
	inode->i_op  = &nnpfs_dead_inode_operations;
	inode->i_fop = &nnpfs_dead_operations;
    }
    
    VNODE_SET_XNODE(inode, newnode);

#ifdef I_NEW
    if (inode->i_state & I_NEW)
	unlock_new_inode(inode);
#endif

    return inode;
}

/*
 * Find the node identified by `node->handle' belong to the filesystem
 * `nnpfsp' or create a new one.  The node is returned with incremented
 * reference count.
 */

int
new_nnpfs_node(struct nnpfs *nnpfsp, struct nnpfs_msg_node *node,
	       struct inode *inode, struct nnpfs_node **xn)
{
    struct nnpfs_node *result;
    int newp = 0;
    int ret;

    NNPFSDEB(XDEBNODE, ("new_nnpfs_node %d.%d.%d.%d\n",
			node->handle.a,
			node->handle.b,
			node->handle.c,
			node->handle.d));
    
    /* Does not allow duplicates */
    ret = nnpfs_node_find(nnpfsp, &node->handle, &result);
    if (ret == -EISDIR) {
	*xn = NULL;
	return ret;
    }

    if (ret == -ENOENT) {
	newp = 1;

	result = nnpfs_alloc(sizeof(*result), NNPFS_MEM_XNODE);
	if (result == NULL) {
	    printk(KERN_EMERG "new_nnpfs_node: node allocation failed\n");
	    return -ENOMEM;
	}
	
	memset(result, 0, sizeof(*result));
	
	result->anonrights = node->anonrights;
	result->handle = node->handle;
	result->flags = 0;
	result->tokens = 0;
	result->offset = 0;
	INIT_LIST_HEAD(&result->inactive_list);

	if (inode == NULL)
	    inode = nnpfs_iget(NNPFS_TO_VFS(nnpfsp),node,result);
	else
	    VNODE_SET_XNODE(inode,result);

	result->vn = inode;
	result->nnpfsp = nnpfsp;
	/* XXX - handle this case better */
    } else {
	/* Node is already cached */
	nnpfs_iref(XNODE_TO_VNODE(result));
    }
    
    /* Init other fields */
    result->attr = node->attr;
    nnpfs_attr2inode (&result->attr, result->vn, 1);
    result->tokens = node->tokens;
    memmove(result->id, node->id, sizeof(result->id));
    memmove(result->rights, node->rights, sizeof(result->rights));
    
    if (newp) {
	down(&nnpfsp->node_sem);

	xn_insert(&nnpfsp->node_head, result);
	nnpfsp->nnodes++;

	up(&nnpfsp->node_sem);
    }

    *xn = result;
    return 0;
}

/*
 * free node
 * call with inactive_sem held
 */

void
nnpfs_free_node(struct nnpfs_node *xn)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_XNODE(xn);

    down(&nnpfsp->node_sem);

    if ((xn->flags & NNPFS_LIMBO) == 0)
	BUG();

    xn_delete(&nnpfsp->node_head, xn);
    nnpfsp->nnodes--;
  
    if (!list_empty(&xn->inactive_list))
	list_del(&xn->inactive_list);

    up(&nnpfsp->node_sem);

    nnpfs_free(xn, NNPFS_MEM_XNODE);
}


/*
 * remove everything about `node'
 * call with inactive_sem held
 */

void
clear_nnpfs_node(struct nnpfs_node *node)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_XNODE(node);

    NNPFSDEB(XDEBNODE, ("clear_nnpfs_node starting\n"));
    
    down(&nnpfsp->node_sem);
    if (node->flags & NNPFS_LIMBO)
	BUG();

    node->flags = NNPFS_LIMBO;

    /* XXX Really need to put back dirty data first. */
    NNPFS_TOKEN_CLEAR(node, ~0,
		      NNPFS_OPEN_MASK | NNPFS_ATTR_MASK |
		      NNPFS_DATA_MASK | NNPFS_LOCK_MASK);
    if (DATA_FROM_XNODE(node)) {
	NNPFS_RESET_I_MAPPING(XNODE_TO_VNODE(node));
	dput(DATA_FROM_XNODE(node));
	DATA_FROM_XNODE(node) = NULL;
    }
    NNPFSDEB(XDEBNODE, ("clear_nnpfs_node xnode: %p inode:%p\n",
			node, XNODE_TO_VNODE(node)));
    VNODE_SET_XNODE(XNODE_TO_VNODE(node), NULL);
    node->vn = NULL;
    
    up(&nnpfsp->node_sem);

    NNPFSDEB(XDEBNODE, ("clear_nnpfs_node done\n"));
}

/*
 * Throw all nodes of the filesystem `nnpfsp'.
 */

void
free_all_nnpfs_nodes(struct nnpfs *nnpfsp)
{

    NNPFSDEB(XDEBNODE, ("free_all_nnpfs_nodes starting\n"));
    
    xn_purge(nnpfsp);

#if 0
    if ((nnpfsp->status & NNPFS_DEVOPEN) == 0)
	nnpfs_empty_inactive_queue(nnpfsp);
#endif

    nnpfsp->root = NULL;
    
    NNPFSDEB(XDEBNODE, ("free_all_nnpfs_nodes done\n"));
}

/*
 * find the node, identifed by `handlep' in `nnpfsp', and put it in
 * *node
 *
 * return 0, -ENOENT, or -EISDIR (for limbo nodes)
 */

int
nnpfs_node_find(struct nnpfs *nnpfsp, nnpfs_handle *handlep, 
		struct nnpfs_node **node)
{
    struct nnpfs_node *x;
    int ret = 0;
    
    down(&nnpfsp->node_sem);
    x = xn_lookup(&nnpfsp->node_head, handlep);

    if (x == NULL)
	ret = -ENOENT;
    else if (x->flags & NNPFS_LIMBO)
	ret = -EISDIR;

    up(&nnpfsp->node_sem);

    *node = x;

    return ret;
}

/*
 * Returns 1 if pag has any rights set in the node
 */

int
nnpfs_has_pag(const struct nnpfs_node *xn, nnpfs_pag_t pag)
{
    int i;

    if (xn == NULL)
	return 0;
    
    for (i = 0; i < NNPFS_MAXRIGHTS; i++)
	if (xn->id[i] == pag)
	    return 1;
    
    return 0;
}

/*
 * Return the number of users of the node `xn'.
 */

int
nnpfs_node_users(struct nnpfs_node *xnode)
{
    struct inode *inode;
    struct list_head *pos;
    int users = 0;

    if (xnode == NULL)
	return 0;

    inode = XNODE_TO_VNODE(xnode);
    
    list_for_each(pos, &inode->i_dentry) {
	struct dentry *dentry = list_entry(pos, struct dentry, d_alias);
	if (nnpfs_dcount(dentry))
	    users++;
    }
    return users;
}

void
nnpfs_print_nodestats(struct nnpfs *nnpfsp)
{
    struct list_head *next;
    struct list_head *xh;
    struct nnpfs_nodelist *xf;
    struct nnpfs_node *xn;
    int i;
    int total = 0;
    int used = 0;
    int nempty = 0;
    int maxlength = 0;

    down(&nnpfsp->node_sem);
    for (i = 0; i < XN_HASHSIZE; i++) {
	int counter = 0;	
	xh = &nnpfsp->node_head.node_lists[i];

	if (list_empty(xh)) {
	    nempty++;
	    continue;
	}

	list_for_each(next, xh) {
	    struct inode *inode;
	    xf = list_entry(next, struct nnpfs_nodelist, node_list);
	    xn = xn_list_entry(xf, struct nnpfs_node, nodes);
	    
	    inode = XNODE_TO_VNODE(xn);
	    if (inode && nnpfs_icount(inode))
		used++;
	    total++;
	    counter++;
	}

	if (counter > maxlength)
	    maxlength = counter;
    }
    
    up(&nnpfsp->node_sem);

    printk("nnodes: %d\n", nnpfsp->nnodes);
    printk("counted nodes: %d used, %d total\n", used, total);
    printk("buckets: %d, empty: %d, maxlength: %d\n",
	   XN_HASHSIZE, nempty, maxlength);
}
