/*
 * Copyright (c) 1997, 1998
 *      Jens Arvid Nilsson, jnilsson@ludd.luth.se. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*$Id: util.c,v 1.14 1998/05/24 14:14:16 jens Exp $*/  
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/uio.h>
#include <net/if.h>
#include <net/if_dl.h>
#include <net/bpf.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <signal.h>
#include <unistd.h>
#include "error.h"
#include "util.h"
#include "readconfig.h"
#include "jque.h"

extern int daemon_proc;
struct if_list sIf_list;
typedef void    Sigfunc(int);   /* for signal handlers */


int
ip_pass_rules(	struct in_addr *from_ip, 
				struct in_addr *for_ip, 
				struct if_elm *pIf_elm)
{
	u_int32_t arpmask 	= pIf_elm->arpmask.s_addr;
	u_int32_t arpnet	= pIf_elm->arpnet.s_addr;
	u_int32_t lfrom_ip	= from_ip->s_addr;
	u_int32_t lfor_ip	= for_ip->s_addr;
	struct ip_elm *pIp_elm;
	struct net_elm *pNet_elm;

	lfrom_ip = lfrom_ip;
#if 0
	printf("from ip: %s ", inet_ntoa(*from_ip));
	printf("for ip: %s\n", inet_ntoa(*for_ip));
	printf("if ip: %s ", inet_ntoa(pIp_rules->ifip));
	printf("if mask: %s\n", inet_ntoa(pIp_rules->ifmask));
	printf("arp mask: %s\n", inet_ntoa(pIp_rules->arpmask));
	printf("arp net: %s\n", inet_ntoa(pIp_rules->arpnet));
#endif

	for (pIp_elm = pIf_elm->sProxy.lh_first; pIp_elm != NULL; 
				pIp_elm = pIp_elm->entries.le_next) {
		u_int32_t proxy = pIp_elm->ip.s_addr;
		if (proxy == lfor_ip)
			return 1; /* proxy arpa fr denna */
	}

	if (!pIf_elm->have_arpnet)
		return 0;

	if ((lfor_ip & arpmask) != (arpnet & arpmask))
		return 0; /* lfor_ip ligger utanfr vad vi svarar som */

	for (pNet_elm = pIf_elm->sNo_net.lh_first; pNet_elm != NULL;
				pNet_elm = pNet_elm->entries.le_next) {
		u_int32_t no_net = pNet_elm->net.s_addr;
		u_int32_t no_mask = pNet_elm->mask.s_addr;
		if ((lfor_ip & no_mask) == (no_net & no_mask))
			return 0; /* Denna ip vill vi inte svara som */
	}

	for (pIp_elm = pIf_elm->sNo_answer.lh_first; pIp_elm != NULL; 
			pIp_elm = pIp_elm->entries.le_next) {
		u_int32_t no_answer = pIp_elm->ip.s_addr;
		if (no_answer == lfor_ip)
			return 0;
	}
	return 1; 
}

#define IF			0
#define ARPNET		1
#define NONET		2
#define DONT		3
#define PROXY		4
#define BROADCAST	5
#define MAX_CONF	5
#define FIELD_SIZE	12

static char fields[][FIELD_SIZE] = {
	"if",
	"arpnet",
	"nonet",
	"dont",
	"proxy",
	"broadcast"
""};

static void
nConf_if(void)
{
	struct if_elm *pIf_elm;
	struct ip_elm *pIp_elm;
	struct net_elm *pNet_elm;
	int have_proxy = 0;
	
	if (nJ_conf_idx() != IF)
		err_quit("config: row %d, expected if directive", nJ_conf_row());
	pIf_elm = Malloc(sizeof(*pIf_elm));
	pIf_elm->have_arpnet = 0;
	pIf_elm->broadcast = 0;
	LIST_INIT(&pIf_elm->sNo_answer);
	LIST_INIT(&pIf_elm->sProxy);
	LIST_INIT(&pIf_elm->sNo_net);

	szJ_Next_token();
	strncpy(pIf_elm->szDevice, szJ_token(), IFNAMSIZ - 1);
	if (szJ_next_token() != NULL)
		err_msg("Warning: line %d, garble at end of line ignored",
				nJ_conf_row());
	Get_if_info(pIf_elm->szDevice, &pIf_elm->ifhwa, NULL);

	for (nJ_next_row(); !bJ_eo_conf() && nJ_conf_idx() != IF; nJ_next_row())
		switch(nJ_conf_idx()) {
		case ARPNET:
			if (pIf_elm->have_arpnet++)
				err_quit("config row: %d allready have arpmask", nJ_conf_row());
			szJ_Next_token();
			Inet_aton(szJ_token(), &pIf_elm->arpnet);
			vJ_Expect_next_token("arpmask");
			szJ_Next_token();
			Inet_aton(szJ_token(), &pIf_elm->arpmask);
			break;
		case NONET:
			pNet_elm = Malloc(sizeof(*pNet_elm));
			szJ_Next_token();
			Inet_aton(szJ_token(), &pNet_elm->net);
			vJ_Expect_next_token("mask");
			szJ_Next_token();
			Inet_aton(szJ_token(), &pNet_elm->mask);
			LIST_INSERT_HEAD(&pIf_elm->sNo_net, pNet_elm, entries);
			break;
		case DONT:
			pIp_elm = Malloc(sizeof(*pIp_elm));
			szJ_Next_token();
			while (szJ_token() != NULL) {
				Inet_aton(szJ_token(), &pIp_elm->ip);
				LIST_INSERT_HEAD(&pIf_elm->sNo_answer, pIp_elm, entries);
				szJ_next_token();
				if (szJ_token() != NULL) 
					pIp_elm = Malloc(sizeof(*pIp_elm));
			}
			break;
		case PROXY:
			have_proxy++;
			pIp_elm = Malloc(sizeof(*pIp_elm));
			szJ_Next_token();
			while (szJ_token() != NULL) {
				Inet_aton(szJ_token(), &pIp_elm->ip);
				LIST_INSERT_HEAD(&pIf_elm->sProxy, pIp_elm, entries);
				szJ_next_token();
				if (szJ_token() != NULL) 
					pIp_elm = Malloc(sizeof(*pIp_elm));
			}
			break;
		case BROADCAST:
			pIf_elm->broadcast++;
			break;
		default:
			err_quit("config row %d, %s", nJ_conf_row(), szJ_conf_orig());
		}
	if (!(pIf_elm->have_arpnet || have_proxy))
		err_quit("config need arpnet or proxy for if = %s", pIf_elm->szDevice);
	LIST_INSERT_HEAD(&sIf_list, pIf_elm, entries);
}
	
void
Read_conf(FILE * pConfigFile)
{
	char	**szFields = (char **) fields;

	pJ_read_config(pConfigFile, szFields, FIELD_SIZE);
	LIST_INIT(&sIf_list);
	while ( !bJ_eo_conf() ) 
		nConf_if();
}

void * Malloc(size_t size)
{
	void *p = malloc(size);

	if (p == NULL)
		err_quit("Malloc: cannot malloc %d", size);
	return p;
}

void
IfIoctl(int fd, unsigned long request, char *argp)
{
	if (ioctl(fd, request, argp) < 0)
		err_sys("ioctl");
}

void
Inet_aton(const char *cp, struct in_addr *addr)
{
	if (inet_aton((char *)cp, addr) == 0)
		err_quit("invalid address %s", cp);
}

void
Ether_aton(const char *s, struct ether_addr *e)
{
	struct ether_addr *eth;

	eth = ether_aton((char *)s);
	if (eth == NULL)
		err_quit("invalid address %s", s);
	memmove(e, eth, 6);
}



Sigfunc *
signal(int signo, Sigfunc *func)
{
	struct sigaction	act, oact;

	act.sa_handler = func;
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	if (signo == SIGALRM) {
#ifdef	SA_INTERRUPT
		act.sa_flags |= SA_INTERRUPT;	/* SunOS 4.x */
#endif
	} else {
#ifdef	SA_RESTART
		act.sa_flags |= SA_RESTART;		/* SVR4, 44BSD */
#endif
	}
	if (sigaction(signo, &act, &oact) < 0)
		return(SIG_ERR);
	return(oact.sa_handler);
}
/* end signal */

Sigfunc *
Signal(int signo, Sigfunc *func)	/* for our signal() function */
{
	Sigfunc	*sigfunc;

	if ( (sigfunc = signal(signo, func)) == SIG_ERR)
		err_sys("signal error");
	return(sigfunc);
}

pid_t
Fork(void)
{
	pid_t   pid;

	if ( (pid = fork()) == -1)
		err_sys("fork error");
	return(pid);
}


void
daemon_init(const char *pname, int facility)
{
	int		i;
	pid_t	pid;

	if ( (pid = Fork()) != 0)
		exit(0);			/* parent terminates */

	/* 1st child continues */
	setsid();				/* become session leader */

	Signal(SIGHUP, SIG_IGN);
	if ( (pid = Fork()) != 0)
		exit(0);			/* 1st child terminates */

	/* 2nd child continues */
	daemon_proc = 1;		/* for our err_XXX() functions */

	chdir("/");				/* change working directory */

	umask(0);				/* clear our file mode creation mask */

	for (i = 0; i < 3; i++)
		close(i);

	openlog(pname, LOG_PID, facility);
}

void
Get_if_info(char *szDevice, struct ether_addr *ifhwa, struct in_addr *ifip)
{
    int     fd;
    struct ifreq ifrcopy;
    char    inbuf[8192];
    const struct sockaddr_dl *sdl = NULL;
    struct ifconf ifc;
    struct ifreq *ifr;
    int     i;
    struct sockaddr_in *sin;


    if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
        err_sys("socket: %s", szDevice);

    strncpy(ifrcopy.ifr_name, szDevice, IFNAMSIZ);

    IfIoctl(fd, SIOCGIFFLAGS, (char *) &ifrcopy);
    if (!(ifrcopy.ifr_flags & IFF_UP))
        err_quit("%s is down", szDevice);
    if (ifrcopy.ifr_flags & IFF_NOARP)
        err_quit("error: arp unenambled on %s", szDevice);

    if (ifip != NULL) {
        IfIoctl(fd, SIOCGIFADDR, (char *) &ifrcopy);
        if (ifrcopy.ifr_addr.sa_family != AF_INET)
            err_quit("error: %s not configured for AF_INET", szDevice);
        sin = (struct sockaddr_in *) &ifrcopy.ifr_addr;
        memcpy(ifip, &sin->sin_addr, sizeof(*ifip));
    }

    if (ifhwa == NULL)
        return;

    ifc.ifc_len = sizeof(inbuf);
    ifc.ifc_buf = inbuf;
    IfIoctl(fd, SIOCGIFCONF, (char *) &ifc);
    close(fd);
    ifr = ifc.ifc_req;
    for (i = 0; i < ifc.ifc_len;) {
        ifr = (struct ifreq *) ((caddr_t) ifc.ifc_req + i);
        i += sizeof(ifr->ifr_name) +
            max(ifr->ifr_addr.sa_len, sizeof(struct sockaddr));
        if (strncmp(ifr->ifr_name, szDevice, sizeof(ifr->ifr_name)) == 0) {
            /* nu har vi hittat den! */
            if (ifr->ifr_addr.sa_family == AF_LINK) {
                sdl = (const struct sockaddr_dl *) & ifr->ifr_addr;
                memcpy(ifhwa, LLADDR(sdl), 6);
            } else {
                err_quit("Get_if_info: not AF_LINK");
            }
            break;
        }
    }
}

