/*==========================================================================
 * Project: ATasm: atari cross assembler
 * File: inc_path.c
 *
 * Provides a linked list store search directories for .INCLUDEd files.
 *==========================================================================
 * Created: 08/03/2002 B. Watson
 * Modifications:
 *  08/05/02 mws added comment blocks, reformatted slightly
 *  03/03/03 mws fopen include now tries to open natural path first, then
 *               explores includes
 *  07/27/03 mws rewrote to reduce number and size of mallocs
 *==========================================================================*
 *  This program is free software; you can redistribute it and/or modify  
 *  it under the terms of the GNU General Public License as published by  
 *  the Free Software Foundation; either version 2 of the License, or     
 *  (at your option) any later version.                                   
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *==========================================================================*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "symbol.h"
#include "inc_path.h"
#include "atasm_err.h"

/*=========================================================================*
 * function init_inc_list
 * parameters: none
 *
 * initializes search directory list;
 *=========================================================================*/
inc_list *init_inc_list() {
  inc_list *head;
  
  head=(inc_list*)malloc(sizeof(inc_list));
  if (head==NULL)
    error("Cannot allocate memory to initialize include list",1);
  head->next = NULL;
  head->path = (char*)malloc(2);
  if (head->path==NULL)
    error("Cannot allocate memory to initialize include list",1);
  head->path[0]='.';
  head->path[1]=0;
  return head;
}

/*=========================================================================*
  funtion free_inc_list
  parameters: head - the head of the include list to free

  This cleans up the include list
 *=========================================================================*/
void free_inc_list() {
  inc_list *kill;
  while(includes) {
    kill=includes;
    includes=includes->next;
    free(kill->path);
    free(kill);
  }
  includes=NULL;
}

/*=========================================================================*
  funtion append_inc_patch
  parameters: head - the current list head;
              path - the path to append

  add another path to the include directory search path
 *=========================================================================*/
void append_inc_path(inc_list *head, char *path) {
  inc_list *append, *walk=head;

  while(walk->next)
    walk=walk->next;
  append=(inc_list *)malloc(sizeof(inc_list));
  if(!append)
    error("Cannot grow include list",1);
  append->path=(char*)malloc(strlen(path)+1);
  if (append->path == NULL)
    error("Cannot grow include list",1);
  strcpy(append->path, path);
  append->next=NULL;
  walk->next=append;
}

/*=========================================================================*
  funtion fopen_include
  parameters: head - the list to search
	      fname - the file to open

  attempts to open a file, checking all include paths
 *=========================================================================*/
FILE *fopen_include(inc_list *head, char *fname) {
  char errbuf[255];
  char *full_path;
  FILE *in;

  /* First, attempt to open file normally... */
  in=fopen(fname,"rt");
  if (in)
    return in;
  
  /* Now test with include paths... */
  full_path = (char*)malloc(MAX_PATH);
  if(full_path==NULL)
    error("Cannot allocate %d bytes in fopen_include",1);
  
  while(head) {
    full_path[0] = '\0';    
    strcpy(full_path, head->path);
    strcat(full_path, DIR_SEP);
    strcat(full_path, fname);
    in=fopen(full_path,"rt");
    if (in) {
      free(full_path);
      return in;
    }
    head=head->next;
  }
  free(full_path);
  snprintf(errbuf,255,"Cannot open file: '%s'", fname);
  error(errbuf, 1);
  return NULL; 
}
/*=========================================================================*/






