/*
 * Copyright (c) 2001-2005 Accense Technology, Inc. All rights reserved.
 *
 * (the Apache like license)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by
 *        Accense Technology, Inc. (http://accense.com/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Accense Technology" must not be used to endorse or promote
 *    products derived from this software without prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL ACCENSE BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * mod_auth_script
 *
 * This module makes it possible authentication/authorization to be done
 * by an external program. The external program can be provided as a CGI,
 * PHP or any other schemes which allow dynamic content to Apache. The program
 * SHOULD print some headers, and MUST NOT print any content body. Recognized
 * headers are as follows.
 *
 *   auth-script
 *       Authentication/authorization result (required)
 *           allow       access allowed
 *           deny        access denied
 *           prompt      access denied and cause browser to prompt the
 *                       browser built-in userid/password dialog
 *
 *   auth-script-user
 *       Set the "REMOTE_USER" CGI variable (optional, at most 1)
 *       The value of this header will be a value of "REMOTE_USER".
 *
 *   auth-script-custom-response
 *       Specify an error document for access denial (optional, at most 1)
 *           /...        internal URI
 *           http://...  external URL
 *           text...     simple text message to display
 *           "text...    simple text message to display
 *
 *   auth-script-debug
 *       Just print a debug message in the apache error_log (optional)
 *       Any number of debug message can be printed by repeating this
 *       header line. However, mod_cgi or other modules may merge them
 *       or ignore them except the last header line.
 *
 * The external program will receive following env variable.
 *
 *   AUTH_SCRIPT_URI    The authorization requesetd URI.
 *                      This is not same as REQUEST_URI, which is the
 *                      originally requested URI by the browser.
 *
 * This module provides following configuration directives:
 *
 *   AuthScriptFile  "OS path to the program"
 *       Specify the program to provide authentication/authorization.
 *       This path should be absolute path or relative to the ServerRoot.
 *
 *   AuthScriptURI   "virtual path"
 *       Specify the program to provide authentication/authorization.
 *       The script should be inside the web content tree.
 *
 *
 * Configuration should be like as follows. AuthType should be "Basic".
 * AuthName should be provided to prompt a browser dialog. Please note that
 * the "require" directive is required, but the actual content of the
 * directive is meaningless in this version of implementation.
 * 
 *   AuthType        Basic
 *   AuthName        "authentication realm"
 *   AuthScriptFile  "OS path to the program"
 *   Require         valid-user
 *
 *
 * This software was written by Shigeru Kanemoto <sgk@ppona.com>.
 *
 */

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_protocol.h"
#include "http_request.h"
#include "http_log.h"
#include <string.h>			/* strcmp() */
#include <sys/stat.h>

static const char* myname = "mod_auth_script";
#define MY_MARK myname,0

typedef struct {
  enum { type_unset, type_file, type_uri } type_;
  char* path_;
} config_rec;

static void*
dir_config(pool* p, char* d)
{
  config_rec* conf = (config_rec*)ap_pcalloc(p, sizeof(config_rec));
  conf->type_ = type_unset;
  conf->path_ = 0;			/* null pointer */
  return conf;
}

static const char*
config_file(cmd_parms* cmd, void* mconfig, char* arg)
{
  if (((config_rec*)mconfig)->path_)
    return "Path to the script already set.";

  ((config_rec*)mconfig)->type_ = type_file;
  ((config_rec*)mconfig)->path_ = ap_server_root_relative(cmd->pool, arg);
  return 0;
}

static const char*
config_uri(cmd_parms* cmd, void* mconfig, char* arg)
{
  if (((config_rec*)mconfig)->path_)
    return "Path to the script already set.";
  if (arg[0] != '/')
    return "URI should start with '/'.";

  ((config_rec*)mconfig)->type_ = type_uri;
  ((config_rec*)mconfig)->path_ = ap_pstrdup(cmd->pool, arg);
  return 0;
}

static const command_rec command_table[] = {
  { "AuthScriptFile",
    config_file, 0,
    OR_AUTHCFG, TAKE1,
    "Set an OS path to a CGI or PHP program to provide authentication/authorization function. The path can be absolute or relative to the ServerRoot." },
  { "AuthScriptURI",
    config_uri, 0,
    OR_AUTHCFG, TAKE1,
    "Set virtual path to a CGI or PHP program to provide authentication/authorization function." },
  { 0 }
};

module MODULE_VAR_EXPORT auth_script_module;

static int
callback_print_debug(void* rec, const char* key, const char* value)
{
  ap_log_rerror(MY_MARK, APLOG_NOTICE, (request_rec*)rec, "debug %s", value);
  return 1;				/* not zero */
}

static int
callback_copy_header(void* t, const char* key, const char* value)
{
  ap_table_add((table*)t, key, value);
  return 1;				/* not zero */
}

static int
check_user_id(request_rec *r)
{
  config_rec* conf;
  request_rec* subreq;
  const char* s;
  int st;

  /* check if there is a request loop. */
  for (subreq = r->main; subreq != 0; subreq = subreq->main) {
    if (strcmp(subreq->uri, r->uri) == 0) {
      ap_log_rerror(MY_MARK, APLOG_ERR, r, "request loop getting '%s'; the script cannot be inside the protected directory itself.", subreq->uri);
      return DECLINED;
    }
  }

  /* get config */
  conf = (config_rec*)ap_get_module_config(
    r->per_dir_config, &auth_script_module);
  if (conf->path_ == 0) {
    ap_log_rerror(MY_MARK, APLOG_ERR, r, "not configured properly");
    return DECLINED;			/* not configured properly */
  }

  /* check if not configured to use this module; thanks to mrueegg@sf. */
  if (conf->type_ == type_unset)
    return DECLINED;			/* not configured */

  /*
   * run the script as a sub request
   */
  /* create the sub request */
  subreq = (conf->type_ == type_file ?
    ap_sub_req_lookup_file(conf->path_, r) :
    ap_sub_req_lookup_uri(conf->path_, r));

  /* make a duplicate copy of the table to avoid overwrite. */
  subreq->headers_in = ap_copy_table(r->pool, r->headers_in);

  /* make sure the CGI don't expect stdin */
  ap_table_unset(subreq->headers_in, "content-type");
  ap_table_unset(subreq->headers_in, "content-length");
  subreq->remaining = 0;

  /* Pass this requested URI (not original URI) to the sub request.
   * The REQUEST_URI env variable is original_uri(r).
   * See apache source code util_script.c ap_add_cgi_vars(). */
  ap_table_setn(subreq->subprocess_env, "AUTH_SCRIPT_URI", r->uri);

  /* run */
  if ((st = ap_run_sub_req(subreq)) != OK) {
    ap_destroy_sub_req(subreq);
    ap_log_rerror(MY_MARK, APLOG_ERR, r, "error on script execution");
    return st;				/* script claims an error */
  }

  /*
   * read the output headers
   */
  /* copy set-cookie headers to r->headers_out */
  ap_table_do(callback_copy_header, (void*)r->headers_out,
    subreq->headers_out, "set-cookie", 0);
  ap_table_do(callback_copy_header, (void*)r->headers_out,
    subreq->err_headers_out, "set-cookie", 0);

  /* auth-script-debug */
  ap_table_do(callback_print_debug, (void*)r,
    subreq->headers_out, "auth-script-debug", 0);
  ap_table_do(callback_print_debug, (void*)r,
    subreq->err_headers_out, "auth-script-debug", 0);

  /* auth-script-custom-response */
  s = ap_table_get(subreq->headers_out, "auth-script-custom-response");
  if (s == 0)
    s = ap_table_get(subreq->err_headers_out, "auth-script-custom-response");
  if (s != 0) {
    char* ss;
    ss = ap_pstrdup(r->pool, s);
    ap_custom_response(r, HTTP_UNAUTHORIZED, ss);
    ap_custom_response(r, HTTP_PROXY_AUTHENTICATION_REQUIRED, ss);
  }

  /* auth-script-user */
  s = ap_table_get(subreq->headers_out, "auth-script-user");
  if (s == 0)
    s = ap_table_get(subreq->err_headers_out, "auth-script-user");
  if (s != 0)
    r->connection->user = ap_pstrdup(r->connection->pool, s);

  /*
   * auth-script
   */
  s = ap_table_get(subreq->headers_out, "auth-script");
  if (s == 0)
    s = ap_table_get(subreq->err_headers_out, "auth-script");
  ap_destroy_sub_req(subreq);
  if (s == 0) {
    ap_log_rerror(MY_MARK, APLOG_ERR, r, "no result from auth script");
    return DECLINED;			/* script do not provide the header */
  }

  /* authentication is ok if "auth-script:allow". */
  if (strcasecmp(s, "allow") == 0) {
    if (r->connection->user == 0) {
      /*
       * This is null because no "auth-script-user" header given.
       * Retrieve userid from header and set it to r->connection->user
       * This is done by calling following API. The returned value
       * and the result of variable 's' is useless.
       */
      (void)ap_get_basic_auth_pw(r, &s);
    }
    return OK;
  }

  /* just return deny if "auth-script:deny". */
  if (strcasecmp(s, "deny") == 0)
    return AUTH_REQUIRED;

  /* prompt the authentication dialog if "auth-script:prompt". */
  if (strcasecmp(s, "prompt") == 0) {
    ap_note_basic_auth_failure(r);
    return AUTH_REQUIRED;
  }

  /* other response is not allowed. */
  ap_log_rerror(MY_MARK, APLOG_ERR, r,
    "unrecognized response '%s' from auth script", s);
  return DECLINED;
}

static int
check_auth(request_rec *r)
{
  config_rec* conf;

  /* Thanks to "chuck.morris at ngc.com" */
  conf = (config_rec*)ap_get_module_config(r->per_dir_config, &auth_script_module);
  if (conf->type_ == type_unset) {
    /* we are not enabled, pass on authentication */
    return DECLINED;
  } else {
    /* don't do anything with Require if we run */
    return OK;
  }
}

module MODULE_VAR_EXPORT auth_script_module =
{
  STANDARD_MODULE_STUFF,
  0,				/* initializer */
  dir_config,			/* dir config creater */
  0,				/* dir merger --- default is to override */
  0,				/* server config */
  0,				/* merge server config */
  command_table,		/* command table */
  0,				/* handlers */
  0,				/* filename translation */
  check_user_id,		/* check_user_id */
  check_auth,			/* check auth */
  0,				/* check access */
  0,				/* type_checker */
  0,				/* fixups */
  0,				/* logger */
  0,				/* header parser */
  0,				/* child_init */
  0,				/* child_exit */
  0				/* post read-request */
};
