/*
 * Copyright (c) 2001 Tommy Bohlin <tommy@gatespace.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* girbil.c
 */

#include <irda.h>

/**********************************************************************
 * Constants
 **********************************************************************/

static const char id_girbil[]="girbil";

/* Control register 1 */
#define GIRBIL_TXEN      0x01 /* Enable transmitter */
#define GIRBIL_RXEN      0x02 /* Enable receiver */
#define GIRBIL_ECAN      0x04 /* Cancel self emmited data */
#define GIRBIL_ECHO      0x08 /* Echo control characters */

/* LED Current Register */
#define GIRBIL_HIGH      0x20
#define GIRBIL_MEDIUM    0x21
#define GIRBIL_LOW       0x22

/* Baud register */
#define GIRBIL_2400      0x30
#define GIRBIL_4800      0x31	
#define GIRBIL_9600      0x32
#define GIRBIL_19200     0x33
#define GIRBIL_38400     0x34	
#define GIRBIL_57600     0x35	
#define GIRBIL_115200    0x36

/* Mode register */
#define GIRBIL_IRDA      0x40
#define GIRBIL_ASK       0x41

/* Control register 2 */
#define GIRBIL_LOAD      0x51 /* Load the new baud rate value */

/**********************************************************************
 * Data structures
 **********************************************************************/

typedef struct Girbil {
  SerialDevice sd;
  SerialPort* sp;
  int state;
  int speed;
} Girbil;

/**********************************************************************
 * Girbil control
 **********************************************************************/

static int encodeSpeed(int speed) {
  switch(speed) {
  case 115200: return GIRBIL_115200;
  case 57600:  return GIRBIL_57600;
  case 38400:  return GIRBIL_38400;
  case 19200:  return GIRBIL_19200;
  case 4800:   return GIRBIL_4800;
  case 2400:   return GIRBIL_2400;
  default:     return GIRBIL_9600;
  }
}

static void speedTimer(void* sd)
{
  Girbil* gir=(Girbil*)sd;
  SerialPort* sp=gir->sp;
  
  switch(gir->state) {
  case 1:
    sp->setLine(sp,LINE_RTS);
    gir->state=2;
    evtSetTimer(20,speedTimer,gir);
    break;
  case 2:
    sp->putChar(sp,GIRBIL_TXEN|GIRBIL_RXEN);
    sp->putChar(sp,encodeSpeed(gir->speed));
    sp->putChar(sp,GIRBIL_LOAD);
    gir->state=3;
    evtSetTimer(100,speedTimer,gir);
    break;
  case 3:
    sp->setLine(sp,LINE_DTR|LINE_RTS);
    if(gir->speed!=9600) sp->setSpeed(sp,gir->speed);
    gir->state=0;
    break;
  }
}

/**********************************************************************
 * Methods
 **********************************************************************/

static int girSetSpeed(SerialDevice* sd, int speed)
{
  Girbil* gir=(Girbil*)sd;
  SerialPort* sp=gir->sp;

  gir->state=1;
  gir->speed=speed;
  sp->setSpeed(sp,9600);
  sp->setLine(sp,LINE_DTR);
  evtSetTimer(5,speedTimer,gir);

  /* Reasonable estimate */
  return 150;
}

static int girGetSpeedMask(SerialDevice* sd)
{
  Girbil* gir=(Girbil*)sd;

  return gir->sp->getSpeedMask(gir->sp)&
    (SPEED_2400|SPEED_9600|SPEED_19200|SPEED_38400|SPEED_57600|SPEED_115200);
}

static int girGetMinTurnaroundMask(SerialDevice* sd)
{
  return MIN_TA_10ms|MIN_TA_5ms;
}

static int girGetChar(SerialDevice* sd)
{
  Girbil* gir=(Girbil*)sd;

  return gir->sp->getChar(gir->sp);
}

static void girPutChar(SerialDevice* sd, int c)
{
  Girbil* gir=(Girbil*)sd;

  if(gir->state==0) gir->sp->putChar(gir->sp,c);
}

static void girClose(SerialDevice* sd)
{
  Girbil* gir=(Girbil*)sd;
  SerialPort* sp=gir->sp;

  evtCancelTimer(speedTimer,gir);
  sp->setLine(sp,0);
  sp->handle=0;
  sp->status=0;
  freeMem(gir);
}

static void girStatus(SerialPort* sp, int event)
{
  Girbil* gir=(Girbil*)sp->handle;

  if(gir->sd.status) gir->sd.status(&gir->sd,event);
}

/**********************************************************************
 * External interface
 **********************************************************************/

SerialDevice* createGIrBILDevice(SerialPort* sp)
{
  Girbil* gir=allocMem(id_girbil,sizeof(Girbil));

  gir->sd.close=girClose;
  gir->sd.setSpeed=girSetSpeed;
  gir->sd.getSpeedMask=girGetSpeedMask;
  gir->sd.getMinTurnaroundMask=girGetMinTurnaroundMask;
  gir->sd.getChar=girGetChar;
  gir->sd.putChar=girPutChar;
  gir->sd.handle=0;
  gir->sd.status=0;
  gir->sp=sp;
  gir->state=0;
  gir->speed=0;

  sp->handle=gir;
  sp->status=girStatus;

  return &gir->sd;
}
