/*
 * Copyright (c) 2001 Tommy Bohlin <tommy@gatespace.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* litelink.c
 */

#include <irda.h>

/**********************************************************************
 * Constants
 **********************************************************************/

static const char id_litelink[]="litelink";

/**********************************************************************
 * Data structures
 **********************************************************************/

typedef struct LiteLink {
  SerialDevice sd;
  SerialPort* sp;
  bool initial;
  int pulses;
  int state;
} LiteLink;

/**********************************************************************
 * LiteLink control
 **********************************************************************/

static void speedTimer(void* sd)
{
  LiteLink* ll=(LiteLink*)sd;
  SerialPort* sp=ll->sp;
  
  switch(ll->state) {
  case 1:
    sp->setLine(sp,LINE_RTS);
    ll->initial=FALSE;
    ll->state=2;
    evtSetTimer(1,speedTimer,ll); /* At least 15us */
    break;
  case 2:
    sp->setLine(sp,LINE_DTR|LINE_RTS);
    if(ll->pulses-->0) {
      ll->state=3;
      evtSetTimer(1,speedTimer,ll);  /* At least 15us */
    } else {
      ll->state=0;
    }
    break;
  case 3:
    sp->setLine(sp,LINE_DTR);    
    ll->state=2;
    evtSetTimer(1,speedTimer,ll);  /* At least 15us */
    break;
  }
}

/**********************************************************************
 * Methods
 **********************************************************************/

static int llSetSpeed(SerialDevice* sd, int speed)
{
  LiteLink* ll=(LiteLink*)sd;
  SerialPort* sp=ll->sp;

  sp->setSpeed(sp,speed);

  switch(speed) {
  case 57600: ll->pulses=1; break;
  case 38400: ll->pulses=2; break;
  case 19200: ll->pulses=3; break;
  case 9600:  ll->pulses=4; break;
  default: /* 115200 */ ll->pulses=0; break;
  }

  if(ll->initial) {
    ll->state=1;
    sp->setLine(sp,LINE_DTR|LINE_RTS);
  } else {
    ll->state=2;
    sp->setLine(sp,LINE_RTS);
  }
  evtSetTimer(1,speedTimer,ll); /* At least 15us */

  /* Reasonable estimate */
  return 40+20*ll->pulses;
}

static int llGetSpeedMask(SerialDevice* sd)
{
  LiteLink* ll=(LiteLink*)sd;
  
  return ll->sp->getSpeedMask(ll->sp)&
    (SPEED_9600|SPEED_19200|SPEED_38400|SPEED_57600|SPEED_115200);
}

static int llGetMinTurnaroundMask(SerialDevice* sd)
{
  return MIN_TA_10ms|MIN_TA_5ms|MIN_TA_1ms|MIN_TA_500us|
    MIN_TA_100us|MIN_TA_50us|MIN_TA_10us;
}

static int llGetChar(SerialDevice* sd)
{
  LiteLink* ll=(LiteLink*)sd;

  return ll->sp->getChar(ll->sp);
}

static void llPutChar(SerialDevice* sd, int c)
{
  LiteLink* ll=(LiteLink*)sd;

  if(ll->state==0) ll->sp->putChar(ll->sp,c);
}

static void llClose(SerialDevice* sd)
{
  LiteLink* ll=(LiteLink*)sd;
  SerialPort* sp=ll->sp;

  evtCancelTimer(speedTimer,ll);
  sp->setLine(sp,0);
  sp->handle=0;
  sp->status=0;
  freeMem(ll);
}

static void llStatus(SerialPort* sp, int event)
{
  LiteLink* ll=(LiteLink*)sp->handle;

  if(ll->sd.status) ll->sd.status(&ll->sd,event);
}

/**********************************************************************
 * External interface
 **********************************************************************/

SerialDevice* createLiteLinkDevice(SerialPort* sp)
{
  LiteLink* ll=allocMem(id_litelink,sizeof(LiteLink));

  ll->sd.close=llClose;
  ll->sd.setSpeed=llSetSpeed;
  ll->sd.getSpeedMask=llGetSpeedMask;
  ll->sd.getMinTurnaroundMask=llGetMinTurnaroundMask;
  ll->sd.getChar=llGetChar;
  ll->sd.putChar=llPutChar;
  ll->sd.handle=0;
  ll->sd.status=0;
  ll->sp=sp;
  ll->initial=TRUE;
  ll->state=0;

  sp->handle=ll;
  sp->status=llStatus;

  return &ll->sd;
}
