/*
 * Copyright (c) 2001 Tommy Bohlin <tommy@gatespace.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* mkobexfile.c
 */

#include <obexformat.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/stat.h>

#define MAXNAME                  100

static int ofs;
static int size;
static char* text;

static void addChar(int c)
{
  if(ofs>=size) {
    size*=2;
    text=realloc(text,size);
  }
  text[ofs++]=c;
}

static void addStr(const char* s)
{
  while(*s) addChar(*s++);
}

int main(int argc, char** argv)
{
  int c,nlen,i;
  int usage=0;
  int date;
  char* file=0;
  char* name=0;

  while((c=getopt(argc,argv,"f:n:h"))!=-1) {
    switch(c) {
    case 'f': file=optarg; break;
    case 'n': name=optarg; break;
    case 'h': usage=1; break;
    case '?': usage=1; break;
    }
  }

  if(usage) {
    fprintf(stderr,"Format OBEX file from stdin or command line\n");
    fprintf(stderr,"Usage: %s [-f file] [-n name] [text ...]\n",argv[0]);
    return 0;
  }

  if(file) {
    FILE* f=fopen(file,"r");
    if(f) {
      struct stat s;

      fstat(fileno(f),&s);
      size=s.st_size;
      date=s.st_mtime;

      text=malloc(size);
      ofs=fread(text,1,size,f);
      fclose(f);
      if(ofs!=size) {
	fprintf(stderr,"Error reading from file\n");
	return -1;
      }

      if(!name) name=file;
    } else {
      fprintf(stderr,"Can't open input file\n");
      return -1;
    }
  } else {
    struct timeval tm;

    gettimeofday(&tm,0);
    date=tm.tv_sec;

    size=1024;
    text=malloc(size);

    if(optind<argc) {
      while(optind<argc) {
	addStr(argv[optind++]);
	if(optind<argc) addChar(' ');
      }
      addChar('\n');
    } else {
      while((c=getchar())!=EOF) addChar(c);
    }

    if(!name) name="file";
  }

  nlen=strlen(name);
  if(nlen>MAXNAME) nlen=MAXNAME;

  putName(name,nlen);
  putLength(ofs);
  putDate(date);

  for(i=0;i<ofs;) {
    int n=ofs-i;

    if(n>65532) {
      n=65532;
      putBody(text+i,n);
    } else {
      putEndOfBody(text+i,n);
    }
    i+=n;
  }

  return 0;
}
