// distribution boxbackup-0.11.1 (svn version: 2821_2827)
// Box Backup, http://www.boxbackup.org/
// 
// Copyright (c) 2003-2010, Ben Summers and contributors.
// All rights reserved.
// 
// Note that this project uses mixed licensing. Any file with this license
// attached, or where the code LICENSE-GPL appears on the first line, falls
// under the "Box Backup GPL" license. See the file COPYING.txt for more
// information about this license.
// 
// ---------------------------------------------------------------------
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
// 
// [http://www.gnu.org/licenses/old-licenses/gpl-2.0.html#SEC4]
// 
// As a special exception to the GPLv2, the Box Backup Project gives
// permission to link any code falling under this license (the Box Backup
// GPL) with any software that can be downloaded from
// the OpenSSL website [http://www.openssl.org] under either the
// "OpenSSL License" or the "Original SSLeay License", and to distribute
// the linked executables under the terms of the "Box Backup GPL" license.
// 
// As a special exception to the GPLv2, the Box Backup Project gives
// permission to link any code falling under this license (the Box Backup
// GPL) with any version of Microsoft's Volume Shadow Copy Service 7.2 SDK
// or Microsoft Windows Software Development Kit (SDK), including
// vssapi.lib, that can be downloaded from the Microsoft website
// [*.microsoft.com], and to distribute the linked executables under the
// terms of the "Box Backup GPL" license.
// --------------------------------------------------------------------------
//
// File
//		Name:    BackupClientCryptoKeys.cpp
//		Purpose: function for setting up all the backup client keys
//		Created: 1/12/03
//
// --------------------------------------------------------------------------

#include "Box.h"

#include <string.h>

#include "BackupClientCryptoKeys.h"
#include "FileStream.h"
#include "BackupStoreFilenameClear.h"
#include "BackupStoreException.h"
#include "BackupClientFileAttributes.h"
#include "BackupStoreFile.h"

#include "MemLeakFindOn.h"

// --------------------------------------------------------------------------
//
// Function
//		Name:    BackupClientCryptoKeys_Setup(const char *)
//		Purpose: Read in the key material file, and set keys to all the backup elements required.
//		Created: 1/12/03
//
// --------------------------------------------------------------------------
void BackupClientCryptoKeys_Setup(const std::string& rKeyMaterialFilename)
{
	// Read in the key material
	unsigned char KeyMaterial[BACKUPCRYPTOKEYS_FILE_SIZE];
	
	// Open the file
	FileStream file(rKeyMaterialFilename);

	// Read in data
	if(!file.ReadFullBuffer(KeyMaterial, BACKUPCRYPTOKEYS_FILE_SIZE, 0))
	{
		THROW_EXCEPTION(BackupStoreException, CouldntLoadClientKeyMaterial)
	}
	
	// Setup keys and encoding method for filename encryption
	BackupStoreFilenameClear::SetBlowfishKey(
		KeyMaterial + BACKUPCRYPTOKEYS_FILENAME_KEY_START,
		BACKUPCRYPTOKEYS_FILENAME_KEY_LENGTH,
		KeyMaterial + BACKUPCRYPTOKEYS_FILENAME_IV_START,
		BACKUPCRYPTOKEYS_FILENAME_IV_LENGTH);
	BackupStoreFilenameClear::SetEncodingMethod(
		BackupStoreFilename::Encoding_Blowfish);

	// Setup key for attributes encryption
	BackupClientFileAttributes::SetBlowfishKey(
		KeyMaterial + BACKUPCRYPTOKEYS_ATTRIBUTES_KEY_START, 
		BACKUPCRYPTOKEYS_ATTRIBUTES_KEY_LENGTH);

	// Setup secret for attribute hashing
	BackupClientFileAttributes::SetAttributeHashSecret(
		KeyMaterial + BACKUPCRYPTOKEYS_ATTRIBUTE_HASH_SECRET_START,
		BACKUPCRYPTOKEYS_ATTRIBUTE_HASH_SECRET_LENGTH);

	// Setup keys for file data encryption
	BackupStoreFile::SetBlowfishKeys(
		KeyMaterial + BACKUPCRYPTOKEYS_ATTRIBUTES_KEY_START,
		BACKUPCRYPTOKEYS_ATTRIBUTES_KEY_LENGTH,
		KeyMaterial + BACKUPCRYPTOKEYS_FILE_BLOCK_ENTRY_KEY_START,
		BACKUPCRYPTOKEYS_FILE_BLOCK_ENTRY_KEY_LENGTH);

#ifndef HAVE_OLD_SSL
	// Use AES where available
	BackupStoreFile::SetAESKey(
		KeyMaterial + BACKUPCRYPTOKEYS_FILE_AES_KEY_START,
		BACKUPCRYPTOKEYS_FILE_AES_KEY_LENGTH);
#endif

	// Wipe the key material from memory
	#ifdef _MSC_VER // not defined on MinGW
		SecureZeroMemory(KeyMaterial, BACKUPCRYPTOKEYS_FILE_SIZE);
	#else
		::memset(KeyMaterial, 0, BACKUPCRYPTOKEYS_FILE_SIZE);
	#endif
}

