// distribution boxbackup-0.11.1 (svn version: 2821_2827)
// Box Backup, http://www.boxbackup.org/
// 
// Copyright (c) 2003-2010, Ben Summers and contributors.
// All rights reserved.
// 
// Note that this project uses mixed licensing. Any file with this license
// attached, or where the code LICENSE-GPL appears on the first line, falls
// under the "Box Backup GPL" license. See the file COPYING.txt for more
// information about this license.
// 
// ---------------------------------------------------------------------
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
// 
// [http://www.gnu.org/licenses/old-licenses/gpl-2.0.html#SEC4]
// 
// As a special exception to the GPLv2, the Box Backup Project gives
// permission to link any code falling under this license (the Box Backup
// GPL) with any software that can be downloaded from
// the OpenSSL website [http://www.openssl.org] under either the
// "OpenSSL License" or the "Original SSLeay License", and to distribute
// the linked executables under the terms of the "Box Backup GPL" license.
// 
// As a special exception to the GPLv2, the Box Backup Project gives
// permission to link any code falling under this license (the Box Backup
// GPL) with any version of Microsoft's Volume Shadow Copy Service 7.2 SDK
// or Microsoft Windows Software Development Kit (SDK), including
// vssapi.lib, that can be downloaded from the Microsoft website
// [*.microsoft.com], and to distribute the linked executables under the
// terms of the "Box Backup GPL" license.
// --------------------------------------------------------------------------
//
// File
//		Name:    StoreStructure.cpp
//		Purpose: 
//		Created: 11/12/03
//
// --------------------------------------------------------------------------

#include "Box.h"

#include "StoreStructure.h"
#include "RaidFileRead.h"
#include "RaidFileWrite.h"
#include "RaidFileController.h"

#include "MemLeakFindOn.h"


// --------------------------------------------------------------------------
//
// Function
//		Name:    StoreStructure::MakeObjectFilename(int64_t, const std::string &, int, std::string &, bool)
//		Purpose: Builds the object filename for a given object, given a root. Optionally ensure that the
//				 directory exists.
//		Created: 11/12/03
//
// --------------------------------------------------------------------------
void StoreStructure::MakeObjectFilename(int64_t ObjectID, const std::string &rStoreRoot, int DiscSet, std::string &rFilenameOut, bool EnsureDirectoryExists)
{
	const static char *hex = "0123456789abcdef";

	// Set output to root string
	rFilenameOut = rStoreRoot;

	// get the id value from the stored object ID so we can do
	// bitwise operations on it.
	uint64_t id = (uint64_t)ObjectID;

	// get leafname, shift the bits which make up the leafname off
	unsigned int leafname(id & STORE_ID_SEGMENT_MASK);
	id >>= STORE_ID_SEGMENT_LENGTH;

	// build pathname
	while(id != 0)
	{
		// assumes that the segments are no bigger than 8 bits
		int v = id & STORE_ID_SEGMENT_MASK;
		rFilenameOut += hex[(v & 0xf0) >> 4];
		rFilenameOut += hex[v & 0xf];
		rFilenameOut += DIRECTORY_SEPARATOR_ASCHAR;

		// shift the bits we used off the pathname
		id >>= STORE_ID_SEGMENT_LENGTH;
	}
	
	// Want to make sure this exists?
	if(EnsureDirectoryExists)
	{
		if(!RaidFileRead::DirectoryExists(DiscSet, rFilenameOut))
		{
			// Create it
			RaidFileWrite::CreateDirectory(DiscSet, rFilenameOut, true /* recusive */);
		}
	}

	// append the filename
	rFilenameOut += 'o';
	rFilenameOut += hex[(leafname & 0xf0) >> 4];
	rFilenameOut += hex[leafname & 0xf];
}


// --------------------------------------------------------------------------
//
// Function
//		Name:    StoreStructure::MakeWriteLockFilename(const std::string &, int, std::string &)
//		Purpose: Generate the on disc filename of the write lock file
//		Created: 15/12/03
//
// --------------------------------------------------------------------------
void StoreStructure::MakeWriteLockFilename(const std::string &rStoreRoot, int DiscSet, std::string &rFilenameOut)
{
	// Find the disc set
	RaidFileController &rcontroller(RaidFileController::GetController());
	RaidFileDiscSet &rdiscSet(rcontroller.GetDiscSet(DiscSet));
	
	// Make the filename
	std::string writeLockFile(rdiscSet[0] + DIRECTORY_SEPARATOR + rStoreRoot + "write.lock");

	// Return it to the caller
	rFilenameOut = writeLockFile;
}


