// distribution boxbackup-0.11.1 (svn version: 2821_2827)
// Box Backup, http://www.boxbackup.org/
// 
// Copyright (c) 2003-2010, Ben Summers and contributors.
// All rights reserved.
// 
// Note that this project uses mixed licensing. Any file with this license
// attached, or where the code LICENSE-DUAL appears on the first line, falls
// under this license. See the file COPYING.txt for more information.
// 
// This file is dual licensed. You may use and distribute it providing that you
// comply EITHER with the terms of the BSD license, OR the GPL license. It is
// not necessary to comply with both licenses, only one.
// 
// The BSD license option follows:
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  
// 3. Neither the name of the Box Backup nor the names of its contributors may
//    be used to endorse or promote products derived from this software without
//    specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 
// [http://en.wikipedia.org/wiki/BSD_licenses#3-clause_license_.28.22New_BSD_License.22.29]
// 
// The GPL license option follows:
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
// 
// [http://www.gnu.org/licenses/old-licenses/gpl-2.0.html#SEC4]
// --------------------------------------------------------------------------
//
// File
//		Name:    RollingChecksum.h
//		Purpose: A simple rolling checksum over a block of data
//		Created: 6/12/03
//
// --------------------------------------------------------------------------

#ifndef ROLLINGCHECKSUM__H
#define ROLLINGCHECKSUM__H

// --------------------------------------------------------------------------
//
// Class
//		Name:    RollingChecksum
//		Purpose: A simple rolling checksum over a block of data -- can move the block
//				 "forwards" in memory and get the next checksum efficiently.
//
//				 Implementation of http://rsync.samba.org/tech_report/node3.html
//		Created: 6/12/03
//
// --------------------------------------------------------------------------
class RollingChecksum
{
public:
	RollingChecksum(const void * const data, const unsigned int Length);

	// --------------------------------------------------------------------------
	//
	// Function
	//		Name:    RollingChecksum::RollForward(uint8_t, uint8_t, unsigned int)
	//		Purpose: Move the checksum forward a block, given the first byte of the current block,
	//				 last byte of the next block (it's rolling forward to) and the length of the block.
	//		Created: 6/12/03
	//
	// --------------------------------------------------------------------------
	inline void RollForward(const uint8_t StartOfThisBlock, const uint8_t LastOfNextBlock, const unsigned int Length)
	{
		// IMPLEMENTATION NOTE: Everything is implicitly mod 2^16 -- uint16_t's will overflow nicely.
		a -= StartOfThisBlock;
		a += LastOfNextBlock;
		b -= Length * StartOfThisBlock;
		b += a;
	}

	// --------------------------------------------------------------------------
	//
	// Function
	//		Name:    RollingChecksum::RollForwardSeveral(uint8_t*, uint8_t*, unsigned int, unsigned int)
	//		Purpose: Move the checksum forward a block, given a pointer to the first byte of the current block,
	//				 and a pointer just after the last byte of the current block and the length of the block and of the skip.
	//		Created: 7/14/05
	//
	// --------------------------------------------------------------------------
	void RollForwardSeveral(const uint8_t * const StartOfThisBlock, const uint8_t * const LastOfNextBlock, const unsigned int Length, const unsigned int Skip);

	// --------------------------------------------------------------------------
	//
	// Function
	//		Name:    RollingChecksum::GetChecksum()
	//		Purpose: Returns the checksum
	//		Created: 6/12/03
	//
	// --------------------------------------------------------------------------	
	inline uint32_t GetChecksum() const
	{
		return ((uint32_t)a) | (((uint32_t)b) << 16);
	}
	
	// Components, just in case they're handy
	inline uint16_t GetComponent1() const {return a;}
	inline uint16_t GetComponent2() const {return b;}
	
	// --------------------------------------------------------------------------
	//
	// Function
	//		Name:    RollingChecksum::GetComponentForHashing()
	//		Purpose: Return the 16 bit component used for hashing and/or quick checks
	//		Created: 6/12/03
	//
	// --------------------------------------------------------------------------
	inline uint16_t GetComponentForHashing() const
	{
		return b;
	}
	
	// --------------------------------------------------------------------------
	//
	// Function
	//		Name:    RollingChecksum::ExtractHashingComponent(uint32_t)
	//		Purpose: Static. Given a full checksum, extract the component used in the hashing table.
	//		Created: 14/1/04
	//
	// --------------------------------------------------------------------------
	static inline uint16_t ExtractHashingComponent(const uint32_t Checksum)
	{
		return Checksum >> 16;
	}
	
private:
	uint16_t a;
	uint16_t b;
};

#endif // ROLLINGCHECKSUM__H

