// distribution boxbackup-0.11.1 (svn version: 2821_2827)
// Box Backup, http://www.boxbackup.org/
// 
// Copyright (c) 2003-2010, Ben Summers and contributors.
// All rights reserved.
// 
// Note that this project uses mixed licensing. Any file with this license
// attached, or where the code LICENSE-DUAL appears on the first line, falls
// under this license. See the file COPYING.txt for more information.
// 
// This file is dual licensed. You may use and distribute it providing that you
// comply EITHER with the terms of the BSD license, OR the GPL license. It is
// not necessary to comply with both licenses, only one.
// 
// The BSD license option follows:
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  
// 3. Neither the name of the Box Backup nor the names of its contributors may
//    be used to endorse or promote products derived from this software without
//    specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 
// [http://en.wikipedia.org/wiki/BSD_licenses#3-clause_license_.28.22New_BSD_License.22.29]
// 
// The GPL license option follows:
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
// 
// [http://www.gnu.org/licenses/old-licenses/gpl-2.0.html#SEC4]
// --------------------------------------------------------------------------
//
// File
//		Name:    ProtocolUncertainStream.h
//		Purpose: Read part of another stream
//		Created: 2003/12/05
//
// --------------------------------------------------------------------------

#include "Box.h"
#include "ProtocolUncertainStream.h"
#include "ServerException.h"
#include "Protocol.h"

#include "MemLeakFindOn.h"

// --------------------------------------------------------------------------
//
// Function
//		Name:    ProtocolUncertainStream::ProtocolUncertainStream(IOStream &, int)
//		Purpose: Constructor, taking another stream.
//		Created: 2003/12/05
//
// --------------------------------------------------------------------------
ProtocolUncertainStream::ProtocolUncertainStream(IOStream &rSource)
	: mrSource(rSource),
	  mBytesLeftInCurrentBlock(0),
	  mFinished(false)
{
}

// --------------------------------------------------------------------------
//
// Function
//		Name:    ProtocolUncertainStream::~ProtocolUncertainStream()
//		Purpose: Destructor. Won't absorb any unread bytes.
//		Created: 2003/12/05
//
// --------------------------------------------------------------------------
ProtocolUncertainStream::~ProtocolUncertainStream()
{
	if(!mFinished)
	{
		BOX_WARNING("ProtocolUncertainStream destroyed before "
			"stream finished");
	}
}

// --------------------------------------------------------------------------
//
// Function
//		Name:    ProtocolUncertainStream::Read(void *, int, int)
//		Purpose: As interface.
//		Created: 2003/12/05
//
// --------------------------------------------------------------------------
int ProtocolUncertainStream::Read(void *pBuffer, int NBytes, int Timeout)
{
	// Finished?
	if(mFinished)
	{
		return 0;
	}
	
	int read = 0;
	while(read < NBytes)
	{
		// Anything we can get from the current block?
		ASSERT(mBytesLeftInCurrentBlock >= 0);
		if(mBytesLeftInCurrentBlock > 0)
		{
			// Yes, let's use some of these up
			int toRead = (NBytes - read);
			if(toRead > mBytesLeftInCurrentBlock)
			{
				// Adjust downwards to only read stuff out of the current block
				toRead = mBytesLeftInCurrentBlock;
			}
			
			BOX_TRACE("Reading " << toRead << " bytes from stream");
	
			// Read it
			int r = mrSource.Read(((uint8_t*)pBuffer) + read, toRead, Timeout);
			// Give up now if it didn't return anything
			if(r == 0)
			{
				BOX_TRACE("Read " << r << " bytes from "
					"stream, returning");
				return read;
			}
			
			// Adjust counts of bytes by the bytes recieved
			read += r;
			mBytesLeftInCurrentBlock -= r;
			
			// stop now if the stream returned less than we asked for -- avoid blocking
			if(r != toRead)
			{
				BOX_TRACE("Read " << r << " bytes from "
					"stream, returning");
				return read;
			}
		}
		else
		{
			// Read the header byte to find out how much there is
			// in the next block
			uint8_t header;
			if(mrSource.Read(&header, 1, Timeout) == 0)
			{
				// Didn't get the byte, return now
				BOX_TRACE("Read 0 bytes of block header, "
					"returning with " << read << " bytes "
					"read this time");
				return read;
			}
			
			// Interpret the byte...
			if(header == Protocol::ProtocolStreamHeader_EndOfStream)
			{
				// All done.
				mFinished = true;
				BOX_TRACE("Stream finished, returning with " <<
					read << " bytes read this time");
				return read;
			}
			else if(header <= Protocol::ProtocolStreamHeader_MaxEncodedSizeValue)
			{
				// get size of the block from the Protocol's lovely list
				mBytesLeftInCurrentBlock = Protocol::sProtocolStreamHeaderLengths[header];
			}
			else if(header == Protocol::ProtocolStreamHeader_SizeIs64k)
			{
				// 64k
				mBytesLeftInCurrentBlock = (64*1024);
			}
			else
			{
				// Bad. It used the reserved values.
				THROW_EXCEPTION(ServerException, ProtocolUncertainStreamBadBlockHeader)	
			}

			BOX_TRACE("Read header byte " << (int)header << ", "
				"next block has " << 
				mBytesLeftInCurrentBlock << " bytes");
		}
	}

	// Return the number read
	return read;
}

// --------------------------------------------------------------------------
//
// Function
//		Name:    ProtocolUncertainStream::BytesLeftToRead()
//		Purpose: As interface.
//		Created: 2003/12/05
//
// --------------------------------------------------------------------------
IOStream::pos_type ProtocolUncertainStream::BytesLeftToRead()
{
	// Only know how much is left if everything is finished
	return mFinished?(0):(IOStream::SizeOfStreamUnknown);
}

// --------------------------------------------------------------------------
//
// Function
//		Name:    ProtocolUncertainStream::Write(const void *, int)
//		Purpose: As interface. But will exception.
//		Created: 2003/12/05
//
// --------------------------------------------------------------------------
void ProtocolUncertainStream::Write(const void *pBuffer, int NBytes)
{
	THROW_EXCEPTION(ServerException, CantWriteToProtocolUncertainStream)
}

// --------------------------------------------------------------------------
//
// Function
//		Name:    ProtocolUncertainStream::StreamDataLeft()
//		Purpose: As interface.
//		Created: 2003/12/05
//
// --------------------------------------------------------------------------
bool ProtocolUncertainStream::StreamDataLeft()
{
	return !mFinished;
}

// --------------------------------------------------------------------------
//
// Function
//		Name:    ProtocolUncertainStream::StreamClosed()
//		Purpose: As interface.
//		Created: 2003/12/05
//
// --------------------------------------------------------------------------
bool ProtocolUncertainStream::StreamClosed()
{
	// always closed
	return true;
}

