/*****************************************************************************/
/*                                                                           */
/*				  lineinfo.c                                 */
/*                                                                           */
/*			Source file line info structure                      */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/* (C) 2001      Ullrich von Bassewitz                                       */
/*               Wacholderweg 14                                             */
/*               D-70597 Stuttgart                                           */
/* EMail:        uz@cc65.org                                                 */
/*                                                                           */
/*                                                                           */
/* This software is provided 'as-is', without any expressed or implied       */
/* warranty.  In no event will the authors be held liable for any damages    */
/* arising from the use of this software.                                    */
/*                                                                           */
/* Permission is granted to anyone to use this software for any purpose,     */
/* including commercial applications, and to alter it and redistribute it    */
/* freely, subject to the following restrictions:                            */
/*                                                                           */
/* 1. The origin of this software must not be misrepresented; you must not   */
/*    claim that you wrote the original software. If you use this software   */
/*    in a product, an acknowledgment in the product documentation would be  */
/*    appreciated but is not required.                                       */
/* 2. Altered source versions must be plainly marked as such, and must not   */
/*    be misrepresented as being the original software.                      */
/* 3. This notice may not be removed or altered from any source              */
/*    distribution.                                                          */
/*                                                                           */
/*****************************************************************************/



/* Note: The line infos kept here are additional line infos supplied by the
 * ".dbg line" command. The native line infos are always kept in the fragments
 * itself (because one fragment always originates from one line). The
 * additional line infos (which may not exist if none are supplied in the
 * source) may have several fragments attached (as is the case with sources
 * generated by the C compiler).
 */



/* common */
#include "coll.h"
#include "xmalloc.h"

/* ca65 */
#include "objfile.h"
#include "lineinfo.h"



/*****************************************************************************/
/*  				     Data                                    */
/*****************************************************************************/



/* Collection containing all line infos */
Collection LineInfoColl = STATIC_COLLECTION_INITIALIZER;
unsigned  LineInfoValid = 0;              /* Valid, that is, used entries */

/* Static pointer to last line info or NULL if not active */
LineInfo* CurLineInfo   = 0;



/*****************************************************************************/
/*     	       	       	      	     Code			     	     */
/*****************************************************************************/



static LineInfo* NewLineInfo (unsigned FileIndex, unsigned long LineNum)
/* Create and return a new line info. Usage will be zero. */
{
    /* Allocate memory */
    LineInfo* LI = xmalloc (sizeof (LineInfo));

    /* Initialize the fields */
    LI->Usage    = 0;
    LI->Index    = 0;           /* Currently invalid */
    LI->Pos.Line = LineNum;
    LI->Pos.Col  = 0;
    LI->Pos.Name = FileIndex;

    /* Insert this structure into the collection */
    CollAppend (&LineInfoColl, LI);

    /* Return the new struct */
    return LI;
}



LineInfo* UseLineInfo (LineInfo* LI)
/* Increase the reference count of the given line info and return it. The
 * function will gracefully accept NULL pointers and do nothing in this case.
 */
{
    if (LI) {
	if (LI->Usage++ == 0) {
	    /* One more valid line info */
	    ++LineInfoValid;
	}
    }
    return LI;
}



void GenLineInfo (unsigned FileIndex, unsigned long LineNum)
/* Generate a new line info */
{
    /* Create a new line info and make it current */
    CurLineInfo = NewLineInfo (FileIndex, LineNum);
}



void ClearLineInfo (void)
/* Clear the current line info */
{
    CurLineInfo = 0;
}



static int CmpLineInfo (void* Data attribute ((unused)), 
			const void* LI1_, const void* LI2_)
/* Compare function for the sort */
{
    /* Cast the pointers */
    const LineInfo* LI1 = LI1_;
    const LineInfo* LI2 = LI2_;

    /* Unreferenced line infos are always larger, otherwise sort by file,
     * then by line.
     */
    if ((LI1->Usage == 0) == (LI2->Usage == 0)) {
	/* Both are either referenced or unreferenced */
       	if (LI1->Pos.Name< LI2->Pos.Name) {
	    return -1;
	} else if (LI1->Pos.Name > LI2->Pos.Name) {
	    return 1;
	} else if (LI1->Pos.Line < LI2->Pos.Line) {
	    return -1;
	} else if (LI1->Pos.Line > LI2->Pos.Line) {
	    return 1;
	} else {
	    return 0;
	}
    } else {
	if (LI1->Usage > 0) {
	    return -1;
	} else {
	    return 1;
	}
    }
}

							   

void MakeLineInfoIndex (void)
/* Sort the line infos and drop all unreferenced ones */
{
    unsigned I;

    /* Sort the collection */
    CollSort (&LineInfoColl, CmpLineInfo, 0);

    /* Walk over the list and index the line infos. */
    for (I = 0; I < LineInfoValid; ++I) {
	/* Get a pointer to this line info */
	LineInfo* LI = CollAtUnchecked (&LineInfoColl, I);
	LI->Index = I;
    }
}



void WriteLineInfo (void)
/* Write a list of all line infos to the object file. */
{
    /* Tell the object file module that we're about to write line infos */
    ObjStartLineInfos ();

    /* Check if debug info is requested */
    if (DbgSyms) {

	unsigned I;

	/* Write the line info count to the list */
       	ObjWriteVar (LineInfoValid);

       	/* Walk through list and write all line infos that have references.
	 * Because of the sort, this are exactly the first LineInfoValid
	 * ones.
	 */
	for (I = 0; I < LineInfoValid; ++I) {
	    /* Get a pointer to this line info */
	    LineInfo* LI = CollAtUnchecked (&LineInfoColl, I);
	    /* Write the source file position */
	    ObjWritePos (&LI->Pos);
	}

    } else {

	/* No line infos */
	ObjWriteVar (0);

    }

    /* End of line infos */
    ObjEndLineInfos ();
}



