/*
 *   cddbd - CD Database Protocol Server
 *
 *   Copyright (C) 1996  Steve Scherf
 *   Email: steve@moonsoft.com
 *   Moondog Software Productions - makers of fine public domain software.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifndef LINT
static char *_mail_c_ident_ = "@(#)$Id: mail.c,v 1.4 1996/12/13 23:35:22 steve Exp $";
#endif

#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include "list.h"
#include "cddbd.h"


/* Preprocessor definitions. */
#define SCAN_ADDR	0
#define SCAN_NAME	1
#define SCAN_MISC	2


/* Prototypes. */
int return_mail(FILE *, char *, char *, char *, int);
int scan_from(char *, char *, int);
int parse_mail(FILE *, char *, char *, char *, unsigned int *, char *, char *,
    char *);

void cpy_sender(char *, char *, char *);


/* Global variables. */
char *charset = "charset";
char *content_encoding = "Content-Transfer-Encoding";
char *content_len = "Content-Length";
char *content_type = "Content-Type";
char *from = "From";
char *mime_ver = "Mime-Version";
char *rpath = "Return-Path";
char *subj = "Subject";
char *to = "To";
char *xsender = "X-Sender";

encode_t encoding_types[] = {
	"quoted-printable",	rfc_1521_qp_map,
	"7bit",			0,
	"8bit",			0,
	{ 0 }
};

char *char_types[] = {
	"us-ascii",
	"iso-8859",
	"iso-8859-1",
	"iso-8859-2",
	"x-iso8859",
	"x-iso88591",
	"x-iso88592",
	0
};

char *failsubj = "Rejected CDDB submission";
char *failsubj2 = "Rejected CDDB submission (fwd)";
char *testsubj = "Successful CDDB test submission";
char *testsubj2 = "Rejected CDDB test submission";
char *mailincl = "> %s\n";

char *testbody[] = {
	"\n",
	"Your CDDB test submission was accepted.\n",
	"\n",
	0
};

char *testsig[] = {
	"\n",
	"Response generated by the CDDB daemon.\n",
	"\n",
	0
};

char *failbody[] = {
	"\n",
	"Your CDDB submission was rejected for the following reason:\n",
	"\n",
	"%s\n",
	"\n",
	"Please fix the problem before you resubmit it. Only the first\n",
	"error in your submission was noted - there may be others. If you\n",
	"continue to have trouble, it may be a due to a bug in your CD\n"
	"player software. If you suspect this, try acquiring a newer version.",
	"\n\n",
	"The rejected submission follows:\n",
	0
};

char *failsig[] = {
	"\n",
	"If you need assistance, send email to: %s\n",
	"\n",
	"Response generated by the CDDB daemon.\n",
	"\n",
	0
};


char *start_type[] = {
	"<",
	"(\"",
	""
};


char *end_type[] = {
	">",
	")\"",
	"<(\""
};


void
cddbd_mail(void)
{
	int i;
	int ret;
	int enc;
	int encflag;
	FILE *fp;
	db_t *db;
	unsigned int discid;
	char to[CDDBBUFSIZ];
	char buf[CDDBBUFSIZ];
	char set[CDDBBUFSIZ];
	char file[CDDBBUFSIZ];
	char rcpt[CDDBBUFSIZ];
	char errstr[CDDBBUFSIZ];
	char category[CDDBBUFSIZ];
	char encoding[CDDBBUFSIZ];

	/* Create a temporary file for the mail. */
	cddbd_snprintf(file, sizeof(file), "%s/%s.%05d", tmpdir, mailprefix,
	    getpid());

	if((fp = fopen(file, "w+")) == NULL) {
		cddbd_snprintf(buf, sizeof(buf),
		    "Can't create mail tmp file %s (%d)", file, errno);

		cddbd_log(LOG_ERR | LOG_MAIL, "%s", buf);

		/* Print to stderr so the bounced mail gets it. */
		fprintf(stderr, "%s\n", buf);

		quit(QUIT_ERR);
	}

	(void)cddbd_fix_file(file, dir_mode, uid, gid);

	/* Put the mail into the temp file. */
	while(fgets(buf, sizeof(buf), stdin) != NULL) {
		if(fputs(buf, fp) == EOF) {
			cddbd_snprintf(buf, sizeof(buf),
			    "Can't write mail tmp file %s (%d)", file, errno);

			cddbd_log(LOG_ERR | LOG_MAIL, "%s", buf);

			/* Print to stderr so the bounced mail gets it. */
			fprintf(stderr, "%s\n", buf);

			unlink(file);
			quit(QUIT_ERR);
		}
	}

	/* Put the pointer back to the start of the file for reading. */
	rewind(fp);

	if(!parse_mail(fp, rcpt, to, category, &discid, encoding, set, errstr)) {
		cddbd_snprintf(buf, sizeof(buf),
		    "Unable to process email submission: %s", errstr);

		cddbd_log(LOG_ERR | LOG_MAIL, "%s", buf);

		/* Print to stderr so the bounced mail gets it. */
		fprintf(stderr, "%s\n", buf);

		unlink(file);
		quit(QUIT_ERR);
	}

	if(set[0] != '\0') {
		/* Look for the charset in the supported list. */
		for(i = 0; char_types[i] != 0; i++)
			if(!cddbd_strcasecmp(set, char_types[i]))
				break;

		/* Unknown charset. */
		if(char_types[i] == 0) {
			cddbd_snprintf(buf, sizeof(buf), "Unsupported %s: %s",
			    charset, set);

			ret = return_mail(fp, rcpt, to, buf, MF_FAIL);

			unlink(file);

			if(ret == 0) {
				cddbd_log(LOG_ERR | LOG_MAIL, "%s", buf);
				quit(QUIT_ERR);
			}

			quit(QUIT_OK);
		}
	}

	if(encoding[0] != '\0') {
		/* Look for the encoding in the supported list. */
		for(i = 0; encoding_types[i].en_type != 0; i++)
			if(!cddbd_strcasecmp(encoding,
			    encoding_types[i].en_type))
				break;

		/* Unknown encoding. */
		if(encoding_types[i].en_type == 0) {
			cddbd_snprintf(buf, sizeof(buf), "Unsupported %s: %s",
			    content_encoding, encoding);

			ret = return_mail(fp, rcpt, to, buf, MF_FAIL);

			unlink(file);

			if(ret == 0) {
				cddbd_log(LOG_ERR | LOG_MAIL, "%s", buf);
				quit(QUIT_ERR);
			}

			quit(QUIT_OK);
		}

		enc = i;
	}
	else
		enc = CE_7BIT;

	switch(enc) {
	case CE_QUOTED_PRINT:
		encflag = DF_REMAP;
		break;

	default: 
		encflag = 0;
		break;
	}

	/* Make sure we have a postdir. */
	if(!write_ok && !test_mail) {
		cddbd_snprintf(buf, sizeof(buf),
		    "Email submissions disallowed");

		ret = return_mail(fp, rcpt, to, buf, MF_FAIL);

		unlink(file);

		if(ret == 0) {
			cddbd_log(LOG_ERR | LOG_MAIL, "%s", buf);
			quit(QUIT_ERR);
		}

		quit(QUIT_OK);
	}

	if(categ_index(category) < 0) {
		cddbd_snprintf(buf, sizeof(buf), "Invalid DB category: %s",
		    category);
		ret = return_mail(fp, rcpt, to, buf, MF_FAIL);

		unlink(file);

		if(ret == 0) {
			cddbd_log(LOG_ERR | LOG_MAIL, "Mail failed: %s", buf);
			quit(QUIT_ERR);
		}

		quit(QUIT_OK);
	}

	db = db_read(fp, errstr, (DF_MAIL | DF_SUBMITTER | encflag));
	if(db == 0) {
		cddbd_snprintf(buf, sizeof(buf), "Invalid DB submission: %s",
		    errstr);
		ret = return_mail(fp, rcpt, to, buf, MF_FAIL);

		unlink(file);

		if(ret == 0) {
			if(db_errno != DE_INVALID) {
				cddbd_log(LOG_ERR | LOG_MAIL, "Mail failed: %s",
				    buf);
			}
			quit(QUIT_ERR);
		}

		quit(QUIT_OK);
	}

	if(!test_mail) {
		cddbd_snprintf(buf, sizeof(buf), "%s/%s", postdir, category);

		if(!db_post(db, buf, discid, errstr)) {
			if(db_errno != DE_INVALID) {
				cddbd_snprintf(buf, sizeof(buf),
				    "Internal DB server error: %s", errstr);
			}
			else
				cddbd_snprintf(buf, sizeof(buf),
				    "Invalid DB file: %s", errstr);

			ret = return_mail(fp, rcpt, to, buf, MF_FAIL);
			unlink(file);

			if(ret == 0) {
				if(db_errno != DE_INVALID) {
					cddbd_log(LOG_ERR | LOG_MAIL,
					    "Mail failed: %s", buf);
				}

				quit(QUIT_ERR);
			}

			quit(QUIT_OK);
		}

		cddbd_log(LOG_WRITE, "Write (via SMTP - %s): %s %08x",
		    to, category, discid);
	}
	else {
		ret = return_mail(fp, rcpt, to, buf, MF_TEST);
		unlink(file);

		if(ret == 0) {
			if(db_errno != DE_INVALID) {
				cddbd_log(LOG_ERR | LOG_MAIL, "Mail failed: %s",
				    buf);
			}

			quit(QUIT_ERR);
		}

		cddbd_log(LOG_INFO, "Test email submission (from %s): %s %08x",
		    to, category, discid);
	}

	fclose(fp);
	unlink(file);
}


int
parse_mail(FILE *fp, char *rcpt, char *to, char *categ,
    unsigned int *discid, char *encoding, char *set, char *errstr)
{
	int blank;
	int header;
	int gotsubj;
	char *p;
	char buf[CDDBBUFSIZ];

	header = 0;
	gotsubj = 0;
	set[0] = '\0';
	encoding[0] = '\0';

	while(fgets(buf, sizeof(buf), fp) != NULL) {
		blank = is_blank(buf, 0);

		if(!strncmp(buf, from, strlen(from))) {
			header++;

			if(header) {
				p = buf + strlen(from);
				while(*p != '\0' && !isspace(*p))
					p++;
				while(*p != '\0' && isspace(*p))
					p++;

				cpy_sender(p, rcpt, to);
			}
		}
		else if(header && !strncmp(buf, subj, strlen(subj))) {
			p = buf + strlen(subj);

			while(*p != '\0' && !isspace(*p))
				p++;
			while(*p != '\0' && isspace(*p))
				p++;

			if(sscanf(p, "cddb %s %08x", categ, discid) != 2) {
				strcpy(errstr, "malformed subject");
				return 0;
			}

			gotsubj++;
		}
		else if(header && !cddbd_strncasecmp(buf, content_type,
		    strlen(content_type)) && ((p = cddbd_strcasestr(buf,
		    charset)) != 0)) {
			/* Skip junk. */
			p += strlen(charset);
			while(*p != '\0' && !isalpha(*p) && !isdigit(*p))
				p++;

			/* Find the charset. */
			if(sscanf(p, "%[a-zA-Z0-9_-]", set) != 1) {
				cddbd_snprintf(errstr, CDDBBUFSIZ,
				    "malformed %s", content_type);
				return 0;
			}
		}
		else if(header && !cddbd_strncasecmp(buf, content_encoding,
		    strlen(content_encoding))) {
			/* Skip junk. */
			p = buf + strlen(content_encoding);
			while(*p != '\0' && !isalpha(*p) && !isdigit(*p))
				p++;

			/* Find the encoding. */
			if(sscanf(p, "%[a-zA-Z0-9_-]", encoding) != 1) {
				cddbd_snprintf(errstr, CDDBBUFSIZ,
				    "malformed %s", content_encoding);

				return 0;
			}

		}
		else if(header && blank) {
			if(!gotsubj) {
				strcpy(errstr, "missing subject");
				return 0;
			}

			return 1;
		}
	}

	strcpy(errstr, "malformed email header");
	return 0;
}


void
cpy_sender(char *str, char *rcpt, char *to)
{
	int gotaddr;
	int gotname;
	int gotmisc;
	char *pa;
	char *pn;
	char addr[CDDBBUFSIZ];
	char name[CDDBBUFSIZ];
	char misc[CDDBBUFSIZ];

	/* Scan the string for the name and address. */
	if(scan_from(str, addr, SCAN_ADDR))
		gotaddr = 1;
	else
		gotaddr = 0;

	if(scan_from(str, name, SCAN_NAME))
		gotname = 1;
	else
		gotname = 0;

	if((!gotaddr || !gotname) && scan_from(str, misc, SCAN_MISC))
		gotmisc = 1;
	else
		gotmisc = 0;


	if(gotaddr) {
		pa = addr;
	}
	else if(gotmisc) {
		pa = misc;
		gotaddr = 1;
		gotmisc = 0;
	}

	if(gotname) {
		pn = name;
	}
	else if(gotmisc) {
		pn = misc;
		gotname = 1;
	}

	/* We couldn't parse the string, use the raw "from". */
	if(!gotaddr) {
		strcpy(rcpt, str);

		pn = &rcpt[strlen(rcpt) - 1];
		if(*pn == '\n')
			*pn = '\0';

		strcpy(to, rcpt);

		return;
	}

	/* Copy the relevant parts. */
	cddbd_snprintf(rcpt, CDDBBUFSIZ, "%s", pa);

	if(gotname)
		cddbd_snprintf(to, CDDBBUFSIZ, "%s (%s)", pa, pn);
	else
		cddbd_snprintf(to, CDDBBUFSIZ, "%s", pa);
}


int
scan_from(char *str, char *buf, int type)
{
	char *p;
	char *p1;

	/* Skip leading white space. */
	p = str;
	while(*p != '\0' && isspace(*p))
		p++;

	/* Scan for a beginning marker. */
	while(*p != '\0' && !is_instr(*p, start_type[type]))
		p++;

	if(*p == '\0')
		return 0;

	/* Strip off marker, if there is one. */
	if(*start_type[type] != '\0')
		p++;

	p1 = buf;

	/* Copy in relevant portion of the string. */
	while(*p != '\0' && !is_instr(*p, end_type[type])) {
		*p1 = *p;
		p++;
		p1++;
	}

	/* Strip off trailing white space. */
	p1--;
	while(isspace(*p1))
		p1--;
	p1++;
	*p1 = '\0';
	
	return 1;
}


int
return_mail(FILE *fp, char *rcpt, char *to, char *errstr, int reason)
{
	int i;
	int len;
	FILE *nfp;
	char *subj;
	char **sig;
	char **body;
	char buf[CDDBBUFSIZ];
	char file[CDDBBUFSIZ];
	struct stat sbuf;

	/* Create a temporary file for the mail. */
	cddbd_snprintf(file, sizeof(file), "%s/%s.%05d", tmpdir,
	    respprefix, getpid());

	if(reason & MF_FAIL) {
		cddbd_log(LOG_ERR | LOG_MAIL, "Email submission failed: %s",
		    errstr);

		if(test_mail)
			subj = testsubj2;
		else
			subj = failsubj;

		sig = failsig;
		body = failbody;
	}
	else {
		sig = testsig;
		body = testbody;
		subj = testsubj;
	}

	if((nfp = fopen(file, "w+")) == NULL) {
		cddbd_log(LOG_ERR | LOG_MAIL,
		    "Can't create mail tmp file %s (%d)", file, errno);

		return 0;
	}

	(void)cddbd_fix_file(file, dir_mode, uid, gid);

	/* Write out the err string. */
	for(i = 0; body[i]; i++) {
		if(fprintf(nfp, body[i], errstr) == EOF) {
			cddbd_log(LOG_ERR | LOG_MAIL,
			    "Can't write mail tmp file %s (%d)",
			    file, errno);

			fclose(nfp);
			unlink(file);
			return 0;
		}
	}

	rewind(fp);

	while(fgets(buf, sizeof(buf), fp) != NULL) {
		strip_crlf(buf);

		if(fprintf(nfp, mailincl, buf) == EOF) {
			cddbd_log(LOG_ERR | LOG_MAIL,
			    "Can't write mail tmp file %s (%d)", file, errno);

			fclose(nfp);
			unlink(file);
			return 0;
		}
	}

	/* Write out the err string. */
	for(i = 0; sig[i]; i++) {
		if(fprintf(nfp, sig[i], admin_email) == EOF) {
			cddbd_log(LOG_ERR | LOG_MAIL,
			    "Can't write mail tmp file %s (%d)",
			    file, errno);

			fclose(nfp);
			unlink(file);
			return 0;
		}
	}

	fflush(nfp);

	if(fstat(fileno(nfp), &sbuf))
		len = 0;
	else
		len = sbuf.st_size;

	if(!smtp_open()) {
		fclose(nfp);
		unlink(file);
		return 0;
	}

	rewind(nfp);

	if(!smtp_transmit(nfp, subj, rcpt, to, CE_7BIT, len)) {
		fclose(nfp);
		unlink(file);
		return 0;
	}

	if((reason & MF_FAIL) && !test_mail) {
		rewind(nfp);

		if(bounce_email[0] != '\0' && !smtp_transmit(nfp, failsubj2,
		    bounce_email, bounce_email, CE_7BIT, len)) {
			fclose(nfp);
			unlink(file);
			return 0;
		}
	}

	smtp_close();
	fclose(nfp);
	unlink(file);

	return 1;
}
