/*
 * CHEST, chess analyst.  For Copyright notice read file "COPYRIGHT".
 *
 * $Source: /home/heiner/ca/chest/RCS/move1gen.c,v $
 * $Id: move1gen.c,v 3.17 1999/07/20 21:06:11 heiner Exp $
 *
 *	generate list of legal moves for mate in 1 move
 */

#include "types.h"
#include "board.h"
#include "mgsubr.h"
#include <stdio.h>
#include "stats.h"
#include "output.h"
#include "move_gen.h"


#if LOW_SEARCH_BITS < 8
# include "/// >>>>> LOW_SEARCH_BITS too small <<<<< ///"
#endif

#if !PROD_LEV && 0	/*CF*/
# define XDB(lv,x)	if( f_xdebug >= (lv) ) { x }else 0
#else
# define XDB(lv,x)	/*empty*/
#endif


#if PROD_LEV
# define MG1_STATS	0
#else	/* !PROD_LEV */
# ifndef MG1_STATS
#  define MG1_STATS	1	/* CF */
# endif
#endif	/* !PROD_LEV */

#undef  THIS_STATS
#define THIS_STATS	MG1_STATS
#include "statsupp.h"

#if MG1_STATS
  static Counter	sc_1gen   [2];		/* [colour] */
  static Counter	sc_empty  [2];		/* [colour] */
  static Counter	sc_expl   [2];		/* [colour] */
  static Counter	sc_impl   [2];		/* [colour] */
  static Counter	sc_implsum[2];		/* [colour] */
  static Counter	sc_explsum[2];		/* [colour] */
#endif	/* MG1_STATS */


    Eximpl void		/*ARGSUSED*/
mg_stats( Bool print )
{
#if MG1_STATS
    register int	c;

    if( print && (f_stats > 0) ) {
	for( c=0 ; c<2 ; ++c ) {
	    if( sc_1gen[c] ) {
		printf("m1g%c", chr_Colour((Colour)c));
		show_sc(1,8, sc_1gen[c]);
		printf(", empty");	show_sc(1,0, sc_empty[c]);
		printf(", expl");	show_scs(1,0, sc_expl[c], "/");
					show_sc (0,0, sc_explsum[c]);
		printf(", impl");	show_scs(1,0, sc_impl[c], "/");
					show_sc (0,0, sc_implsum[c]);
		printf("\n");
	    }
	}
    }
				/* Reinitialize: */
    for( c=0 ; c<2 ; ++c ) {
	sc_1gen   [c] = 0;
	sc_empty  [c] = 0;
	sc_expl   [c] = 0;
	sc_impl   [c] = 0;
	sc_implsum[c] = 0;
	sc_explsum[c] = 0;
    }
#endif	/* MG1_STATS */
}


/*
 * king_escapes()
 *	Scan the 3x3 square around the specified king (to be mated),
 *	and collect the following sets:
 *	(a) Fields which are escapes for the king,
 *	    and not even have any indirect attack in them.
 *	(b) Fields which are (now) escapes for the king.
 *	(c) Which pieces cause the difference between (a) and (b).
 *	(d) Fields which are (now) NO king escape,
 *	    for the sole reason that there is exactly 1 (one) direct attack,
 *	    but no indirect.
 *	(e) The pieces which span the (d) set.
 */
typedef struct
{
    EscSet	escs0i;		/* (a) */
    EscSet	escs;		/* (b) */
    PieceSet	doesi;		/* (c) */
    PieceSet	does1d0i;	/* (e) */
    EscSet	noescs1d0i;	/* (d) */
} EscResult;


    static void
king_escapes(
    register Xconst Board*	bp,
    register Xconst Field*	kingp,
    register EscResult*		erp)
{
    register int		dir;
    register Xconst Field*	p;
    register Xconst Field*	popp;
    register rPieceSet		atts;
    register rPieceSet		iatts;
    register rEscSet		escs0i;		/* (a) */
    register rEscSet		escs;		/* (b) */
    register rPieceSet		doesi;		/* (c) */
    register rEscSet		noescs1d0i;	/* (d) */
    register rPieceSet		does1d0i;	/* (e) */

    XDB(1, printf(">  king_escapes\n"); );
    escs       = 0;
    escs0i     = 0;
    doesi      = 0;
    noescs1d0i = 0;
    does1d0i   = 0;
    atts = COLOUR_MASK(opp_colour(kingp->f_c));
    for( dir=MIN_E_DIR ; dir<MAX_E_DIR ; ++dir ) {
	p = kingp + dam_mov[dir];
	if( (p->f_c == border)
	 || ((p->f_c == kingp->f_c) && !zero_dir(dir)) ) {
	    continue;
	}
	iatts = F_IATT(p);
			/* approximate indir attacks that stop in the king: */
	if( ! zero_dir(dir) ) {
	    popp = kingp - dam_mov[dir];
	    if( popp->f_c != border ) {
		iatts |= F_IATT(kingp) & (F_DATT(popp) | F_IATT(popp));
	    }
	}
	iatts &= atts;
	if( F_DATT(p) & atts ) {
	    if( !iatts && max1elems(F_DATT(p) & atts) ) {
		noescs1d0i |= (1 << dir);	/* (d) */
		does1d0i |= F_DATT(p);		/* (e) */
	    }
	    continue;
	}
	escs |= (1 << dir);			/* (b) */
	if( ! iatts ) {
	    escs0i |= (1 << dir);		/* (a) */
	}else {
	    doesi |= iatts;			/* (c) */
	}
    }
    erp->escs       = escs;
    erp->escs0i     = escs0i;
    erp->doesi      = doesi;
    erp->noescs1d0i = noescs1d0i;
    erp->does1d0i   = does1d0i & atts;
#if 1
    XDB(1, printf("$  escs %x, escs0i %x, doesi %x, noescs1d0i %x\n",
		    escs, escs0i, doesi, noescs1d0i); );
#endif
    XDB(1, printf("<  king_escapes\n"); );
}


/*
 * move1gen()
 *	Generate all legal mate moves for the specified Board,
 *	i.e. those which cannot be mate moves can be left out.
 *	Return, whether any legal move found.
 */
    Eximpl Bool
move1gen(
    register Xconst Board*	bp,
    register Movelist*		lp)
{
    Colour			self;
    Colour			enemy;
    Xconst Field*		eking;
    EscSet			escs;
    const EscSet*		covp;
    const EscInfo*		eip;
    int				ifig;
    int				minifig;
    Position			pos;
    Position			kpos;
    Position			tpos;
    Position			ekpos;
    Position			ckpos;
    int				ckdir;		/* kpos -> ckpos */
    register Xconst Field*	p;
    register Xconst Field*	tp;
    Xconst Field*		xp;
    int				pindir;
    int				idir;
    register int		i;
    int				imax;
    int				delta;
    Move			m;
    Position			bfpos;
    Position			btpos;
    int				bkdir;
    register rPieceSet		atts;
    PieceSet			ckatts;		/* enemy datts in kpos */
    PieceSet			backatts;
    PieceSet			selfmask;
    PieceSet			enemymask;
    PieceSet			nonBallowed;
    PieceSet			ifigset;
    FigSet			resfigs;	/* set of resulting fig types */
    EscResult			escres;
    static Figure		max_uncover[MAX_FIGURES] ={
					/* BSLTDK */
					dame, dame, turm, laeufer, bauer, dame
				};

    XDB(1, printf("> move1gen\n"); );
    self  = bp->b_tomove;
    scinc(sc_1gen[self]);
    enemy = opp_colour(self);
    enemymask = COLOUR_MASK(enemy);
    selfmask  = COLOUR_MASK(self );
    clear_list(lp);
    m.m_value = 0;
    m.m_attr  = 0;
    /*
     * First, collect the irregular move types, which do not fit into
     * the simple sceme, that just one field is cleared and another one
     * gets the cleared piece:
     *  - promotions (piece changes figure type)
     *  - e.p. (a second field is cleared)
     *  - castlings (a second figure is moved)
     */
#define WITH_PROM	1
#if ! WITH_PROM
				/* Append possible promotions */
    if( bp->b_bau[self].fs_line[BAS_LIN(enemy)] ) {
	mg_app_prom(bp, lp, TRUE);	/* ... that might check */
    }
#endif
				/* Append possible e.p.'s: */
    if( ! no_pos(bp->b_ep) ) {
	mg_app_ep(bp, lp);
    }
				/* Append possible castlings: */
    if( bp->b_castle[self] ) {
	mg_app_castle(bp, lp);
    }
				/* Analyse escapes of king to mate */
    kpos  = K_POS(bp, self );
    ekpos = K_POS(bp, enemy);
    eking = &(bp->b_f[ekpos]);
    king_escapes(bp, eking, &escres);
    if( ! cov_esc[escres.escs0i]->figcan ) {
	scinc(sc_empty[self]);
	mg_link(lp);
	XDB(1, printf("< move1gen empty\n"); );
	return ! empty_list(lp);
    }
    /*
     * If self is in double check, only king moves are legal.
     * Then we set "ifig" and "ifigmax" appropriate.
     * If self is in single check, this check must be removed by
     *  - either moving the king itself
     *  - or moving a non-K onto either the checker's position,
     *	  or between the K and the checker (block).
     * Here, we collect "nonBallowed", the set of non-bauer pieces,
     * which are still allowed to move (as they may block/beat the checker).
     */
    minifig = COLOUR_IDX(self);
    ifig = minifig + bp->b_max_piece[self] - 1;	/* maximum (inclusive) */
    ckpos = NO_POS;	/* single check: from where */
    ckdir = NO_DIR;	/* single check: from where */
    nonBallowed = selfmask;
    if( (ckatts = (F_DATT(&(bp->b_f[kpos])) & enemymask)) ) {  /* self in check */
		register int	zeroes;
	XDB(1, printf("$  m1g ckatts %x\n", ckatts); );
	/*
	 * If there is a checker which is not a B, and it is
	 * not yet pinned, then moving the K cannot mate, since a
	 * possibly opened check may always legally be blocked by that
	 * checker on the former K position (any additional pinning
	 * would go through that field and not be left).
	 * Exception: the K beats this checking attacker.
	 */
	i = 0;
	atts = ckatts;
	do {
	    while( ! (atts & 01) ) {
		zeroes = MIN_LOW_ZERO(atts);
		i += zeroes; atts >>= zeroes;
	    }
	    tp = &(bp->b_f[ ckpos = bp->b_piece[i] ]);
	    if( (tp->f_f != bauer)
	     && (   ! dam_dir(att_dir(ckpos, ekpos))		/* unpinnable */
		 || ! (F_DATT(tp) & F_IATT(eking) & selfmask)	/* unpinned */
		)
	     && !(    (F_DATT(tp) & BOTH_KINGS & selfmask)	/* K attacks */
		  && !(F_DATT(tp) & enemymask)			/* undefended */
		 )
	      ) {
		i = -2;		/* remember to exclude K moves */
		break;
	    }
	}while( ++i, (atts >>= 1) );
	if( max1elems(ckatts) ) {		/* in single check: */
	    if( i < 0 ) {
		atts = 0;		/* exclude K */
		if( KING_IDX == 0 ) {
		    ++minifig;
		}
	    }else {
		atts = BOTH_KINGS;	/* allow K to move */
	    }
	    /*
	     * Collect direct attacks along the check path in order
	     * to restrict the moving non-bauer pieces.
	     *    If the checking piece is completely unpinnable, and
	     * there is no indirect check possible, all pieces doing
	     * a blocking move can be beaten back by the current checker,
	     * what would be a legal move (hence the blocking move no mate).
	     * Then the checker must be beaten or the K moved.
	     */
	    atts |= F_DATT(tp);		/* allow to beat him, FFS: K */
	    ckdir = att_dir(kpos, ckpos);
	    if( dam_dir(ckdir) ) {	/* in check by B or LTD */
		if( dam_dir(att_dir(ckpos, ekpos))	/* pinnable */
		 || (F_IATT(eking) & selfmask)		/* indir check poss. */
		  ) {
		    delta = dam_mov[ckdir];
		    while( (tp -= delta)->f_c == empty ) {
			atts |= F_DATT(tp);
		    }
		}
	    }
	    nonBallowed = atts & selfmask;
	}else {					/* in double check: */
	    if( i < 0 ) {	/* K moves excluded, already */
		goto rdy;	/* no further chance */
	    }
	    ckpos = NO_POS;	/* set only for single check */
	    minifig += KING_IDX;
	    ifig = minifig;
	}
    }
    /*
     * Now we scan over all our pieces (inclusive the K) ...
     */
    XDB(1, printf("$  m1g minifig %d, ifig %d\n", minifig, ifig); );
    for( ifigset=SET1(ifig) ; ifig >= minifig ; ifigset>>=1, --ifig ) {
	pos = bp->b_piece[ifig];
	if( no_pos(pos) )
	    continue;		/* skip empty slot (beaten piece) */
	p = &(bp->b_f[pos]);
	XDB(1, printf("$  m1g ifig %d %d\n", ifig, p->f_f); );
	switch( p->f_f ) {
	 case springer:
	    if( ifigset & escres.does1d0i ) {
		continue;	/* would take away necessary attack */
	    }
	    /*FALLTHROUGH*/
	 default:
	    if( ! (ifigset & nonBallowed) ) {
		continue;
	    }
	    resfigs = (1 << p->f_f);
	    break;
	 case bauer:
#if WITH_PROM
	    if( LIN64(p->f_pos64) == BAS_LIN(enemy) ) {	/* promotion */
		resfigs = ( (ifigset & escres.does1d0i)
			  ? (                           (1<<turm)|(1<<dame))
			  : ((1<<springer)|(1<<laeufer)|(1<<turm)|(1<<dame))
			  );
	    }else
#endif
	    {
		if( ifigset & escres.does1d0i ) {
		    continue;	/* would take away necessary attack */
		}
		resfigs = (1 << p->f_f);
	    }
	    break;
	}
	/*
	 * Determine the set of escapes which must be covered directly
	 * by this piece (all - uncovered attacks) in "escs".
	 *
	 * This piece may uncover some attack to a potential escape, if
	 *  - there is a direct attack from one of those 3x3 indirect ones,
	 *  - it is not a dame (which would already attack direct)
	 *  - has appropriate relation to some of those escapes,
	 *    which might disappear by uncovering.
	 */
	escs = escres.escs;		/* hopefully, if nothing uncovered */
	if( (F_DATT(p) & escres.doesi) && (p->f_f != dame) ) {
	    escs &= ~(fig_cov[max_uncover[p->f_f]][pos-ekpos] & ~escres.escs0i);
	}
	/*
	 * Those 1d0i fields around the king, which are covered by this
	 * piece, need still be covered by this piece.
	 * Thus, for this piece, they are put into the set of escapes.
	 */
	if( ifigset & escres.does1d0i ) { /* there are such of this piece */
	    for( i=MIN_D_DIR ; i<MAX_D_DIR ; ++i ) {
		if( escres.noescs1d0i & (1 << i) ) {
		    if( F_DATT(&(eking[dam_mov[i]])) & ifigset ) {
			escs |= (1 << i);
		    }
		}
	    }
	}
	eip = cov_esc[escs];
	if( ! (resfigs &= eip->figcan) ) {
	    continue;		/* this figure type cannot cover the escapes */
	}
	/*
	 * An especially annoying case is when "escs" is empty, as all legal
	 * moves seem to be able to do it (looks like nothing to be done).
	 * But, we know that we need a check, which is not yet there.
	 * Therefore, we exactly analyse the possibility of an indirect
	 * check (by removing this piece).  The D direction of the
	 * uncovered attack is put into "idir".
	 * In fact, this analysis is useful, whenever ZERO_DIR is not yet
	 * in "escs".
	 */
	if( ! (escs & (1 << ZERO_DIR)) ) {	/* might be better */
	    if( p->f_f == dame ) {		/* D cannot uncover attacks */
		i = NO_DIR;
	    }else {
		i = att_dir(ekpos, pos);
		if( ! dam_dir(i)
		 || ! (atts = F_IATT(eking) & F_DATT(p) & selfmask)
		 ) {			/* cannot be indirect */
		    i = NO_DIR;
		}else {			/* might be indirect */
		    /*
		     * Standard pinning test tricks:
		     * When piece is not near to king we use the indirect
		     * attacks one step away to get a unique result.
		     * When the piece is near on T direction, we are
		     * already sure.
		     * When near on L direction, we have to search.
		     */
		    delta = dam_mov[i];
		    if( (ekpos + delta) != pos ) {	/* not near to K */
			if( ! (atts & F_IATT(&(p[-delta]))) ) {
			    i = NO_DIR;
			}
		    }else {				/* is near to K */
			if( ! trm_dir(i) ) {
			    tp = p;
			    do {
				tp += delta;
			    }while( tp->f_c == empty );
			    if( (tp->f_c != self)
			     || ! (atts & SET1(tp->f_idx)) ) {
				i = NO_DIR;
			    }
			}
		    }
		}
	    }
	    if( no_dir(idir = i) ) {	/* no indirect check possible */
		escs |= (1 << ZERO_DIR);	/* must check directly */
		eip = cov_esc[escs];
		if( ! (resfigs &= eip->figcan) ) {
		    continue;	/* this figure type cannot cover the escapes */
		}
	    }
	}
	/*
	 * "escs" is the set of escapes of the enemy's king (who is to be
	 * set mate), which must be covered directly, i.e. not by removing
	 * the piece but by direct attacks at its target position.
	 * If ZERO_DIR is in this set, then such a move cannot be an indirect
	 * check (as we found the K must be attacked directly).
	 */
	m.m_from = pos;
	m.m_idx  = ifig;
#if WITH_PROM
# define PUT_prom(ekdelta, sltd)				\
		if( (resfigs & (1<<(sltd)))			\
		 && !(escs & ~fig_cov[sltd][ekdelta]) ) {	\
		    m.m_fig = (sltd); app_move(&m, lp);		\
		}
	if( resfigs != (1 << p->f_f) ) {	/* promotion */
	    pindir = att_dir(kpos, pos);
	    if( dam_dir(pindir) && ! is_pinned(bp, pos) ) {
		pindir = NO_DIR;		/* not pinned */
	    }
					/* Try to do single step: */
	    if( ! dam_dir(pindir) ) {	/* else: is blocked or leaves pinning */
		delta = bau_mov[self];
		tp = p + delta;
		if( tp->f_c == empty ) {
		    m.m_to = tpos = pos + delta;
		    if( ckatts ) {	/* must block existing check */
			if( (att_dir(kpos, tpos ) != ckdir)
			 || (att_dir(tpos, ckpos) != ckdir) ) {
			    goto prom_forw_rdy;
			}
		    }
		    imax = tpos - ekpos;
		    PUT_prom(imax, dame)
		    PUT_prom(imax, springer)
		    PUT_prom(imax, laeufer)
		    PUT_prom(imax, turm)
		}
	    }
    prom_forw_rdy:;
					/* Try to beat in both directions: */
	    for( i=0 ; i<2 ; ++i ) {
		delta = (i ? bau_right : bau_left)[self];
		if( dam_dir(pindir) ) {
		    if( trm_dir(pindir)
		     || (lin_dir(att_dir(0, delta)) != lin_dir(pindir)) ) {
			continue;
		    }
		}
		tp = p + delta;
		if( tp->f_c == enemy ) {
		    m.m_to = tpos = pos + delta;
		    if( ckatts ) {	/* must block or beat existing check */
			if( (tpos != ckpos)
			 && (   (att_dir(kpos, tpos ) != ckdir)
			     || (att_dir(tpos, ckpos) != ckdir)
			    )
			  ) {
			    continue;
			}
		    }
		    imax = tpos - ekpos;
		    PUT_prom(imax, dame)
		    PUT_prom(imax, springer)
		    PUT_prom(imax, laeufer)
		    PUT_prom(imax, turm)
		}
	    } /* for left/right */
	    continue;				/* with next piece */
	}
# undef PUT_prom
#endif	/* WITH_PROM */
	m.m_fig = p->f_f;
	if( eip->fighow[p->f_f].count > 0 ) {	/* use explicit deltas */
		register const LegalDelta*	dp;
	    XDB(1, printf("$> m1g expl\n"); );
	    scinc(sc_expl[self]);
	    scsub(sc_explsum[self], list_length(lp));
	    for( dp = eip->fighow[p->f_f].deltas ; *dp ; ++dp ) {
		tp = eking + *dp;
		/*
		 * If this target is legally reachable by our piece, it might
		 * be mate.  First test pseudo-legality (reachable),
		 * then true legality (existing checks removed).
		 */
		if( (tp->f_c == border) || (tp->f_c == self) ) {
		    continue;
		}
		if( p->f_f == bauer ) {
#if ! WITH_PROM
		    if( LIN64(tp->f_pos64) == PROM_LIN(self) ) {
			continue;		/* promotions not here */
		    }
#endif
		    if( F_DATT(tp) & ifigset ) {	/* test beat */
			if( tp->f_c != enemy ) {
			    continue;
			}
		    }else {				/* test move */
			if( tp->f_c != empty ) {
			    continue;
			}
			if( (p + bau_mov[self]) != tp ) {
			    if( ((p + 2*bau_mov[self]) != tp)
			     || (LIN64(p->f_pos64) != BAS_LIN(self))
			     || (p[bau_mov[self]].f_c != empty) ) {
				continue;
			    }
			}
		    }
		}else {	/* non-B */
		    if( ! (F_DATT(tp) & ifigset) ) {
			continue;			/* cannot reach this */
		    }
		} /* non-B */
						/* is pseudo-legal */
		/*
		 * The K might beat back (even in double check):
		 */
		if( (F_DATT(tp) & enemymask & BOTH_KINGS)
		 && ! (F_DATT(tp) & ~ifigset & selfmask)
		 && ! (F_IATT(tp) & F_DATT(p) & selfmask)
		  ) {
		    continue;
		}
						/* test legality */
		tpos = pos64_pos[tp->f_pos64];
		if( p->f_f == koenig ) {
		    if( F_DATT(tp) & enemymask ) {
			continue;
		    }
		    /*
		     * The following test is confusingly simple.
		     * We have to find exactly those indirect attacks,
		     * which are stopped just by the king.  "ckatts" are
		     * the enemies direct attacks in our king.
		     * In single check we have "ckdir" to test explicitly
		     * the single source of such an attack.
		     * Only LTD form indirect attacks.  If such a checker
		     * is not near to the K, any indirect attack near the K
		     * (from ckatts) is bad for the K.
		     * In case of double check one checker must not be near
		     * the K (opened), and the other one cannot be a D.
		     * But, only a D can form a nonunique indirect attack,
		     * a near L or T cannot.
		     */
		    if( (F_IATT(tp) & ckatts)
		     && (no_dir(ckdir) || (att_dir(tpos, pos) == ckdir)) ) {
			continue;
		    }
		}else {				/* test non-K legality */
		    pindir = att_dir(kpos, pos);
		    if( dam_dir(pindir)
		     && (att_dir(kpos, tpos) != pindir)
		     && is_pinned(bp, pos) ) {
			continue;			/* pinned */
		    }
		    if( ckatts ) {
			if( (att_dir(kpos, tpos) != ckdir)
			 || ((tpos != ckpos) && (att_dir(tpos, ckpos) != ckdir))
			  ) {
			    continue;
			}
		    }
		}
#if 1
		/*
		 * Some non-K might beat back the direct checker:
		 */
		if( (escs & (1 << ZERO_DIR))	/* need direct check */
		 && (atts = F_DATT(tp) & enemymask & ~BOTH_KINGS)
		  ) {				/* exist non-K defenders */
		    i = 0;
		    do {
			while( ! (atts & 01) ) {
				register int	zeroes;
			    zeroes = MIN_LOW_ZERO(atts);
			    i += zeroes; atts >>= zeroes;
			}
			{	register rPosition	bpos;
				register int		bdir;
			    bpos = bp->b_piece[i];
			    bdir = att_dir(ekpos, bpos);
			    if( !dam_dir(bdir)	/* unpinnable */
			     || (att_dir(ekpos, tpos) == bdir)	/* not left */
			     || (  ! (  F_IATT(eking) & F_DATT(&(bp->b_f[bpos]))
				      & selfmask
				     )		/* not yet pinned */
				&& (  (att_dir(ekpos, pos) != bdir)
				    /*FFS*/
				   )
				)
			      ) {
				goto next_expl;
			    }
			}
		    }while( ++i, atts >>= 1 );
		}
#endif
		m.m_to = tpos;
		app_move(&m, lp);
    next_expl:  ;
	    } /* for deltas */
	    scadd(sc_explsum[self], list_length(lp));
	    XDB(1, printf("$< m1g expl\n"); );
	}else {				/* no explicit deltas provided */
	    XDB(1, printf("$> m1g impl\n"); );
	    scinc(sc_impl[self]);
	    scsub(sc_implsum[self], list_length(lp));
					/* Analyse the pinning of this piece: */
	    pindir = NO_DIR;
	    atts = F_IATT(&(bp->b_f[kpos])) & F_DATT(p) & enemymask;
	    if( atts ) {
		i = att_dir(pos, kpos);
		if( dam_dir(i) ) {
		    delta = dam_mov[i];
		    if( (pos + delta) != kpos ) {
			if( atts & F_IATT(&(p[delta])) ) {
			    pindir = i;
			}
		    }else {
			/* Piece to be tested is near to its king.
			 * Therefore there is only one f_iatt,
			 * which alone is not sufficient for
			 * check of direction.
			 * If T-dir, we are already sure to be pinned.
			 */
			if( trm_dir(i) ) {
			    pindir = i;
			}else {			/* L-dir, search: */
			    tp = p;
			    do {
				tp -= delta;
			    }while( tp->f_c == empty );
			    if( (tp->f_c == enemy)
			     && (atts & SET1(tp->f_idx)) ) {
				pindir = i;
			    }
			}
		    }
		}
	    }
	    /*
	     * Still the implicit case.
	     * Enumerate possibilities by type of moving figure.
	     * Remember that special moves are handled elsewhere.
	     */
	    switch( p->f_f ) {
	     case bauer:		/* implicit B */
#if ! WITH_PROM
		if( LIN64(p->f_pos64) == BAS_LIN(enemy) ) {
		    break;		/* promotions not here */
		}
#endif
					/* Try single and double step: */
		delta = bau_mov[self];
		if( no_dir(pindir)
		 || (delta == dam_mov[pindir])
		 || (delta == dam_mov[opp_dir(pindir)]) ) {
					/* Not pinned or staying in pinning */
		    tpos = pos;
		    for( i=0 ; i<2 ; ++i ) {
			if( i && (LIN64(p->f_pos64) != BAS_LIN(self)) )
			    break;
			tpos += delta;
			if( bp->b_f[tpos].f_c != empty )
			    break;
					/* must check somehow: */
			if( ! dam_dir(att_dir(pos, ekpos))
			 || (att_dir(ekpos, pos) == att_dir(ekpos, tpos))
			 || ! (F_IATT(eking) & F_DATT(p) & selfmask) ) {
					/* must check directly: */
			    if( ((tpos + bau_left [self]) != ekpos)
			     && ((tpos + bau_right[self]) != ekpos) ) {
				continue;
			    }
			}
			if( ckatts ) {
			    if( (att_dir(kpos, tpos) != ckdir)
			     || (  (tpos != ckpos)
				&& (att_dir(tpos, ckpos) != ckdir))
			      ) {
				continue;
			    }
			}
			m.m_to = tpos;
			app_move(&m, lp);
		    }
		}
					/* Try to beat in both directions: */
		for( i=0 ; i<2 ; ++i ) {
		    delta = (i ? bau_right : bau_left)[self];
		    if( ! no_dir(pindir)
		     && (delta != dam_mov[pindir])
		     && (delta != dam_mov[opp_dir(pindir)]) ) {
			continue;
		    }
					/* Not pinned or staying in pinning */
		    tpos = pos + delta;
		    tp = &(bp->b_f[tpos]);
		    if( tp->f_c == enemy ) {	/* try normal beat: */
			if( ckatts ) {
			    if( (att_dir(kpos, tpos) != ckdir)
			     || (  (tpos != ckpos)
				&& (att_dir(tpos, ckpos) != ckdir))
			      ) {
				continue;
			    }
			}
			m.m_to = tpos;
			app_move(&m, lp);
		    }
		}
		break;
	     case springer:		/* implicit S */
		if( ! no_dir(pindir) )
		    break;			/* pinning always left by S */
		for( i=0 ; i<8 ; ++i ) {
		    tpos = pos + spr_mov[i];
		    if( bp->b_f[tpos].f_c == border )
			continue;
		    if( bp->b_f[tpos].f_c == self )
			continue;
		    if( ckatts ) {		/* we are in check */
			if( (att_dir(kpos, tpos) != ckdir)
			 || (  (tpos != ckpos)
			    && (att_dir(tpos, ckpos) != ckdir))
			  ) {
			    continue;		/* neither beats nor blocks */
			}
		    }
		    m.m_to = tpos;
		    app_move(&m, lp);
		}
		break;
	     case laeufer:		/* implicit L */
		if( escs ) {
		    i    = MIN_L_DIR;
		    imax = MAX_L_DIR;
		    goto ltd_rev;
		}
		if( no_dir(pindir) ) {
		    i    = MIN_L_DIR;
		    imax = MAX_L_DIR;
		}else if( lfr_dir(pindir) ) {
		    i    = pindir & ~1;
		    imax = i + 2;
		}else {		/* trm_dir(pindir) : cannot move */
		    break;
		}
		goto ltd;
	     case turm:			/* implicit T */
		if( escs ) {
		    i    = MIN_T_DIR;
		    imax = MAX_T_DIR;
		    goto ltd_rev;
		}
		if( no_dir(pindir) ) {
		    i    = MIN_T_DIR;
		    imax = MAX_T_DIR;
		}else if( trm_dir(pindir) ) {
		    i    = pindir & ~1;
		    imax = i + 2;
		}else {		/* lfr_dir(pindir) : cannot move */
		    break;
		}
		goto ltd;
	     case dame:			/* implicit D */
		if( escs ) {
		    i    = MIN_D_DIR;
		    imax = MAX_D_DIR;
		    goto ltd_rev;
		}
		if( no_dir(pindir) ) {
		    i    = MIN_D_DIR;
		    imax = MAX_D_DIR;
		}else {
		    i    = pindir & ~1;
		    imax = i + 2;
		}
	     ltd:		/* implicit LTD */
		/*
		 * When pinned, we here already have restricted
		 * the directions to pindir and its complement.
		 *
		 * "escs" is empty. We suspect indirect check.
		 * But, we have analysed "idir", above.
		 * If it were not there, "escs" would not be empty.
		 * So, here we have a D direction "idir".  We know implicitly,
		 * that the moving piece would be able to beat the king,
		 * if it were able to stay within that "idir".
		 * Hence, "idir" we will leave it and uncover a check.
		 * FFS: search blockers on the path, which are fatal,
		 * if it is not also a direct check (double check).
		 */
#if 0
		btpos = NO_POS;		/* not yet found */
		if( bp->b_cur_piece[enemy] > 1 ) {
		    tp = eking; tpos = ekpos;
		    delta = dam_mov[idir];
		    do {
			tp += delta; tpos += delta;
			if( atts = F_DATT(tp) & ~BOTH_KINGS & enemymask ) {
				register int	zeros;
			    i = 0;
			    do {
				while( ! (atts & 01) ) {
				    i += (zeros = MIN_LOW_ZERO(atts));
				    atts >>= zeros;
				}
				bfpos = bp->b_piece[i];
				if( bp->b_f[bfpos].f_f == bauer )
				    continue;	/* not suitable */
				if( dam_dir(att_dir(ekpos, bfpos)) ) {
				    continue;	/* might be pinnable */
				}
				btpos = tpos;
				bkdir = att_dir(bfpos, btpos);
				goto bk_rdy;
			    }while( ++i, atts >>= 1 );
			}
		    }while( (tp->f_c == empty) || (tp == p) );
		}
	bk_rdy: ;
#endif
		covp = fig_cov[p->f_f];
		atts = enemymask & BOTH_KINGS;
		for( ; i<imax ; ++i ) {
		    tp = p;
		    tpos = pos;
		    delta = dam_mov[i];
		    do {
			tp += delta;
			if( (tp->f_c == border) || (tp->f_c == self) )
			    break;
			tpos += delta;
			/*
			 * Special: The enemies K might beat back,
			 * except we are supported, already.
			 */
			if( (F_DATT(tp) & atts)
			 && ! (F_DATT(tp) & ~ifigset & selfmask)
			 && ! (F_IATT(tp) & F_DATT(p) & selfmask)
			  ) {
			    continue;
			}
			if( ckatts ) {		/* we are in check ourselves */
			    /* Since we move a non-K, it is no double check. */
			    if( (att_dir(kpos, tpos) != ckdir)
			     || (  (tpos != ckpos)
				&& (att_dir(tpos, ckpos) != ckdir))
			      ) {
				continue;
			    }
			}
#if 0
			if( ! no_pos(btpos)
			 && ! (covp[tpos-ekpos] & (1<<ZERO_DIR))  /* !dirck */
			 && (tpos != bfpos)			/* ! beaten */
			 && (   (att_dir(bfpos,tpos) != bkdir)
			     || (att_dir(tpos,btpos) != bkdir)	/* ! blocked */
			    )
			  ) {
			    continue;
			}
#endif
			m.m_to = tpos;
			app_move(&m, lp);
		    }while( tp->f_c == empty );
		}
		break;
	ltd_rev: ;
		XDB(1, printf("$  m1g ltd_rev\n"); );
		/*
		 * There is a field, which must be attacked directly.
		 * We prefer the central one (with the king),
		 * and from there search for direct attacks of the piece.
		 * If not starting at the central field, we must search
		 * through the central field (K does not block attacks).
		 * We also search through the move candidate at 'p', as
		 * it will attack its own source by moving away.
		 */
		if( escs & (1 << ZERO_DIR) ) {
		    xp = eking;			/* must check direct */
		}else {
		    xp = eking + dam_mov[MIN_LOW_ZERO(escs)];
		}
		XDB(1, printf("$  m1g escs %x, xp %2o\n", escs, xp->f_pos64); );
		covp = fig_cov[p->f_f];
		for( ; i<imax ; ++i ) {
		    XDB(1, printf("$  m1g dir %d\n", i); );
		    tp = xp;
		    delta = dam_mov[i];
		    do {
			tp += delta;
			if( tp->f_c == border )
			    break;		/* stops direction */
			XDB(1, printf("$  m1g tp %2o\n", tp->f_pos64); );
			if( tp->f_c == self ) {
			    if( tp == p )
				continue;	/* illegal empty move */
			    break;		/* stops direction */
			}
			tpos = pos64_pos[tp->f_pos64];
			if( xp == eking ) {	/* can't be indir check */
			    backatts = F_DATT(tp) & ~BOTH_KINGS & enemymask;
			    /*
			     * If there is an unpinnable back-attack,
			     * which is not a B, then it could beat.
			     * If it comes from another direction,
			     * it can also always block for further targets
			     * in this direction, what stops this direction.
			     * Note: as (xp == eking), we know that
			     *		att_dir(ekpos, tpos) == i
			     * for this direction "i".
			     */
			    if( (atts = backatts) ) {
				    register int	zeros;
				    register int	bfig;
				    register Position	bpos;
				    register int	bdir;
				bfig = 0;
				do {
				    XDB(1, printf("$  m1g bkatts %x\n",atts); );
				    while( ! (atts & 01) ) {
					zeros = MIN_LOW_ZERO(atts);
					atts >>= zeros; bfig += zeros;
				    }
				    bpos = bp->b_piece[bfig];
				    if( bp->b_f[bpos].f_f == bauer )
					continue;	/* not yet suitable */
				    bdir = att_dir(ekpos, bpos);
				    if( ! dam_dir(bdir) ) {	/* unpinnable */
					goto ndir_ltd_rev;	/* is blocker */
				    }
				    if( i == bdir ) {
					/*
					 *	ekpos -i-> tpos
					 *  and
					 *	ekpos -i-> tpos
					 * looks like
					 *	ekpos -i-> tpos -i-> bpos
					 * Hence, the next chance is to beat
					 * in bpos, as all others can also
					 * be beaten back.
					 * The direct attack bit tells us,
					 * nothing is between tpos & bpos.
					 * Nothing is between ekpos & tpos
					 * is known from scanning.  If now
					 * tpos is not empty, this direction
					 * is done.
					 */
					XDB(1, printf(
				"$  xp%2o %d tp%2o bp%2o %x ifig%d bfig%d\n",
						xp->f_pos64, delta,
						tp->f_pos64,
						bp->b_f[bpos].f_pos64,
						F_DATTioRD(tp),
						ifig, bfig
						); );
					if( tp->f_c != empty ) {
					    goto ndir_ltd_rev;
					}
					tp = &(bp->b_f[bpos - delta]);
					goto next_ltd_rev;
				    }
				    if( ! (F_DATT(&(bp->b_f[bpos])) & F_IATT(eking)
					  & selfmask
					  )	/* not yet pinned */
				     && (  (p->f_f == dame)
					|| (att_dir(ekpos, pos) != bdir)
					)	/* unpinnable by opening pos */
				      ) {
					goto ndir_ltd_rev;
				    }
				}while( ++bfig, atts >>= 1 );
			    }
			}else {
			    backatts = 0;
			}
			if( ! (F_DATT(tp) & ifigset) ) {
			    continue;		/* cannot reach target */
			}
			if( dam_dir(pindir)
			 && (lin_dir(pindir) != lin_dir(att_dir(pos,tpos))) ) {
			    continue;		/* illegal: pinning left */
			}
			if( (covp[tpos-ekpos] & escs) != escs ) {
			    continue;		/* some escape not covered */
			}
					/* the K might beat back: */
			if( (F_DATT(tp) & BOTH_KINGS & enemymask)
			 && ! (F_DATT(tp) & ~ifigset & selfmask)
			 && ! (F_IATT(tp) & F_DATT(p) & selfmask)
			  ) {
			    continue;
			}
					/* someone else might beat back: */
			if( (atts = backatts) ) {	/* enum backatts: */
				register int		zeros;
				register int		bfig;
				register Position	bpos;
				register int		bdir;
			    bfig = 0;
			    do {
				    XDB(1, printf("$  m1g bkatt2 %x\n",atts); );
				while( ! (atts & 01) ) {
				    zeros = MIN_LOW_ZERO(atts);
				    atts >>= zeros; bfig += zeros;
				}
				bpos = bp->b_piece[bfig];
				if( bp->b_f[bpos].f_f != bauer )
				    continue;		/* analysed above */
				bdir = att_dir(ekpos, bpos);
				if( ! dam_dir(bdir) ) {	/* unpinnable */
				    goto next_ltd_rev;
				}
				if( att_dir(ekpos, tpos) == bdir ) {
				    goto next_ltd_rev;	/* won't leave */
				}
				if( ! (F_DATT(&(bp->b_f[bpos])) & F_IATT(eking)
				      & selfmask
				      )
				 && (  (p->f_f == dame)
				    || (att_dir(ekpos, pos) != bdir)
				    )
				  ) {
				    goto next_ltd_rev;
				}
			    }while( ++bfig, atts >>= 1 );
			}
			if( ckatts ) {
			    if( (att_dir(kpos, tpos) != ckdir)
			     || (  (tpos != ckpos)
				&& (att_dir(tpos, ckpos) != ckdir))
			      ) {
				continue;	/* illegal: ck not blocked */
			    }
			}
			XDB(1, printf("$  m1g ltdrev appm\n"); );
			m.m_to = tpos;
			app_move(&m, lp);
	next_ltd_rev:	;
		    }while( (tp->f_c == empty) || (tp == p) || (tp == eking) );
	ndir_ltd_rev:;
		}
		XDB(1, printf("$  m1g ltdrev rdy\n"); );
		break;
	     case koenig:		/* implicit K */
		pindir = att_dir(pos,ekpos);		/* myK -> hisK */
		if( ! dam_dir(pindir)
		 || ! (F_DATT(p) & F_IATT(eking) & F_IATT(&(p[dam_mov[pindir]]))
		      & selfmask
		      )
		  ) {
		    break;		/* cannot uncover => no check */
		}
		pindir = lin_dir(pindir);
		for( i=MIN_D_DIR ; i<MAX_D_DIR ; ++i ) {
		    if( lin_dir(i) == pindir )
			continue;
		    delta = dam_mov[i];
		    tp = p + delta;
		    if( (tp->f_c == border) || (tp->f_c == self) )
			continue;
		    if( F_DATT(tp) & enemymask )
			continue;
		    if( (F_IATT(tp) & ckatts)
		     && (no_dir(ckdir) || (opp_dir(i) == ckdir)) ) {
			continue;
		    }
		    m.m_to = pos + delta;
		    app_move(&m, lp);
		}
		break;
	     default:
		panic("move1gen: figure %d", p->f_f);
		break;
	    } /* implicit switch over figure type */
	    scadd(sc_implsum[self], list_length(lp));
	    XDB(1, printf("$< m1g impl\n"); );
	} /* end of implicit case */
    }
rdy:;
    mg_link(lp);
    XDB(1, printf("< move1gen\n"); );
    return ! empty_list(lp);
}
