/*
    Copyright 2004 Brian Smith (brian@smittyware.com)
    This file is part of CM2GPX.

    CM2GPX is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    CM2GPX is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with CM2GPX; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"
#include "dates.h"

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

CDateParser::CDateParser()
{
	m_bValid = 0;
}

void CDateParser::Initialize()
{
#if defined(HAVE_LOCALE_H) && defined(HAVE_SETLOCALE)
	setlocale(LC_TIME, "");

	// Get sample to detect local date format
	struct tm tmi;
	char buf[16];
	bzero(&tmi, sizeof(tmi));
	tmi.tm_year = 104;
	tmi.tm_mon = 0;
	tmi.tm_mday = 28;
	strftime(buf, 15, "%x", &tmi);

	// Scan formatted date for delimiter
	int n = strlen(buf);
	m_sFormat.empty();
	while (n--)
	{
		char ch = buf[n];
		if (ch < '0' || ch > '9')
		{
			char fmt[10];
			sprintf(fmt, "%%d%c%%d%c%%d", ch, ch);
			m_sFormat = fmt;
			break;
		}
	}

	if (m_sFormat.empty())
		return;	// Shouldn't happen, but just in case...

	int n1, n2, n3;
	if (sscanf(buf, m_sFormat.c_str(), &n1, &n2, &n3) != 3)
		return; // Not a fully numeric date format

	// What order are the numbers in?
	if (n1 == 4 || n1 == 2004)
	{
		if (n2 == 1)
			m_nFormat = DATE_YMD;
		else
			m_nFormat = DATE_YDM;
	}
	else if (n1 == 1)
	{
		if (n2 == 28)
			m_nFormat = DATE_MDY;
		else
			m_nFormat = DATE_MYD;
	}
	else if (n1 == 28)
	{
		if (n2 == 1)
			m_nFormat = DATE_DMY;
		else
			m_nFormat = DATE_DYM;
	}

	// If we made it this far, we've successfully found the date format
	m_bValid = 1;
#endif
}

string CDateParser::Normalize(string sDate)
{
	if (!m_bValid)
		return "";	// Format deduction didn't work

	int n1, n2, n3;
	if (sscanf(sDate.c_str(), m_sFormat.c_str(), &n1, &n2, &n3) != 3)
		return "";	// Not a numeric date format

	int month, day, year;
	switch (m_nFormat)
	{
	case DATE_YMD: year = n1; month = n2; day = n3; break;
	case DATE_YDM: year = n1; month = n3; day = n2; break;
	case DATE_MDY: year = n3; month = n1; day = n2; break;
	case DATE_MYD: year = n2; month = n1; day = n3; break;
	case DATE_DMY: year = n3; month = n2; day = n1; break;
	case DATE_DYM: year = n2; month = n3; day = n1; break;
	}

	// If the local date format doesn't match that in the CacheMate 
	// reocrd, one of these checks may fail.  Return an empty date in
	// that case
	if (month < 1 || month > 12)
		return "";
	if (day < 1 || day > 31)
		return "";

	if (year < 100)
	{	// Make a guess as to the century part
		if (year >= 30)
			year += 1900;
		else
			year += 2000;
	}

	char buf[32];
	sprintf(buf, "%04d-%02d-%02dT00:00:00", year, month, day);
	return buf;
}
