/*
    Copyright 2004 Brian Smith (brian@smittyware.com)
    This file is part of CM2GPX.

    CM2GPX is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    CM2GPX is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with CM2GPX; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"
#include "gpxwriter.h"
#include "util.h"

CGPXWriter::CGPXWriter()
{
	m_bToStdout = 0;
	m_bFudgeTZ = 0;
	m_bKeepName = 0;
	m_bCategInType = 0;
}

string CGPXWriter::FormatDouble(double v)
{
	char buf[32];
	sprintf(buf, "%.6f", v);

	int i = strlen(buf);
	while (i--)
	{
		if (buf[i] != '0')
			break;

		buf[i] = 0;
	}

	if (buf[i] == '.')
		buf[i] = 0;

	return buf;
}

CXmlNode* CGPXWriter::BuildLogExt(CPDBRecord *pRec)
{
	int count = 0;
	char buf[32];

	CXmlNode *pRoot = new CXmlNode("cmlog:log");
	pRoot->AddAttr("version", "1.0");
	pRoot->AddAttr("found",
		(pRec->m_nFlags & CMREC_FOUND) ? "true" : "false");
	pRoot->AddAttr("xmlns:cmlog",
		"http://www.smittyware.com/schema/cmlog/1/0");
 
	if (pRec->m_nLogYear && pRec->m_nLogMonth && pRec->m_nLogDay)
	{
		sprintf(buf, "%04d-%02d-%02dT%02d:%02d:00",
			pRec->m_nLogYear, pRec->m_nLogMonth,
			pRec->m_nLogDay, pRec->m_nLogStartHour,
			pRec->m_nLogStartMin);
		pRoot->AddChild("cmlog:start_time", buf);

		sprintf(buf, "%04d-%02d-%02dT%02d:%02d:00",
			pRec->m_nLogYear, pRec->m_nLogMonth,
			pRec->m_nLogDay, pRec->m_nLogEndHour,
			pRec->m_nLogEndMin);
		pRoot->AddChild("cmlog:end_time", buf);

		count++;
	}

	pRoot->AddChild("cmlog:notes", pRec->m_aFields[FLD_NOTES]);
	if (!pRec->m_aFields[FLD_NOTES].empty())
		count++;

	if (pRec->m_nFlags & CMREC_FOUND)
		count++;

	if (count == 0)
	{
		delete pRoot;
		return NULL;
	}

	return pRoot;
}

CXmlNode* CGPXWriter::BuildGroundspeakExt(CPDBRecord *pRec)
{
	if (pRec->m_aFields[FLD_WAYPOINT] == pRec->m_aFields[FLD_NAME])
		return NULL;

	int bArchive, bAvail;
	bArchive = (pRec->m_aFields[FLD_STATUS] == "Archived");
	bAvail = (pRec->m_aFields[FLD_STATUS] != "Inactive");

	CXmlNode *pRoot = new CXmlNode("groundspeak:cache");
	pRoot->AddAttr("available", bAvail ? "True" : "False");
	pRoot->AddAttr("archived", bArchive ? "True" : "False");
	pRoot->AddAttr("xmlns:groundspeak",
		"http://www.groundspeak.com/cache/1/0");

	CXmlNode *pNode;
	string sFld = pRec->m_aFields[FLD_NAME];
	if (!sFld.empty())
		pRoot->AddChild("groundspeak:name", sFld);
	sFld = pRec->m_aFields[FLD_OWNER];
	if (!sFld.empty())
		pRoot->AddChild("groundspeak:owner", sFld);
	sFld = pRec->m_aFields[FLD_TYPE];
	if (!sFld.empty())
		pRoot->AddChild("groundspeak:type", sFld);
	sFld = pRec->m_aFields[FLD_CONTAINER];
	if (!sFld.empty())
		pRoot->AddChild("groundspeak:container", sFld);
	sFld = pRec->m_aFields[FLD_DIFFICULTY];
	if (!sFld.empty())
		pRoot->AddChild("groundspeak:difficulty", sFld);
	sFld = pRec->m_aFields[FLD_TERRAIN];
	if (!sFld.empty())
		pRoot->AddChild("groundspeak:terrain", sFld);
	sFld = pRec->m_aFields[FLD_COUNTRY];
	if (!sFld.empty())
		pRoot->AddChild("groundspeak:country", sFld);
	sFld = pRec->m_aFields[FLD_STATE];
	if (!sFld.empty())
		pRoot->AddChild("groundspeak:state", sFld);
	sFld = pRec->m_aFields[FLD_DESC];
	if (!sFld.empty())
	{
		pNode = pRoot->AddChild("groundspeak:long_description", 
			sFld);
		pNode->AddAttr("html", "False");
	}

	return pRoot;
}

CXmlNode* CGPXWriter::BuildRecord(CPDBRecord *pRec)
{
	CXmlNode *pWpt;

	if (pRec->m_dLat < m_dMinLat)
		m_dMinLat = pRec->m_dLat;
	if (pRec->m_dLon < m_dMinLon)
		m_dMinLon = pRec->m_dLon;
	if (pRec->m_dLat > m_dMaxLat)
		m_dMaxLat = pRec->m_dLat;
	if (pRec->m_dLon > m_dMaxLon)
		m_dMaxLon = pRec->m_dLon;

	pWpt = new CXmlNode("wpt");
	pWpt->AddAttr("lat", FormatDouble(pRec->m_dLat));
	pWpt->AddAttr("lon", FormatDouble(pRec->m_dLon));

	CXmlNode *pGS = BuildGroundspeakExt(pRec);
	int bGeocache = 0;
	if (pGS)
	{
		string sCheck = pRec->m_aFields[FLD_TYPE];
		CUtil::LowercaseString(sCheck);

		bGeocache = !pRec->m_aFields[FLD_DIFFICULTY].empty() &&
			!pRec->m_aFields[FLD_TERRAIN].empty();

		if (!bGeocache)
		{
			if (sCheck.find("cache") != string::npos)
				bGeocache = 1;
		}

		pWpt->AddChild(pGS);
	}

	string sDesc, sType, sSym, sCheck;
	if (bGeocache)
	{
		int bFound = (pRec->m_nFlags & CMREC_FOUND);
		sSym = bFound ? "Geocache Found" : "Geocache";

		sCheck = pRec->m_aFields[FLD_TYPE];
		CUtil::LowercaseString(sCheck);

		sType = "Geocache";
		if (sCheck != "geocache")
		{
			sType += "|";
			sType += pRec->m_aFields[FLD_TYPE];
		}
		if (bFound)
			sType += "|Found";

		sDesc = pRec->m_aFields[FLD_NAME];
		if (!pRec->m_aFields[FLD_OWNER].empty())
		{
			sDesc += " by ";
			sDesc += pRec->m_aFields[FLD_OWNER];
		}

		if (sCheck != "geocache")
		{
			sDesc += ", ";
			sDesc += pRec->m_aFields[FLD_TYPE];
		}

		if (!pRec->m_aFields[FLD_DIFFICULTY].empty() &&
			!pRec->m_aFields[FLD_TERRAIN].empty())
		{
			sDesc += " (";
			sDesc += pRec->m_aFields[FLD_DIFFICULTY] + "/";
			sDesc += pRec->m_aFields[FLD_TERRAIN] + ")";
		}
	}
	else
	{
		sType = pRec->m_aFields[FLD_TYPE];
		sSym = m_sDefSymbol;
		sDesc = pRec->m_aFields[FLD_DESC];

		if (sDesc.empty() || (pGS && m_bKeepName))
			sDesc = pRec->m_aFields[FLD_NAME];
		if (sType.empty() || (sType == "(none)"))
			sType = m_sDefType;
	}

	if (m_bCategInType)
	{
		sType += "|";
		sType += pRec->m_sCategory;
	}

	if (!sType.empty())
		pWpt->InsertChild("type", sType);
	if (!sSym.empty())
		pWpt->InsertChild("sym", sSym);
	if (!sDesc.empty())
		pWpt->InsertChild("desc", sDesc);
	pWpt->InsertChild("name", pRec->m_aFields[FLD_WAYPOINT]);

	if (!pRec->m_aFields[FLD_DATE].empty())
	{
		string sTime = pRec->m_aFields[FLD_DATE];
		if (m_bFudgeTZ)
			sTime += "Z";

		pWpt->InsertChild("time", sTime);
	}

	CXmlNode *pLog = BuildLogExt(pRec);
	if (pLog)
		pWpt->AddChild(pLog);

	return pWpt;
}

int CGPXWriter::WriteGPX(string sFile, RecordList &rList)
{
	CXmlNode *pRoot, *pNode;
	int bSuccess = 0;

	m_dMaxLat = m_dMaxLon = -200.0;
	m_dMinLat = m_dMinLon = 200.0;

	pRoot = new CXmlNode("gpx");
	pRoot->AddAttr("xmlns:xsd", "http://www.w3.org/2001/XMLSchema");
	pRoot->AddAttr("xmlns:xsi",
		"http://www.w3.org/2001/XMLSchema-instance");
	pRoot->AddAttr("version", "1.0");
	pRoot->AddAttr("creator", PACKAGE_STRING);
	pRoot->AddAttr("xsi:schemaLocation",
		"http://www.topografix.com/GPX/1/0 "
		"http://www.topografix.com/GPX/1/0/gpx.xsd "
		"http://www.groundspeak.com/cache/1/0 "
		"http://www.groundspeak.com/cache/1/0/cache.xsd "
		"http://www.smittyware.com/schema/cmlog/1/0 "
		"http://www.smittyware.com/schema/cmlog/1/0/cmlog.xsd");
	pRoot->AddAttr("xmlns", "http://www.topografix.com/GPX/1/0");

	int count = 0;
	RecordList::iterator iter = rList.begin();
	while (iter != rList.end())
	{
		CPDBRecord *pRec = *(iter++);
		pRoot->AddChild(BuildRecord(pRec));
		delete pRec;
		count++;
	}

	rList.clear();

	// Summary info/metadata
	if (count > 0)
	{
		pNode = pRoot->InsertChild("bounds");
		pNode->AddAttr("minlat", FormatDouble(m_dMinLat).c_str());
		pNode->AddAttr("minlon", FormatDouble(m_dMinLon).c_str());
		pNode->AddAttr("maxlat", FormatDouble(m_dMaxLat).c_str());
		pNode->AddAttr("maxlon", FormatDouble(m_dMaxLon).c_str());
	}

	FILE *fp;
	if (m_bToStdout)
		fp = stdout;
	else
		fp = fopen(sFile.c_str(), "w");

	if (fp)
	{
		fprintf(fp, "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
		fprintf(fp, "%s", pRoot->GetXML().c_str());
		bSuccess = 1;
	}

	delete pRoot;
	return bSuccess;
}

