/*
    Copyright 2004-2005 Brian Smith (brian@smittyware.com)
    This file is part of CM2GPX.

    CM2GPX is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    CM2GPX is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with CM2GPX; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"
#include "pdbreader.h"
#include "util.h"

// Byte-swapper utility functions

static UInt32 pdbLongSwap(UInt32 val)
{
#ifdef WORDS_BIGENDIAN
	return val;
#else
	UInt32 out;
	UInt8 *pIn = (UInt8*)&val, *pOut = (UInt8*)&out;
	int i;

	for (i=0; i<4; i++)
		pOut[3-i] = pIn[i];

	return out;
#endif
}

static UInt16 pdbShortSwap(UInt16 val)
{
#ifdef WORDS_BIGENDIAN
	return val;
#else
	UInt16 out;
	UInt8 *pIn = (UInt8*)&val, *pOut = (UInt8*)&out;
	int i;

	for (i=0; i<2; i++)
		pOut[1-i] = pIn[i];

	return out;
#endif
}

// PDB record object

CPDBRecord::CPDBRecord()
{
	m_pData = NULL;

	m_nFlags = 0;
	m_nLogYear = m_nLogMonth = m_nLogDay = m_nLogStartHour =
	m_nLogStartMin = m_nLogEndHour = m_nLogEndMin = 0;
}

CPDBRecord::~CPDBRecord()
{
	if (m_pData)
		free(m_pData);
}

int CPDBRecord::ProcessRawRecord(int nType, CDateParser *pDate)
{
	string sData;

	// Separate header and text out of raw record
	if (nType == CM_IMPORT_DB)
		sData = m_pData;
	else
	{
		CM_RecHeader *pHdr = (CM_RecHeader*)m_pData;
		sData = m_pData + sizeof(CM_RecHeader);

		m_nFlags = pdbLongSwap(pHdr->flags);
		m_nLogYear = pdbShortSwap(pHdr->year);
		m_nLogMonth = pdbShortSwap(pHdr->month);
		m_nLogDay = pdbShortSwap(pHdr->day);
		m_nLogStartHour = pdbShortSwap(pHdr->start_hour);
		m_nLogStartMin = pdbShortSwap(pHdr->start_min);
		m_nLogEndHour = pdbShortSwap(pHdr->end_hour);
		m_nLogEndMin = pdbShortSwap(pHdr->end_min);
	}

	// Delete raw record data (don't need it anymore)
	free(m_pData);
	m_pData = NULL;

	// Parse delimited string for fields
	int index, i=0;
	while (i < NUM_DB_FIELDS)
	{
		index = sData.find(1);
		if (index == string::npos)
			break;

		m_aFields[i++] = sData.substr(0, index);
		sData = sData.substr(index+1);
	}

	if (!ParseCoords())
		return 0;

	ParseDescHeaders(pDate);

	i = NUM_FIELDS;
	while (i--)
		CUtil::StripWhitespace(m_aFields[i]);

	return !m_aFields[FLD_WAYPOINT].empty();
}

void CPDBRecord::ParseDescHeaders(CDateParser *pDate)
{
	string sDesc, sHeader, sLine;
	string sOwner, sDate, sCountry, sState, sCont, sBugs, sStatus;
	int index;

	sDesc = m_aFields[FLD_DESC];
	index = sDesc.find("\n\n");
	if (index == string::npos)
		return;

	sHeader = sDesc.substr(0, index+1);
	sDesc = sDesc.substr(index+2);

	index = sHeader.find('\n');
	while (index != string::npos)
	{
		sLine = sHeader.substr(0, index);
		sHeader = sHeader.substr(index+1);

		if (sLine.substr(0, 7) == "Owner: ")
			sOwner = sLine.substr(7);
		else if (sLine.substr(0, 6) == "Date: ")
			sDate = pDate->Normalize(sLine.substr(6));
		else if (sLine.substr(0, 11) == "Container: ")
			sCont = sLine.substr(11);
		else if (sLine.substr(0, 6) == "Bugs: ")
			sBugs = sLine.substr(6);
		else if (sLine.substr(0, 8) == "Status: ")
			sStatus = sLine.substr(8);
		else if (sLine.substr(0, 10) == "Location: ")
		{
			sLine = sLine.substr(10);
			index = sLine.rfind(", ");

			if (index != string::npos)
			{
				sCountry = sLine.substr(index+2);
				sState = sLine.substr(0, index);
			}
			else
				sCountry = sLine;
		}
		else
			return;

		index = sHeader.find('\n');
	}

	m_aFields[FLD_DESC] = sDesc;
	m_aFields[FLD_OWNER] = sOwner;
	m_aFields[FLD_DATE] = sDate;
	m_aFields[FLD_COUNTRY] = sCountry;
	m_aFields[FLD_STATE] = sState;
	m_aFields[FLD_CONTAINER] = sCont;
	m_aFields[FLD_BUGS] = sBugs;
	m_aFields[FLD_STATUS] = sStatus;
}

#define MAX_PARTS 9
int CPDBRecord::ParseCoords()
{
	int i, n, j = 0, bNum = 0;
	char *szParse, *aParts[MAX_PARTS], szMap[MAX_PARTS+1];
	int bSuccess = 0, bHeadChar = 0;
	double lat, lon;

	szParse = (char*)malloc(m_aFields[FLD_COORD].size()+1);
	strcpy(szParse, m_aFields[FLD_COORD].c_str());
	n = strlen(szParse);

	// Parse coordinate string into relevant pieces
	for (i=0; i<n; i++)
	{
		char ch = szParse[i];
		if (ch >= 'a' && ch <= 'z')
		{
			ch = ch - 'a' + 'A';
			szParse[i] = ch;
		}

		if ((ch >= '0' && ch <= '9') || ch == '.' ||
			((ch == '-') && !bHeadChar && !bNum))
		{
			if (!bNum)
			{
				szMap[j] = '#';
				aParts[j++] = &szParse[i];
				bNum = 1;
			}
		}
		else if (bNum)
		{
			bNum = 0;
			szParse[i] = 0;
		}
		else if (ch == 'N' || ch == 'S')
		{
			bHeadChar = 1;
			szMap[j] = 'N';
			aParts[j++] = &szParse[i];
		}
		else if (ch == 'E' || ch == 'W')
		{
			bHeadChar = 1;
			szMap[j] = 'E';
			aParts[j++] = &szParse[i];
		}

		if (j == MAX_PARTS)
			break;
	}

	szMap[j] = 0;

	if (!strcmp(szMap, "##"))
	{	// DDD.ddddd
		lat = atof(aParts[0]);
		lon = atof(aParts[1]);

		bSuccess = 1;
	}
	else if (!strcmp(szMap, "####"))
	{	// DDD mm.mmm
		double deg;

		lat = atof(aParts[1]);
		deg = atof(aParts[0]);
		if (deg < 0)
			lat = -lat;
		lat = deg + (lat / 60);

		lon = atof(aParts[3]);
		deg = atof(aParts[2]);
		if (deg < 0)
			lon = -lon;
		lon = deg + (lon / 60);

		bSuccess = 1;
	}
	else if (!strcmp(szMap, "N#E#"))
	{	// hddd.ddddd
		lat = atof(aParts[1]);
		if (aParts[0][0] == 'S')
			lat = -lat;

		lon = atof(aParts[3]);
		if (aParts[2][0] == 'W')
			lon = -lon;

		bSuccess = 1;
	}
	else if (!strcmp(szMap, "N##E##"))
	{	// hddd mm.mmm
		lat = atof(aParts[2]);
		lat = atof(aParts[1]) + (lat / 60);
		if (aParts[0][0] == 'S')
			lat = -lat;

		lon = atof(aParts[5]);
		lon = atof(aParts[4]) + (lon / 60);
		if (aParts[3][0] == 'W')
			lon = -lon;

		bSuccess = 1;
	}
	else if (!strcmp(szMap, "N###E###"))
	{	// hddd mm ss.s
		lat = atof(aParts[3]);
		lat = atof(aParts[2]) + (lat / 60);
		lat = atof(aParts[1]) + (lat / 60);
		if (aParts[0][0] == 'S')
			lat = -lat;

		lon = atof(aParts[7]);
		lon = atof(aParts[6]) + (lon / 60);
		lon = atof(aParts[5]) + (lon / 60);
		if (aParts[4][0] == 'W')
			lon = -lon;

		bSuccess = 1;
	}

	if (bSuccess)
	{	// Coordinate range checking
		if (lat < -90.0 || lat > 90.0)
			bSuccess = 0;
		if (lon < -180.0 || lon >= 180.0)
			bSuccess = 0;
	}

	if (bSuccess)
	{
		m_dLat = lat;
		m_dLon = lon;
	}

	free(szParse);
	return bSuccess;
}

// PDB file reader/container

CPDBReader::CPDBReader()
{
	m_bFoundFilt = 0;
	m_bBkmkFilt = 0;
	m_bExactMatch = 0;
}

CPDBReader::~CPDBReader()
{
	RecordList::iterator iter = m_entries.begin();
	while (iter != m_entries.end())
		delete (CPDBRecord*)(*iter++);

	m_entries.clear();
}

PDBHeader *CPDBReader::ReadHeader(int fd)
{
	int rc;
	u_long len;
	UInt16 num;
	PDBHeader *pHdr;

	pHdr = (PDBHeader*)malloc(sizeof(PDBHeader));
	if (!pHdr)
		return NULL;

	rc = read(fd, pHdr, sizeof(PDBHeader));
	if (rc < sizeof(PDBHeader))
	{
		free(pHdr);
		return NULL;
	}

	num = pdbShortSwap(pHdr->recordList.numRecords);
	free(pHdr);
	len = sizeof(PDBHeader) + num * sizeof(PDBRecordEntry);
	pHdr = (PDBHeader*)malloc(len);
	if (!pHdr)
		return NULL;

	lseek(fd, 0, SEEK_SET);
	rc = read(fd, pHdr, len);
	if (rc < len)
	{
		free(pHdr);
		return NULL;
	}

	PDBRecordEntry *pRecList = 
		(PDBRecordEntry*)&(pHdr->recordList.dummy);
	u_long lastOfs = pdbLongSwap(pRecList[num-1].localChunkID);
	m_nFileLen = lseek(fd, 0, SEEK_END);
	if (lastOfs > m_nFileLen)
	{
		free(pHdr);
		return NULL;
	}

	return pHdr;
}

void CPDBReader::ReadAppInfoBlock(int fd, PDBHeader *pHdr)
{
	int bSuccess = 0;

	u_long ofs = pdbLongSwap(pHdr->appInfoID);
	if (ofs > 0)
	{
		if (lseek(fd, ofs, SEEK_SET) == ofs)
		{
			int len = sizeof(ItemAppInfoType);
			if (read(fd, &m_AppInfo, len) == len)
				bSuccess = 1;
		}
	}

	if (!bSuccess)
	{
		bzero(&m_AppInfo, sizeof(m_AppInfo));
		strcpy(m_AppInfo.categoryLabels[0], "Unfiled");
	}
}

int CPDBReader::CheckFilterString(string sFilter, string sCheck)
{
	if (sFilter.empty())
		return 1;
	if (sCheck.empty())
		return 0;

	CUtil::LowercaseString(sCheck);
	CUtil::LowercaseString(sFilter);

	int nPos = sFilter.find(':');
	while (nPos != string::npos)
	{
		string sSub = sFilter.substr(0, nPos);
		sFilter = sFilter.substr(nPos+1);

		if (sSub.empty())
			return 0;

		if (m_bExactMatch)
		{
			if (sCheck == sSub)
				return 1;
		}
		else
		{
			if (strstr(sCheck.c_str(), sSub.c_str()))
				return 1;
		}

		nPos = sFilter.find(':');
	}

	if (sFilter.empty())
		return 0;

	if (m_bExactMatch)
	{
		if (sCheck == sFilter)
			return 1;
	}
	else
	{
		if (strstr(sCheck.c_str(), sFilter.c_str()))
			return 1;
	}

	return 0;
}

int CPDBReader::ReadRecords(int fd, PDBHeader *pHdr)
{
	m_entries.clear();

	u_long len, i, n;
	PDBRecordEntry *pRecList;

	n = pdbShortSwap(pHdr->recordList.numRecords);
	pRecList = (PDBRecordEntry*)&(pHdr->recordList.dummy);

	if (n > 0)
		ReadAppInfoBlock(fd, pHdr);

	for (i=0; i<n; i++)
	{
		u_long cur, next;

		cur = pdbLongSwap(pRecList[i].localChunkID);
		if (i == (n-1))
			next = m_nFileLen;
		else
			next = pdbLongSwap(pRecList[i+1].localChunkID);

		len = next - cur;

		if (lseek(fd, cur, SEEK_SET) != cur)
			return 0;

		char *pBuf = (char*)malloc(len);
		if (!pBuf)
			return 0;
		if (read(fd, pBuf, len) < len)
		{
			free(pBuf);
			return 0;
		}

		CPDBRecord *pRec = new CPDBRecord;
		pRec->m_pData = pBuf;

		int nCateg = pRecList[i].attributes & 15;
		pRec->m_sCategory = m_AppInfo.categoryLabels[nCateg];
		int bAdd = pRec->ProcessRawRecord(m_nType, &m_Date);

		if (bAdd)
		{
			if (m_bFoundFilt &&
				!(pRec->m_nFlags & CMREC_FOUND))
				bAdd = 0;
			if (m_bBkmkFilt &&
				!(pRec->m_nFlags & CMREC_BOOKMARK))
				bAdd = 0;

			if (!CheckFilterString(m_sCategFilt, pRec->m_sCategory))
				bAdd = 0;
		}
		
		if (bAdd)
			m_entries.push_back(pRec);
		else
			delete pRec;
	}

	return 1;
}

void CPDBReader::GetDBName(PDBHeader *pHdr)
{
	char buf[33];

        bzero(buf, 33);
	strncpy(buf, (char*)pHdr->name, 32);
	m_sDBName = buf;

	int len = m_sDBName.size();
	if (m_nType == CM_MAIN_DB)
	{
		if (m_sDBName.size() > 10)
		{
			if (m_sDBName.substr(len-10) == "Items-cMat")
				m_sDBName = m_sDBName.substr(0, len-10);
		}
		else
			m_sDBName = "Items";
	}
	else
	{
		if (m_sDBName.substr(0, 5) == "cMat-")
			m_sDBName = m_sDBName.substr(5);
	}
}

int CPDBReader::ReadFile(string sPath, int bQuiet)
{
	int fd;
	int bSuccess = 0;

	m_Date.Initialize();

	fd = open(sPath.c_str(), PDB_OPEN_FLAGS, 0600);
	if (fd >= 0)
	{
		PDBHeader *pHdr = ReadHeader(fd);
		if (pHdr)
		{
			m_nType = CM_UNKNOWN;

			if (strncmp((char*)&pHdr->creator, PDB_CREATOR, 4))
				printf("Not a CacheMate PDB file!\n");
			else if (!strncmp((char*)&pHdr->type, PDB_TYPE_MAIN, 4))
				m_nType = CM_MAIN_DB;
			else if (!strncmp((char*)&pHdr->type, PDB_TYPE_IMPORT, 4))
				m_nType = CM_IMPORT_DB;

			if (m_nType != CM_UNKNOWN)
			{
				GetDBName(pHdr);
				bSuccess = ReadRecords(fd, pHdr);
			}
			else
				printf("Unknown CacheMate database type.\n");

			free(pHdr);
		}
		else
			printf("PDB file sanity checks failed.\n");

		close(fd);
		return bSuccess;
	}

	return 0;
}
