/*
    Copyright 2004 Brian Smith (brian@smittyware.com)
    This file is part of CM2GPX.

    CM2GPX is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    CM2GPX is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with CM2GPX; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"
#include "xmlnode.h"

CXmlNode::CXmlNode()
{
	m_pParent = NULL;
	m_bAttr = 0;
}

CXmlNode::CXmlNode(string sName, string sContent)
{
	m_sName = sName;
	m_sContent = sContent;
	m_pParent = NULL;
	m_bAttr = 0;
}

CXmlNode::~CXmlNode()
{
	XmlNodeList::iterator iter = m_Kids.begin();
	while (iter != m_Kids.end())
	{
		CXmlNode *pNode = *(iter++);
		delete pNode;
	}

	if (m_pParent)
		m_pParent->m_Kids.remove(this);
}

CXmlNode* CXmlNode::GetParent()
{
	return m_pParent;
}

CXmlNode* CXmlNode::GetChildByIndex(int nIndex)
{
	if (nIndex < 0)
		return NULL;

	XmlNodeList::iterator iter = m_Kids.begin();
	while (iter != m_Kids.end())
	{
		CXmlNode *pNode = *(iter++);
		if (pNode->m_bAttr)
			continue;
		if (nIndex-- == 0)
			return pNode;
	}

	return NULL;
}

CXmlNode* CXmlNode::GetChildByNameIndex(string sName, int nIndex)
{
	if (nIndex < 0)
		return NULL;

	XmlNodeList::iterator iter = m_Kids.begin();
	while (iter != m_Kids.end())
	{
		CXmlNode *pNode = *(iter++);
		if (pNode->m_bAttr)
			continue;
		if (sName != pNode->m_sName)
			continue;
		if (nIndex-- == 0)
			return pNode;
	}

	return NULL;
}

int CXmlNode::GetChildCount(string sName)
{
	int nCount = 0;

	XmlNodeList::iterator iter = m_Kids.begin();
	while (iter != m_Kids.end())
	{
		CXmlNode *pNode = *(iter++);
		if (pNode->m_bAttr)
			continue;

		if (!sName.empty())
		{
			if (sName != pNode->m_sName)
				continue;
		}

		nCount++;
	}

	return nCount;
}

void CXmlNode::AddChild(CXmlNode *pNode)
{
	pNode->m_pParent = this;
	m_Kids.push_back(pNode);
}

CXmlNode* CXmlNode::AddChild(string sName, string sValue)
{
	CXmlNode *pNode = new CXmlNode(sName, sValue);
	AddChild(pNode);
	return pNode;
}

void CXmlNode::InsertChild(CXmlNode *pNode)
{
	pNode->m_pParent = this;
	m_Kids.push_front(pNode);
}

CXmlNode* CXmlNode::InsertChild(string sName, string sValue)
{
	CXmlNode *pNode = new CXmlNode(sName, sValue);
	InsertChild(pNode);
	return pNode;
}

void CXmlNode::AddAttr(string sName, string sValue)
{
	CXmlNode *pAttr = AddChild(sName, sValue);
	pAttr->m_bAttr = 1;
}

CXmlNode* CXmlNode::GetAttrNode(string sName)
{
	XmlNodeList::iterator iter = m_Kids.begin();
	while (iter != m_Kids.end())
	{
		CXmlNode *pNode = *(iter++);
		if (!pNode->m_bAttr)
			continue;
		if (sName != pNode->m_sName)
			continue;

		return pNode;
	}

	return NULL;
}

int CXmlNode::GetAttrValue(string sName, string &rValue)
{
	CXmlNode *pNode = GetAttrNode(sName);
	if (pNode)
	{
		rValue = pNode->m_sContent;
		return 1;
	}
	else
	{
		rValue.erase();
		return 0;
	}
}

// Map of redundant Unicode characters (for conversion back)
typedef struct stCharMap
{
	uint16_t uch;
	uint16_t ach;
} stCharMap;
static const stCharMap aCharMap[] = {
	{ 0x0152, 0x8c },
	{ 0x0153, 0x9c },
	{ 0x0160, 0x8a },
	{ 0x0161, 0x9a },
	{ 0x0178, 0x9f },
	{ 0x0192, 0x83 },
	{ 0x02c6, 0x88 },
	{ 0x02dc, 0x98 },
	{ 0x2013, 0x96 },
	{ 0x2014, 0x97 },
	{ 0x2018, 0x91 },
	{ 0x2019, 0x92 },
	{ 0x201A, 0x82 },
	{ 0x201E, 0x84 },
	{ 0x201c, 0x93 },
	{ 0x201d, 0x94 },
	{ 0x2020, 0x86 },
	{ 0x2021, 0x87 },
	{ 0x2022, 0x95 },
	{ 0x2030, 0x89 },
	{ 0x2039, 0x8b },
	{ 0x203a, 0x9b },
	{ 0x20AC, 0x80 },
	{ 0x2122, 0x99 },
	{ 0, 0 }
};

WCHAR_T* CXmlNode::StringToWCS(string sStr, int &rLen)
{
	WCHAR_T *pBuf, *pTmp;
	int n = sStr.size();

	pBuf = new WCHAR_T[n+1];
	memset(pBuf, 0, (n+1)*sizeof(WCHAR_T));

#ifdef HAVE_WCHAR_H
	mbstowcs(pBuf, sStr.c_str(), n+1);
	rLen = wcslen(pBuf);
#else
	const char *szStr = sStr.c_str();
	pTmp = pBuf;
	while (*szStr)
		*(pTmp++) = *(szStr++);
	rLen = n;
#endif

	pTmp = pBuf;
	while (*pTmp)
	{
		const stCharMap *pRec = aCharMap;
		while (pRec->ach)
		{
			if (pRec->ach == *pTmp)
			{
				*pTmp = pRec->uch;
				break;
			}

			pRec++;
		}

		pTmp++;
	}

	return pBuf;
}

string CXmlNode::EncodeUTF8(string sData)
{
	WCHAR_T *pTmp;
	u_char *pResult;
	int i, j, n = sData.size();

	pTmp = StringToWCS(sData, n);

	int nb = n*6+1;
	pResult = new u_char[nb];
	memset(pResult, 0, nb);
	j = 0;

	for (i=0; i<n; i++)
	{
		unsigned long ch = pTmp[i];
		
		if (ch < 128)
			pResult[j++] = (ch & 127);
		else if (ch < 2048)
		{
			pResult[j++] = 192 + (ch >> 6);
			pResult[j++] = 128 + (ch & 63);
		}
		else if (ch < 65536)
		{
			pResult[j++] = 224 + (ch >> 12);
			pResult[j++] = 128 + ((ch >> 6) & 63);
			pResult[j++] = 128 + (ch & 63);
		}
		else if (ch < 2097152L)
		{
			pResult[j++] = 240 + (ch >> 18);
			pResult[j++] = 128 + ((ch >> 12) & 63);
			pResult[j++] = 128 + ((ch >> 6) & 63);
			pResult[j++] = 128 + (ch & 63);
		}
		else if (ch < 67108864L)
		{
			pResult[j++] = 248 + (ch >> 24);
			pResult[j++] = 128 + ((ch >> 18) & 63);
			pResult[j++] = 128 + ((ch >> 12) & 63);
			pResult[j++] = 128 + ((ch >> 6) & 63);
			pResult[j++] = 128 + (ch & 63);
		}
		else
		{
			pResult[j++] = 252 + (ch >> 30);
			pResult[j++] = 128 + ((ch >> 24) & 63);
			pResult[j++] = 128 + ((ch >> 18) & 63);
			pResult[j++] = 128 + ((ch >> 12) & 63);
			pResult[j++] = 128 + ((ch >> 6) & 63);
			pResult[j++] = 128 + (ch & 63);
		}
	}

	string sResult = (char*)pResult;

	delete[] pTmp;
	delete[] pResult;

	return sResult;
}

void CXmlNode::ReplaceString(string &rStr, string sOld, string sNew)
{
	string tmp, remain = rStr;
	int oldlen = sOld.size();
	int idx;

	if (!oldlen)
		return;

	idx = remain.find(sOld);
	while (idx >= 0)
	{
		tmp += remain.substr(0, idx);
		tmp += sNew;
		remain = remain.substr(idx+oldlen);

		idx = remain.find(sOld);
	}

	rStr = tmp + remain;
}

string CXmlNode::CheckForSpecialCharacters(string sStr)
{
	ReplaceString(sStr, "&", "&amp;");
	ReplaceString(sStr, "\"", "&quot;");
	ReplaceString(sStr, "<", "&lt;");
	ReplaceString(sStr, ">", "&gt;");
	return sStr;
}

string CXmlNode::GetXML(int nIndent)
{
	string sOutput, sIndent, sTmp;

	int i = nIndent;
	while (i--)
		sIndent += "  ";

	sOutput += sIndent;
	sOutput += "<";
	sOutput += m_sName;

	XmlNodeList::iterator iter = m_Kids.begin();
	while (iter != m_Kids.end())
	{
		CXmlNode *pNode = *(iter++);
		if (!pNode->m_bAttr)
			continue;

		string sValue;
		sValue = CheckForSpecialCharacters(pNode->m_sContent);
		sValue = EncodeUTF8(sValue);

		sOutput += " ";
		sOutput += pNode->m_sName;
		sOutput += "=\"";
		sOutput += sValue;
		sOutput += "\"";
	}

	if (GetChildCount() > 0)
	{
		sOutput += ">\n";

		XmlNodeList::iterator iter = m_Kids.begin();
		while (iter != m_Kids.end())
		{
			CXmlNode *pNode = *(iter++);
			if (pNode->m_bAttr)
				continue;

			sOutput += pNode->GetXML(nIndent+1);
		}

		sOutput += sIndent;
		sOutput += "</";
		sOutput += m_sName;
	}
	else if (!m_sContent.empty())
	{
		sTmp = CheckForSpecialCharacters(m_sContent);
		sTmp = EncodeUTF8(sTmp);

		sOutput += ">";
		sOutput += sTmp;
		sOutput += "</";
		sOutput += m_sName;
	}
	else
		sOutput += " /";

	sOutput += ">\n";
	return sOutput;
}
