/*
 * cripple - command line CD ripper/encoder wrapper with cddb support
 *
 * Copyright (C) 2003-2005 Neil Phillips
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * perform cddb lookup by http GET requests.
 *
 * currently not very well tested - suck it and see!
 *
 * seems to work fine directly to freedb servers (HTTP/1.1)
 * and also through a HTTP/1.0 proxy - HTTP/1.1 proxies not tested.
 */

#include "cripple.h"

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <ctype.h>

#if HAVE_STRINGS_H
#  include <strings.h>
#endif

static __inline__ void terminate_line(char *str)
{
	for(; *(unsigned char*)str >= 0x20; str++);
	*str = '\0';
}


static __inline__ int http_cddb_query(char *cddb_query, const char *base_url,
		const char *host)
{
	int i, len;

	len = sprintf(cddb_query, "GET %s?cmd=cddb+query+%08lx+%d",
			base_url, cd.discid, cd.tracks);
	
	for (i=0; i < cd.tracks; i++)
		len += sprintf(cddb_query+len, "+%d", cd.cdtoc[i].frame);
	
	len += sprintf(cddb_query+len, "+%d&hello="USER"+"HOST"+"CLIENT"+"
			VERSION"&proto=5 HTTP/1.1\r\n"
			"Host: %s\r\n\r\n"
			, cd.cdtoc[cd.tracks].frame / 75, host);	
	return len;	/* not including '\0' */
}



int connect_to_server(struct in_addr ip_addr, unsigned short port)
{
	int sock_fd;
	struct sockaddr_in remote_addr = {AF_INET};

	if((sock_fd = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
		perror("socket()");
		exit(1);
	}

	remote_addr.sin_addr = ip_addr;
	remote_addr.sin_port = port;

	fprintf(stderr, "connecting to %s:%hu...\n", inet_ntoa(ip_addr),
			ntohs(port));

	if( connect(sock_fd, (struct sockaddr *)&remote_addr,
				sizeof(struct sockaddr)) ) {
		perror("connect()");
		exit(1);
	}

	return sock_fd;
}



static __inline__ char *find_header_line(char *str, const char *cmp_str, int len)
{
	for(; *str && strncasecmp(str, cmp_str, len); str++)
		if( !(str = strchr(str, '\n')) ) return NULL;
	return str + len;
}



int recv_chunk(int sock_fd, int len)
{
	int i, rlen = 0;

	if( (mb.off + len + 1) > mb.size )
		extend_mb(&mb, mb.off + len + 1);
	/*
	 * don't use MSG_WAITALL as some systems e.g. cygwin don't support it.
	 */
	do {
		i = recv(sock_fd, mb.buf+mb.off, len-rlen, 0);
		if(i < 0) {
			perror("recv()");
			break;
		}
		mb.off += i;
		rlen += i;	
	} while(rlen < len && i);

	mb.buf[mb.off] = '\0';
	
	return rlen;
}



/*
 * recieve line to mb.buf+mb.off, return line length
 */
int recv_line(int sock_fd)
{
	int len = 1;

	while(recv(sock_fd, mb.buf + mb.off, 1, 0) &&
			mb.buf[mb.off++] != '\n') {
		len++;
		if(mb.off+1 >= mb.size)
			extend_mb(&mb, mb.size + BUFSIZE);
	}
	mb.buf[mb.off] = '\0';

	return len;
}



static __inline__ void get_chunked_body(int sock_fd)
{
	/* chunk format: chunk-size(hex) CRLF chunk-data CRLF */
	int i, chunk_len;

	DEBUG_PUTS("using chuncked method...\n");

	do {
		i = mb.off;
		recv_line(sock_fd);
		mb.off = i;
		chunk_len = strtol(mb.buf + mb.off, 0, 16);

		DEBUG_PRINTF("getting %d bytes (+CRLF)...\n", chunk_len );

		recv_chunk(sock_fd, chunk_len+2);	/* + trailing CRLF */
		mb.off -= 2;
		mb.buf[mb.off] = '\0';
			
		DEBUG_PRINTF("recieved %d bytes (+CRLF):\n***%s***\n",
				mb.off-i, mb.buf+i);
	} while(chunk_len);
}



/*
 * return values:
 * < 0: error.
 *   0: success, connection still open.
 *   1: success, connection closed by server.
 * > 1: http response code if <200 or >300.
 */
int make_http_request(int sock_fd, char *request, int request_len,
			const char *content_type)
{
	int i, response_code, chunk_len, status = 0;
	char *te, *ct;
	
	if( ! (i = send(sock_fd, request, request_len, 0)) ) {
		fprintf(stderr, "error send()ing request:\n\n%s\n", request);
		perror("send()");
		return -1;
	}
	DEBUG_PRINTF(
		"--------------------- sent %d bytes ---------------------\n%s"
		"----------------------------------------------------------\n\n"
		, i, request);

	fputs("sent request to server, getting HTTP header...\n", stderr);
	
	/*
	 * get first line of header
	 */
	mb.off = 0;
	recv_line(sock_fd);
	if( strncmp(mb.buf, "HTTP/", 5) ) {
		fprintf(stderr, "header does not start with 'HTTP/'\n");
		return -1;
	}
	for(i=0; mb.buf[i] != ' '; i++);
	
	response_code = atoi(mb.buf + i);
	if(response_code < 200 || response_code >= 300) {
		fprintf(stderr, "HTTP respose code = %d, check "
				"server/proxy/ports\n\n", response_code);
		return response_code;
	}

	/*
	 * get rest of header
	 */
	while(recv_line(sock_fd) > 2);

	DEBUG_PRINTF(
		"--------------------- recieved header --------------------\n%s"
		"----------------------------------------------------------\n\n"
		, mb.buf);
	fputs("got header, getting message-body...\n", stderr);

	/*
	 * Content-Type: type/subtype; par1; par2;...
	 */
	if(( ct = find_header_line(mb.buf, "Content-Type: ", 14) ))
		terminate_line(ct);
	else
		fputs("Warning: no Content-Type recieved\n", stderr);

	if(content_type && ct) {
		char *cte = strchr(ct, ';');
		if(cte) *cte = '\0';
		if(strcasecmp(ct, content_type)) {
		    fprintf(stderr, "wrong content-type: wanted %s, got %s.\n",
				content_type, ct);
		    return -1;
		}
	}
	
	status = find_header_line(mb.buf, "Connection: close", 17) ||
			find_header_line(mb.buf, "Proxy-Connection: close", 23);

	if( (te = find_header_line(mb.buf, "Transfer-Encoding: ", 19)) )
		terminate_line(te);

	DEBUG_PRINTF("recieved transfer-encoding: %s\n", te);

	/*
	 * done with header, so reset offset
	 */
	mb.off = 0;
	
	if(te && !strcasecmp(te, "chunked")) {
		get_chunked_body(sock_fd);
	} else {
		/*
		 * if there is a Content-Length header field it will be in
		 * decimal.
		 */
		if((te = find_header_line(mb.buf, "Content-Length: ", 16))) {
			if( ! (chunk_len = strtol(te, 0, 10)) ) {
				fprintf(stderr, "zero Content-Length!!!\n");
				return -1;
			}
			DEBUG_PRINTF("getting %d bytes...\n", chunk_len);
			recv_chunk(sock_fd, chunk_len);
			DEBUG_PRINTF("recieved %d bytes:\n\n***%s***\n\n",
					mb.off, mb.buf);
		} else {
			/*
			 * non-persistent connection, where close signals end
			 * of data (i.e. 'Connection: close',
			 * 'Proxy-Connection: close' or pre HTTP/1.1 server)
			 */
			DEBUG_PUTS("no Content-Length or Transfer-Encoding, "
				"so getting data until server closes "
				" connection...\n");
			while(recv_chunk(sock_fd, BUFSIZE) == BUFSIZE);
			DEBUG_PRINTF("recieved %d bytes before connection "
					"closed\n", mb.off);
			status = 1;
		}
	}
	return status;
}



void get_cddb_info(const char *serv_addr, unsigned short serv_port,
		   const char *cgi_path,
		   const char *proxy_addr, unsigned short proxy_port)
{
	const char	def_path[] = DEF_CGI_PATH;
	unsigned short	connect_port;
	char		request[512], base_url[128], *chrptr;
	const char	*connect_addr;
	int		sock_fd, request_len, i, j, categ, response_code;
	struct in_addr	connect_ip;
	struct hostent	*server_hostent;
	
	if(!serv_port) serv_port = DEF_HTTP_PORT;
	if(!cgi_path || !*cgi_path) cgi_path = def_path;
	
	sprintf(base_url, "http://%s:%hu%s", serv_addr, serv_port, cgi_path);
	
	connect_addr = proxy_addr ? proxy_addr : serv_addr;
	connect_port = htons(proxy_port ? proxy_port : serv_port);
	
	fprintf(stderr, "resolving %s ...\n", connect_addr);
	if( ! (server_hostent = gethostbyname(connect_addr)) ) {
		fputs("host lookup failure\n", stderr);
		exit(1);
	}

	for(i=0; server_hostent->h_addr_list[i]; i++)
		fprintf(stderr, "found IP address: %s\n", inet_ntoa(
			*(struct in_addr *)server_hostent->h_addr_list[i] ));

	/*
	 * take the first IP address, as gethostbyname() returns them in
	 * random order anyway
	 */
	connect_ip = *(struct in_addr *)*(server_hostent->h_addr_list);
	fprintf(stderr, "using address:    %s\n\n", inet_ntoa(connect_ip));
	
	/*
	 * "cddb query discid tot_trks frames1 frames2 ... framesN tot_secs"
	 */
	request_len = http_cddb_query(request, base_url, serv_addr);

	sock_fd = connect_to_server(connect_ip, connect_port);

	switch(make_http_request(sock_fd, request, request_len, "text/plain")) {
		case 0 : break;			/* OK, connection still open*/
		case 1 : close(sock_fd);	/* OK, connection closed */
			 sock_fd = 0;
			 break;
		default: fputs("error performing 'cddb query', exiting\n\n",
					 stderr);
			 exit(1);
	}

	response_code = atoi(mb.buf);

	fprintf(stderr, "\n"
		"--------------------- query response ---------------------\n%s"
		"----------------------------------------------------------\n\n"
		"response code: %d --> ", mb.buf, response_code);
	switch(response_code) {
		case 200:	/* one exact match (200 categ discid DTITLE) */
			fputs("exact match!\n\n", stderr);
			for(i=0; !islower((int)mb.buf[i]); i++);
			break;
		case 210:		/* multiple exact matches */
		case 211:		/* inexact matches */
			fputs("multiple / inexact matches, using first "
					"suggestion.\n\n", stderr);
			for(i=1; (mb.buf[i-1] != '\n') && mb.buf[i]; i++);
			if(mb.buf[i]) break;
		default:
			fputs("no match (or invalid reply from server).\n\n",
					stderr);
			close(sock_fd);
			exit(1);
	}

	/*
	 * use category and discid from response for following read.
	 */
	for(categ = i; islower((int)mb.buf[i]); i++);
	mb.buf[i++] = '+';
	for(j = i+8; i<j && isxdigit((int)mb.buf[i]); i++);
	if(i != j || mb.buf[i] != ' ')
		fputs("Warning: server returned invalid discid!\n", stderr);
	mb.buf[i] = '\0';
	
	DEBUG_PRINTF("category: %s (offset %d)\n\n", mb.buf+categ, categ);
	
	/*
	 * read request: "cddb read categ discid"
	 */
	request_len = sprintf(request, "GET %s?cmd=cddb+read+%s"
			"&hello="USER"+"HOST"+"CLIENT"+"VERSION
			"&proto=5 HTTP/1.1\r\nHost: %s\r\n"
			"Connection: close\r\n\r\n",
			base_url, mb.buf+categ, serv_addr);
	if( ! sock_fd )
		sock_fd = connect_to_server(connect_ip, connect_port);
	
	if(make_http_request(sock_fd, request, request_len, "text/plain") < 0) {
		fputs("error performing 'cddb read', exiting\n\n", stderr);
		exit(1);
	}
	close(sock_fd);

	if( strncmp(mb.buf, "210", 3)) {
		fprintf(stderr, "bad reply code - should be 210");
		exit(1);
	}

	/*
	 * strip first line (response code) and final ".\r\n"
	 */
	if(!(chrptr = strchr(mb.buf, '\n'))) {
		fputs("invalid reply from server\n\n", stderr);
		exit(1);
	}
	mb.off -= (int)(chrptr - mb.buf) + 4;
	memmove(mb.buf, chrptr+1, mb.off);
	mb.buf[mb.off] = '\0';
}
