//Please refer to http://dansguardian.org/?page=copyright2
//for the license for this code.
//Written by Daniel Barron (daniel@jadeb//.com).
//For support go to http://groups.yahoo.com/group/dansguardian

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "autoconf/platform.h"
#include <syslog.h>
#include "ImageContainer.hpp"
#include "ConnectionHandler.hpp"
#include "DataBuffer.hpp"
#include "Socket.hpp"
#include "UDSocket.hpp"
#include "Ident.hpp"
#ifdef __BSD
	#include <sys/wait.h>
#else
	#include <wait.h>
#endif

#include "FDTunnel.hpp"
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <cstdio>
#include <algorithm>
#include <iostream>
#include <netdb.h>
#include <cstdlib>
#include <unistd.h>
#include <sys/time.h>

#ifdef __GCCVER3
    #include <istream>
#else
    #include <istream.h>
#endif

extern OptionContainer o;

void ConnectionHandler::handleConnection(int peerfd, String ip, int port) {
    Socket peerconn;
    peerconn.close();
    peerconn.setFD(peerfd);

    struct timeval thestart;
    struct timezone notused;
    gettimeofday(&thestart, &notused);

    peerconn.setTimeout(10);

    HTTPHeader header;  // to hold the incoming client request header

    header.setTimeout(10);  // set a timeout as we don't want blocking 4 eva

    HTTPHeader docheader;  // to hold the returned page header from proxy

    docheader.setTimeout(20);

    DataBuffer docbody;  // to hold the returned page

    docbody.setTimeout(120);

    bool waschecked = false;  // flags
    bool wasrequested = false;
    bool isexception = false;
    bool isourwebserver = false;
    bool wasclean = false;
    bool cachehit = false;
    bool forceauthrequest = false;
    bool isbypass = false;
    bool iscookiebypass = false;
    int bypasstimestamp = 0;
    bool ispostblock = false;
    bool pausedtoobig = false;

    if (o.preemptive_banning == 0) {
            forceauthrequest = true;
    }

    std::string mimetype = "-";

    String url;
    String urld;
    String urldomain;

    std::string exceptionreason;  // to hold the reason for not blocking

    int docsize = 0;  // to store the size of the returned document for loggin

    Ident ident;  // for holding

    std::string clientip = ip.toCharArray();  // hold the clients ip

    #ifdef DGDEBUG  // debug stuff surprisingly enough
        std::cout << "got connection" << std::endl;
        std::cout << clientip << std::endl;
    #endif

    Socket proxysock;  // to hold connection to proxy

    try {

        // connect to proxy
        int rc = proxysock.connect(o.proxy_ip, o.proxy_port);

        if (rc) {
            #ifdef DGDEBUG
                std::cerr << "Error connecting to proxy" << std::endl;
            #endif
            syslog(LOG_ERR, "%s","Error connecting to proxy");
            return;  // if we can't connect to the proxy, there is no point
                 // in continuing
        }

        header.in(&peerconn);  // get header from client
        url = header.url();
        urld = header.decode(url);
        if (url.after("://").contains("/")) {
            urldomain = url.after("//").before("/");
        }
        else {
            urldomain = url.after("//");
        }

        if (header.malformedURL(url)) {
            // checks for bad URLs to prevent security hole
            try { // writestring throws exception on error/timeout
                peerconn.writeString("HTTP/1.0 400 Bad Request\n");
                peerconn.writeString("Content-Type: text/html\n\n");
                peerconn.writeString("<HTML><HEAD><TITLE>DansGuardian - 400 Bad Request</TITLE></HEAD><BODY><H1>DansGuardian - 400 Bad Request</H1> ");
                peerconn.writeString(o.language_list.getTranslation(200));
                // The requested URL is malformed.
                peerconn.writeString("</BODY></HTML>\n");
            } catch (exception& e) {}
            try {
                proxysock.close();  // close connection to proxy
            } catch (exception& e) {}
            return;
        }

        if (o.use_xforwardedfor == 1) {
            std::string xforwardip = header.getXForwardedForIP();
            if (xforwardip.length() > 6) {
                clientip = xforwardip;
            }
            #ifdef DGDEBUG
                std::cout << "using x-forwardedfor:" << clientip << std::endl;
            #endif
        }

        std::string clientuser = ident.getUsername(&header, &clientip, port);
                                                   // extract username

        #ifdef DGDEBUG
            std::cout << "About to determine group" << std::endl;
        #endif

        int filtergroup = determineGroup(&clientuser);
        if (filtergroup < 0) {
            filtergroup = determineGroup(&clientip);
        }
        if (filtergroup < 0) {
            filtergroup = 0; //default group - one day configurable?
        }

        #ifdef DGDEBUG
            std::cout << "filtergroup:" << filtergroup << std::endl;
        #endif

        if (o.forwarded_for == 1) {
            header.addXForwardedFor(clientip);  // add squid-like entry
        }
        if ((*o.fg[filtergroup]).bypass_mode != 0) {
            #ifdef DGDEBUG
                std::cout << "About to check for bypass..." << std::endl;
            #endif
            bypasstimestamp = header.isBypassURL(&url, (*o.fg[filtergroup]).magic.c_str(), clientip.c_str());
            if (bypasstimestamp > 0) {
                #ifdef DGDEBUG
                    std::cout << "Bypass URL match" << std::endl;
                #endif
                header.chopBypass(url);
                url = header.url();
                urld = header.decode(url);
                if (bypasstimestamp > 1) {  // not expired
                    isbypass = true;
                    exceptionreason = o.language_list.getTranslation(606);
                }
            }
            else if (header.isBypassCookie(&urldomain, (*o.fg[filtergroup]).cookie_magic.c_str(), clientip.c_str())) {
                #ifdef DGDEBUG
                    std::cout << "Bypass cookie match" << std::endl;
                #endif
                iscookiebypass = true;
                isbypass = true;
                exceptionreason = o.language_list.getTranslation(607);
            }
            #ifdef DGDEBUG
                std::cout << "Finished bypass checks." << std::endl;
            #endif
        }
        if (isbypass) {
            #ifdef DGDEBUG
                std::cout << "Bypass activated!" << std::endl;
            #endif
        }
        else if (o.inipexceptions(&clientip)) {  // admin pc
            isexception = true;
            exceptionreason = o.language_list.getTranslation(600);
            // Exception client IP match.
        }
        else if (o.inuserexceptions(&clientuser)) { // admin user
            isexception = true;
            exceptionreason = o.language_list.getTranslation(601);
            // Exception client user match.
        }
        else if ((*o.fg[filtergroup]).inexceptions(urld)) {  // allowed site
            if ((*o.fg[0]).iswebserver(url)) {
                isourwebserver = true;
            }
            else {
                isexception = true;
                exceptionreason = o.language_list.getTranslation(602);
                // Exception site match.
            }
        }
        else if ((*o.fg[filtergroup]).inurlexceptions(urld)) {  // allowed url
            isexception = true;
            exceptionreason = o.language_list.getTranslation(603);
            // Exception url match.
        }



        #ifdef DGDEBUG
            std::cout << "extracted url:" << urld << std::endl;
        #endif


	if ( (isourwebserver || isexception || iscookiebypass)
	    // don't filter exception and local web server
	    // Cookie bypass so don't need to add cookie so just CONNECT
	    && !o.inBannedIPList(&clientip)		// bad users pc
	    && !o.inBannedUserList(&clientuser) ) { 	// bad user

	    proxysock.readyForOutput(10);  // exception on timeout or error
            header.out(&proxysock);  // send proxy the request
            try {
                FDTunnel fdt;  // make a tunnel object
                // tunnel from client to proxy and back
                fdt.tunnel(proxysock.getFD(), peerconn.getFD()); // not expected to exception
                docsize = fdt.throughput;
                if (!isourwebserver) {  // don't log requests to the web server
                    decideHowToLog(clientuser, clientip, url.toCharArray(), header.port, exceptionreason, header.requesttype().toCharArray(), docsize, o.ll, false, isexception, o.log_exception_hits, false, &thestart, cachehit, 200, mimetype);
                }

            } catch (exception& e) {}
            try {
                proxysock.close();  // close connection to proxy
            } catch (exception& e) {}
            return;  // connection dealt with so exit
        }


        NaughtyFilter checkme;  // our filter object
        checkme.filtergroup = filtergroup;

        char* i;

        // Improved IF structure as suggested by AFN

        if ((!forceauthrequest || header.requesttype().startsWith("CONNECT")) && !isbypass) {
            // if its a connect and we don't do filtering on it now then
            // it will get tunneled and not filtered.  We can't tunnel later
            // as its ssl so we can't see the return header etc
            // So preemptive banning is forced on with ssl unfortunately.
            // It is unlikely to cause many problems though.
            requestChecks(&header, &checkme, &urld, &clientip, &clientuser, filtergroup, &ispostblock);
        }


        if (!checkme.isItNaughty && header.requesttype().startsWith("CONNECT")) {
            // can't filter content of CONNECT
            proxysock.readyForOutput(10);  // exception on timeout or error
            header.out(&proxysock);  // send proxy the request
            try {
                FDTunnel fdt;  // make a tunnel object
                // tunnel from client to proxy and back
                fdt.tunnel(proxysock.getFD(), peerconn.getFD()); // not expected to exception
                docsize = fdt.throughput;
                decideHowToLog(clientuser, clientip, url.toCharArray(), header.port, exceptionreason, header.requesttype().toCharArray(), docsize, o.ll, false, isexception, o.log_exception_hits, false, &thestart, cachehit, 200, mimetype);
            } catch (exception& e) {}
            try {
                proxysock.close();  // close connection to proxy
            } catch (exception& e) {}
            return;  // connection dealt with so exit
        }

        if (!checkme.isItNaughty) {
            proxysock.readyForOutput(10);
            header.out(&proxysock);  // send header to proxy
            proxysock.checkForInput(120);
            docheader.in(&proxysock);  // get header from proxy
            #ifdef DGDEBUG
                std::cout << "got header from proxy" << std::endl;
            #endif
            wasrequested = true;  // so we know where we are later

            if (isbypass) {
	        docheader.setCookie("GBYPASS", hashedCookie(&urldomain, filtergroup, &clientip, bypasstimestamp).toCharArray());
	    }

            mimetype = docheader.getcontenttype().toCharArray();
            unsigned int p = (*o.fg[filtergroup]).banned_mimetype_list;
//            i = o.banned_mimetype_list.findInList((char*)mimetype.c_str());
            if ((i = (*o.lm.l[p]).findInList((char*)mimetype.c_str())) != NULL) {
                checkme.whatIsNaughty = o.language_list.getTranslation(800);
                // Banned MIME Type:
                checkme.whatIsNaughty += i;
                checkme.whatIsNaughtyLog = checkme.whatIsNaughty;
                checkme.isItNaughty = true;
            }
            #ifdef DGDEBUG
                std::cout << mimetype.length() << std::endl;
                std::cout << ":" << mimetype;
                std::cout << ":" << std::endl;
            #endif

            if (!checkme.isItNaughty && !docheader.isRedirection()) {
                // Can't ban file extensions of URLs that just redirect
                String tempurl = urld;
                String tempdispos = docheader.disposition();
                if (tempdispos.length() > 1) {
                    // dispos filename must take presidense
                    #ifdef DGDEBUG
                         std::cout << "Disposition filename:" << tempdispos << ":" << std::endl;
                    #endif
                    // The function expects a url so we have to
                    // generate a psudo one.
                    tempdispos = "http://foo.bar/" + tempdispos;
                    if ((i = (*o.fg[filtergroup]).inBannedExtensionList(tempdispos)) != NULL) {
                        checkme.whatIsNaughty = o.language_list.getTranslation(900);
                        // Banned extension:
                        checkme.whatIsNaughty += i;
                        checkme.whatIsNaughtyLog = checkme.whatIsNaughty;
                        checkme.isItNaughty = true;
                    }
                }
                else {
                    if (!tempurl.contains("?")) {
//                        i = o.inBannedExtensionList(tempurl);
                        if ((i = (*o.fg[filtergroup]).inBannedExtensionList(tempurl)) != NULL) {
                            checkme.whatIsNaughty =  o.language_list.getTranslation(900);
                            // Banned extension:
                            checkme.whatIsNaughty += i;
                            checkme.whatIsNaughtyLog = checkme.whatIsNaughty;
                            checkme.isItNaughty = true;
                        }
                    }
                    if (String(mimetype.c_str()).contains("application/")) {
                        while (tempurl.endsWith("?")) {
                            tempurl.chop();
                        }
                        while(tempurl.contains("/")) {  // no slash no url
                            if ((i = (*o.fg[filtergroup]).inBannedExtensionList(tempurl)) != NULL) {
                                checkme.whatIsNaughty =  o.language_list.getTranslation(900);
                                // Banned extension:
                                checkme.whatIsNaughty += i;
                                checkme.whatIsNaughtyLog = checkme.whatIsNaughty;
                                checkme.isItNaughty = true;
                                break;
                            }
                            while (tempurl.contains("/") && !tempurl.endsWith("?")) {
                                tempurl.chop();
                            }
                            tempurl.chop();  // get rid of the ?

                        }
                    }
                }
            }

            if (!checkme.isItNaughty && forceauthrequest && !docheader.authRequired()) {
                requestChecks(&header, &checkme, &urld, &clientip, &clientuser, filtergroup, &ispostblock);
            }

            if (docheader.iscontenttype("text") && !checkme.isItNaughty) {
                // here we check if its a known good one
                // if so we skip content checking
                waschecked = true;
                proxysock.checkForInput(120);
                if (docheader.isCompressed()) {
                    docbody.setDecompress(docheader.contentEncoding());
                }
                #ifdef DGDEBUG
                    std::cout << docheader.contentEncoding() << std::endl;
                    std::cout << "about to get body from proxy" << std::endl;
                #endif
                pausedtoobig = docbody.in(&proxysock);  // get body from proxy
                #ifdef DGDEBUG
                    if (pausedtoobig) {
                        std::cout << "got PARTIAL body from proxy" << std::endl;
                    }
                    else {
                        std::cout << "got body from proxy" << std::endl;
                    }
                #endif

                int dblen = docbody.length();
                docsize = dblen;

                if (o.url_cache_number > 0) {
                    if (wasClean(urld)) {
                        wasclean = true;
                        cachehit = true;
                        #ifdef DGDEBUG
                            std::cout << "url was clean skipping content checking" << std::endl;
                        #endif
                    }
                    // was not clean
                    else {
                        #ifdef DGDEBUG
                            system("date");
                        #endif
                        if (dblen <= o.max_content_filter_size
                            || o.max_content_filter_size == 0) {
                            checkme.checkme(&docbody);  // content filtering
                        }
                        #ifdef DGDEBUG
                            else {
                                std::cout << "content length large so skipping content filtering" << std::endl;
                            }
                            system("date");
                        #endif
                    }
                }
                // urlcache is not used
                else {
                        #ifdef DGDEBUG
                            system("date");
                        #endif
                        if (dblen <= o.max_content_filter_size
                            || o.max_content_filter_size == 0) {
                            checkme.checkme(&docbody);  // content filtering
                        }
                        #ifdef DGDEBUG
                            else {
                                std::cout << "content length large so skipping content filtering" << std::endl;
                            }
                            system("date");
                        #endif
                }
                bool contentmodified = false;
                if (dblen <= o.max_content_filter_size || o.max_content_filter_size == 0) {
                    contentmodified = docbody.contentRegExp(filtergroup);  // content modifying
                                          // uses global variable
                }
                #ifdef DGDEBUG
                    else {
                        std::cout << "content length large so skipping content modifying" << std::endl;
                    }
                    system("date");
                #endif

                if (contentmodified) {
                    #ifdef DGDEBUG
                        std::cout << "content modification made" << std::endl;
                    #endif
                    if (docheader.isCompressed()) {
                        docheader.removeEncoding(dblen);
                        // need to modify header to mark as not compressed
                        // it also modifies Content-Length as well
                    }
                    else {
                        docheader.setContentLength(docbody.buffer_length);
                    }
                }
                else {
                    docbody.swapbacktocompressed();
                    // if we've not modified it might as well go back to
                    // the original compressed version (if there) and send
                    // that to the browser
                }
                // here if its OK then request addition to good url database
            }
        }

        if (checkme.isException) {
            isexception = true;
            exceptionreason = checkme.whatIsNaughtyLog;
        }

        if (docheader.isRedirection()) {
            checkme.isItNaughty = false;
        }

        if (o.url_cache_number > 0) {
            if (!wasclean && !checkme.isItNaughty && docheader.iscontenttype("text") && header.requesttype() == "GET") {

                addToClean(urld);
            }
        }

        if (checkme.isItNaughty && !isbypass) {  // then we deny, unless we were told to bypass the block
            decideHowToLog(clientuser, clientip, url.toCharArray(), header.port, checkme.whatIsNaughtyLog, header.requesttype().toCharArray(), docsize, o.ll, true, false, false, false, &thestart, cachehit, 403, mimetype);
            if (denyAccess(&peerconn, &proxysock, &header, &docheader, &url, &checkme, &clientuser, &clientip, filtergroup, ispostblock)) {
                return;  // not stealth mode
            }
            // if get here in stealth mode
        }

        if (wasrequested == false) {
            proxysock.readyForOutput(10); // exceptions on error/timeout
            header.out(&proxysock); // exceptions on error/timeout
            proxysock.checkForInput(120); // exceptions on error/timeout
            docheader.in(&proxysock);  // get reply header from proxy
        }

        #ifdef DGDEBUG
            std::cout << "sending header to client" << std::endl;
        #endif
        peerconn.readyForOutput(10);  // exceptions on error/timeout
        docheader.out(&peerconn);  // send header to client
        #ifdef DGDEBUG
             std::cout << "sent header to client" << std::endl;
        #endif
        if (waschecked) {
            if(!docheader.authRequired() && !pausedtoobig) {
                decideHowToLog(clientuser, clientip, url.toCharArray(), header.port, exceptionreason, header.requesttype().toCharArray(), docsize, o.ll, false, isexception, o.log_exception_hits, docheader.iscontenttype("text"), &thestart, cachehit, 200, mimetype);
            }

            #ifdef DGDEBUG
                std::cout << "sending body to client" << std::endl;
            #endif
            peerconn.readyForOutput(10); // check for error/timeout needed
            docbody.out(&peerconn); // send doc body to client
            #ifdef DGDEBUG
                if (pausedtoobig) {
                    std::cout << "sent PARTIAL body to client" << std::endl;
                }
                else {
                    std::cout << "sent body to client" << std::endl;
                }
            #endif
            if (pausedtoobig) {
                #ifdef DGDEBUG
                    std::cout << "about to start tunnel to send the rest" << std::endl;
                #endif

                FDTunnel fdt;
                #ifdef DGDEBUG
                    std::cout << "tunnel activated" << std::endl;
                #endif
                fdt.tunnel(proxysock.getFD(), peerconn.getFD());
                docsize += fdt.throughput;
                decideHowToLog(clientuser, clientip, url.toCharArray(), header.port, exceptionreason, header.requesttype().toCharArray(), docsize, o.ll, false, isexception, o.log_exception_hits, docheader.iscontenttype("text"), &thestart, cachehit, 200, mimetype);
            }
        }
        else {  // was not supposed to be checked
            FDTunnel fdt;
            #ifdef DGDEBUG
                std::cout << "tunnel activated" << std::endl;
            #endif
            fdt.tunnel(proxysock.getFD(), peerconn.getFD());
            docsize = fdt.throughput;

            decideHowToLog(clientuser, clientip, url.toCharArray(), header.port, exceptionreason, header.requesttype().toCharArray(), docsize, o.ll, false, isexception, o.log_exception_hits, docheader.iscontenttype("text"), &thestart, cachehit, 200, mimetype);

        }
    } catch (exception& e) {
        #ifdef DGDEBUG
            std::cout << "connection handler caught an exception" << std::endl;
        #endif
        try {
            proxysock.close();  // close connection to proxy
        } catch (exception& e) {}
        return;
    }

    try {
        proxysock.close();  // close conection to squid
    } catch (exception& e) {}
    try {
        peerconn.readyForOutput(10);
    }
    catch (exception& e) {
        return;
    }
    return;
}

// if we don't do this the browsers complain
std::string ConnectionHandler::miniURLEncode(std::string s) {
    std::string encoded;
    char* buf = new char[16];  // way longer than needed
    unsigned char c;
    for(int i=0; i < (signed)s.length(); i++) {
        c = s[i];
        if ((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') || (c >= '0' && c <= '9')) {  // allowed characters in a url that have non special meaning
            encoded += c;
            continue;
        }
        sprintf(buf, "%x", c);
        encoded += "%";
        encoded += buf;
    }
    delete[] buf;
    return encoded;
}

void ConnectionHandler::doTheLogMan(std::string who, std::string from, std::string where, std::string what, std::string how, int size, struct timeval *thestart, bool cachehit, int code, std::string mimetype) {
    std::string logline,year,month,day,hour,min,sec,when,ssize,duration, utime, hitmiss, hier;
    struct timeval theend;
    struct timezone notused;
    gettimeofday(&theend, &notused);
    long durationsecs, durationusecs;
    durationsecs = theend.tv_sec - (*thestart).tv_sec;
    durationusecs = theend.tv_usec - (*thestart).tv_usec;
    durationusecs = (durationusecs / 1000) + durationsecs * 1000;
    String temp = String((int)durationusecs);
    while (temp.length() < 6) {
        temp = " " + temp;
    }
    duration = temp.toCharArray();
    temp = String((int)(theend.tv_usec / 1000));
    while (temp.length() < 3) {
        temp = "0" + temp;
    }
    if (temp.length() > 3) {
        temp = "999";
    }
    utime = temp.toCharArray();
    utime = "." + utime;
    utime = String((int)theend.tv_sec).toCharArray() + utime;

    if (what.length() > 3000) {
        what = what.substr(0, 2999);
    }

    if (code == 403) {
        hitmiss = "TCP_DENIED/403";
    }
    else {
        if (cachehit) {
            hitmiss = "TCP_HIT/";
            hitmiss += String((int)code).toCharArray();
        }
        else {
           hitmiss = "TCP_MISS/";
           hitmiss += String((int)code).toCharArray();
        }
    }
    hier = "DEFAULT_PARENT/";
    hier += o.proxy_ip;

    time_t tnow;  // to hold the result from time()
    struct tm *tmnow;  // to hold the result from localtime()
    time(&tnow);  // get the time after the lock so all entries in order
    tmnow = localtime(&tnow);  // convert to local time (BST, etc)
    year = String(tmnow->tm_year + 1900).toCharArray();
    month = String(tmnow->tm_mon + 1).toCharArray();
    day = String(tmnow->tm_mday).toCharArray();
    hour = String(tmnow->tm_hour).toCharArray();
    temp = String(tmnow->tm_min);
    if (temp.length() == 1) { temp = "0" + temp;}
    min = temp.toCharArray();
    temp = String(tmnow->tm_sec);
    if (temp.length() == 1) { temp = "0" + temp;}
    sec = temp.toCharArray();
    ssize = String(size).toCharArray();
    when = year + "." + month + "." + day + " " + hour + ":" + min + ":" + sec;
    switch(o.log_file_format){
        case 4:
            logline = when +"\t"+ who + "\t" + from + "\t" + where + "\t" + what + "\t" + how + "\t" + ssize + "\n";
            break;
        case 3:
            logline = utime + " " + duration + " " + from + " " + hitmiss + " " + ssize + " " + how + " " + where + " " + who + " " + hier + " " + mimetype + "\n";
            break;

        case 2:
            logline = "\"" + when +"\",\""+ who + "\",\"" + from + "\",\"" + where + "\",\"" + what + "\",\"" + how + "\",\"" + ssize + "\"\n";
            break;
        default:
            logline = when +" "+ who + " " + from + " " + where + " " + what + " " + how + " " + ssize + "\n";
    }
    UDSocket ipcsock;
    if (ipcsock.getFD() < 0) {
        syslog(LOG_ERR, "%s","Error creating ipc socket to log");
        return;
    }
    if (ipcsock.connect((char*) o.ipc_filename.c_str()) < 0) {  // connect to dedicated logging proc
        syslog(LOG_ERR, "%s","Error connecting via ipc to log");
        ipcsock.close();
        return;
    }
    ipcsock.writeString(logline.c_str());
    ipcsock.close();
}



void ConnectionHandler::decideHowToLog(std::string who, std::string from, std::string where, unsigned int port, std::string what, std::string how, int size, int loglevel, bool isnaughty, bool isexception, int logexceptions, bool istext, struct timeval *thestart, bool cachehit, int code, std::string mimetype) {

    if (loglevel == 0) {
        return;
    }
    if (port != 0 && port != 80) {
        String newwhere = where.c_str();
        if (newwhere.after("://").contains("/")) {
            String proto, host, path;
            proto = newwhere.before("://");
            host = newwhere.after("://");
            path = host.after("/");
            host = host.before("/");
            newwhere = proto;
            newwhere += "://";
            newwhere += host;
            newwhere += ":";
            newwhere += String((int)port);
            newwhere += "/";
            newwhere += path;
            where = newwhere.toCharArray();
        }
        else {
            where += ":";
            where += String((int)port).toCharArray();
        }
    }
    if (isnaughty) {
        what = "*DENIED* " + what;
        // make it stand out in the logs and also
        // more easily findable with a search
    }
    if (isexception) {
       if (logexceptions == 1) {
            what = "*EXCEPTION* " + what;
        }
        else {
            what = "";
        }
    }
    if ((isexception && logexceptions == 1)
         || isnaughty
         || loglevel == 3
         || (loglevel == 2 && istext)) {
        doTheLogMan(who, from, where, what, how, size, thestart, cachehit, code, mimetype);
    }
}



bool ConnectionHandler::wasClean(String url) {
    url = url.after("://");
    UDSocket ipcsock;
    if (ipcsock.getFD() < 0) {
        syslog(LOG_ERR, "%s","Error creating ipc socket to url cache");
        return false;
    }
    if (ipcsock.connect((char*) o.urlipc_filename.c_str()) < 0) {  // conn to dedicated url cach proc
        syslog(LOG_ERR, "%s","Error connecting via ipc to url cache");
        ipcsock.close();
        return false;
    }
    url += "\n";
    char* reply = new char[8];
    #ifdef DGDEBUG
        std::cout << "sending clean request:" << url.toCharArray() << std::endl;
    #endif
    try {
        ipcsock.writeString(url.toCharArray());  // throws on err
    }
    catch (exception& e) {
        #ifdef DGDEBUG
            std::cerr << "Exception writing to url cache" << std::endl;
            std::cerr << e.what() << std::endl;
        #endif
        syslog(LOG_ERR, "%s","Exception writing to url cache");
        syslog(LOG_ERR, "%s", e.what());
    }
    try {
        ipcsock.getline(reply, 7, 6);  // throws on err
    }
    catch (exception& e) {
        #ifdef DGDEBUG
            std::cerr << "Exception reading from url cache" << std::endl;
            std::cerr << e.what() << std::endl;
        #endif
        syslog(LOG_ERR, "%s","Exception reading from url cache");
        syslog(LOG_ERR, "%s", e.what());
    }
    ipcsock.close();
    if (reply[0] == 'Y') {
        delete[] reply;
        return true;
    }
    delete[] reply;
    return false;
}


void ConnectionHandler::addToClean(String url) {
    url = url.after("://");
    UDSocket ipcsock;
    if (ipcsock.getFD() < 0) {
        syslog(LOG_ERR, "%s","Error creating ipc socket to url cache");
        return;
    }
    if (ipcsock.connect((char*) o.urlipc_filename.c_str()) < 0) {  // conn to dedicated url cach proc
        syslog(LOG_ERR, "%s","Error connecting via ipc to url cache");
        #ifdef DGDEBUG
            std::cout << "Error connecting via ipc to url cache" << std::endl;
        #endif
        return;
    }
    url += "\n";
    url = "A " + url;
    try {
        ipcsock.writeString(url.toCharArray());  // throws on err
    }
    catch (exception& e) {
        #ifdef DGDEBUG
            std::cerr << "Exception adding to url cache" << std::endl;
            std::cerr << e.what() << std::endl;
        #endif
        syslog(LOG_ERR, "%s","Exception adding to url cache");
        syslog(LOG_ERR, "%s", e.what());
    }
    ipcsock.close();
}

void ConnectionHandler::requestChecks(HTTPHeader *header, NaughtyFilter *checkme, String *urld, std::string *clientip, std::string *clientuser, int filtergroup, bool *ispostblock) {
    char *i;
    int j;
    String temp;
    temp = (*urld);
    bool igsl = (*o.fg[filtergroup]).inGreySiteList(temp);
    bool igul = (*o.fg[filtergroup]).inGreyURLList(temp);

    if ((*o.fg[filtergroup]).blanketblock == 1 && !igsl && !igul) {
        (*checkme).isItNaughty = true;
        (*checkme).whatIsNaughty = o.language_list.getTranslation(502);
        // Blanket Block is active and that site is not on the white list.
        (*checkme).whatIsNaughtyLog = (*checkme).whatIsNaughty;
    }
    else if (o.inBannedIPList(clientip)) {
        (*checkme).isItNaughty = true;
        (*checkme).whatIsNaughtyLog = o.language_list.getTranslation(100);
                // Your IP address is not allowed to web browse:
        (*checkme).whatIsNaughtyLog += (*clientip);
        (*checkme).whatIsNaughty = o.language_list.getTranslation(101);
                // Your IP address is not allowed to web browse.
    }
    else if (o.inBannedUserList(clientuser)) {
        (*checkme).isItNaughty = true;
        (*checkme).whatIsNaughtyLog = o.language_list.getTranslation(102);
                // Your username is not allowed to web browse:
        (*checkme).whatIsNaughtyLog += (*clientuser);
        (*checkme).whatIsNaughty = (*checkme).whatIsNaughtyLog;
    }


    if (!(*checkme).isItNaughty && (*o.fg[filtergroup]).blanketblock == 0) {
        if ((*o.fg[filtergroup]).blanket_ip_block == 1 && isIPHostnameStrip(temp)) {
            (*checkme).isItNaughty = true;
            (*checkme).whatIsNaughty = o.language_list.getTranslation(502);
            //Blanket IP Block is active and that address is an IP only address.
            (*checkme).whatIsNaughtyLog = (*checkme).whatIsNaughty;
        }
        else {
            if (!igsl && !igul && ((i = (*o.fg[filtergroup]).inBannedSiteList(temp)) != NULL)) {
                (*checkme).whatIsNaughty = o.language_list.getTranslation(500); // banned site
                (*checkme).whatIsNaughty += i;
                (*checkme).whatIsNaughtyLog = (*checkme).whatIsNaughty;
                (*checkme).isItNaughty = true;
            }
        }
    }
    if (!(*checkme).isItNaughty) {
        if (!igsl && !igul && ((i = (*o.fg[filtergroup]).inBannedURLList(temp)) != NULL)) {
            (*checkme).whatIsNaughty = o.language_list.getTranslation(501);
            // Banned URL:
            (*checkme).whatIsNaughty += i;
            (*checkme).whatIsNaughtyLog = (*checkme).whatIsNaughty;
            (*checkme).isItNaughty = true;
        }
        else if (!igsl && !igul && ((j = (*o.fg[filtergroup]).inBannedRegExpURLList(temp)) >= 0)) {
            (*checkme).isItNaughty = true;
            (*checkme).whatIsNaughtyLog = o.language_list.getTranslation(503);
            // Banned Regular Expression URL:
            (*checkme).whatIsNaughtyLog += (*o.fg[filtergroup]).banned_regexpurl_list_source[j].toCharArray();
            (*checkme).whatIsNaughty = o.language_list.getTranslation(504);
            // Banned Regular Expression URL found.
        }
        else if (o.max_upload_size > -1) {
            if ((*header).isPostUpload()) {
                #ifdef DGDEBUG
                    std::cout << "is post upload" << std::endl;
                #endif
                if (o.max_upload_size == 0) {
                    (*checkme).whatIsNaughty = o.language_list.getTranslation(700);
                    // Web upload is banned.
                    (*checkme).whatIsNaughtyLog = (*checkme).whatIsNaughty;
                    (*checkme).isItNaughty = true;
                    (*ispostblock) = true;
                }
                else if ((*header).contentlength() > o.max_upload_size) {
                    (*checkme).whatIsNaughty = o.language_list.getTranslation(701);
                    // Web upload limit exceeded.
                    (*checkme).whatIsNaughtyLog = (*checkme).whatIsNaughty;
                    (*checkme).isItNaughty = true;
                    (*ispostblock) = true;
                }
            }
        }
    }
}

bool ConnectionHandler::isIPHostnameStrip(String url) {
    url.removePTP();  // chop off the ht(f)tp(s)://
    if (url.contains("/")) {
        url = url.before("/");  // chop off any path after the domain
    }
    return (*o.fg[0]).isIPHostname(url);
}

int ConnectionHandler::determineGroup(std::string *user) {
    String u = (*user).c_str();

    if (u.length() < 1 || u == "-") {

        return -1;
    }
    String ue = u;
    ue += "=";

    char *i = o.filter_groups_list.findStartsWithPartial(ue.toCharArray());

    if (i == NULL) {
        #ifdef DGDEBUG
            std::cout << "User not in filter groups list:" << ue << std::endl;
        #endif
        return -1;
    }
    #ifdef DGDEBUG
        std::cout << "User found:" << i << std::endl;
    #endif


    ue = i;

    if (ue.before("=") == u) {
        ue = ue.after("=filter");
        int l = ue.length();
        if (l < 1 || l > 2) {
            return -1;
        }
        int g = ue.toInteger();
        if (g > o.numfg) {
            return -1;
        }
        if (g > 0) {
            g--;
        }
        return g;
    }

    return -1;
}

// based on patch by Aecio F. Neto (afn@harvest.com.br) - Harvest Consultoria (http://www.harvest.com.br)
bool ConnectionHandler::denyAccess (Socket *peerconn, Socket *proxysock, HTTPHeader *header, HTTPHeader *docheader, String *url, NaughtyFilter *checkme, std::string *clientuser, std::string *clientip, int filtergroup, bool ispostblock) {
    try { // writestring throws exception on error/timeout
        if (o.reporting_level == 3) {
            (*proxysock).close();  // finished with proxy
            (*peerconn).readyForOutput(10);
            if ((*header).requesttype().startsWith("CONNECT")) {
                if (o.preemptive_banning == 1) {
                    String redirhttps = (*url).after("://");
                    if (!redirhttps.contains("/")) {
                        redirhttps += "/";
                    }
                    redirhttps = "http://" + redirhttps;
        //  The idea is that redirecting it back to the http page
        // of itself will also get blocked but won't confuse the
        // browser when it gets unencrypted content

                    try { // writestring throws exception on error/timeout
                        String writestring = "HTTP/1.0 302 Redirect\nLocation: ";
                        writestring += redirhttps;
                        writestring += "\n\n";
                        (*peerconn).writeString(writestring.toCharArray());
                     } catch (exception& e) {}
                }
                else {

                    // if preemptive banning is not in place then a redirect
                    // is not guaranteed to ban the site so we have to write
                    // an access denied page.  Unfortunately IE does not
                    // work with access denied pages on SSL more than a few
                    // hundred bytes so we have to use a crap boring one
                    // instead.  Nothing can be done about it - blame
                    // mickysoft.

                    String writestring = "HTTP/1.0 403 ";
                    writestring += o.language_list.getTranslation(500); // banned site
                    writestring += "\nContent-Type: text/html\n\n<HTML><HEAD><TITLE>DansGuardian - ";
                    writestring += o.language_list.getTranslation(500); // banned site
                    writestring += "</TITLE></HEAD><BODY><H1>DansGuardian - ";
                    writestring += o.language_list.getTranslation(500); // banned site
                    writestring += "</H1>";
                    writestring += (*url);
                    writestring += "</BODY></HTML>\n";

                    try { // writestring throws exception on error/timeout
                        (*peerconn).writeString(writestring.toCharArray());
                    } catch (exception& e) {}
                }
            }
            else {
                bool replaceimage = false;
                if (o.use_custom_banned_image == 1) {

                    // It would be much nicer to do a mime comparison
                    // and see if the type is image/* but the header
                    // never (almost) gets back from squid because
                    // it gets denied before then.
                    // This method is prone to over image replacement
                    // but will work most of the time.

                    String lurl = (*url);
                    lurl.toLower();
                    if (lurl.endsWith(".gif") ||
                        lurl.endsWith(".jpg") ||
                        lurl.endsWith(".jpeg") ||
                        lurl.endsWith(".jpe") ||
                        lurl.endsWith(".png") ||
                        lurl.endsWith(".bmp") ||
                        (*docheader).iscontenttype("image/")) {
                            replaceimage = true;
                    }
                }
                if (replaceimage) {
                    (*peerconn).writeString("HTTP/1.0 200 OK\n");
                      o.banned_image.display(peerconn);
                }
                else {
//Mod by Ernest W Lessenger Mon 2nd February 2004
//Other bypass code mostly written by Ernest also

                    String hashed;
                    if ((*o.fg[filtergroup]).bypass_mode != 0 && !ispostblock) {
                        hashed = hashedURL(url, filtergroup, clientip);
                    }
                    (*peerconn).writeString("HTTP/1.0 200 OK\nPragma: no-cache\nCache-control: no-cache\nContent-type: text/html\n\n");
                    o.html_template.display(peerconn,
                        (*url).toCharArray(),
                        (*checkme).whatIsNaughty.c_str(),
                        (*checkme).whatIsNaughtyLog.c_str(),
                        (*clientuser).c_str(),
                        (*clientip).c_str(),
                        String(filtergroup + 1),
                        hashed.toCharArray()
                    );

                }
            }
        }
        else if (o.reporting_level > 0) {
            (*proxysock).close();  // finshed with proxy
            (*peerconn).readyForOutput(10);
            if ((*checkme).whatIsNaughty.length() > 2048) {
                (*checkme).whatIsNaughty = String((*checkme).whatIsNaughty.c_str()).subString(0, 2048).toCharArray();
            }
            if ((*checkme).whatIsNaughtyLog.length() > 2048) {
                (*checkme).whatIsNaughtyLog = String((*checkme).whatIsNaughtyLog.c_str()).subString(0, 2048).toCharArray();
            }
            String writestring = "HTTP/1.0 302 Redirect\n";
            writestring += "Location: ";
            writestring += o.access_denied_address.c_str();

            if (o.non_standard_delimiter == 1) {
                writestring += "?DENIEDURL==";
                writestring += miniURLEncode((*url).toCharArray()).c_str();
                writestring += "::IP==";
                writestring += (*clientip).c_str();
                writestring += "::USER==";
                writestring += (*clientuser).c_str();
                if ((*o.fg[filtergroup]).bypass_mode != 0 && !ispostblock) {
                    //String timecode(time(NULL) + 300);
                   // String hashed = (*url).md5(std::string((*o.fg[filtergroup]).magic + timecode.toCharArray()).c_str());
                    //hashed += timecode;
                    writestring += "::HASH==";
                    writestring += hashedURL(url, filtergroup, clientip).after("GBYPASS=").toCharArray();
                }
                writestring += "::REASON==";
            }
            else {
                writestring += "?DENIEDURL=";
                writestring += miniURLEncode((*url).toCharArray()).c_str();
                writestring += "&IP=";
                writestring += (*clientip).c_str();
                writestring += "&USER=";
                writestring += (*clientuser).c_str();
                if ((*o.fg[filtergroup]).bypass_mode != 0 && !ispostblock) {
                    //String timecode(time(NULL) + 300);
                    //String hashed = (*url).md5(std::string((*o.fg[filtergroup]).magic + timecode.toCharArray()).c_str());
                    //hashed += timecode;
                    writestring += "&HASH=";
                    writestring += hashedURL(url, filtergroup, clientip).after("GBYPASS=").toCharArray();
                }
                writestring += "&REASON=";
            }
            if (o.reporting_level == 1) {
                writestring += miniURLEncode((*checkme).whatIsNaughty).c_str();
            }
            else {
                writestring += miniURLEncode((*checkme).whatIsNaughtyLog).c_str();
            }
            writestring += "\n\n";
            (*peerconn).writeString(writestring.toCharArray());
            #ifdef DGDEBUG  // debug stuff surprisingly enough
                std::cout << "******* redirecting to:" << std::endl;
                std::cout << writestring << std::endl;
                std::cout << "*******" << std::endl;
            #endif
        }
        else if (o.reporting_level == 0) {
            (*proxysock).close();  // finshed with proxy
            String writestring = "HTTP/1.0 200 OK\n";
            writestring += "Content-type: text/html\n\n";
            writestring += "<HTML><HEAD><TITLE>DansGuardian - ";
            writestring += o.language_list.getTranslation(1);  // access denied
            writestring += "</TITLE></HEAD><BODY><CENTER><H1>DansGuardian - ";
            writestring += o.language_list.getTranslation(1);  // access denied
            writestring += "</H1></CENTER></BODY></HTML>";
            (*peerconn).readyForOutput(10);
            (*peerconn).writeString(writestring.toCharArray());
            #ifdef DGDEBUG  // debug stuff surprisingly enough
                std::cout << "******* displaying:" << std::endl;
                std::cout << writestring << std::endl;
                std::cout << "*******" << std::endl;
            #endif
        }
        else if (o.reporting_level == -1) {  // stealth
            (*checkme).isItNaughty = false;  // dont block
        }
    } catch (exception& e) {}
    if ((*checkme).isItNaughty) { // not stealth mode then
        try {
            (*peerconn).readyForOutput(10);  //as best a flush as I can
        } catch (exception& e) {}
        try {
            (*proxysock).close();  // close connection to proxy
        } catch (exception& e) {}
       return true;  // we said no, so return true saying exit
    }
    return false;
}

String ConnectionHandler::hashedURL(String *url, int filtergroup, std::string *clientip) {
    String timecode(time(NULL) + (*o.fg[filtergroup]).bypass_mode);
    String magic = (*o.fg[filtergroup]).magic.c_str();
    magic += (*clientip).c_str();
    magic += timecode;
    String res = "GBYPASS=";
    if (!(*url).after("://").contains("/")) {
        String newurl = (*url);
        newurl += "/";
        res += newurl.md5(magic.toCharArray());
    }
    else {
        res += (*url).md5(magic.toCharArray());
    }
    res += timecode;
    return res;
}

String ConnectionHandler::hashedCookie(String *url, int filtergroup, std::string *clientip, int bypasstimestamp) {
    String timecode(bypasstimestamp);
    String magic = (*o.fg[filtergroup]).cookie_magic.c_str();
    magic += (*clientip).c_str();
    magic += timecode;
    String res = (*url).md5(magic.toCharArray());
    res += timecode;

    #ifdef DGDEBUG
        std::cout << "hashedCookie=" << res << std::endl;
    #endif
    return res;
}
