/**
 * Taken from TUDOS.
 * Author Bjoern Doebel, Dirk Vogt
 * License GLPv2
 **/ 
#include <linux/blkdev.h>
#include <linux/genhd.h>
#include <linux/bio.h>
#include <linux/hdreg.h>

#include "ddekit_bd.h"

struct gendisk* ddekit_find_disk(const char *name, int partno);
void ddekit_release_disk(struct gendisk *disk,  int partno);

struct ddekit_bd_disk *ddekit_bd_create(const char *name, int partno)
{
	struct ddekit_bd_disk *d =
		kmalloc(sizeof(struct ddekit_bd_disk), GFP_KERNEL);

	if (!d)
		return 0;
	d->disk = ddekit_find_disk(name, partno);
	d->partno = partno;

	if (!d->disk)
		goto out_kfree;

	struct hd_struct *part = d->disk->part_tbl->part[partno];
	if (!part)
		goto out_kfree;

	d->start_sector = part->start_sect;
	d->sectors = part->nr_sects;
	return d;

out_kfree:
	kfree(d);
	return 0;
}

void ddekit_bd_destroy(struct ddekit_bd_disk *disk)
{
	if (!disk)
		return;
	ddekit_release_disk(disk->disk, disk->partno);

	kfree(disk);
}


void ddekit_bd_page_cache_add(unsigned long data_address,
                              unsigned long data_size)
{
	unsigned long addr;

	for (addr = 0; addr < data_size; addr += PAGE_SIZE)
	{
		unsigned long va = data_address + addr;
		struct page *p = kmalloc(sizeof(struct page), GFP_KERNEL);
		p->virtual = (void *)va;
		dde_page_cache_add(p);
	}
}

void ddekit_bd_page_cache_remove(unsigned long data_address,
                                 unsigned long data_size)
{
	unsigned long addr;

	for (addr = 0; addr < data_size; addr += PAGE_SIZE)
	{
		struct page *p = dde_page_lookup(data_address + addr);
		if (p)
			dde_page_cache_remove(p);
			kfree(p);
	}
}

struct bio *ddekit_bd_create_bio(struct ddekit_bd_disk *d,
                                 unsigned long start_sector,
								 unsigned sg_count)
{
	struct gendisk *disk = d->disk;

	if (!disk)
		return 0;

	struct bio *b = bio_alloc(GFP_NOIO, sg_count);

	if (!b)
		return 0;

	b->bi_sector = start_sector;
	b->bi_bdev   = bdget_disk(disk, d->partno);
	b->bi_vcnt   = sg_count;
	b->bi_size   = 0;

	return b;
}

void ddekit_bd_destroy_bio(struct bio *b)
{}


int ddekit_bd_add_bio_io(struct bio *b,
                         unsigned i,
						 unsigned long addr,
						 unsigned long size)
{
	struct page *p = dde_page_lookup(addr & PAGE_MASK);

	/*
	 * If this page is not yet registered with the page cache,
	 * add it now.
	 */
	if (!p) {
		ddekit_printf("Warning:bio is not in page cache\n");
#if 0
		p = kmalloc(sizeof(struct page), GFP_KERNEL);
		p->virtual = (void *)(addr & PAGE_MASK);
		dde_page_cache_add(p);
#endif
	}

	b->bi_io_vec[i].bv_page = p;
	b->bi_io_vec[i].bv_len  = size;
	b->bi_io_vec[i].bv_offset = addr - (addr & PAGE_MASK);

	b->bi_size += size;

	return 0;
}


static bio_end_callback_t bio_end_callback = NULL;

bio_end_callback_t ddekit_bd_register_callback(bio_end_callback_t callback)
{
	bio_end_callback_t old = bio_end_callback;
	bio_end_callback = callback;

	return old;
}

static void do_end_io_callback(struct bio *b, int err)
{
	if (bio_end_callback)
			bio_end_callback(b->bi_private, err);
	bio_put(b);
}

void ddekit_bd_submit_bio(struct bio *b, void *private, int is_read)
{
	b->bi_end_io = do_end_io_callback;
	b->bi_private = private;

	submit_bio(is_read ? READ : WRITE, b);
	generic_unplug_device(b->bi_bdev->bd_disk->queue);
}

int ddekit_bd_get_geom(struct ddekit_bd_disk *disk,
                       struct ddekit_bd_geometry *geom)
{
	int result;
	struct block_device	* bdev;

	bdev = bdget_disk(disk->disk, 0);

	if (!bdev) {
		return -1;
	}
	result = disk->disk->fops->getgeo(bdev, (struct hd_geometry*) geom);
}


int ddekit_bd_wc_set(struct ddekit_bd_disk *disk,
                     int wc_enable)
{
	/* TODO*/
	return 0;
}

int ddekit_bd_wc_get(struct ddekit_bd_disk *d)
{

	return 0;
}

int ddekit_bd_flush(struct ddekit_bd_disk *d)
{
	struct block_device *bdev;

	if (!d->disk)
		return ENODEV;

	bdev = bdget_disk(d->disk, 0);

	return blkdev_issue_flush(bdev, 0);
}

unsigned long ddekit_bd_get_sectorcount(struct ddekit_bd_disk *d)
{
	return d->disk->part0.nr_sects;
}
