/**
 * usb_stub_driver.c
 *
 * Copyright (c) 2010 Dirk Vogt, <dvogt@few.vu.nl>
 *
 * This Linunx USB stub driver implements the ddekit_usb interface.
 */

#include <linux/usb.h>
#include <linux/list.h>
#include <linux/kthread.h>
#include <linux/wait.h>
#include <ddekit/usb.h>

static inline void dump_linux_urb(struct urb *urb) {
	printk("================\n");
	printk("DUMP: LX urb (0x%p)\n", urb);
	printk("================\n");
	printk("= usb_device: %p\n", urb->dev);
	printk("= pipe %d\n", urb->pipe);
	printk("=    type: %d\n", usb_pipetype(urb->pipe));
	printk("=    direction: %d\n", usb_pipein(urb->pipe));
	printk("= status: %d\n", urb->status);
	printk("= error_count: %d\n", urb->error_count);
	printk("= size: %d\n", urb->transfer_buffer_length);
	printk("= actual_length: %d\n", urb->actual_length);
	printk("= transfer_flags %x\n", urb->transfer_flags);
	printk("= setup_packet: \n");
	if (urb->setup_packet) {
		printk("=   bRequestType: 0x%x \n", ((struct usb_ctrlrequest *)urb->setup_packet)->bRequestType);
		printk("=   bRequest 0x%x \n", ((struct usb_ctrlrequest *)urb->setup_packet)->bRequest);
		printk("=   wValue: 0x%x \n", ((struct usb_ctrlrequest *)urb->setup_packet)->wValue);
		printk("=   wIndex: 0x%x \n", ((struct usb_ctrlrequest *)urb->setup_packet)->wIndex);
		printk("=   wLength: 0x%x \n", ((struct usb_ctrlrequest *)urb->setup_packet)->wLength);
	} else {
		printk("+ None. \n");
	}
	printk("===============\n");
}

#define MAX_CLIENTS

#if 0
#define DEBUG_MSG(fmt, ...) ddekit_printf("%s : "fmt"\n", __func__, ##__VA_ARGS__ ) 
#else
#define DEBUG_MSG(fmt, ...)
#endif

struct completion_context
{
	unsigned int transfer_flags;
	struct urb *urb;
	struct ddekit_usb_urb *d_urb;
	struct list_head list;
};

struct ddekit_usb_dev {
	struct list_head list;

	struct usb_device *usb_dev;    /* pointer to linux usb_device */

	int intf_count;        /* bitmask of interfaces */
	unsigned int interfaces;        /* bitmask of interfaces */
	
	struct  work_struct work;      /* Work struct and data for doing
	                                 synchronous calls */

	__u16 alternate;               /* the data for the set_interface */
	__u16 interface;               /* work                           */
	
	struct completion_context *context;
	int pipe;
	int ref_count;
	void *data;
};

LIST_HEAD(gbl_dev_list);

/* the callbacks */
static ddekit_usb_completion_cb completion_cb = NULL;
static ddekit_usb_connect_cb    connect_cb    = NULL;
static ddekit_usb_disconnect_cb disconnect_cb = NULL;

static struct usb_device_id usb_stub_driver_id_table [] = {
	{ .driver_info = 1 }, /* every device! */
	{ }     /* Terminating entry */
};

static void completion_handler(struct urb *urb);


/*
 * this function is part of the interface hack
 * (see usb/core/message.c:1735)
 */
int ddekit_usbintf_hack(int portnum, int interface)
{
	return 1;
}


static void
set_interface_work
(struct work_struct * work)
{
	struct ddekit_usb_dev *dev =  container_of(work,
	                                     struct ddekit_usb_dev, work);

	int ret;

	DEBUG_MSG("cancel all pending urbs!\n");

	schedule_timeout_interruptible(msecs_to_jiffies(1000));

	DEBUG_MSG("Set Interface: %d altsetting: %d\n", dev->interface,
	    dev->alternate);

	ret = usb_set_interface(dev->usb_dev,
	          dev->interface, dev->alternate);

	DEBUG_MSG("Set Interface returned %d\n",ret);

	/*
	 * success isn't enough because usb device may return -EPIPE 
	 * (see Linux' usb_set_interface())
	 */
	ret = usb_control_msg(dev->usb_dev,
	                      usb_sndctrlpipe(dev->usb_dev, 0),
						  USB_REQ_SET_INTERFACE,
						  USB_RECIP_INTERFACE,
						  dev->alternate,
						  dev->interface, 
						  NULL, 0, 5000);
	
	/*
	 * returning the status to the urb should be enough...
	 */
	((struct completion_context *)dev->context)->d_urb->status = ret;

	completion_handler(((struct completion_context *)dev->context)->urb);

}


static void 
clear_halt_work
(struct work_struct *work)
{
	int ret;

	struct ddekit_usb_dev *dev = container_of(work,
	                                    struct ddekit_usb_dev, work);
	
	DEBUG_MSG("CLEAR HALT: dev = %p\n");
	ret = usb_clear_halt(dev->usb_dev,dev->pipe);

	/* returning the status to the client should be enough...*/
	((struct completion_context *)dev->context)->d_urb->status = ret;

	completion_handler(((struct completion_context *)dev->context)->urb);
	
}


static struct completion_context *
create_context
(struct ddekit_usb_urb *d_urb)
{
	struct completion_context *context = NULL;

	context = (struct completion_context *)
		kmalloc(sizeof(struct completion_context),GFP_KERNEL);

	if (!context) {
		DEBUG_MSG("out of memory");
		return NULL;
	}

	context->d_urb        = d_urb;
	context->transfer_flags = d_urb->transfer_flags;

	return context;
}


static struct urb *
create_urb_common
(struct ddekit_usb_urb *d_urb, struct completion_context *ctx)
{
	struct urb *urb = NULL;

	/* allocate a URB */
	if (d_urb->type == DDEKIT_USB_TRANSFER_ISO) {
		urb = usb_alloc_urb(d_urb->number_of_packets, GFP_KERNEL);
	} else {
		urb = usb_alloc_urb(0, GFP_KERNEL);
	}
	
	if(urb == NULL) {
		DEBUG_MSG("Out of mem");
		return NULL;
	}

	/* setup field commonly used by all transfer types */
	
	urb->dev = d_urb->dev->usb_dev;
	
	/* drivers are not allowed to set these! */
	urb->transfer_flags          = d_urb->transfer_flags
		& ~URB_NO_TRANSFER_DMA_MAP
		& ~URB_NO_SETUP_DMA_MAP;

	urb->transfer_buffer         = d_urb->data;
	urb->transfer_buffer_length  = d_urb->size;
	urb->context                 = ctx;
	urb->complete                = completion_handler;
	urb->actual_length           = 0;

	return urb;
}


static int
setup_int_urb
(struct urb *urb, struct ddekit_usb_urb *d_urb)
{
	urb->interval = d_urb->interval;
	switch (d_urb->direction) {
		case DDEKIT_USB_IN:
			urb->pipe = usb_rcvintpipe(urb->dev, d_urb->endpoint);
			break;
		case DDEKIT_USB_OUT:
			urb->pipe = usb_sndintpipe(urb->dev, d_urb->endpoint);
			break;
		default:
			DEBUG_MSG("neither in nor out?");
			return EINVAL;
	}
	return 0;
}


static int
setup_blk_urb
(struct urb *urb, struct ddekit_usb_urb *d_urb)
{
	
	switch (d_urb->direction) {
		case DDEKIT_USB_IN:
			urb->pipe = usb_rcvbulkpipe(urb->dev, d_urb->endpoint);
			break;
		case DDEKIT_USB_OUT:
			urb->pipe = usb_sndbulkpipe(urb->dev, d_urb->endpoint);
			break;
		default:
			DEBUG_MSG("neither in nor out?");
			return EINVAL;
	}
	return 0;
}


static int 
setup_iso_urb
(struct urb *urb, struct ddekit_usb_urb *d_urb)
{

	urb->interval         = d_urb->interval;
	urb->number_of_packets = d_urb->number_of_packets;
	urb->start_frame      = d_urb->start_frame;

	memcpy(urb->iso_frame_desc, d_urb->iso_desc,
		urb->number_of_packets * sizeof(struct ddekit_usb_iso_packet_desc));
		
	switch (d_urb->direction) {
		case DDEKIT_USB_IN:
			urb->pipe = usb_rcvisocpipe(urb->dev, d_urb->endpoint);
			break;
		case DDEKIT_USB_OUT:
			urb->pipe = usb_sndisocpipe(urb->dev, d_urb->endpoint);
			break;
		default:
			DEBUG_MSG("neither in nor out?");
			return EINVAL;
	}

	return 0;
}


static int
setup_ctrl_urb
(struct urb *urb, struct ddekit_usb_urb *d_urb) 
{
	switch (d_urb->direction) {
		case DDEKIT_USB_IN:
			urb->pipe = usb_rcvctrlpipe(urb->dev, d_urb->endpoint);
			break;
		case DDEKIT_USB_OUT:
			urb->pipe = usb_sndctrlpipe(urb->dev, d_urb->endpoint);
			break;
		default:
			DEBUG_MSG("neither in nor out?");
			return EINVAL;
	}

	urb->setup_packet = d_urb->setup_packet;

	/* there are some control calls drivers are not allowed to do directly... */

	struct usb_ctrlrequest * req = (struct usb_ctrlrequest *) d_urb->setup_packet;

	/* ...set interface,... */

	if ((req->bRequest == USB_REQ_SET_INTERFACE) &&
		(req->bRequestType == USB_RECIP_INTERFACE))	{
		/*
		 * if so, we have to schedule the setinterface work of this device 
		 * */
		d_urb->dev->alternate = le16_to_cpu(req->wValue);
		d_urb->dev->interface = le16_to_cpu(req->wIndex);

		d_urb->dev->context   = urb->context;

		INIT_WORK(&d_urb->dev->work, set_interface_work);
		schedule_work(&d_urb->dev->work);
		return 1;
	}

	/* ... and clear halt */

	if ((req->bRequest == USB_REQ_CLEAR_FEATURE)  &&
		(req->bRequestType == USB_RECIP_ENDPOINT) &&
		(req->wValue == USB_ENDPOINT_HALT))	{

		int target_endp = le16_to_cpu(req->wIndex) & 0x000f;
		int target_dir  = le16_to_cpu(req->wIndex) & 0x0080;

		if (target_dir) {
			d_urb->dev->pipe = usb_rcvctrlpipe (urb->dev, target_endp);
		} else {
			d_urb->dev->pipe = usb_sndctrlpipe (urb->dev, target_endp);
		}

		d_urb->dev->context = urb->context;

		INIT_WORK(&d_urb->dev->work, clear_halt_work);
		schedule_work(&d_urb->dev->work);

		return 1;
	}

	return 0;
}


int
ddekit_usb_submit_urb
(struct ddekit_usb_urb *d_urb)
{
	struct urb *urb = NULL;
	struct completion_context *context=NULL;
	struct ddekit_usb_dev *ddev= NULL;
	int res;

	ddev = d_urb->dev;

	if (!ddev) {
		return EINVAL;
	}

	context = create_context(d_urb);

	if (context == NULL) {
		return ENOMEM;
	}

	/* create a urb filled up with commonly used fields */
	urb = create_urb_common(d_urb, context);

	if (urb == NULL) {
		res = ENOMEM;
		goto ERROR_FREE_CONTEXT;
	}

	context->urb = urb;

	/*  now do the transfertype specific stuff */
	switch (d_urb->type) {
		case DDEKIT_USB_TRANSFER_INT:
			DEBUG_MSG("DDEKIT_USB_TRANSFER_INT");
			res = setup_int_urb(urb, d_urb);
			break;
		case DDEKIT_USB_TRANSFER_BLK:
			DEBUG_MSG("DDEKIT_USB_TRANSFER_BLK");
			res = setup_blk_urb(urb, d_urb);
			break;
		case DDEKIT_USB_TRANSFER_ISO:
			DEBUG_MSG("DDEKIT_USB_TRANSFER_ISO");
			res = setup_iso_urb(urb, d_urb);
			break;
		case DDEKIT_USB_TRANSFER_CTL:
			DEBUG_MSG("DDEKIT_USB_TRANSFER_CTL");
			res = setup_ctrl_urb(urb, d_urb);
			if (res) {
				return 0;
			}
			break;
		default: 
			DEBUG_MSG("illegal transfer type");
			goto ERROR_FREE_URB;
	}
	if(res != 0) {
		res=0;
		goto ERROR_FREE_URB;
	}

	res = usb_submit_urb(urb, GFP_KERNEL);
	return res;

ERROR_FREE_URB:
	usb_free_urb(urb);
ERROR_FREE_CONTEXT:
	kfree(context);
	return res;
}


int
ddekit_usb_cancle_urb
(struct ddekit_usb_urb *d_urb)
{
	if (!d_urb->ddekit_priv) {
		return EINVAL;
	}

	/* get the corresponding linux URB */

	struct urb *urb = (struct urb*) d_urb->ddekit_priv;
	int res;

	res = usb_unlink_urb(urb);

	return res;
}


int
ddekit_usb_dev_set_data
(struct ddekit_usb_dev *ddev, void *data)
{
	ddev->data = data;
	return 0;
}


void *
ddekit_usb_dev_get_data
(struct ddekit_usb_dev *ddev)
{
	return ddev->data;
}


static void
completion_handler
(struct urb *urb)
{
	struct completion_context * context;
	struct ddekit_usb_urb *d_urb;

	context = (struct completion_context *) urb->context;

	d_urb = context->d_urb;

	d_urb->status         = urb->status;
	d_urb->actual_length  = urb->actual_length;
	d_urb->transfer_flags = context->transfer_flags;

	if (d_urb->type == DDEKIT_USB_TRANSFER_ISO) {
		d_urb->error_count    = urb->error_count;
		d_urb->start_frame = urb->start_frame;
		memcpy(d_urb->iso_desc, urb->iso_frame_desc,
			urb->number_of_packets * sizeof(struct ddekit_usb_iso_packet_desc));
	}

	DEBUG_MSG("giving back urb with status: %d", urb->status);

	completion_cb(d_urb->priv);
	usb_free_urb(urb);
	kfree(context);
}


static void
setup_interface_bitmap
(struct ddekit_usb_dev *ddev, struct usb_device *usb_dev)
{
	struct usb_host_config *conf = usb_dev->actconfig;
	int intf_count = conf->desc.bNumInterfaces;
	int i;

	ddev->intf_count = intf_count;
	for (i = 0; i < intf_count; i++) {
		struct usb_host_interface *intf;
		intf = conf->interface[i]->cur_altsetting;
		ddev->interfaces |= (1 << intf->desc.bInterfaceNumber);
	}
}



/*
 * usb_stub_driver_probe
 *
 * This function is called whenever a new device is attached to the USB bus.
 * We need to setup a ddekit_usb_device and forward it to the ddekit_usb
 * user.
 */
static int
usb_stub_driver_probe
(struct usb_interface * intf, const struct usb_device_id *id)
{
	struct usb_device *usb_dev  = interface_to_usbdev(intf);
	struct ddekit_usb_dev *ddev = NULL;
	struct list_head *entry;

	/* allready created a device for this usb_device? */
	list_for_each(entry, &gbl_dev_list) {
		struct ddekit_usb_dev *tmp_dev=
		    list_entry(entry, struct ddekit_usb_dev, list);
		if (tmp_dev->usb_dev == usb_dev) {
			ddev = tmp_dev;
			break;
		}
	}

	/* if device does not exist yet create one */
	if (ddev == NULL) {

		ddev = (struct ddekit_usb_dev *)
			kmalloc(sizeof(struct ddekit_usb_dev), GFP_KERNEL);
		if (ddev==NULL) {
			DEBUG_MSG("out of memory");
			return ENOMEM;
		}
		memset(ddev, 0, sizeof (struct ddekit_usb_dev));

		ddev->usb_dev   = usb_dev;
		ddev->ref_count = 0;

		/* add to device list */
		list_add_tail(&ddev->list, &gbl_dev_list);

		/* find the interfaces and set bitmap */
		setup_interface_bitmap(ddev, usb_dev);

		usb_get_dev(usb_dev);
	}

	/* remember this interface is associated with the ddev */
	usb_set_intfdata(intf, ddev);

	/* increment reference counter of intf and dev in linux*/
	usb_get_intf(intf);
	/* increment also our refcount, so we now when we can free the
	   ddekit dev */
	ddev->ref_count++;

	/* if it's a new device, announce it */
	if (ddev->ref_count == ddev->intf_count) {
		connect_cb(ddev, ddev->interfaces);
	}

	return 0;
}


static void
usb_stub_driver_disconnect
(struct usb_interface *intf)
{
	struct ddekit_usb_dev *ddev = NULL;

	/* get ddekit_usb_device */
	ddev = (struct ddekit_usb_dev *) usb_get_intfdata(intf);

	usb_put_intf(intf);
	
	ddev->ref_count--;

	/* clean up the device? */
	if (ddev->ref_count == 0) {
		list_del(&ddev->list);
		disconnect_cb(ddev);
		usb_put_dev(ddev->usb_dev);
		kfree(ddev);
	}
}


struct usb_driver usb_stub_driver = {
	.name           = "ddekit_usb stub driver",
	.probe          = usb_stub_driver_probe,
	.disconnect     = usb_stub_driver_disconnect,
	.id_table       = usb_stub_driver_id_table,
};

static void *my_malloc(size_t size) {
	return kmalloc(size, GFP_KERNEL);
}


static void my_free(void *ptr) {
	kfree(ptr);
}


int
ddekit_usb_init
(struct ddekit_usb_driver *drv, ddekit_usb_malloc_fn *_m,
ddekit_usb_free_fn   *_f)
{
	completion_cb = drv->completion;
	connect_cb    = drv->connect;
	disconnect_cb = drv->disconnect;

	*_m = my_malloc;
	*_f = my_free;

	usb_register(&usb_stub_driver);
	return 0;
}

/**********************************************************************/
/* device descriptor helpers                                          */
/**********************************************************************/
char *
_ddekit_usb_get_manufacturer
(struct ddekit_usb_dev *ddev)
{
	return ddev->usb_dev->manufacturer;
}

char *
_ddekit_usb_get_product
(struct ddekit_usb_dev *ddev)
{
	return ddev->usb_dev->product;
}

char *
_ddekit_usb_get_serial
(struct ddekit_usb_dev *ddev)
{
	return ddev->usb_dev->serial;
}



struct usb_device_descriptor *
_ddekit_usb_get_device_desc
(struct ddekit_usb_dev *ddev) {
	return &ddev->usb_dev->descriptor;
}

struct usb_interface_descriptor * 
_ddekit_usb_get_interface_desc
(struct ddekit_usb_dev *ddev, int inum)
{
	int i;
	struct usb_host_config *conf = ddev->usb_dev->actconfig;

	for (i=0; i < conf->desc.bNumInterfaces; i++ ) {
		struct usb_interface_descriptor *intf = 
			&conf->interface[i]->cur_altsetting->desc;
		if (intf->bInterfaceNumber == inum) {
			return intf;
		}
	}
	return NULL;
}

