/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: Collection_TargetDBGenerate.c,v 1.6 1995/02/09 18:13:31 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "Hint.h"
#include "File.h"
#include "FileSystemImage.h"
#include "DepotConf.h"
#include "TargetDB.h"
#include "Collection.h"

static void Collection_CheckSourceForExternalReferences();
static void Collection_CheckTargetForExternalReferences();

TARGETDB *
Collection_TargetDBGenerate(collectionp, hintdb_old)
     COLLECTION *collectionp;
     HINTDB hintdb_old;
{
  register DEPOTCONFDB dp;
  register char **sp;
  TARGETDB *targetdbp;
  char *source, *target, *commandlabel;
  int targetlistsize;


  if ((COLLECTION_DepotConfDB(collectionp) == NULL)
      || (*COLLECTION_DepotConfDB(collectionp) == NULL)) {
    FatalError(E_NODEPOTCONFSPEC,
	       "No installation specifications found for collection %s\n",
	       COLLECTION_Name(collectionp));
  }
  targetdbp = NULL;
  for (dp = *COLLECTION_DepotConfDB(collectionp);
       (PROGRAM_ErrorNo == E_NULL) && (*dp != NULL);
       dp++) {
    switch (DEPOTCONF_Spec(*dp)) {
    case DC_MAP:
      if ((StringArray_Size(DEPOTCONF_ConfValue(*dp)) != 2)
	  || (STRINGARRAY_String(DEPOTCONF_ConfValue(*dp), 0) == NULL)
	  || (STRINGARRAY_String(DEPOTCONF_ConfValue(*dp), 1) == NULL)) {
	FatalError
	  (E_BADDEPOTCONFDB,
	   "mapping parameters for collection %s absent in database.\n",
	   COLLECTION_Name(collectionp));
      }
      if (PROGRAM_ErrorNo == E_NULL) {
	source = STRINGARRAY_String(DEPOTCONF_ConfValue(*dp), 0);
	target = STRINGARRAY_String(DEPOTCONF_ConfValue(*dp), 1);
	Collection_CheckSourceForExternalReferences(source, collectionp);
	Collection_CheckTargetForExternalReferences(target, collectionp);
      }
      if (PROGRAM_ErrorNo == E_NULL) {
	if (COLLECTION_Image(collectionp) != NULL) {
	  targetdbp = Collection_FSImageMap(targetdbp,
					    source, target,
					    collectionp,
					    hintdb_old);
	} else {
	  targetdbp = Collection_PathMap(targetdbp,
					 source, target,
					 collectionp,
					 hintdb_old);
	}
      }
      break;
    case DC_DELETE:
      if ((StringArray_Size(DEPOTCONF_ConfValue(*dp)) != 1)
	  || (STRINGARRAY_String(DEPOTCONF_ConfValue(*dp), 0) == NULL)) {
	FatalError
	  (E_BADDEPOTCONFDB,
	   "deletion parameters for collection %s absent in database.\n",
	   COLLECTION_Name(collectionp));
      }
      if (PROGRAM_ErrorNo == E_NULL) {
	target = STRINGARRAY_String(DEPOTCONF_ConfValue(*dp), 0);
	Collection_CheckTargetForExternalReferences(target, collectionp);
      }
      if (PROGRAM_ErrorNo == E_NULL) {
	targetdbp = Collection_DeleteTarget(targetdbp,
					    target,
					    collectionp);
      }
      break;
    case DC_COMMAND:
      targetlistsize = StringArray_Size(DEPOTCONF_ConfValue(*dp)) - 1;
      if ((targetlistsize < 0)
	  || (STRINGARRAY_String(DEPOTCONF_ConfValue(*dp), 0) == NULL)) {
	FatalError
	  (E_BADDEPOTCONFDB,
	   "command parameters for collection %s absent in database.\n",
	   COLLECTION_Name(collectionp));
      }
      if ((PROGRAM_ErrorNo == E_NULL) && (targetlistsize > 0)) {
	commandlabel = STRINGARRAY_String(DEPOTCONF_ConfValue(*dp), 0);
	sp = STRINGARRAY_Values(DEPOTCONF_ConfValue(*dp)) + 1;
	while ((PROGRAM_ErrorNo == E_NULL) && (*sp != NULL)) {
	  target = *sp;
	  Collection_CheckTargetForExternalReferences(target,
						      collectionp);
	  if (PROGRAM_ErrorNo == E_NULL) {
	    targetdbp = Collection_CommandTarget(targetdbp,
						 commandlabel,
						 target,
						 collectionp);
	  }
	  sp++;
	}
      }
      break;
    case DC_UNKNOWN:
    default:
      {
	FatalError
	  (E_BADDEPOTCONFDB,
	   "Unknown configuration spec for collection %s in database.\n",
	   COLLECTION_Name(collectionp));
      }
      break;
    }
  }

  return (PROGRAM_ErrorNo == E_NULL) ? targetdbp : NULL;
}



static void 
Collection_CheckSourceForExternalReferences(source, collectionp)
     char *source;
     COLLECTION *collectionp;
{
  register char **sp;

  Boolean LocatedExternalReference;
  STRINGARRAY *sourcestringarray = NULL;
  char **sourcearray;
  char sourcehier[MAXPATHLEN + 1];
  char sourcehierpath[MAXPATHLEN + 1];
  FILESYSTEMIMAGE *fsimagenodep;
  FILESTAT filestatbuffer;

  LocatedExternalReference = FALSE;
  if (source[0] == '/') {
    if (source[1] != '\0') {
      /* flush the leading / if it exists in the path */
      source++;
    }
  } else {			/* disallow ..s and symlinks in path to
				 * source */
    sourcestringarray = StringToStringArray(source,
					    '/' /* delimited by /s */ ,
					    -1 /* no quotechar */ );
    if (PROGRAM_ErrorNo == E_NULL) {
      if (StringArray_Empty(sourcestringarray)) {
	FatalError
	  (E_BADTARGETSOURCEPATH, "Bad source %s for collection %s.\n",
	   source, COLLECTION_Name(collectionp));
      } else {
	sourcearray = STRINGARRAY_Values(sourcestringarray);
      }
    }
    if ((PROGRAM_ErrorNo == E_NULL) && (sourcearray != NULL)) {
      (void) strcpy(sourcehierpath, COLLECTION_Path(collectionp));
      (void) strcpy(sourcehier, "/");
      sp = sourcearray;
      while ((PROGRAM_ErrorNo == E_NULL)
	     && !LocatedExternalReference && (*sp != NULL)) {
	if (String_Comparator(*sp, "..") == 0) {
	  LocatedExternalReference = TRUE;
	} else if (*(sp + 1) != NULL) {		/* disallow symlinks except
						 * in leaf */
	  if (COLLECTION_Image(collectionp) != NULL) {
	    if (String_Comparator(sourcehier, "/") == 0) {
	      (void) strcpy(sourcehier, *sp);
	    } else {
	      (void) strcat(sourcehier, "/");
	      (void) strcat(sourcehier, *sp);
	    }
	    fsimagenodep =
	      FileSystemImage_LocateNode(COLLECTION_Image(collectionp),
					 sourcehier,
					 COLLECTION_Name(collectionp));
	    if ((PROGRAM_ErrorNo == E_NULL)
		&& (FILESYSTEMIMAGE_Type(fsimagenodep) & FS_SYMLNK)) {
	      LocatedExternalReference = TRUE;
	    }
	  } else {
	    (void) strcat(sourcehierpath, "/");
	    (void) strcat(sourcehierpath, *sp);
	    (void) File_GetStatus(sourcehierpath,
				  &filestatbuffer,
				  FALSE /* followlinks */ );
	    if ((PROGRAM_ErrorNo == E_NULL)
		&& (FILESTAT_Type(&filestatbuffer) & F_LNK)) {
	      LocatedExternalReference = TRUE;
	    }
	  }
	}
	sp++;
      }
    }
  }

  if ((PROGRAM_ErrorNo == E_NULL) && LocatedExternalReference) {
    FatalError
      (E_BADTARGETSOURCEPATH,
       "Potential external reference in source %s for collection %s.\n",
       source, COLLECTION_Name(collectionp));
  }

  if (sourcestringarray != NULL)
    StringArray_Free(sourcestringarray);

  return;
}

static void 
Collection_CheckTargetForExternalReferences(target, collectionp)
     char *target;
     COLLECTION *collectionp;
{
  register char **tp;

  Boolean LocatedExternalReference;
  STRINGARRAY *targetstringarray = NULL;
  char **targetarray;

  LocatedExternalReference = FALSE;
  if (target[0] == '/') {
    if (target[1] != '\0')
      /* "/" is OK, everything else is invalid */
    {
      target++;
    }
  } else {			/* disallow ..s in target */
    targetstringarray = StringToStringArray(target,
					    '/' /* delimited by /s */ ,
					    -1 /* no quotechar */ );
    if (PROGRAM_ErrorNo == E_NULL) {
      if (StringArray_Empty(targetstringarray)) {
	FatalError
	  (E_BADTARGETPATH, "Invalid target %s for collection %s.\n",
	   target, COLLECTION_Name(collectionp));
      } else {
	targetarray = STRINGARRAY_Values(targetstringarray);
      }
    }
    if ((PROGRAM_ErrorNo == E_NULL) && (targetarray != NULL)) {
      tp = targetarray;
      while ((PROGRAM_ErrorNo == E_NULL)
	     && !LocatedExternalReference && (*tp != NULL)) {
	if (String_Comparator(*tp, "..") == 0) {
	  LocatedExternalReference = TRUE;
	}
	tp++;
      }
    }
  }

  if ((PROGRAM_ErrorNo == E_NULL) && LocatedExternalReference) {
    FatalError
      (E_BADTARGETPATH,
       "Potential external reference in target %s for collection %s.\n",
       target, COLLECTION_Name(collectionp));
  }

  if (targetstringarray != NULL)
    StringArray_Free(targetstringarray);

  return;
}

/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/018/src/lib/Collection/RCS/Collection_TargetDBGenerate.c,v $ */
