/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: Depot.c,v 1.15 1994/11/01 15:56:56 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */


#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "Lock.h"
#include "File.h"
#include "Hint.h"
#include "FileSystemImage.h"
#include "Preference.h"
#include "DepotConf.h"
#include "TargetDB.h"
#include "Collection.h"
#include "CollectionList.h"
#include "DepotDB.h"
#include "HintService.h"
#include "Command.h"

#include "config.h"
#include "Depot.h"

static void ProcessLockAction();
static void ProcessUpdates();


int 
Depot(TargetPath, ImagePath, UpdateMode, LockAction)
     char *TargetPath;
     char *ImagePath;
     unsigned UpdateMode;
     int LockAction;
{
  DIR *dp;
  char DepotLockFileName[MAXPATHLEN + 1];
  (void) sprintf(DepotLockFileName,
		 "%s/%s", DEPOTSPECIALDIRECTORY, DEPOTDBLOCKFILE);

  if (chdir(TargetPath) != 0) {
    FatalError(E_CHDIRFAILED,
	       "Could not chdir to target directory %s: %s\n",
	       TargetPath, strerror(errno));
    exit(E_CHDIRFAILED);
  }
  if ((dp = opendir(DEPOTSPECIALDIRECTORY)) == NULL) {
    FatalError(E_BADPATHNAME, "Invalid target directory: %s\n\tUnable to open %s subdirectory: %s.\n",
	       TargetPath, DEPOTSPECIALDIRECTORY, strerror(errno));
    exit(E_BADPATHNAME);
  }
  closedir(dp);

  if (PROGRAM_ErrorNo == E_NULL) {
    if (UpdateMode & M_LOCKONLY)
      ProcessLockAction(DepotLockFileName, LockAction);
    else
      ProcessUpdates(DepotLockFileName, ImagePath, UpdateMode);
  }
  return PROGRAM_ErrorNo;
}

static void 
ProcessLockAction(DepotLockFileName, LockAction)
  char *DepotLockFileName;
  int LockAction;
{
  char *rval;
 
  switch (LockAction) {
  case L_FORCELOCK:
    rval = Lock_SetLock(DepotLockFileName);
    if (PROGRAM_ErrorNo != E_NULL) {
      FatalError(E_FOOLPROOFLOCK,
		 "Setting lock failed - still locked by %s.\n", rval);
      return;
    }
    if (rval == NULL) {
      Message("Environment locked.\n");
      return;  /* lock successfully set */
    }
    Lock_RemoveLock(DepotLockFileName);
    if (PROGRAM_ErrorNo != E_NULL) {
      FatalError(E_FOOLPROOFLOCK,
		 "Unable to remove old lock file  - still locked by %s.\n", rval);
      return;
    }
    (void)Lock_SetLock(DepotLockFileName);
    if (PROGRAM_ErrorNo == E_NULL) {
      Message("Environment locked.\n");
    } else {
      FatalError(E_FOOLPROOFLOCK,
		 "Unable to reset lock file  - still locked by %s.\n", rval);
    }      
    break;

  case L_LOCK:
    rval = Lock_SetLock(DepotLockFileName);
    if (PROGRAM_ErrorNo != E_NULL)
      return;
    if (rval == NULL) {
      Message("Environment locked.\n");
    } else {
      FatalError(E_CANTLOCK, "Lock failed - locked by %s\n", rval);
    }
    break;

  case L_FORCEUNLOCK:
    Lock_RemoveLock(DepotLockFileName);
    if (PROGRAM_ErrorNo == E_NULL)
      Message("Environment unlocked.\n");
    break;

  case L_UNLOCK:
    rval = Lock_UnsetLock(DepotLockFileName);
    if (PROGRAM_ErrorNo != E_NULL)
      return;
    if (rval == NULL) {
      Message("Environment Unlocked.\n");
    } else {
      FatalError(E_CANTLOCK, "Unlock failed - locked by %s\n", rval);
    }
    break;
  }

  return;
}


static void 
ProcessUpdates(DepotLockFileName, ImagePath, UpdateMode)
  char *DepotLockFileName;
  char *ImagePath;
  unsigned UpdateMode;
{
  char *DepotLockOwner;		/* whoever has the depot databse locked now */
  Boolean PointofNoReturn;	/* TRUE if updates made to target directory */

  PointofNoReturn = FALSE;
  if (!(UpdateMode & M_SHOWACTIONSONLY)) {	/* lock database if necessary */
    DepotLockOwner = Lock_QueryLock(DepotLockFileName); 

    if (PROGRAM_ErrorNo != E_NULL)
      return;

    if (DepotLockOwner != NULL) {
      if (UpdateMode & M_IMPLICITLOCKINGALLOWED) {
	Message("DEPOT WARNING: environment is currently locked by %s\n", DepotLockOwner);
      } else {
	FatalError(E_CANTLOCK,
		   "Environment currently locked by %s\n", DepotLockOwner);
	return;
      }
    }
    (void)Lock_SetLock(DepotLockFileName); /* try to lock it */

  }
  if (PROGRAM_ErrorNo == E_NULL) {
    if (UpdateMode & M_DELTA) {
      Depot_RunDeltaMode(ImagePath, UpdateMode, &PointofNoReturn);
    } else {
      Depot_RunBuildMode(ImagePath, UpdateMode, &PointofNoReturn);
    }
  }
  if (!(UpdateMode & M_SHOWACTIONSONLY)) {
    /* unlock database ,  if no error or no updates  to target  directory */
    if (PROGRAM_ErrorNo == E_NULL) {
      Lock_UnsetLock(DepotLockFileName);
    }
  }
  return;
}

/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/022/src/lib/Depot/RCS/Depot.c,v $ */
