/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: TargetDB_SetTargetFileStatus.c,v 1.10 1994/09/28 18:04:03 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "Preference.h"
#include "File.h"
#include "TargetDB.h"



static char *TARGETLISTVALUE_NULL[] =
{
  NULL
};

static STRINGARRAY TARGETLIST_NULL =
{
  0, TARGETLISTVALUE_NULL
};


static void TargetDB_NodeSetTargetFileStatus();
static void TargetDB_TreeSetTargetFileStatus();
static void TargetDB_SetTargetFileStatusPath();

static void 
TargetDB_NodeSetTargetFileStatus(nodep, statusp, flags)
     TARGETDB *nodep;
     FILESTAT *statusp;
     unsigned int flags;
{
  if (flags == U_NULL) {
    TARGETDB_UpdateSpec(nodep) &= ~(U_OWNER | U_GROUP | U_MODE | U_SETUID | U_SETGID);
    if (TARGETDB_FileStatus(nodep) != NULL) {
      (void) free((void *) TARGETDB_FileStatus(nodep));
      TARGETDB_FileStatus(nodep) = NULL;
    }
    return;
  } 
  
  if (statusp == NULL) {
    FatalError(E_BADTARGETFILESTATUS,
	       "Attempt to set file status of target database node with NULL status.\n");
    return;
  }
  if (TARGETDB_FileStatus(nodep) == NULL) {
    TARGETDB_FileStatus(nodep) = (FILESTAT *) emalloc(sizeof(FILESTAT));
    if (PROGRAM_ErrorNo == E_NULL) {
      /* initialize the structure and reset potentially bad values */
      (void)memset(TARGETDB_FileStatus(nodep), 0, sizeof(FILESTAT));

      FILESTAT_Uid(TARGETDB_FileStatus(nodep)) = -1;
      FILESTAT_Gid(TARGETDB_FileStatus(nodep)) = -1;
      FILESTAT_Mode(TARGETDB_FileStatus(nodep)) = 0500;
      /* 
       * $$$WARNING$$$
       * Don't use 07777 !
       * $$$KLUDGE$$$
       * Be more elegant than using something like 0777,
       * define the appropriate thing in FileOps.h
       * Also decide whether we should allow masks to inhibit
       * suids & guids, since this idea would be
       * potentially non-portable across OSs.
       *        -- Sohan
       * 
       * Modified to be 0500 as to avoid the possibility of 
       * introducing a security hole. -- Walter
       */
    }
  }
  if (PROGRAM_ErrorNo != E_NULL) 
      return;

  /* NOTE: None of these calls can generate an error so
   * let's not check for PROGRAM_ErrorNo
   */
  if (flags & U_OWNER) {
    TARGETDB_UpdateSpec(nodep) |= (flags & U_OWNER);
    FILESTAT_Uid(TARGETDB_FileStatus(nodep)) = FILESTAT_Uid(statusp);
  }

  /* setuid must come afer U_OWNER so that setuid can override it */
  if (flags & U_SETUID) {
    TARGETDB_UpdateSpec(nodep) |= (flags & U_SETUID);
    FILESTAT_Uid(TARGETDB_FileStatus(nodep)) = FILESTAT_Uid(statusp);
  }

  if (flags & U_GROUP) {
    TARGETDB_UpdateSpec(nodep) |= (flags & U_GROUP);
    FILESTAT_Gid(TARGETDB_FileStatus(nodep)) = FILESTAT_Gid(statusp);
  }

  if (flags & U_SETGID) {
    TARGETDB_UpdateSpec(nodep) |= (flags & U_SETGID);
    FILESTAT_Gid(TARGETDB_FileStatus(nodep)) = FILESTAT_Gid(statusp);
  }

  if (flags & U_MODE) {
    TARGETDB_UpdateSpec(nodep) |= U_MODE;
    FILESTAT_Mode(TARGETDB_FileStatus(nodep)) =
      FILESTAT_Mode(statusp);
  }
    
  return;
}



static void 
TargetDB_TreeSetTargetFileStatus(targetdbp, statusp, flags)
     TARGETDB *targetdbp;
     FILESTAT *statusp;
     unsigned flags;
{
  register int i;
  register TARGETDB **childp;

  TargetDB_NodeSetTargetFileStatus(targetdbp, statusp, flags);

  for (i = 0, childp = TARGETDB_Children(targetdbp);
       (PROGRAM_ErrorNo == E_NULL) && (i < TARGETDB_ChildCount(targetdbp));
       i++, childp++) {
    TargetDB_TreeSetTargetFileStatus(*childp, statusp, flags);
  }

  return;
}



static void 
TargetDB_SetTargetFileStatusPath(targetdbp, targetpath, statusp, flags)
     TARGETDB *targetdbp;
     char *targetpath;
     FILESTAT *statusp;
     unsigned flags;
{
  register char **sp;

  TARGETDB *childnodep;
  char **targetpatharray;
  STRINGARRAY *targetpathstringarray;

  /* split target path into array of its components */
  targetpathstringarray = StringToStringArray(targetpath,
					      '/' /* delimited by /s */ ,
					      -1 /* no quotechar */ );
  if (PROGRAM_ErrorNo == E_NULL) {
    if (targetpathstringarray == NULL) {
      targetpatharray = NULL;
    } else {
      targetpatharray = STRINGARRAY_Values(targetpathstringarray);
    }
  }
  /* go down the targetdatabase setting the approriate update spec. */
  if ((PROGRAM_ErrorNo == E_NULL) && (targetpatharray != NULL)) {
    if (flags & (U_OWNER | U_GROUP | U_MODE | U_SETUID | U_SETGID)) {
      TARGETDB_UpdateSpec(targetdbp) |= U_STATUSPATH;
    }
    for (sp = targetpatharray, childnodep = targetdbp;
       (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL) && (childnodep != NULL);
	 sp++) {
      childnodep = TargetDB_LocateChildNode(childnodep, *sp, TDB_LAX);
      if ((PROGRAM_ErrorNo == E_NULL) && (childnodep != NULL)) {
	if (flags & (U_OWNER | U_GROUP | U_MODE | U_SETUID | U_SETGID)) {
	  TARGETDB_UpdateSpec(childnodep) |= U_STATUSPATH;
	}
      }
    }
  }
}



/*
 *  -- the iterations on (owner, group, setuid, setgid, masks)
 *      are coded sequentially
 *  get a list of owners, groups, setuids, setgids and masks to set
 *  for each of owner, group, setuid, setgid and mask lists
 *    get a list of targets for set for each of those ids or masks
 *    for each of those targets in the list
 *      set status etc. as appropriate
 */
TARGETDB *
TargetDB_SetTargetFileStatus(targetdbp, preferencedbp)
     TARGETDB *targetdbp;
     PREFERENCEDB *preferencedbp;
{
  register char **cpp;
  register char **tp;
  register int i;

  TARGETDB *newtargetdbp;

  STRINGSET *OwnerList, *GroupList, *SetUidList, *SetGidList, *MaskList;
  STRINGSET *TargetList;
  unsigned short *MaskModeList;
  STRINGSET **MaskTargetListList;
  int *MaskTargetIndexList;
  int current;
  Boolean EndOfLists;
  int TargetComparison;
  FILESTAT targetfilestatus;
  TARGETDBENTRY *targetdbnodep;

  newtargetdbp = targetdbp;

  /* first reset all Target FileStatuses specified in the tree to the default 
   */
  if (newtargetdbp != NULL) {
    TargetDB_TreeSetTargetFileStatus(newtargetdbp, NULL, 0);
  }

  /* XXXXX We should check for conflicts here, but we aren't going to
   *           basically, we need to ensure that you don't have the situation
   *           where you have two target.* setting things to the exact same path.
   */
  OwnerList = Preference_GetOwnerNamesToSet(preferencedbp);
  if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(OwnerList)) {
    for (cpp = STRINGSET_Values(OwnerList);
	 (PROGRAM_ErrorNo == E_NULL) && (*cpp != NULL);
	 cpp++) {
      FILESTAT_Uid(&targetfilestatus) = UserIDFromName(*cpp);
      if (PROGRAM_ErrorNo == E_NULL) {
	TargetList = Preference_GetStringSet(preferencedbp,
					     *cpp,
					     "target.owner",
					     &TARGETLIST_NULL,
					     0 /* don't allocate space */ );
      }
      if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(TargetList)) {
	for (tp = STRINGSET_Values(TargetList);
	     (PROGRAM_ErrorNo == E_NULL) && (*tp != NULL);
	     tp++) {
	  targetdbnodep = TargetDB_LocateNode(newtargetdbp,
					      *tp,
					      TDB_LAX);
	  if (targetdbnodep != NULL) {
	    TargetDB_TreeSetTargetFileStatus(targetdbnodep,
					     &targetfilestatus,
					     U_OWNER);
	    TargetDB_SetTargetFileStatusPath(newtargetdbp,
					     *tp,
					     &targetfilestatus,
					     U_OWNER);
	  }
	}
      }
    }
    StringArray_Free(OwnerList);
  }
  GroupList = Preference_GetGroupNamesToSet(preferencedbp);
  if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(GroupList)) {
    for (cpp = STRINGSET_Values(GroupList);
	 (PROGRAM_ErrorNo == E_NULL) && (*cpp != NULL);
	 cpp++) {
      FILESTAT_Gid(&targetfilestatus) = GroupIDFromName(*cpp);
      if (PROGRAM_ErrorNo == E_NULL) {
	TargetList = Preference_GetStringSet(preferencedbp,
					     *cpp,
					     "target.group",
					     &TARGETLIST_NULL,
					     0 /* don't allocate space */ );
      }
      if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(TargetList)) {
	for (tp = STRINGSET_Values(TargetList);
	     (PROGRAM_ErrorNo == E_NULL) && (*tp != NULL);
	     tp++) {
	  targetdbnodep = TargetDB_LocateNode(newtargetdbp,
					      *tp,
					      TDB_LAX);
	  if (targetdbnodep != NULL) {
	    TargetDB_TreeSetTargetFileStatus(targetdbnodep,
					     &targetfilestatus,
					     U_GROUP);
	    TargetDB_SetTargetFileStatusPath(newtargetdbp,
					     *tp,
					     &targetfilestatus,
					     U_GROUP);
	  }
	}
      }
    }
    StringArray_Free(GroupList);
  }
  SetUidList = Preference_GetSetUIDNamesToSet(preferencedbp);
  if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(SetUidList)) {
    for (cpp = STRINGSET_Values(SetUidList);
	 (PROGRAM_ErrorNo == E_NULL) && (*cpp != NULL);
	 cpp++) {
      FILESTAT_Uid(&targetfilestatus) = UserIDFromName(*cpp);
      if (PROGRAM_ErrorNo == E_NULL) {
	TargetList = Preference_GetStringSet(preferencedbp,
					     *cpp,
					     "target.setuid",
					     &TARGETLIST_NULL,
					     0 /* don't allocate space */ );
      }
      if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(TargetList)) {
	for (tp = STRINGSET_Values(TargetList);
	     (PROGRAM_ErrorNo == E_NULL) && (*tp != NULL);
	     tp++) {
	  targetdbnodep = TargetDB_LocateNode(newtargetdbp,
					      *tp,
					      TDB_LAX);
	  if (targetdbnodep != NULL) {
	    TargetDB_TreeSetTargetFileStatus(targetdbnodep,
					     &targetfilestatus,
					     U_SETUID);
	    TargetDB_SetTargetFileStatusPath(newtargetdbp,
					     *tp,
					     &targetfilestatus,
					     U_SETUID);
	  }
	}
      }
    }
    StringArray_Free(SetUidList);
  }
  SetGidList = Preference_GetSetGIDNamesToSet(preferencedbp);
  if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(SetGidList)) {
    for (cpp = STRINGSET_Values(SetGidList);
	 (PROGRAM_ErrorNo == E_NULL) && (*cpp != NULL);
	 cpp++) {
      FILESTAT_Gid(&targetfilestatus) = GroupIDFromName(*cpp);
      if (PROGRAM_ErrorNo == E_NULL) {
	TargetList = Preference_GetStringSet(preferencedbp,
					     *cpp,
					     "target.setgid",
					     &TARGETLIST_NULL,
					     0 /* don't allocate space */ );
      }
      if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(TargetList)) {
	for (tp = STRINGSET_Values(TargetList);
	     (PROGRAM_ErrorNo == E_NULL) && (*tp != NULL);
	     tp++) {
	  targetdbnodep = TargetDB_LocateNode(newtargetdbp,
					      *tp,
					      TDB_LAX);
	  if (targetdbnodep != NULL) {
	    TargetDB_TreeSetTargetFileStatus(targetdbnodep,
					     &targetfilestatus,
					     U_SETGID);
	    TargetDB_SetTargetFileStatusPath(newtargetdbp,
					     *tp,
					     &targetfilestatus,
					     U_SETGID);
	  }
	}
      }
    }
    StringArray_Free(SetGidList);
  }
  /* 
   * get a list of masks to set
   * get the list of targets for each of the masks
   * iteratively set the appropriate mask to the shortest target in the lists
   */
  MaskList = Preference_GetModeMasksToSet(preferencedbp);
  if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(MaskList)) {
    MaskModeList =
      (unsigned short *) emalloc(STRINGSET_Size(MaskList) * sizeof(unsigned short));
    MaskTargetListList =
      (STRINGSET **) emalloc(STRINGSET_Size(MaskList) * sizeof(STRINGSET *));
    MaskTargetIndexList =
      (int *) emalloc(STRINGSET_Size(MaskList) * sizeof(int));
    for (i = 0;
	 (PROGRAM_ErrorNo == E_NULL) && (i < STRINGSET_Size(MaskList));
	 i++) {
      *(MaskModeList + i) = OctalNumber(STRINGSET_String(MaskList, i));
      *(MaskTargetListList + i) =
	Preference_GetStringSet(preferencedbp,
				STRINGSET_String(MaskList, i),
				"target.modemask",
				&TARGETLIST_NULL, 1	/* allocate space for 
							 * this one */ );
      *(MaskTargetIndexList + i) = 0;
    }

    EndOfLists = FALSE;
    while ((PROGRAM_ErrorNo == E_NULL) && !EndOfLists) {
      EndOfLists = TRUE;	/* until we find an unfinished target list */
      current = -1;
      if (!StringSet_Empty(*MaskTargetListList)
	  && (STRINGSET_String(*MaskTargetListList, *MaskTargetIndexList)
	      != NULL)) {
	EndOfLists = FALSE;
      }
      for (i = 0;
	   (PROGRAM_ErrorNo == E_NULL) && (i < STRINGSET_Size(MaskList));
	   i++) {
	if (!StringSet_Empty(*(MaskTargetListList + i))
	    && (STRINGSET_String(*(MaskTargetListList + i),
				 *(MaskTargetIndexList + i))
		!= NULL)) {
	  EndOfLists = FALSE;
	  if (current < 0) {
	    current = i;
	  } else {
	    TargetComparison =
	      String_Comparator
	      (STRINGSET_String(*(MaskTargetListList + current),
				*(MaskTargetIndexList + current)),
	       STRINGSET_String(*(MaskTargetListList + i),
				*(MaskTargetIndexList + i)));
	    if (TargetComparison == 0) {
	      if (*(MaskModeList + current) != *(MaskModeList + i)) {
		FatalError(E_MODEMASKCONFLICT,
			   "Attempt to apply conflicting mode masks %o and %o to target %s.\n",
			   *(MaskModeList + current),
			   *(MaskModeList + i),
			   STRINGSET_String
			   (*(MaskTargetListList + i),
			    *(MaskTargetIndexList + i)));
	      } else {
		(*(MaskTargetIndexList + i))++;
	      }
	    } else if (TargetComparison > 0) {
	      current = i;
	    }
	  }
	}
      }

      if ((PROGRAM_ErrorNo == E_NULL) && !EndOfLists) {
	FILESTAT_Mode(&targetfilestatus) = *(MaskModeList + current);
	targetdbnodep =
	  TargetDB_LocateNode(newtargetdbp,
			   STRINGSET_String(*(MaskTargetListList + current),
					  *(MaskTargetIndexList + current)),
			      TDB_LAX);
	if (targetdbnodep != NULL) {
	  TargetDB_TreeSetTargetFileStatus(targetdbnodep,
					   &targetfilestatus,
					   U_MODE);
	  TargetDB_SetTargetFileStatusPath
	    (newtargetdbp,
	     STRINGSET_String(*(MaskTargetListList + current),
			      *(MaskTargetIndexList + current)),
	     &targetfilestatus,
	     U_MODE);
	}
	(*(MaskTargetIndexList + current))++;
      }
    }

    (void) free((void *) MaskModeList);
    for (i = 0;
	 (PROGRAM_ErrorNo == E_NULL) && (i < STRINGSET_Size(MaskList));
	 i++) {
      if (*(MaskTargetListList + i) != NULL) {
	StringArray_Free(*(MaskTargetListList + i));
      }
    }
    (void) free((void *) MaskTargetListList);
    (void) free((void *) MaskTargetIndexList);
    StringArray_Free(MaskList);
  }
  return (PROGRAM_ErrorNo == E_NULL) ? newtargetdbp : NULL;
}


/* $Source: /afs/andrew/system/src/local/depot2/013/src/lib/TargetDB/RCS/TargetDB_SetTargetFileStatus.c,v $ */
