#!/usr/local/bin/perl
# $Id: tree_diff.pl,v 1.2 1994/08/01 21:54:16 ww0r Exp $
#
# (c) 1994 Carnegie Mellon University
#
# tree_diff dir1 dir2
#       prints difference between directory dir1 and directory dir2
# arguments
#      -b  keep output buffered
#      -t  show differences in mtime/ctime
#      -q  don't warn about devices
#      -m  don't cross AFS mountpoints
#      -u  ignore uid
#      -g  ignore group
#

# XXX features to add
#  show diff in dev's
#  make -m cross all mount points (including NFS ones)
#  format for printing the initial stuff so things line up properly


require "getopts.pl";

$debug = 0;

$quiet = 0;
$difftime = 0;
$buffer = 0;
$ignore_uid=0;
$ignore_gid=0;

$crossmt = 1;


&Getopts('gubtqmd');
$debug = 1
  if (defined $opt_d);

$quiet = 1
  if (defined $opt_q);
$difftime = 1
  if (defined $opt_t);
$buffer = 1
  if (defined $opt_b);
$ignore_uid = 1
  if (defined $opt_u);
$ignore_gid = 1
  if (defined $opt_g);

$crossmt = 0
  if (defined $opt_m);

&usage() 
  unless (@ARGV == 2);

$dir1 = $ARGV[0];
$dir2 = $ARGV[1];

&usage() 
  unless (defined($dir1) && defined($dir2));

die "Not a directory: $dir1\n"
  unless (-d "$dir1");

die "Not a directory: $dir2\n"
  unless (-d "$dir2");


if (! $buffer) {
  select(STDOUT); $| = 1;
}

&do_diff($dir1, $dir2);

exit(0);
# -- end of main program


#
sub
usage {
  print STDERR "tree_diff: Invalid arguments.\n";
  print STDERR "usage: tree_diff dir1 dir2\n";
  exit(-1);
}


#
sub
print_only {
  local($dir, @items) = @_;

  foreach (@items) {
    print "Only in $dir: $_\n";
  }
}


#
sub
stat_entry {
  local($entry) = @_;
  local(@stb);
  local(@retval);

  print "Checking $entry\n"
    if ($debug);

  @retval = ();
  (@stb = lstat($entry))
    || die "Error lstat'ing: $entry: $!\n";

  if (-f _) {			# entry is a file
    local($mode) = $stb[2] & 0007777;
    local($uid)  = $stb[4];
    local($gid)  = $stb[5];
    local($sz)   = $stb[7];
    
    @retval = ('F', "$mode", "$uid", "$gid", "$sz");
  } elsif (-l _) {		# entry is a symbolic link
    (local($to) = readlink($entry))
      || die "Error reading link for $entry: $!\n";
    
    @retval = ('L', $to);

  } elsif (-d _) {		# entry is a directory
    local($mode) = $stb[2] & 0007777;
    local($uid)  = $stb[4];
    local($gid)  = $stb[5];
    local($afsmt) = !($stb[1] & 1);
    
    @retval = ('D', $mode, $uid, $gid, $afsmt);

  } elsif ((-c _) || (-b _)) {	# entry is a device
    @retval = ('V');

  } elsif ((-p _) || (-S _)) {	# entry is a named pipe or socket
    @retval = ('I');
  }
  print "Returing: @retval\n"
    if ($debug);

  @retval;
}


#
sub 
fmt_mode {
  local($entry) = @_;

  sprintf ("%o", $entry);
}

#
sub
fmt_user {
  local($uid) = @_;
  local($user);

  if (defined $uid_cache{$uid}) {
    return $uid_cache{$uid};
  } 
  ($user) = getpwuid($uid);
    if ($user) {
      $uid_cache{$uid} = $user;
    } else {
      $uid_cache{$uid} = $uid;
      $user = $uid;
    }
 
  $user;
}

#
sub
fmt_group {
  local($gid) = @_;
  local($group);

  if (defined $gid_cache{$gid}) {
    return $gid_cache{$gid};
  }

  ($group) = getgrgid($gid);
  if ($group) {
    $gid_cache{$gid} = $group;
  } else {
    $gid_cache{$gid} = $gid;
    $group = $gid;
  }

  $group;
}


#
sub
print_entry {
  local(@entry) = @_;
  local($mode, $user, $group, $size);

# XXX look into use formats for this?

  if ($entry[0] eq "D") {
    $mode = &fmt_mode($entry[1]);
    $user = &fmt_user($entry[2]);
    $group = &fmt_group($entry[3]);

    print "D $mode $user $group\n";
  } elsif ($entry[0] eq "F") {
    $mode = &fmt_mode($entry[1]);
    $user = &fmt_user($entry[2]);
    $group = &fmt_group($entry[3]);
    $size  = $entry[4];

    print "F $mode $user $group $size\n";
  } elsif ($entry[0] eq "L") {
    print "L $entry[1]\n";
  } else {
    print "@entry";
  }
}
    
  
# 
sub
cmp_diff {
  local($d1, $d2, $file,
	*e1, *e2) = @_;		# have to pass the address to these 
                                # to get two lists passed in
  local($different) = 0;
  local($i);

  $different = 0;
  if (@e1 == @e2) {
    $i = 0;
    foreach (@e1) {
      next 
	if ($ignore_uid && ($i == 2));
      next
	if ($ignore_gid && ($i == 3));
      $different = 1
	if ($_ ne @e2[$i++]);
    }
  } else {
    $different = 1;
  }

  if ($different) {
    print "$file:\t< $d1\n\t> $d2\n";
    print "< ";
    &print_entry(@e1);
    print "> ";
    &print_entry(@e2);
  }
  $different;
}

#
sub
do_diff {
  local($d1, $d2) = @_;
  local(*DP);
  local(@items, @items);
  local(@only1, @only2, @shared);
  local($entry);
  local(@travdir);
  
  print "Processing $d1 vs $d2\n"
    if ($debug);

  opendir(DP, "$d1") 
    || die "Unable to open directory: $d1: $!\n";
  @items1 =  grep(!/^\.\.?$/,readdir(DP));
  closedir(DP);

  opendir(DP, "$d2") 
    || die "Unable to open directory: $d2: $!\n";
  @items2 =  grep(!/^\.\.?$/,readdir(DP));
  closedir(DP);

  # compute differences based on names
  %mark = ();
  grep($mark{$_}++, @items1);
  @only2 = sort grep(!$mark{$_}, @items2);
  @shared = sort grep($mark{$_}, @items2);

  %mark = ();
  grep($mark{$_}++, @shared);
  @only1 = sort grep(!$mark{$_}, @items1);
  
  if ($debug) { 
    print "only1 = @only1\n";
    print "shared = @shared\n";
    print "only2 = @only2\n";
  }
  &print_only($d1, @only1);
  &print_only($d2, @only2);

  @travdir = ();
  foreach $entry (@shared) {
    local($file);
    local(@s1, @s2);
    local($diff_found) = 0;

    @s1 = &stat_entry("$d1/$entry");
    @s2 = &stat_entry("$d2/$entry");
    
    $diff_found = &cmp_diff($d1, $d2, $entry, *s1, *s2);
# ****** XXXX    clean up logic here ... 
    if (!$diff_found && ($s1[0] eq "D")) {
      if ($crossmt) {
	push(@travdir, $entry);
      } elsif ($s1[4] != $s2[4]) { 
        # both are not mount points so we have to traverse them
	print STDERR "WARNING: Crossing AFS mount point in ";
	print STDERR "$d1 as $d2 "
	  if $s1[4];
	print STDERR "$d2 as $d1 "
	  if $s2[4];
	print STDERR "is not a mount point.\n";
	push(@travdir, $entry);
      } elsif ($s1[4] == 0) {	# neither are mount points
	push(@travdir, $entry);
      }
    }
  }				# end loop
# ****************************************************************
	
  foreach (sort @travdir) {
    &do_diff("$d1/$_", "$d2/$_");
  }
  return;
}

sub
print_uid {
}

sub
print_mode{
}

sub
print_group{
}
# $Source: /afs/andrew.cmu.edu/system/src/local/depot2/021/tools/conv_tools/RCS/tree_diff.pl,v $


