/*
  lserial.c
  routines for handling the PC/SHARC serial interface

  (C) 1998 PinPoint Corporation
  This software is available for unlimited personal use and may only
  be redistributed in unmodified form.  Above all, this notice may
  not be modified or removed.
  Contact tim.wall@pinpointco.com for commercial use

  */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include <termios.h>
#include <fcntl.h>
#include <time.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#ifdef __NetBSD__
#include <errno.h>
#endif

#include "serial.h"

/* use a string of all character values to ensure the serial port
   works properly for all of them (i.e. is set to raw mode) */
static char lockname[] = "/tmp/DSPLOCK";
static char verify_string[256+8];
static int SerialFD = -1;
static int debug = 0;
static float dsp_timeout = DEFAULT_TIMEOUT;
static int in_send = 0;
static int send_error = 0;

static int closing = 0;
static int caught_signal = 0;
static int signals[] = { SIGHUP, SIGINT, SIGQUIT, SIGABRT, SIGTERM };
#define NSIGNALS (sizeof(signals)/sizeof(signals[0]))
static void (*(old_handler[NSIGNALS]))(int sig) = {0};
static void (*(saved_handler[NSIGNALS]))(int sig) = {0};

static void LinuxSetSpeed(speed_t speed);
static void LinuxSerialInit(int port, int verbose);

typedef struct 
{
    unsigned short dst_id;           /* always zero for EZ-LITE */
    unsigned short src_id;
    unsigned short length;
    unsigned short pkt_id;
    union
    {
        unsigned char _8[1024];
        unsigned short _16[512];
        unsigned long _32[254];
    } data;
} Packet; /* maximum size is 256 long words */

enum Messages {
    DSP_WRITEDM32 = 0x0005,
    DSP_WRITEPM48 = 0x000E,
    DSP_READDM32 = 0x0105,
    DSP_READPM48 = 0x010E,
    DSP_READCORE = 0x010F,
    DSP_START = 0x0201,
    DSP_RESETCPU = 0x0203,
    DSP_RESETBRD = 0x0204,
    DSP_SETSPEED = 0x0206,
    DSP_VERIFY = 0x0207,
    DSP_RESYNC = 0x0208
};

static unsigned gBaudCode = Baud9600;
static unsigned long baudRate[] =
{ -1,300,1200,2400,4800,9600,19200,38400,57600,115200 };

static void
alive ()
{
    static int spin = 0;
    static char spinner[] = { "|/-\\" };
    fprintf (stderr, "%c\b", spinner[spin]);
    if (++spin == 4) spin = 0;
}

static void
signal_handler (int sig)
{
    if (debug > 2) fprintf (stderr, "Signal %d caught in DSPSend\n", sig);
    caught_signal = sig;
}

/*
 * Main communications routine for transferring data to and from the DSP
 */
int
DSPSend (unsigned short msg, char *data, int nbytes,
         char *response, unsigned short *rnbytes)
{
    static int retries = 0;
    const int MAX_RETRIES = 3;
    char txbuf[2*1024];
    Packet *txpkt = (Packet *)txbuf;
    Packet rxpkt;
    int rxcnt;
    int readlen, writelen;
    time_t start, now;
    int i;

    in_send = 1;
    
    /* don't allow signals to interrupt a DSP transfer */
    /* signal_handler saves the signal for later processing */
    for (i=0;i < NSIGNALS;i++)
        saved_handler[i] = signal (signals[i], signal_handler);

    if (retries >= MAX_RETRIES)
    {
        fprintf (stderr,
                 "\nThe SHARC kernel is not responding\n"
                 "Please reset the target\n");
        send_error = 1;
        exit (1);
    }
    txpkt->dst_id = txpkt->src_id = 0;
    txpkt->length =  nbytes/4 + 2;
    txpkt->pkt_id = msg;
    
    if (nbytes)
        memcpy (txpkt->data._8, data, nbytes);
    
    if (debug > 0)
        fprintf (stderr, "DSPSend ");

    writelen = write (SerialFD, txbuf, nbytes+8);

    if (debug > 0)
        fprintf (stderr, "-> 0x%x ", (int)msg);

    if (writelen  < 0)
    {
        perror ("write");
        exit (1);
    }
    else if (writelen != nbytes + 8)
    {
        perror ("write size");
        exit (1);
    }
    
    /* if the command was to switch speeds, the response will come back at
       the new speed so the serial port must be set to new speed*/
    if (msg == DSP_SETSPEED)
    {
        speed_t speed = ((gBaudCode == Baud300) ? B300 :
                         (gBaudCode == Baud1200) ? B1200 :
                         (gBaudCode == Baud4800) ? B4800 :
                         (gBaudCode == Baud19200) ? B19200 :
                         (gBaudCode == Baud38400) ? B38400 :
                         (gBaudCode == Baud57600) ? B57600 :
                         (gBaudCode == Baud115200) ? B115200 : B9600);
        
        if (debug > 2) fprintf (stderr, "Changing serial port speed to %ld\n",
                                baudRate[gBaudCode]);
        tcdrain (SerialFD);
        LinuxSetSpeed(speed);
        /* flush the response, if any */
        tcflush (SerialFD, TCIFLUSH);
        retries = 0;
        goto done;
    }
    /*
     * Receive the response header (8 bytes, or 2 32-bit words)
     */
    start = time (0);
    
    rxcnt = 0;
    while (rxcnt < 8)
    {
        int len;

        len = read (SerialFD, (char *)&rxpkt + rxcnt, 8 - rxcnt);
        
        if (len < 0)
        {
            if (errno == EWOULDBLOCK)
                len = 0;
            else
            {
                perror ("read (hdr)");
                exit (1);
            }
        }
        
        if (len == 0)
        {
            now = time (0);

            if (difftime (now, start) > dsp_timeout)
            {
                if (debug > 0)
                    fprintf (stderr, "rx hdr timeout (%d)\n", rxcnt);
                ++retries;
                goto done;
            }
        }
        else
            start = time(0);
        
        rxcnt += len;
    }
            
    if (debug > 0) fprintf (stderr, "<- 0x%x ", rxpkt.pkt_id);
    
    /*
     * Parse the packet header
     */
    if (rxpkt.pkt_id == (msg | 0x8000))
        /* a good response means the DSP is still alive; clear retry count */
        retries = 0;
    else
    {
        fprintf (stderr,  "Invalid response 0x%x/0x%x",
                 rxpkt.pkt_id, (msg | 0x8000));
        if (rxpkt.pkt_id & 0x8000)
            fprintf (stderr, ", len %d\n", rxpkt.length);
        else
            fprintf (stderr, "\n");
        
        /* dispose of the rest of the packet */
        start = time (0);
        if (rnbytes)
            *rnbytes = 0;
        
        tcflush (SerialFD, TCIFLUSH);
        
        retries = 0;
        goto done;        
    }

    /* read the packet byte count (excluding header) */
    readlen = (rxpkt.length-2)*4;
    if (debug > 1 && readlen > 0) fprintf (stderr, "len=%d\n", readlen);

    if (rxpkt.pkt_id == 0x8105)
    {
        rxcnt = ((unsigned long *)data)[1] * 4;
        if (readlen-8 != rxcnt)
            fprintf (stderr,  "\nRequest/return mismatch %d/%d\n",
                     readlen-8, rxcnt);
    }
    
    rxcnt = 0;
    start = time(0);
    while (rxcnt < readlen)
    {
        int len;

        len = read (SerialFD, response+rxcnt, readlen - rxcnt);
        if (len < 0)
        {
            if (errno == EWOULDBLOCK)
                len = 0;
            else
            {
                perror ("read");
                exit (1);
            }
        }
        if (len == 0)
        {
            now = time (0);
            if (difftime (now, start) > dsp_timeout)
            {
                if (debug > 0)
                {
                    fprintf (stderr,  "%s (%d/%d)\n",
                             rxcnt ? "partial packet" : "rx timeout",
                             rxcnt, readlen);
                    for (len=2;len < readlen/4;len++)
                        fprintf (stderr, "rxbuf[%d]=0x%lx\n",
                                len, *((unsigned long *)response+len));
                }
                ++retries;
                goto done;
            }
        }
        else
            start = time(0);
        
        rxcnt += len;

        if (rnbytes)
            *rnbytes = rxcnt;
        
    }
    retries = 0;

  done:

    if (debug)
        fprintf (stderr, "\n");

    in_send = 0;

    /* restore signal handlers, then raise any pending signals */
    for (i=0;i < NSIGNALS;i++)
        signal (signals[i], saved_handler[i]);

    if (caught_signal && !closing)
    {
        if (debug > 1)
            fprintf (stderr, "Caught signal %d rethrown\n", caught_signal);
        raise (caught_signal);
    }

    return retries;
}

/* length LEN is in 32-bit words */
void
DSPWrite (unsigned long addr, unsigned long *data, int len)
{
	int i;

    while (len > 0)
    {
#define MAX_TXMSG 256
        unsigned long txbuf[MAX_TXMSG];
        int count = (len > MAX_TXMSG-2) ? MAX_TXMSG-2 : len;
        
        txbuf[0] = addr;
        txbuf[1] = count;

        for (i=0;i < count;i++)
            txbuf[i+2] = data[i];

        /*if (debug) fprintf (stderr, "writing %d words to 0x%lx\n", count, addr);*/
        while (DSPSend (DSP_WRITEDM32, (char *)txbuf, (count+2)*4, 0, 0) != 0)
            alive ();

        addr += count;
        data += count;
        len -= count;
    }
}

/* write 48-bit data to program memory; data is contiguous data, len is
   number of 48-bit words to write
   */
void
DSPWrite48 (unsigned long addr, unsigned char *data, int len)
{
	int i;
	int byte;

    while (len > 0)
    {
        unsigned long txbuf[MAX_TXMSG];
        int count = (len > (MAX_TXMSG-2)/2) ? (MAX_TXMSG-2)/2 : len;
        /* copy up to 157 48-bit words into txbuf and send */
        txbuf[0] = addr;
        txbuf[1] = count;

        /* unpack 48-bit quantities into two 32-bit words */
        for (i=0;i < count;i++)
            for (byte=0;byte < 6;byte++)
                *((unsigned char *)&txbuf[2+i*2] + byte) = data[i*6+byte];

        while (DSPSend (DSP_WRITEPM48,
                        (char *)txbuf, (count*2+2)*4, 0, 0) != 0)
            alive ();

        addr += count;
        data += count*6;
        len -= count;
    }
}

/* length LEN is in 32-bit words  */
int
DSPRead (unsigned long addr, unsigned long *data, int len)
{
    int total = 0;

    while (len > 0)
    {
#define MAX_RXMSG 256
        int count = (len > MAX_RXMSG-2) ? MAX_RXMSG-2 : len;
        unsigned short rxlen;
        unsigned long txbuf[2];
        unsigned long rxbuf[MAX_RXMSG];

        txbuf[0] = addr;
        txbuf[1] = count;
        
        /*if (debug) fprintf (stderr, "reading %d words from 0x%lx\n", count, addr);*/
        while (DSPSend (DSP_READDM32, (char *)txbuf, sizeof(txbuf),
                        (char *)rxbuf, &rxlen) != 0)
            alive ();
        
        memcpy ((char *)data, (char *)rxbuf+8, count*4);

        /* track total number of longwords received */
        total += (rxlen-8)/4;
        data += count;
        len -= count;
        addr += count;
    }

    /* return the number of longwords received */
    return total;
}

/* 
   Increasing baud rate is not entirely reliable
 */
void
DSPSpeed (baud_t baud_code)
{
    unsigned long txbuf[2];
    unsigned long divisor[] = {
        /* baud rate divisors:
           none  300  1200 2400 4800 9600 19200 38400 57600 115200 */
             0, 3840, 960, 480, 240, 120, 60,   30,   20,   10
    };

    if (gBaudCode == baud_code)
        return;

    txbuf[0] = divisor[baud_code] & 0xFF;
    txbuf[1] = (divisor[baud_code]>>8) & 0xFF;

	/*fprintf(stderr,"divisor is %d %d\n",txbuf[0],txbuf[1]);*/
    gBaudCode = baud_code;

    if (DSPSend (DSP_SETSPEED, (char *)txbuf, sizeof(txbuf), 0, 0) != 0)
        fprintf (stderr, "S\b");

    /* if the speed change didn't work, bail */
    if (DSPVerify (verify_string, sizeof(verify_string), debug))
        exit (1);
}

void
DSPSync (void)
{
    char initstr[] = { 0xb4, 0x0d, 0xdc, 0xba };
    int i;

    for (i=0;i < sizeof(initstr);i++)
        write (SerialFD,(char *) &initstr[i], 1);
}

void
DSPClose (void)
{
    if (debug > 2) fprintf (stderr, "DSPClose()\n");

    if (!send_error && in_send)
        fprintf (stderr, "Warning: DSPSend interrupted\n");
    
    /* don't make repeated attempts to close communications with the DSP if
       there's an error */ 
    if (SerialFD != -1)
    {
        if (!closing)
        {
            closing = 1;
            
            if (!send_error)
            {
                if (gBaudCode != Baud9600)
                    DSPSpeed(Baud9600);
                
                DSPSend (DSP_RESYNC, 0, 0, 0, 0);
            }
        }
        /* always make sure the file lock gets removed */
        close (SerialFD);
        SerialFD = -1;
        closing = 0;
        /* remove the lockfile */
        unlink (lockname);
    }
}

void
DSPResetCPU (void)
{
    while (DSPSend (DSP_RESETCPU, 0, 0, 0, 0) != 0)
        alive ();
}

/* note that the first two 32-bit fields of the message are clobbered
   in the response
 */
int
DSPVerify (char *message, unsigned short len, int verbose)
{
    char rxbuf[MAX_RXMSG*4];
    unsigned short rlen;

    //fprintf (stderr, "Sending \"%s\"\n", message);
    while (DSPSend (DSP_VERIFY, message, len, rxbuf, &rlen) != 0)
        alive ();

    if (rlen != len)
        fprintf (stderr,  "Error -- incorrect verify length\n");
    else
    {
        long err = *(unsigned long *)rxbuf;
        int index;
        
        if (err)
            fprintf (stderr, "Power-on error code %ld\n", err);
        else if (verbose)
            fprintf (stderr, "DSP OK -- ");
        if (verbose)
            fprintf (stderr, "Kernel version 0x%lx\n",
                    *((unsigned long *)rxbuf + 1));
        for (index=0;index < 8;index++)
            rxbuf[index] = 'X';
        
        for (index=8;index < len;index++)
        {
            if (message[index] != rxbuf[index])
            {
                fprintf (stderr,  "Error -- incorrect verify message"
                         " at index %d 0x%x/0x%x\n",
                         index, message[index], rxbuf[index]);
                err = 1;
                break;
            }
        }
        if (!err && verbose)
            fprintf (stderr, "DSP message \"%s\" verified\n", rxbuf);
        
        return err;
    }
    return -1;
}

void
DSPRun (void)
{
    while (DSPSend (DSP_START, 0, 0, 0, 0) != 0)
        alive ();
}

static void
termination_handler (int sig)
{
    if (sig == SIGINT)
        printf ("^C\n");

    if (debug > 1) fprintf (stderr, "Got signal %d\n", sig);

    exit (1);
}

int
DSPInit (int port, baud_t baud, int verbose, float timeout)
{
    int i;

    dsp_timeout = timeout;

    if (SerialFD == -1)
    {
        for (i=0;i<256;i++)
            verify_string[i+8] = i;
        
        LinuxSerialInit(port, verbose);
        
        DSPSync ();
        
        /* make sure DSP communication is properly terminated */
        for (i=0;i < NSIGNALS;i++)
            old_handler[signals[i]] = signal (signals[i],
                                              termination_handler);
        atexit (DSPClose); /* just in case */
        
        if (DSPVerify (verify_string, sizeof(verify_string), debug))
            return -1;
        
        if (verbose || debug) printf ("DSP OK\n");
        
        if (baud != Baud9600)
            DSPSpeed (baud);

    }
	return 0;
}

int
DSPInUse (void)
{
    struct stat statbuf;

    return (stat (lockname, &statbuf) == 0);
}

static void
LinuxSerialInit(int port, int verbose)
{
    struct termios options;
    char devname[] = "/dev/tty00";
    FILE *fp;

#ifdef __NetBSD__
    sprintf (devname, "/dev/tty0%d", port - 1);
#else
    sprintf (devname, "/dev/ttyS%d", port - 1);
#endif

    if (DSPInUse ())
    {
        int pid = -1;
        fp = fopen (lockname, "r");;

        if (fp != NULL)
        {
            fscanf (fp, "%d", &pid);
            fclose (fp);
        }
        fprintf (stderr, "\n\nThe DSP is currently in use by ");
        if (pid == -1)
        {
            fprintf (stderr, "an unknown process.\n");
            fprintf (stderr, "Reset the DSP and run this program"
                     "again.\n");
            unlink (lockname);
        }
        else
        {
            fprintf (stderr, "process number %d.\n", pid);
            fprintf (stderr, "If you *really* need to use the DSP, do\n");
            fprintf (stderr, "\n\t%% kill %d\n\n", (int)pid);
            fprintf (stderr, "If no such process exists, then remove the\n"
                     "file %s.  Then run this program again.\n\n",
                     lockname);
        }
        exit (1);
    }

    /* if no lock, set one */
    fp = fopen (lockname, "w");
    if (!fp)
    {
        fprintf (stderr, "Error setting %s, aborting\n", lockname);
        exit (1);
    }
    fprintf (fp, "%d\n", (int)getpid ());
    fclose (fp);

    if (debug > 0 || verbose)
    { printf("Opening %s...", devname);fflush (stdout); }
    SerialFD = open(devname,O_RDWR | O_NOCTTY | O_NDELAY);

    if (SerialFD == -1)
    {
        fprintf (stderr, "Error opening %s\n", devname);
        exit (1);
    }

    /* make sure reads return immediately */
    fcntl ( SerialFD,F_SETFL,FNDELAY);

    tcgetattr(SerialFD,&options);

    cfsetispeed(&options,B9600); 
    cfsetospeed(&options,B9600); 
    cfmakeraw(&options); /* raw, binary I/O */
/*
  cfmakeraw sets the terminal attributes as follows:
  termios_p->c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP
                          |INLCR|IGNCR|ICRNL|IXON);
  termios_p->c_oflag &= ~OPOST;
  termios_p->c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
  termios_p->c_cflag &= ~(CSIZE|PARENB);
  termios_p->c_cflag |= CS8;
 */
    /* define this if your system supports HW handshaking (RTS/CTS) */
#ifdef HW_HANDSHAKE
    options.c_cflag |= /*CLOCAL |*/ CREAD | CRTSCTS; /* HW flow control */
#else
    options.c_cflag |= CLOCAL | CREAD /*| CRTSCTS*/; /* no HW flow control */
#endif
    options.c_cflag &= ~CSTOPB; /* one stop bit */

#ifdef __NetBSD__
    options.c_iflag &= ~(IXOFF); /* no SW flow control*/
    options.c_oflag &= ~(ONLCR);
#else
    options.c_iflag &= ~(IXOFF|IUCLC); /* no SW flow control, no mapping */
    options.c_oflag &= ~(OLCUC|ONLCR|OCRNL|ONOCR|ONLRET);
#endif
    options.c_iflag |= IGNBRK;

    tcsetattr(SerialFD,TCSANOW,&options);

}

static void
LinuxSetSpeed(speed_t speed)
{
    struct termios options;
    struct termios offoptions;

    tcgetattr(SerialFD,&options);
    offoptions = options;
    cfsetispeed(&options,speed);
    cfsetospeed(&options,speed); 
    cfsetispeed(&offoptions,B0);
    cfsetospeed(&offoptions,B0);
    tcsetattr(SerialFD,TCSANOW,&offoptions);
    sleep (1);
    tcsetattr(SerialFD,TCSANOW,&options);
}

void
DSPDebug (int level)
{
    debug = level;
}

int
DSPReady (void)
{
    int msr;
    
    if (SerialFD == -1)
        return 0;
    
    if (debug)
        fprintf (stderr, "ioctl(TIOCMGET)\n");

    if (ioctl (SerialFD, TIOCMGET, &msr) < 0)
    {
        perror ("ioctl");
        return 0;
    }

    if (msr & TIOCM_CTS)
        return 1;

    /* throttle repeated calls to this function; if they come
       too fast, they interfere with serial port operation
       this value experimentally chosen on a P233MMX
       */
    usleep (500);

    return 0;
}
