/*
   File: value.c
   Defines operations on values

   CVSID: "$Id: ebs_value.c,v 1.2 2002/11/12 10:58:37 marcs Exp $"
*/

/* global includes */
#include <stdio.h>

/* libebs includes */
#include <ebs_global.h>
#include <ebs_error.h>
#include <ebs_memalloc.h>
#include <ebs_textstorage.h>
#include <ebs_bst.h>
#include <ebs_cst.h>
#include <ebs_primio.h>
#include <ebs_value.h>

/*
   Some storage administration of values
   copy_value en free_value maken??
*/
private value free_values;
private value new_value (int tag)
	{ value new;
	  if (free_values != value_nil)
	     { new = free_values;
	       free_values = free_values -> u.free;
	     }
	  else new = (value) ckmalloc (sizeof (struct value_rec));
	  new -> admin_nr = 0;
	  new -> ref_count = 1;
	  new -> dptr = NULL;
	  new -> tag = tag;
	  return (new);
	};

public value new_undefined_value ()
	{ return (new_value (undefined_value));
	};

public value new_string_value (string s)
	{ string rs = addto_names (s);
	  value new = new_value (string_value);
	  new -> u.str = rs;
	  return (new);
	};

public value new_integer_value (int nr)
	{ value new = new_value (integer_value);
	  new -> u.nr = nr;
	  return (new);
	};

public value new_tuple_value (value_list vl)
	{ value new = new_value (tuple_value);
	  new -> u.tuple = vl;
	  return (new);
	};

public value new_tuple_value_from_array (int size, value *array)
	{ value_list vl = new_value_list (size);
	  value new = new_value (tuple_value);
	  int ix;
	  vl -> size = size;
	  for (ix = 0; ix < size; ix++) vl -> array[ix] = array[ix];
	  new -> u.tuple = vl;
	  return (new);
	};

public value new_small_lattice_value (int slat, string *lnames)
	{ value new = new_value (small_lattice_value);
	  new -> dptr = (void *) lnames;
	  new -> u.slat = slat;
	  return (new);
	};

public value new_large_lattice_value (int size, int *elat, string *lnames)
	{ int_list il = new_int_list ();
	  value new = new_value (large_lattice_value);
	  int ix;
	  room_int_list (il, size);
	  il -> size = size;
	  for (ix = 0; ix < size; ix++) il -> array[ix] = (elat)?(elat[ix]):0;
	  new -> dptr = (void *) lnames;
	  new -> u.elat = il;
	  return (new);
	};

public value new_singleton_value (int eltnr, int size, string *lnames)
	{ value new;
	  if (size > 32)
	     { int asize = size/32 + 1;
	       new = new_large_lattice_value (asize, NULL, lnames);
	       new -> u.elat -> array[asize - 1 - eltnr/32] =
				(1 << (eltnr % 32));
	     }
	  else new = new_small_lattice_value (1 << eltnr, lnames);
	  return (new);
	};

/* recursively free a value */
public void rfre_value (value val)
	{ if (val == value_nil) return;
	  val -> ref_count--;
	  if (val -> ref_count) return;
	  switch (val -> tag)
	     { case tuple_value: rfre_value_list (val -> u.tuple); break;
	       case large_lattice_value: rfre_int_list (val -> u.elat);
	       default: break;
	     };
	  val -> u.free = free_values;
	  free_values = val;
	};

public value rdup_value (value val)
	{ if (val != value_nil) val -> ref_count++;
	  return (val);
	};

/*
   Define a total order for values
*/
public int less_value (value v1, value v2)
	{ if (v1 == value_nil) return (0);
	  if (v2 == value_nil) return (0);
	  if (v1 == v2) return (0);
	  if (v1 -> tag != v2 -> tag) return (v1 -> tag < v2 -> tag);
	  switch (v1 -> tag)
	     { case undefined_value: return (0);
	       case string_value:
		  return (strcmp (v1 -> u.str, v2 -> u.str) < 0);
	       case integer_value: return (v1 -> u.nr < v2 -> u.nr);
	       case tuple_value:
		  return (less_value_list (v1 -> u.tuple, v2 -> u.tuple));
	       case small_lattice_value:
		  if (v1 -> u.slat < v2 -> u.slat) return (1);
		  if (v2 -> u.slat < v1 -> u.slat) return (0);
		  break;
	       case large_lattice_value:
		  if (less_int_list (v1 -> u.elat, v2 -> u.elat)) return (1);
		  if (less_int_list (v2 -> u.elat, v1 -> u.elat)) return (0);
		  break;
	       default: bad_tag (v1 -> tag, "less_value");
	     };
	  return (v1 -> admin_nr < v2 -> admin_nr);
	};

public int equal_value (value v1, value v2)
	{ if (v1 == value_nil) return (0);
	  if (v2 == value_nil) return (0);
	  if (v1 == v2) return (1);
	  if (v1 -> tag != v2 -> tag) return (0);
	  switch (v1 -> tag)
	     { case undefined_value: return (0);
	       case string_value: return (!strcmp (v1 -> u.str, v2 -> u.str));
	       case integer_value: return (v1 -> u.nr == v2 -> u.nr);
	       case tuple_value:
		  return (equal_value_list (v1 -> u.tuple, v2 -> u.tuple));
	       case small_lattice_value: return (v1 -> u.slat == v2 -> u.slat);
	       case large_lattice_value:
		  return (equal_int_list (v1 -> u.elat, v2 -> u.elat));
	       default: bad_tag (v1 -> tag, "equal_value");
	     };
	  return (0);
	};

public void save_value (FILE *f, value v)
	{ save_int (f, v -> tag);
	  switch (v -> tag)
	     { case string_value: save_string (f, v -> u.str); break;
	       case integer_value: save_int (f, v -> u.nr); break;
	       case tuple_value: save_value_list (f, v -> u.tuple); break;
	       case small_lattice_value: save_int (f, v -> u.slat); break;
	       case large_lattice_value: save_int_list (f, v -> u.elat); break;
	       default: bad_tag (v -> tag, "save_value");
	     };
	};

public int load_value (FILE *f, value *v, string *lnames)
	{ value new;
	  int tag;
	  if (!load_int (f, &tag)) return (0);
	  new = new_value (tag);
	  new -> dptr = (void *) lnames;
	  switch (tag)
	     { case string_value:
		  if (!load_string (f, &new -> u.str)) return (0);
		  break;
	       case integer_value:
		  if (!load_int (f, &new -> u.nr)) return (0);
		  break;
	       case tuple_value:
		  if (!load_value_list (f, &new -> u.tuple, lnames)) return (0);
		  break;
	       case small_lattice_value:
		  if (!load_int (f, &new -> u.slat)) return (0);
		  break;
	       case large_lattice_value:
		  if (!load_int_list (f, &new -> u.elat)) return (0);
		  break;
	       default:
		  bad_tag (tag, "load_value");
	     };
	  *v = new;
	  return (1);
	};

/* Announce to use 'room' values in a value_list */
public void room_value_list (value_list vl, int room)
	{ if (room <= vl -> room) return;
	  vl -> array = (value *) ckrecalloc (vl -> array, room,
					      sizeof (value));
	  vl -> room = room;
	};

/* Allocate a new value list */
public value_list new_value_list (int room)
	{ value_list new =
		(value_list) ckmalloc (sizeof (struct value_list_rec));
	  int mroom = (room <= 0)?1:room;
	  new -> size = 0;
	  new -> room = mroom;
	  new -> array = (value *) ckcalloc (mroom, sizeof (value));
	  return (new);
	};

/* Recursively copy a value list */
public value_list rdup_value_list (value_list vl)
	{ value_list new = new_value_list (vl -> size);
	  int ix;
	  new -> size = vl -> size;
	  for (ix = 0; ix < vl -> size; ix++)
	     new -> array[ix] = rdup_value (vl -> array[ix]);
	  return (new);
	};

/* Recursively free a value list */
public void rfre_value_list (value_list old)
	{ int ix;
	  if (old == value_list_nil) return;
	  for (ix = 0; ix < old -> size; ix++)
	     rfre_value (old -> array[ix]);
	  free (old -> array);
	  free (old);
	};

/* Append value to value_list */
public void app_value_list (value_list vl, value v)
	{ if (vl -> size == vl -> room) room_value_list (vl, vl -> size << 1);
	  vl -> array [vl -> size] = v;
	  vl -> size++;
	};

/*
   Define a total order on value lists
*/
public int less_value_list (value_list vl1, value_list vl2)
	{ int ix;
	  if (vl1 == value_list_nil) return (0);
	  if (vl2 == value_list_nil) return (0);
	  if (vl1 -> size != vl2 -> size) return (vl1 -> size < vl2 -> size);
	  for (ix = 0; ix < vl1 -> size; ix++)
	     if (less_value (vl1 -> array[ix], vl2 -> array[ix])) return (1);
	     else if (less_value (vl2 -> array[ix], vl1 -> array[ix]))
		return (0);
	  return (0);
	};

public int equal_value_list (value_list vl1, value_list vl2)
	{ int ix;
	  if (vl1 == value_list_nil) return (0);
	  if (vl2 == value_list_nil) return (0);
	  if (vl1 -> size != vl2 -> size) return (0);
	  for (ix = 0; ix < vl1 -> size; ix++)
	     if (!equal_value (vl1 -> array[ix], vl2 -> array[ix]))
		return (0);
	  return (1);
	};

public void save_value_list (FILE *f, value_list vl)
	{ int ix;
	  save_int (f, vl -> size);
	  for (ix = 0; ix < vl -> size; ix++)
	     save_value (f, vl -> array[ix]);
	};

public int load_value_list (FILE *f, value_list *vl, string *lnames)
	{ int ix, size;
	  value_list new;
	  if (!load_int (f, &size)) return (0);
	  new = new_value_list (size);
	  new -> size = size;
	  new -> room = size;
	  for (ix = 0; ix < size; ix++)
	     if (!load_value (f, &new -> array[ix], lnames)) return (0);
	  *vl = new;
	  return (1);
	};

/* value calculation */
public value concatenate_values (value_list vl)
	{ int ix;
	  switch (vl -> array[0] -> tag)
	     { case string_value:
		  { register char *dptr = strstore;
		    for (ix = 0; ix < vl -> size; ix++)
		       { register char *sptr = vl -> array[ix] -> u.str;
		         while (*sptr) *dptr++ = *sptr++;
		       };
		    *dptr = '\0';
		    return (new_string_value (strstore));
		  };
	       case integer_value:
		  { int sum = 0;
		    for (ix = 0; ix < vl -> size; ix++)
		       sum += vl -> array[ix] -> u.nr;
		    return (new_integer_value (sum));
		  };
	       default: bad_tag (vl -> array[0] -> tag, "concatenate_values");
	     };
	  return (value_nil);
	};

public value join_lattice_values (value_list vl)
	{ int ix;
	  string *lnames = (string *) vl -> array[0] -> dptr;
	  switch (vl -> array[0] -> tag)
	     { case small_lattice_value:
		  { int uni = 0;
		    for (ix = 0; ix < vl -> size; ix++)
		       uni |= vl -> array[ix] -> u.slat;
		    return (new_small_lattice_value (uni, lnames));
	          };
	       case large_lattice_value:
		  { int size = vl -> array[0] -> u.elat -> size;
		    value new = new_large_lattice_value (size, NULL, lnames);
		    for (ix = 0; ix < vl -> size; ix++)
		       { int_list il = vl -> array[ix] -> u.elat;
		         int iy;
		         for (iy = 0; iy < size; iy++)
			    new -> u.elat -> array[iy] |= il -> array[iy];
		       };
		    return (new);
		  };
	       default: bad_tag (vl -> array[0] -> tag, "join_lattice_values");
	     };
	  return (value_nil);
	};

public int meet_lattice_values (value v1, value v2, value *vret)
	{ if ((v1 == value_nil) || (v2 == value_nil)) return (0);
	  if (v1 -> tag != v2 -> tag) return (0);
	  switch (v1 -> tag)
	     { case small_lattice_value:
		  { int met = v1 -> u.slat & v2 -> u.slat;
		    if (!met) return (0);
		    *vret = new_small_lattice_value (met,
						     (string *) v1 -> dptr);
		    return (1);
		  };
	       case large_lattice_value:
		  { int_list il1 = v1 -> u.elat;
		    int_list il2 = v2 -> u.elat;
		    int size = il1 -> size;
		    value new = new_large_lattice_value (size, NULL,
							 (string *) v1 -> dptr);
		    int_list ilt = new -> u.elat;
		    int lmet = 0;
		    int ix;
		    for (ix = 0; ix < size; ix++)
		       { int met = il1 -> array[ix] & il2 -> array[ix];
			 if (met) lmet = 1;
			 ilt -> array[ix] = met;
		       };
		    if (lmet)
		       { *vret = new;
			 return (1);
		       }
		    rfre_value (new);
		    return (0);
		  };
	       default: break;
	     };
	  return (0);
	};

/*
   Formatted output for value logging and tracing
*/
public void dump_value (value val)
	{ if (val == value_nil)
	     { eprint_log ("nil");
	       return;
	     };
	  switch (val -> tag)
	     { case undefined_value:
		  eprint_log ("\bot");
		  break;
	       case string_value:
		  output_string (stderr, val -> u.str);
		  break;
	       case integer_value:
		  eprint_log ("%d", val -> u.nr);
		  break;
	       case tuple_value:
		  { value_list vl = val -> u.tuple;
		    eprint_log ("<");
		    if (vl != value_list_nil)
		       { int ix;
			 for (ix = 0; ix < vl -> size; ix++)
			    { if (ix != 0) eprint_log (" * ");
			      dump_value (vl -> array[ix]);
			    };
		       };
		    eprint_log (">");
		  }; break;
	       case small_lattice_value:
		  if (val -> dptr == NULL)
		     eprint_log ("{ %08x }", val -> u.slat);
		  else
		     { string *lnames = (string *) val -> dptr;
		       int nfirst = 0;
		       int ix;
		       eprint_log ("{ ");
		       for (ix = 0; ix < 32; ix++)
			  if (val -> u.slat & (1 << ix))
			     { eprint_log ("%s%s", (nfirst)?", ":"",
					   lnames[ix]);
			       nfirst = 1;
			     };
		       eprint_log (" }");
		     };
		  break;
	       case large_lattice_value:
		  { string *lnames = (string *) val -> dptr;
		    int nfirst = 0;
		    int lidx = 0;
		    int_list il = val -> u.elat;
		    int ix;
		    eprint_log ("{ ");
		    if (lnames == NULL)
		       for (ix = 0; ix < il -> size; ix++)
			  eprint_log ("%08x", il -> array[ix]);
		    else
		       for (ix = il -> size - 1; 0 <= ix; ix--)
			  { int iy;
			    for (iy = 0; iy < 32; iy++, lidx++)
			       if (il -> array[ix] & (1 << iy))
				  { eprint_log ("%s%s", (nfirst)?", ":"",
						lnames [lidx]);
				    nfirst = 1;
				  };
			  };
		    eprint_log (" }");
		  }; break;
	       default: bad_tag (val -> tag, "dump_value");
	     };
	};

/*
   unformatted output for affix output at end of parse etc.
*/
public void output_value (FILE *out, value val)
	{ if (val == value_nil)
	     { fputs ("<value_nil>", out);
	       return;
	     };
	  switch (val -> tag)
	     { case undefined_value: break;
	       case string_value: 
		  fputs (val -> u.str, out);
		  break;
	       case integer_value:
		  fprintf (out, "%d", val -> u.nr);
		  break;
	       case tuple_value:
		  { value_list vl = val -> u.tuple;
		    fputc ('<', out);
		    if (vl != value_list_nil)
		       { int ix;
			 for (ix = 0; ix < vl -> size; ix++)
			    { if (ix != 0) fputs (" * ", out);
			      output_value (out, vl -> array[ix]);
			    };
		       };
		    fputc ('>', out);
		  }; break;
	       case small_lattice_value:
		  if (val -> dptr == NULL)
		     fprintf (out, "{ %08x }", val -> u.slat);
		  else
		     { string *lnames = (string *) val -> dptr;
		       int nfirst = 0;
		       int ix;
		       fprintf (out, "{ ");
		       for (ix = 0; ix < 32; ix++)
			  if (val -> u.slat & (1 << ix))
			     { fprintf (out, "%s%s", (nfirst)?", ":"",
					lnames[ix]);
			       nfirst = 1;
			     };
		       fprintf (out, " }");
		     };
		  break;
	       case large_lattice_value:
		  { string *lnames = (string *) val -> dptr;
		    int nfirst = 0;
		    int lidx = 0;
		    int_list il = val -> u.elat;
		    int ix;
		    fprintf (out, "{ ");
		    if (lnames == NULL)
		       for (ix = 0; ix < il -> size; ix++)
			  eprint_log ("%08x", il -> array[ix]);
		    else
		       for (ix = il -> size - 1; 0 <= ix; ix--)
			  { int iy;
			    for (iy = 0; iy < 32; iy++, lidx++)
			       if (il -> array[ix] & (1 << iy))
				  { fprintf (out, "%s%s", (nfirst)?", ":"",
						  lnames [lidx]);
				    nfirst = 1;
				  };
			  };
		    fprintf (out, " }");
		  }; break;
	       default: bad_tag (val -> tag, "output_value");
	     };
	};

/* Initialization */
public void init_value ()
	{ free_values = value_nil;
	};
