static char rcsid[] = "@(#)$Id: charset.c,v 1.45 2001/06/06 18:08:56 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.45 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *
 *  Partially based on mime_parse.c, which is initially 
 *     written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "headers.h"
#include "s_me.h"
#include "cs_imp.h"

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

DEBUG_VAR(Debug,__FILE__,"charset");

struct  charset_type * cs_first_type = &cs_unknown;

/* ---------------------------------------------------------------------- */

#define INVARIANT &precompiled_sets[0]
#define ASCII &precompiled_sets[1]
#define LATIN1 &precompiled_sets[2]

static struct charcode_info precompiled_sets[] = {
    { &cs_iso646, &map_invariant, SET_valid,  "INVARIANT",  NULL, NULL  },
    { &cs_ascii, &map_ascii, SET_valid,  "US-ASCII",
      INVARIANT, &(sets_iso_8859_X[0]) },
    { &cs_ascii, &map_latin1, SET_valid,  "ISO-8859-1", 
      ASCII, &(sets_iso_8859_X[1]) },
    { &cs_ascii, NULL, SET_valid,  "ISO-8859-2", 
      ASCII, &(sets_iso_8859_X[2]) },
    { &cs_ascii, NULL, SET_valid,  "ISO-8859-3", 
      ASCII, &(sets_iso_8859_X[3]) },
    { &cs_ascii, NULL, SET_valid,  "ISO-8859-4", 
      ASCII, &(sets_iso_8859_X[4]) },
    { &cs_ascii, NULL, SET_valid,  "ISO-8859-5", 
      ASCII, &(sets_iso_8859_X[5]) },
    { &cs_ascii, NULL, SET_valid,  "ISO-8859-6", 
      ASCII, &(sets_iso_8859_X[6]) },
    { &cs_ascii, NULL, SET_valid,  "ISO-8859-7",
      ASCII, &(sets_iso_8859_X[7]) },
    { &cs_ascii, NULL, SET_valid,  "ISO-8859-8", 
      ASCII, &(sets_iso_8859_X[8]) },
    { &cs_ascii, NULL, SET_valid,  "ISO-8859-9", 
      ASCII, &(sets_iso_8859_X[9]) },
    { &cs_ascii, NULL, SET_valid,  "ISO-8859-10", ASCII, NULL },
    { &cs_ascii, NULL, SET_valid,  "ISO-8859-13", ASCII, NULL },
    { &cs_ascii, NULL, SET_valid,  "ISO-8859-14", ASCII, &set_latin8 },
    { &cs_ascii, NULL, SET_valid,  "ISO-8859-15", ASCII, NULL },
    { &cs_ascii, &map_koi8r, SET_valid,  "KOI8-R", ASCII, NULL },
    { &cs_ascii, &map_koi8u, SET_valid,  "KOI8-U", ASCII, NULL },
    { &cs_ascii, &map_cp1251, SET_valid, "Windows-1251", ASCII, NULL },
    { &cs_ascii, &map_cp1252, SET_valid, "windows-1252", LATIN1, NULL },
    { &cs_utf8,  &map_utf8,  SET_valid,  "UTF-8",  NULL, &set_utf8 },
    { &cs_utf7,  &map_utf7,  SET_valid,  "UTF-7",  NULL, NULL },
    { NULL, NULL, 0,       NULL, NULL, NULL }

};

/* 'Raw' byte buffer -- treated as ascii -- used on buffering !!!!!!  */

static struct charcode_info raw_buffer = {
    &cs_ascii, &map_ascii, SET_valid,  "*RAW*", ASCII, &(sets_iso_8859_X[0])
};
CONST charset_t RAW_BUFFER = &raw_buffer;


CONST charset_t ASCII_SET = ASCII;
charset_t display_charset = ASCII;  /* current charset used by terminal */
charset_t system_charset  = ASCII;  /* the charset, the locale supports */

static struct extra_sets {
    struct charcode_info set;
    struct extra_sets    * next;
} * extra_sets = NULL, * extra_sets_tail = NULL;
static int extra_sets_count = 0;

union compare_info {
    char *name;
    struct charcode_info * part;
};

typedef  int compare_set P_((struct charcode_info *set, 
			     union compare_info data));

static struct charcode_info * find_set P_(( compare_set *func,
					    union compare_info data ));

static struct charcode_info * find_set(func,data)
     compare_set *func;
     union compare_info data;
{
    int i;
    struct extra_sets *walk;

#define TEST(ITEM) \
    if (0 != (ITEM.flags & SET_valid) && func(&(ITEM),data)) { \
      DPRINT(Debug,9,(&Debug,"find_set: found=%p (%s)\n",&(ITEM),\
		       ITEM.MIME_name ? ITEM.MIME_name : "<no MIME name>")); \
    return &(ITEM); }

#define SCAN(SET) \
    for (i = 0; i < sizeof SET / sizeof (struct charcode_info); i++) {\
	DPRINT(Debug,49,(&Debug,"find_set: (scan#%d) %p -- flags=%d\n",\
			 i,&(SET[i]),SET[i].flags)); \
	TEST(SET[i]); } 


    DPRINT(Debug,40,(&Debug, 
		     "find_set: precompiled_sets\n"));
    SCAN(precompiled_sets);
    
    DPRINT(Debug,40,(&Debug, "find_set: extra_sets\n"));
    for (walk = extra_sets; walk; walk = walk -> next) {
	DPRINT(Debug,49,(&Debug, "find_set: (walk) %p -- flags=%d\n",
		    &(walk->set),walk->set.flags)); 
	TEST(walk->set);
    }

    DPRINT(Debug,40,(&Debug, "find_set: NOT FOUND\n"));
    return NULL;
}	
#undef SCAN
#undef SET

static struct charcode_info * add_set P_((struct charcode_info s));
static struct charcode_info * add_set(s)
     struct charcode_info s;
{
    struct extra_sets *tmp = safe_malloc( sizeof (struct extra_sets));
    
    tmp->set  = s;
    tmp->next = NULL;

    if (extra_sets_tail)
	extra_sets_tail->next = tmp;    
    extra_sets_tail = tmp;    
    if (!extra_sets)
	extra_sets = tmp;
    
    extra_sets_count++;

    if (tmp->set.MIME_name) {
	DPRINT(Debug,1,(&Debug, 
			"charset: Adding charset %s (type %s)\n",
		  tmp->set.MIME_name,
		  tmp->set.charset_type->type_name));	
    }

    return &(tmp->set);
}

static int same_name P_((struct charcode_info *set, union compare_info data));
static int same_name (set,data)
     struct charcode_info *set;
     union compare_info data;
{
    DPRINT(Debug,40,(&Debug, 
		     "same_name -- set=%p (MIME_name=%s,data.name=%s)\n",
	       set,set->MIME_name ? set->MIME_name : "<NULL>",data.name));
    if (!set->MIME_name)
	return 0;
    return 0 == istrcmp(data.name,set->MIME_name);
}

static struct charset_alias {
       char *                name;
       charset_t             set;
} *    charset_alias_list  = NULL;
static int charset_alias_count = 0;

static int locate_charset_alias P_((const char * name));
static int locate_charset_alias(name)
     CONST char * name;
{
    int i;

    for (i = 0; i < charset_alias_count; i++)
	if (0 == istrcmp(charset_alias_list[i].name,name))
	    return i;
    return -1;
}

static int add_charset_alias P_((const char * name, charset_t set));
static int add_charset_alias(name,set)
     CONST char * name;
     charset_t set;
{
    charset_alias_list = safe_realloc(charset_alias_list,
				      (charset_alias_count +1) *
				      sizeof (struct charset_alias));
    charset_alias_list[charset_alias_count].name = safe_strdup(name);
    charset_alias_list[charset_alias_count].set = set;
    return charset_alias_count++;
}


charset_t MIME_name_to_charset (name,create)
     char *name;
     int create;
{
    int i;
    charset_t ret;
    struct charcode_info new;
    
    union compare_info D;
    D.name =  name;
    
    ret = find_set(same_name,D);

    if (!ret && 0 == istrcmp(name,"US-ASCII")) {
	DPRINT(Debug,7,(&Debug, 
		   "MIME_name_to_charset: US-ASCII not found -- using builting constant...\n"));
	ret = ASCII_SET;
    }

    if (ret) {
	/* Initialize associated map if not done already ... */       
	if (ret->map_info && !ret->map_info->map_initialized)
	    ret->map_info->map_init_it(ret->map_info);

	DPRINT(Debug,9,(&Debug, "MIME_name_to_charset(%s), found=%p\n",
		   name,ret));
	return ret;
    }

    /* Check aliases ... */
    if (0 <= (i = locate_charset_alias(name)) &&
	(ret = charset_alias_list[i].set) &&
	ret->MIME_name &&
	/* Ignore aliases which point invalidated charsets.
	   These charsets may me invalidated by setting
	   explicit compatcahrsets on user's elmrc
	*/
	(0 != (ret-> flags & SET_valid))) {

	/* Initialize associated map if not done already ... */       
	if (ret->map_info && !ret->map_info->map_initialized)
	    ret->map_info->map_init_it(ret->map_info);

	DPRINT(Debug,9,(&Debug,
		   "MIME_name_to_charset(%s), found=%p (alias of %s)\n",
		   name,ret,
		   ret->MIME_name));
	return ret;
    }


    if (!create) {
	DPRINT(Debug,12,(&Debug,"MIME_name_to_charset(%s), not found\n",
			 name));
	return NULL;
    }    

    new.flags          = SET_nodata|SET_valid;
    new.map_info       = NULL;
    new.MIME_name      = safe_strdup(name);
    new.charset_type   = &cs_unknown;
    new.subtype        = NULL;
    new.iso2022_info   = NULL;

    ret = add_set(new);

    DPRINT(Debug,12,(&Debug,"MIME_name_to_charset(%s), adding=%p\n",
		     name,ret));

    return ret;
}

static int add_it_ P_((charset_t ascii_ptr,
		       struct charcode_info *item,
		       struct charcode_info **list,
		       int *count, int max));


static int add_it_ (ascii_ptr,item,list,count,max)
     charset_t ascii_ptr;
     struct charcode_info *item;
     struct charcode_info **list;
     int *count; 
     int max;
{
    /* Pick only MIME charsets ... */

    if (item->MIME_name) 
	if (item->subtype == ascii_ptr) 		    
	    if (*count < max) 
		list[(*count)++] = item; 
	    else { 
		lib_error(FRM("charset: Internal error -- overflow")); 
		return 0;
	    }
    
    return 1;
}


static int array_walk_ P_((charset_t ascii_ptr,
			   struct charcode_info *arr,
			   int arrsize,
			   struct charcode_info **list,
			   int *count, int max));


static int array_walk_ (ascii_ptr,arr,arrsize, list, count, max)
     charset_t ascii_ptr;
     struct charcode_info *arr;
     int arrsize;
     struct charcode_info **list;
     int *count;
     int max;
{
    int i;

    
    for(i = 0; i < arrsize; i++) 
	if (0 != (arr[i].flags & SET_valid))
	    if (!add_it_(ascii_ptr,&arr[i],list,count,max))
		return 0;

    return 1;
}

static int name_ok P_((CONST char *name));
static int name_ok(name)
     CONST char *name;
{
    if ('\0' == *name)
	return 0;

    return strlen(name) == 
	strspn(name,
	       "ABCDEFGHIJKLMNOPQRSTUVWXYZ-0123456789abcdefghijklmnopqrstuvwxyz");
}

#define ARRSIZE(SET) sizeof SET / sizeof (struct charcode_info)

int charset_compatfunc (value,enter)
     char **value;
     int enter;
{
    struct extra_sets *walk;
    int ret = 0;
    charset_t ascii_ptr = MIME_name_to_charset("US-ASCII",0);

    CONST int size = ARRSIZE(precompiled_sets) + 
	extra_sets_count;

    struct charcode_info **need_modify_list =
	safe_malloc(size * sizeof(struct charcode_info *));
    int count = 0;

    if (!ascii_ptr)
	panic("CHARSET PANIC",__FILE__,__LINE__,"charset_compatfunc",
	      "US-ASCII not found",0);

#define HANDLE(SET) if (!array_walk_(ascii_ptr,SET,ARRSIZE(SET),need_modify_list,&count,size)) goto fail;
#define ADD_IT(ITEM) if (!add_it_(ascii_ptr,ITEM,need_modify_list,&count,size)) goto fail;
        
    HANDLE(precompiled_sets);

    for (walk = extra_sets; walk; walk = walk -> next) 
	if (0 != (walk->set.flags & SET_valid))
	    ADD_IT(&(walk->set));

    if (enter) {
	char * temp = safe_strdup(*value);
	char *wrk,*next;
	int j;
	
	for (j = 0; j < count; j++)
	    need_modify_list[j]->flags |= SET_mark;

	for (wrk = temp; wrk && *wrk; wrk = next) {
	    charset_t charset_ptr = NULL;

	    next = qstrpbrk(wrk," ,;");
	retry:
	    if (next) {
		char *a = next;
		if (';' != *next)
		    next++;
		while (' ' == *next)
		    next++;
		if (';' == *next) {
		    next++;
		    while (' ' == *next)
			next++;
		    next = qstrpbrk(next," ,");
		    goto retry;
		}
		*a = '\0';
	    }

	    if (!load_charset_map_info(&charset_ptr,wrk)) 
		charset_ptr = MIME_name_to_charset(wrk,1); 

	    if (0 != (charset_ptr -> flags & SET_mark)) {
		charset_ptr -> flags &=  ~SET_mark;
		charset_ptr -> flags &=  ~SET_nodata;
		if (&cs_unknown == charset_ptr -> charset_type)
		    charset_ptr -> charset_type  = &cs_ascii;

		DPRINT(Debug,39,(&Debug, 
			   "charset_compatfunc: modify=%p (%s), flags=%d, type=%s\n",
			   charset_ptr,
			   charset_ptr->MIME_name ? 
			   charset_ptr->MIME_name : "<no MIME name>",
			   charset_ptr->flags,
			   charset_ptr->charset_type->type_name));

	    } else {  /* new charset create information */
		
		charset_ptr -> charset_type  = &cs_ascii;
		charset_ptr -> flags         = SET_valid;
		charset_ptr -> subtype       = ascii_ptr;
		DPRINT(Debug,39,(&Debug, 
				 "charset_compatfunc: new=%p (%s), flags=%d, type=%s\n",
				 charset_ptr,
				 charset_ptr->MIME_name ? 
				 charset_ptr->MIME_name : "<no MIME name>",
				 charset_ptr->flags,
				 charset_ptr->charset_type->type_name));
	    }

	    if (charset_ptr->map_info && 
		charset_ptr->map_info->map_type != charset_ptr->charset_type) {

		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeCompatcharsetsDiscardMap,
				  "WARNING: %s redefined by compatcharsets, discarding map defination"),
			  charset_ptr->MIME_name ?
			  charset_ptr->MIME_name : "<no MIME name>");
		charset_ptr->map_info = NULL;		
	    }

	}
	     
	free(temp);

	/* invalidate rest */
	for (j = 0; j < count; j++)
	    if (0 != (need_modify_list[j]->flags & SET_mark)) {
		need_modify_list[j] -> flags &=  ~SET_valid;
		DPRINT(Debug,1,(&Debug,"charset: Invalidating charset %s\n",
				need_modify_list[j]->MIME_name));	
	    }

    } else {
	int j;
	
	/* static pointer to buffer accross invocations */
	static char * return_buffer = NULL;

	return_buffer = strmcpy(return_buffer,"");    /* Reset result */
	for (j = 0; j < count; j++) {
	    char * s = need_modify_list[j]->MIME_name;
	    
	    if (return_buffer[0])
		return_buffer = strmcat(return_buffer," ");
	    
	    if (name_ok(s))
		return_buffer = strmcat(return_buffer,
					need_modify_list[j]->MIME_name);
	    else {
		char * tmp = elm_message(FRM("%Q;!"),s);
		return_buffer = strmcat(return_buffer,tmp);
		free(tmp);
	    }
	}

	*value = return_buffer;
    }
    ret = 1;
    
	fail:
    free(need_modify_list);
    return ret;
}
#undef ADD_IT
#undef HANDLE

int charset_iso646func (value,enter)
     char **value;
     int enter;
{
    struct extra_sets *walk;
    int ret = 0;
    charset_t invariant_ptr = MIME_name_to_charset("INVARIANT",0);

    CONST int size = ARRSIZE(precompiled_sets) + 
	extra_sets_count;

    struct charcode_info **need_modify_list =
	safe_malloc(size * sizeof(struct charcode_info *));
    int count = 0;

    if (!invariant_ptr)
	invariant_ptr = INVARIANT;

#define HANDLE(SET) if (!array_walk_(invariant_ptr,SET,ARRSIZE(SET),need_modify_list,&count,size)) goto fail;
#define ADD_IT(ITEM) if (!add_it_(invariant_ptr,ITEM,need_modify_list,&count,size)) goto fail;
        
    HANDLE(precompiled_sets);

    for (walk = extra_sets; walk; walk = walk -> next) 
	if (0 != (walk->set.flags & SET_valid))
	    ADD_IT(&(walk->set));

    if (enter) {
	char * temp = safe_strdup(*value);
	char *wrk,*next;
	int j;
	
	for (j = 0; j < count; j++)
	    need_modify_list[j]->flags |= SET_mark;

	for (wrk = temp; wrk && *wrk; wrk = next) {
	    charset_t charset_ptr = NULL;

	    next = qstrpbrk(wrk," ,;");
	retry:
	    if (next) {
		char *a = next;
		if (';' != *next)
		    next++;
		while (' ' == *next)
		    next++;
		if (';' == *next) {
		    next++;
		    while (' ' == *next)
			next++;
		    next = qstrpbrk(next," ,");
		    goto retry;
		}
		*a = '\0';
	    }

	    if (!load_charset_map_info(&charset_ptr,wrk)) 
		charset_ptr = MIME_name_to_charset(wrk,1); 

	    if (0 != (charset_ptr -> flags & SET_mark)) {
		charset_ptr -> flags &=  ~SET_mark;
		charset_ptr -> flags &=  ~SET_nodata;
		if (&cs_unknown == charset_ptr -> charset_type)
		    charset_ptr -> charset_type  = &cs_iso646;

		DPRINT(Debug,39,(&Debug, 
			   "charset_iso646func: modify=%p (%s), flags=%d, type=%s\n",
			   charset_ptr,
			   charset_ptr->MIME_name ? 
			   charset_ptr->MIME_name : "<no MIME name>",
			   charset_ptr->flags,
			   charset_ptr->charset_type->type_name));

	    } else {  /* new charset create information */
		charset_ptr -> charset_type  = &cs_iso646;
		charset_ptr -> flags         = SET_valid;
		charset_ptr -> subtype       = invariant_ptr;
		DPRINT(Debug,39,(&Debug, 
			   "charset_iso646func: new=%p (%s), flags=%d, type=%s\n",
			   charset_ptr,
			   charset_ptr->MIME_name ? 
			   charset_ptr->MIME_name : "<no MIME name>",
			   charset_ptr->flags,
			   charset_ptr->charset_type->type_name));
	    }

	    if (charset_ptr->map_info && 
		charset_ptr->map_info->map_type != charset_ptr->charset_type) {
		
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeIso646charsetsDiscardMap,
				  "WARNING: %s redefined by iso646-charsets, discarding map defination"),
			  charset_ptr->MIME_name ?
			  charset_ptr->MIME_name : "<no MIME name>");
		charset_ptr->map_info = NULL;		
	    }

	}
	     
	free(temp);

	/* invalidate rest */
	for (j = 0; j < count; j++)
	    if (0 != (need_modify_list[j]->flags & SET_mark)) {
		need_modify_list[j] -> flags &=  ~SET_valid;
		DPRINT(Debug,1,(&Debug,"charset: Invalidating charset %s\n",
				need_modify_list[j]->MIME_name));	
	    }

    } else {
	int j;
	
	/* static pointer to buffer accross invocations */
	static char * return_buffer = NULL;

	return_buffer = strmcpy(return_buffer,"");    /* Reset result */
	for (j = 0; j < count; j++) {
	    char * s = need_modify_list[j]->MIME_name;
	    
	    if (return_buffer[0])
		return_buffer = strmcat(return_buffer," ");
	    
	    if (name_ok(s))
		return_buffer = strmcat(return_buffer,
					need_modify_list[j]->MIME_name);
	    else {
		char * tmp = elm_message(FRM("%Q;!"),s);
		return_buffer = strmcat(return_buffer,tmp);
		free(tmp);
	    }
	}

	*value = return_buffer;
    }
    ret = 1;
    
	fail:
    free(need_modify_list);
    return ret;
}
#undef ADD_IT
#undef HANDLE

#undef ARRSIZE

char * dequote_opt(source,len)
     CONST char *source; 
     int len;
{
    int size = len + 1;
    int ptr = 0;
    char * store = safe_malloc(size);
    CONST char *p;
    int q = 0;
    
    for (p = source; p - source < len && ptr < size-1; p++) {
	switch(*p) {
	case '"':
	    q = !q;
	    break;
	case '\\':
	    if (q) {
		p++;
		if (*p != '\0')
		    store[ptr++] = *p;
		else {
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeTrailingBackslash,
				      "Trailing backslash (\\): %.20s..."),
			      source);
		}
		break;
	    }
	    /* FALLTHRU */
	default:
	    if (q || *p != ' ')
		store[ptr++] = *p;
	    break;
	}    
    }
    store[ptr] = '\0';

    if (q)
	lib_error(CATGETS(elm_msg_cat, MeSet, 
			  MeUnbalancedQuote,
			  "Unbalanced quote (\"): %.20s..."),
		  source);


    DPRINT(Debug,20,(&Debug, 
		    "dequote_opt(%.*s (len=%d))=%s\n",
		    len,source,len,store));

    return store;
}

int load_charset_map_info(buffer,data)
     charset_t *buffer; 
     CONST char *data;
{
    char * temp = safe_strdup(data);
    char *c = qstrpbrk(temp,";");
    char *store,*opt;
    struct charcode_info new_vals;
    int reallocation_needed = 0;
    char ** alias_vector = NULL;
    int alias_count      = 0;
    struct setlist     new_setlist;
    int         setcount = 0;

    if (!c) {
	*buffer = NULL;
	
	DPRINT(Debug,7,(&Debug, 
		     "load_charset_map_info(\"%s\") = 0 (FAIL)\n", 
		     data));
	free(temp);
	return 0;
    }
    store = dequote_opt(temp,c - temp);
    DPRINT(Debug,11,(&Debug, 
		 "load_charset_map_info: charset=%s\n", 
		 store));

    DPRINT(Debug,11,(&Debug, 
		 "load_charset_map_info: info=%s\n", c));

    *buffer = MIME_name_to_charset(store,0);
    if (!*buffer) {
	reallocation_needed   = 1;
	new_vals.charset_type = &cs_unknown;
	new_vals.map_info     = NULL;
	new_vals.flags        = SET_valid;
	new_vals.MIME_name    = safe_strdup(store);
	new_vals.subtype      = NULL;
	new_vals.iso2022_info = NULL;
    } else 
	new_vals = **buffer;
    
    c++;

    if ('\0' == *c || 0 == strcmp(c,"!")) {
	DPRINT(Debug,9,(&Debug,"charset: %s: no data\n",store));
	new_vals.flags       |= SET_nodata;
    } else { 
	int i;
       
	if (0 <= (i = locate_charset_alias(store)) &&
	    charset_alias_list[i].set == *buffer) {
	    DPRINT(Debug,1,(&Debug,
			    "DANGER: Defining charset %s which is also alias of %s\n",
			    store,
			    charset_alias_list[i].set->MIME_name ?
			    charset_alias_list[i].set->MIME_name :
			    "<no MIME name>"));
	    alias_vector = safe_realloc(alias_vector,
					(alias_count+1) *
					sizeof (char *));
	    alias_vector[alias_count++] = safe_strdup(store);

	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeTryingCharsetAlias,
			      "Trying redefine alias %s as charset"),
		      store);
	}

	for (opt = mime_parse_content_opts(c); 
	     opt; 
	     opt = mime_parse_content_opts(NULL)) {
	    char * q = strchr(opt,'=');
	    
	    DPRINT(Debug,20,(&Debug, "mime_parse_content_opts gives: %s\n",
			     opt));

	    if (q) {
		char *val; 
		*q++ = '\0';
		val = dequote_opt(q,strlen(q));
		
		if (0 == strcmp(opt,"MIME-subset")) {
		    reallocation_needed = 1;
		    new_vals.flags  &= ~SET_nodata;

		    new_vals.subtype  = MIME_name_to_charset(val,0);
		    if (!new_vals.subtype) {
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeUnknownMIMEsubset,
					  "Unknown MIME-subset %s"),
				  val);				  
		    } 
		    
		    if (0 == istrcmp(val,"US-ASCII") &&
			&cs_unknown == new_vals.charset_type)
			new_vals.charset_type = &cs_ascii;
		    if (0 == istrcmp(val,"INVARIANT") &&
			&cs_unknown == new_vals.charset_type)
			new_vals.charset_type = &cs_iso646;
		    free(val);


		} else if (0 == strcmp(opt,"map")) {
		    struct  map_info * r;
		    if (new_vals.charset_type == &cs_unknown) {
			lib_error(CATGETS(elm_msg_cat, MeSet,
                                          MeCharsetTypeBeforeMap,
                                          "Charset type must be given before map -parameter"));
		    }
		    r = new_vals.charset_type->find_map_type(val);
		    if (!r) {
			lib_error(CATGETS(elm_msg_cat, MeSet,
                                          MeUnknownMap,
					  "No map %s on charset type %s"),
				  val,new_vals.charset_type->type_name);
		    } else {
			reallocation_needed = 1;
			new_vals.flags     &= ~SET_nodata;
			new_vals.map_info   = r;
		    }

		} else if (0 == strcmp(opt,"type")) {
		    struct  charset_type * walk;

		    for (walk = cs_first_type; walk; walk = walk -> next_type)
			if (0 == strcmp(val,walk->type_name))
			    break;
		    if (!walk) {
			lib_error(CATGETS(elm_msg_cat, MeSet,
                                          MeUnknownCharsetType,
                                          "Unknown charset type %s"),
                                  val);
			walk = &cs_unknown;
		    }	
		    reallocation_needed   = 1;
		    new_vals.flags       &= ~SET_nodata;
		    new_vals.charset_type = walk;
		} else if (0 == strcmp(opt,"alias")) {
		    int i = locate_charset_alias(val);
		    charset_t p1;

		    if (i < 0 &&
			(p1 = MIME_name_to_charset(val,0))) {
			lib_error(CATGETS(elm_msg_cat, MeSet,
                                          MeTryingAliasCharset,
					  "Trying redefine charset %s as alias"),
				  p1->MIME_name ? p1->MIME_name : val);
			free(val);
		    } else {
			alias_vector = safe_realloc(alias_vector,
						    (alias_count+1) *
						    sizeof (char *));
			alias_vector[alias_count++] = val;
		    }
		} else if (parse_iso2022_specification(opt,val,
						       &setcount,
						       &new_setlist)) {
		    /* none */
		} else {
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeUnsupportedCharsetOption,
				      "Unsupported charset option %s (value %.20s...)"),
			      opt,val);				  
		    free(val);
		}
	    } else {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeUnsupportedCharsetOptionNoValue,
				  "Unsupported charset option %s (no value)"),
			  opt);				  
	    }
	}
    }

    if (setcount > 0) {
	int i;
	int ptr_94 = -1;
	int ptr_96 = -1;
	int banks[4];

	for (i = 0; i < 4; i++)
	    banks[i] = -1;

	/* Non ISO2022 charsets there may be only on 94 and one 96 set */

	for (i = 0; i < setcount; i++) {
	    switch (new_setlist.sets[i]->type) {
	    case iso2022_94x94: 
		if (new_vals.charset_type == &cs_ascii ||
		    new_vals.charset_type == &cs_onebyte ||
		    new_vals.charset_type == &cs_iso646) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeNoMultibyte,
				      "Charset type %s does not allow multibyte bank specifications."),
			      new_vals.charset_type->type_name);
		    goto iso2022_fail; /* Discard bank defination */
		}
		/* FALLTRU */
	    case iso2022_94: 
		if (-1 == ptr_94)
		    ptr_94 = i;
		else {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeNoSeveral94,
				      "Charset type %s does not allow several 94 or 94x94 bank specifications"),
			      new_vals.charset_type->type_name);
		    goto iso2022_fail; /* Discard bank defination */
		}
		break;
	    case iso2022_96x96:
		if (new_vals.charset_type == &cs_ascii ||
		    new_vals.charset_type == &cs_onebyte ||
		    new_vals.charset_type == &cs_iso646) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeNoMultibyte,
				      "Charset type %s does not allow multibyte bank specifications."),
			      new_vals.charset_type->type_name);
		    goto iso2022_fail; /* Discard bank defination */
		}
		/* FALLTRU */
	    case iso2022_96: 
		if (-1 == ptr_96)
		    ptr_96 = i;
		else {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeNoSeveral96,
				      "Charset type %s does not allow several 96 or 96x96 bank specifications"),
			      new_vals.charset_type->type_name);
		    goto iso2022_fail; /* Discard bank defination */
		}
		break;		
	    }
	    if (new_setlist.sets[i]->bank >= 0 &&
		new_setlist.sets[i]->bank <= 4) {
		if (-1 == banks[new_setlist.sets[i]->bank])
		    banks[new_setlist.sets[i]->bank] = i;
		else {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeNoSeveralBank,
				      "Charset type %s does not allow several bank-G%d specifications"),
			      new_vals.charset_type->type_name,
			      new_setlist.sets[i]->bank);
		    goto iso2022_fail; /* Discard bank defination */
		}
	    }

	    /* Auto add ASCII bank */
	    if (new_vals.charset_type == &cs_ascii &&
		banks[0] == -1 && -1 == ptr_94 &&
		setcount < sizeof (new_setlist.sets) / 
		sizeof (new_setlist.sets[0])) {
		
		new_setlist.sets[setcount++] = ASCII_BANK;
	    }
	}

	for (i = setcount; 
	     i < sizeof (new_setlist.sets) / sizeof (new_setlist.sets[0]);
	     i++)
	    new_setlist.sets[i] = NULL;
	
	new_vals.iso2022_info = loc_setlist(new_setlist);
	reallocation_needed   = 1;
	new_vals.flags       &= ~SET_nodata;
    }
 iso2022_fail:

    if (reallocation_needed) {
	/* Reset incompatble map types ... */
	if (new_vals.map_info && 
	    new_vals.map_info->map_type != new_vals.charset_type) {
	    DPRINT(Debug,1,(&Debug,
		      "charset: Resetting incompatible map %s (type %s) from charset %s (type %s)\n",
		      new_vals.map_info->map_name,
		      new_vals.map_info->map_type->type_name,
		      new_vals.MIME_name ? 
		      new_vals.MIME_name : "<no MIME name>",
		      new_vals.charset_type->type_name));
		   new_vals.map_info = NULL;
	}
	
	if (!new_vals.map_info && &cs_onebyte == new_vals.charset_type) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeNeedsMapParamater,
			      "Charset type %s requires map parameter"),
		      new_vals.charset_type->type_name);

	    if (new_vals.subtype == ASCII) {
		DPRINT(Debug,1,(&Debug,
			  "charset: Charset %s: setting type to %s instead of %s\n",
			  (*buffer)->MIME_name ?
			  (*buffer)->MIME_name : "<no MIME name>",
			  cs_ascii.type_name,
			  new_vals.charset_type->type_name));
		new_vals.charset_type = &cs_ascii;
	    }
	    if (new_vals.subtype == INVARIANT) {
		DPRINT(Debug,1,(&Debug,
			  "charset: Charset %s: setting type to %s instead of %s\n",
			  (*buffer)->MIME_name ?
			  (*buffer)->MIME_name : "<no MIME name>",
			  cs_iso646.type_name,
			  new_vals.charset_type->type_name));
		new_vals.charset_type = &cs_iso646;
	    }
	}


	if (*buffer) {
	    (*buffer) -> flags  &= ~SET_valid;
	    DPRINT(Debug,1,(&Debug,
		      "charset: Invalidating charset %s (and regenerating)\n",
		      (*buffer)->MIME_name ? 
		      (*buffer)->MIME_name : "<no MIME name>"));	
	}
	*buffer = add_set(new_vals);
	DPRINT(Debug,9,(&Debug,
			"charset: Adding charset %s (type %s)\n",
			(*buffer)->MIME_name ? 
			(*buffer)->MIME_name : "<no MIME name>",
			(*buffer)->charset_type->type_name));	
    }

    if (alias_vector) {
	int i;
	for (i = 0; i < alias_count; i++) {
	    int j =  locate_charset_alias(alias_vector[i]);
	    if (j < 0)
		j = add_charset_alias(alias_vector[i],*buffer);

	    /* In case of we redefine alias ... */
	    charset_alias_list[j].set->flags &= ~SET_havealias;

	    (*buffer) -> flags  |= SET_havealias;

	    DPRINT(Debug,9,(&Debug,
			    "load_charset_map_info: Setting %s as alias of %s\n",
			    charset_alias_list[j].name,
			    (*buffer)->MIME_name ?
			    (*buffer)->MIME_name : "<no MIME name>"));

	    charset_alias_list[j].set = *buffer;

	    free(alias_vector[i]);
	    alias_vector[i] = NULL;
	}
	free(alias_vector);
    }
    DPRINT(Debug,9,(&Debug,
		    "load_charset_map_info(\"%s\") = 1 (SUCCEED)\n", 
		    data));    
    free(temp);
    free(store);
    return 1;    
}

static void dump_alias_info P_((FILE *f,struct charcode_info *info));
static void dump_alias_info(f,info)
     FILE *f;
     struct charcode_info *info;
{
    int i;
    for (i = 0; i < charset_alias_count; i++)
	if (info == charset_alias_list[i].set &&
	    charset_alias_list[i].name) {
	    if (name_ok(charset_alias_list[i].name))
		elm_fprintf(f,FRM(";alias=%s"),charset_alias_list[i].name);
	    else
		elm_fprintf(f,FRM(";alias=%Q"),charset_alias_list[i].name);
	}   
}

void dump_map_info(f,info)
     FILE *f;
     struct charcode_info *info;
{
    if (!info->MIME_name) {
	fputs("# <No MIME name> ",f);
    } else if (name_ok(info->MIME_name)) {
	fputs(info->MIME_name,f);
    } else {
	elm_fprintf(f,FRM("%Q"),info->MIME_name);
    }
    
    if (0 != (info->flags & SET_nodata)) {
	fputs(";!",f);
	return;
    }

    if (&cs_unknown != info->charset_type)
	elm_fprintf(f,FRM(";type=%s"),info->charset_type->type_name);
    if (info->map_info && name_ok(info->map_info->map_name))
	elm_fprintf(f,FRM(";map=%s"),info->map_info->map_name);
    else if (info->map_info)
	elm_fprintf(f,FRM(";map=%Q"),info->map_info->map_name);

    if (info->subtype && 
	info->subtype->MIME_name &&
	name_ok(info->subtype->MIME_name))
	elm_fprintf(f,FRM(";MIME-subset=%s"),info->subtype->MIME_name);
    else if (info->subtype && 
	     info->subtype->MIME_name)
	elm_fprintf(f,FRM(";MIME-subset=%Q"),info->subtype->MIME_name);

    if (info->iso2022_info)
	print_setlist(f,info->iso2022_info);

    if (0 != (info->flags & SET_havealias)) {
	dump_alias_info(f,info);
    }
}

int get_charset_map_info (buffer,data,size)
     char *buffer;
     CONST char *data; 
     int size;
{
    charset_t set;

    if (!load_charset_map_info(&set,data) ||
	!set->MIME_name) {
	buffer[0] = '\0';

	DPRINT(Debug,7,(&Debug, 
		     "get_charset_map_info(\"%s\") = 0 (FAIL)\n", 
		     data));

	return 0;
    }

    strfcpy(buffer,set->MIME_name,size);

    DPRINT(Debug,11,(&Debug, 
		 "get_charset_map_info(\"%s\") = 1 (SUCCEED)\n", 
		 data));
    return 1;
}

static int charset_superset_of_p P_((charset_t charset, 
				     charset_t subset,
				     int level));

static int charset_superset_of_p(charset,subset,level)
     charset_t charset,subset;
     int level;
{

    if (level > 10) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeCharsetSubsetLoop,
			  "Charset subset defination loop detected for %s"),
		  charset->MIME_name ? charset->MIME_name : "<no MIME name>");
	return 0;
    }

    if (charset->subtype == subset)
	return 1;

    if (charset->subtype &&
	charset_superset_of_p(charset->subtype,subset,level+1))
	return 1;
    return 0;
}

int charset_superset_of(charset,subset)
     charset_t charset,subset;
{
    int res = charset_superset_of_p(charset,subset,0);

    DPRINT(Debug,6,(&Debug, 
		    "charset_superset_of(%p (%s),%p (%s))=%d\n",
	      charset,
	      charset->MIME_name ? charset->MIME_name :"<no MIME name>",
	      subset,subset->MIME_name ? subset->MIME_name :"<no MIME name>",
	      res));
    return res;
}

int charset_ok_p(ptr)
     charset_t ptr;
{
    charset_t ascii_ptr = MIME_name_to_charset("US-ASCII",0);

    if (!ascii_ptr)
	panic("CHARSET PANIC",__FILE__,__LINE__,"charset_ok_p",
	      "US-ASCII not found",0);
    return charset_superset_of(ptr,ascii_ptr);
}

int charset_properties(ptr)
     charset_t ptr;
{
    int prop = 0;

    /* We know at least ASCII part from charset if it have type cs_ascii */
    if (&cs_ascii == ptr->charset_type)  
	prop |= CS_printable | CS_mapping;
	
    /* We know at least that part of charset which maps to ASCII */
    if (&cs_onebyte == ptr->charset_type && ptr->map_info)
	prop |= CS_mapping | CS_printable; 
    if (&cs_utf8 == ptr->charset_type)  
	prop |= CS_printable | CS_mapping;
    if (&cs_utf7 == ptr->charset_type)  
	prop |= CS_printable | CS_mapping;
    if (&cs_imap == ptr->charset_type)  
	prop |= CS_printable | CS_mapping;


    /* We know at least invariant part of charset */
    if (&cs_iso646 == ptr->charset_type)
	prop |= CS_mapping | CS_printable; 


#ifndef ASCII_CTYPE
    /* We know printable characters from locale */
    if ((&cs_ascii   == ptr->charset_type ||
	 &cs_unknown == ptr->charset_type ||
	 &cs_onebyte == ptr->charset_type) &&
	system_charset == ptr)
	prop |= CS_printable;
#endif

    DPRINT(Debug,7,(&Debug, 
		    "charset_properties(%p '%s'; type=%p)=%d\n",
	       ptr,
	       ptr->MIME_name ? ptr->MIME_name : "<none>",
	       ptr->charset_type,
	       prop));
    return prop;
}

struct locale_map_item  * load_locale_map(filename,errors) 
     CONST char *filename;
     int *errors;
{
    struct locale_map_item *result;
    int result_len = 0;
    FILE * f = fopen(filename,"r");
    int max_result = 0;
    int c,l1;
    char buf[LONG_STRING];

    if (!f) {
	int err = errno;
	DPRINT(Debug,2,(&Debug, 
			"load_locale_map: %s: %s\n",
			filename,error_description(err)));
	return NULL;
    }

    while(EOF != (c = fgetc(f)))
	if ('\n' == c)
	    max_result++;

    DPRINT(Debug,11,(&Debug, 
		     "load_locale_map: %s, max_result=%d\n",
		     filename,max_result));

    if (!max_result) {
	fclose(f);
	return NULL;
    }
    rewind(f);

    result = safe_malloc((max_result +1) * sizeof (struct locale_map_item));

    while (result_len < max_result &&
	   (l1 = mail_gets(buf,sizeof buf, f)) > 0) {
	char * c = buf,*c1;
	
	if ('\n' == buf[l1 -1]) 
	    buf[l1 - 1] = '\0';
	else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLine,
			      "%30s: Too long line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    break;
	}
	
	while (l1-- > 0 && whitespace(buf[l1]))
	    buf[l1] = '\0';
	
	c = buf;
	while (*c && whitespace (*c)) /* skip leading whitespace */
	    c++;
	if ('#' == *c)
	    continue;
	if (!*c)
	    continue;

	c1 = strpbrk(c," \t");

	if (!c1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
			      "%30s: Bad line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    break;	    
	}
	*c1 = '\0';

	c1++;

	while (*c1 && whitespace (*c1)) /* skip leading whitespace */
	    c1++;
	if (!*c1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
			      "%30s: Bad line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    break;	    
	}

	if (0 == strcmp(c,"-")) {
	    result[result_len].match = NULL;
	    if (!load_charset_map_info(&result[result_len].charset,c1)) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeNotCharsetDeclaration,
				  "%30s: Not charset declaration: %.30s..."),
			  filename,c1);
		(*errors) ++;
		break;	    
	    }
	    result_len++;
	} else {
	    result[result_len].match = safe_strdup(c);

	    if (0 == strcmp(c1,"(unsupported)")) {
		/* Adding nameless unsupported charset */
		struct charcode_info new;
		charset_t ret;

		new.flags          = 0;
		new.map_info       = NULL;
		new.MIME_name      = NULL;
		new.charset_type   = &cs_unknown;
		new.subtype        = NULL;
		new.iso2022_info   = NULL;

		ret = add_set(new);

		DPRINT(Debug,11,(&Debug,"Adding (unsupported) =%p\n",
				 ret));

		result[result_len].charset = ret;
	    } else if (!load_charset_map_info(&result[result_len].charset,c1))
		result[result_len].charset = MIME_name_to_charset(c1,1); 
	    
	    result_len++;
	}
	DPRINT(Debug,11,(&Debug,"Added map[%d] %-15s   %-15s  (flags %X)\n",
			 result_len-1,
			 result[result_len-1].match ?
			 result[result_len-1].match : "- (define)",
			 result[result_len-1].charset->MIME_name ?
			 result[result_len-1].charset->MIME_name :
			 "(unsupported -- no MIME name)",
			 result[result_len-1].charset->flags));

    }
    result[result_len].match   = NULL;
    result[result_len].charset = NULL;

    DPRINT(Debug,11,(&Debug,"load_locale_map: %s, result_len=%d\n",
	       filename,result_len));
    
    return result;
}

void dump_locale_map(f,map)
     FILE *f; 
     struct locale_map_item *map;
{
    int i;
    struct locale_map_item *ptr;

    for (ptr = map; ptr && ptr->charset; ptr++) {
	ptr->charset->flags &= ~SET_mark;
	ptr->charset->flags &= ~SET_printed;
	DPRINT(Debug,25,(&Debug, 
			"dump_locale_map: map[%d] %p - '%s' - flags=%X\n",
			 ptr-map,ptr->charset,
			 ptr->charset->MIME_name ? ptr->charset->MIME_name :"<none>",
			 ptr->charset->flags));
    }

    /* For precompiled sets we print only alias definations ... */
    for (i = 0; i < sizeof precompiled_sets / sizeof (struct charcode_info); 
	 i++) {
	precompiled_sets[i].flags |= SET_mark;
	DPRINT(Debug,25,(&Debug, 
			 "dump_locale_map: pre[%d] %p - '%s' - flags=%X\n",
			 i,&(precompiled_sets[i]),
			 precompiled_sets[i].MIME_name ? 
			 precompiled_sets[i].MIME_name :"<none>",
			 precompiled_sets[i].flags));
    }
    for (ptr = map; ptr && ptr->charset; ptr++) {
	if (0 != (ptr->charset->flags & SET_mark) &&
	    0 != (ptr->charset->flags & SET_havealias) &&
	    0 == (ptr->charset->flags & SET_printed)) {
	    fputs("-\t",f);
	    if (!ptr->charset->MIME_name) {
		fputs("# <No MIME name> ",f);
	    } else if (name_ok(ptr->charset->MIME_name)) {
		fputs(ptr->charset->MIME_name,f);
	    } else {
		elm_fprintf(f,FRM("%Q"),ptr->charset->MIME_name);
	    }
	    dump_alias_info(f,ptr->charset);
	    ptr->charset->flags |= SET_printed;
	    fputc('\n',f);
	    DPRINT(Debug,25,(&Debug, 
			     "dump_locale_map: map[%d] %p - '%s' - flags=%X\n",
			     ptr-map,ptr->charset,
			     ptr->charset->MIME_name ? ptr->charset->MIME_name :"<none>",
			     ptr->charset->flags));
	}
    }

    for (ptr = map; ptr && ptr->charset; ptr++) {
	if (ptr->charset->MIME_name) {
	    if (0 == (ptr->charset->flags & SET_mark) &&
		0 == (ptr->charset->flags & SET_nodata) &&
		0 == (ptr->charset->flags & SET_printed)) {
		fputs("-\t",f);
		dump_map_info(f,ptr->charset);
		ptr->charset->flags |= SET_printed;
		fputc('\n',f);
	    }
	    ptr->charset->flags |= SET_mark;
	    DPRINT(Debug,25,(&Debug,"dump_locale_map: map[%d] %p - '%s' - flags=%X\n",
		       ptr-map,ptr->charset,
		       ptr->charset->MIME_name ? ptr->charset->MIME_name :"<none>",
		       ptr->charset->flags));
	}
    }

    for (ptr = map; ptr && ptr->charset; ptr++) {
	if (ptr->match) {
	    fputs(ptr->match,f);
	    fputc('\t',f);
	    if (!ptr->charset->MIME_name)
		fputs("(unsupported)",f);
	    else if ((0 != (ptr->charset->flags & SET_mark) ||
		      0 != (ptr->charset->flags & SET_nodata) ||
		      0 != (ptr->charset->flags & SET_printed)) && 
		     ptr->charset->MIME_name &&
		     NULL == strpbrk(ptr->charset->MIME_name,";\""))
		fputs(ptr->charset->MIME_name,f);
	    else
		dump_map_info(f,ptr->charset);
	    ptr->charset->flags |= SET_printed;
	    fputc('\n',f);
	    DPRINT(Debug,25,(&Debug,
			     "dump_locale_map: map[%d] %p - '%s' - flags=%X\n",
			     ptr-map,ptr->charset,
			     ptr->charset->MIME_name ? ptr->charset->MIME_name :"<none>",
			     ptr->charset->flags));

	}
    }
}

char * mime_parse_content_opts (str)
     char *str;
{
    /* A routine for parsing the options in a Content-Type: - like field.  
     * The important point here is to skip the semi-colon if it appears
     * inside of quotes.  This works sort of like strtok, except that
     * the token is already known.
     */
    static char *ptr;
    char *ret;
    int in_quote = 0;
    
    /* This is the initialization call */
    if (str) {
	DPRINT(Debug,20,(&Debug, 
			"mime_parse_content_opts: string=%s\n",
		   str));
	ptr = str;
    }    
    /* skip leading spaces */
    while (*ptr && isascii(*ptr) && isspace (*ptr))
	ptr++;

    if (*ptr == '\0') {
	DPRINT(Debug,20,(&Debug, "mime_parse_content_opts=NULL: EOS\n"));
	return NULL;
    }
    
    ret = ptr;
    while (*ptr) {
	if (*ptr == '\\' && in_quote) {		  
	    /* \ escapes next character  
	     * (not allowed outside of quotes) */
	    ptr++;
	    if (*ptr == '\0')
		break;
	} else if (*ptr == '\"') {
	    if (in_quote)
		in_quote = 0;
	    else
		in_quote = 1;
	}
	else if (! in_quote) {
	    if (*ptr == ';') {
		*ptr++ = '\0';
		DPRINT(Debug,20,(&Debug, "mime_parse_content_opts=%s (token)\n",
				 ret));
		return (ret);
	    }
	} 
	ptr++;
    }
    DPRINT(Debug,20,(&Debug, "mime_parse_content_opts=%s (token)\n",
		     ret));
    return (ret);
}

/* Used to get charset argument */
int mime_get_param (name, value, opts, size)
     char *value;
     CONST char *opts, *name;
     int size;
{
    char *c, tmp[VERY_LONG_STRING];
    int i = 0, quoted = FALSE, found = FALSE;
    
    value[0] = '\0';
    
    if (!opts) {
	DPRINT(Debug,20,(&Debug, 
			 "mime_get_param: name=\"%s\", opts=NULL\n",
			 name));
	return 0;
    }

    DPRINT(Debug,20,(&Debug, "mime_get_param: name=\"%s\", opts=\"%s\"\n",
		     name,opts));
    
    /* Make sure not to harm opts */
    strfcpy (tmp, opts, sizeof (tmp));
    rfc822_reap_comments (tmp, NULL, 0);
    
    c = tmp;
    while ((c = mime_parse_content_opts (c)) != NULL && !found) {
	char * d  = strchr(c,'=');
	char * d2 = d;
	if (!d) {
	    c = NULL;
	    continue;    /* bad paramater */
	}
	while (d2 > c && (whitespace (*(d2-1))))
	    d2--;
	*d2 = '\0';
	
	while (*c && whitespace(*c))
	    c++;
	
	if (istrcmp (c, name) == 0) {
	    found = TRUE;
	    
	    c = d+1;
	    while (*c && whitespace(*c))
		c++;
	    if (*c == '"') {
		c++;
		quoted = TRUE;
	    }
	    /* Either look for a trailing quoted (if quoted==TRUE) or a SPACE */
	    while (*c && ((quoted && *c != '"') || (!quoted && *c != ' '))) {
		if (*c == '\\' && quoted) {
		    /* \ escapes next character */
		    c++;
		    if (!*c)
			break;
		}
		if (i >= size-1)
		    break;    /* Avoid buffer overflow */
		value[i++] = *c++;
	    }
	    value[i] = '\0';
	    break;
	}
	c = NULL;
    }
    
    DPRINT(Debug,20,(&Debug, 
		     "mime_get_param: found=%d, value=%s\n",found,value));
    
    return found;
}

/* Removes comments from string */
void rfc822_reap_comments (ptr, comments, size) 
     char *ptr, *comments;
     int size;
{
    char *w_ptr = ptr, *c_ptr = comments;
    int comment_level = 0, saved_level = 0;
    int in_quote = 0;
  
    while (*ptr) {
	if (*ptr == '\\' && (in_quote || comment_level > 0)) {		  
	    /* \ escapes next character  
	     * (not allowed outside of quotes or comments) */
	    ptr++;
	    if (*ptr == '\0')
		break;
	    if (comment_level > 0 && comments) {
		if (c_ptr < comments + size - saved_level -3) {
		    *c_ptr++ = '\\';
		    *c_ptr++ = *ptr;
		}
	    }
	    if (comment_level == 0) {
		*w_ptr++ = '\\';
		*w_ptr++ = *ptr;
	    }
	    ptr++;
	    continue;
	} else if (comment_level > 0) {
	    if (*ptr == ')')
		comment_level --;
	    if (*ptr == '(')
		comment_level ++;
	    if (comments && c_ptr < comments + size - saved_level -3) {
		*c_ptr++ = *ptr;
		saved_level = comment_level;
	    }
	} else if (*ptr == '\"') {
      if (in_quote)
	  in_quote = 0;
      else
	  in_quote = 1;
	} else if (!in_quote && *ptr == '(') {
	    comment_level ++;
	    if (comments && c_ptr < comments + size - saved_level -4) {
		if (c_ptr != comments)
		    *c_ptr++ = ' ';
		*c_ptr++ = *ptr;
		saved_level = comment_level;
	    }
	    *w_ptr++ = ' ';  /* RFC 822 (STD 11) says that 
				comments represents one space */
	}
	if (comment_level == 0 && (in_quote || *ptr != ')'))
	    *w_ptr++ = *ptr;
	ptr++;
    }
    while (comments && saved_level > 0) {
	*c_ptr++ = ')';
	saved_level--;
    }
    
    if (comments)
	*c_ptr = '\0';
    *w_ptr = '\0';
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */
