static char rcsid[] = "@(#)$Id: service_list.c,v 1.14 2001/06/06 18:09:02 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.14 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *****************************************************************************/

#include "headers.h"
#include "ss_imp.h"
#include "mbx_imp.h"
#include "shared_imp.h"
#include "s_me.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"net");

#ifdef REMOTE_MBX

/* Seems that h_errno is macro on AIX */
#ifndef h_errno
extern int h_errno;
#endif

#include <errno.h>
extern int errno;

static int valid_option_type P_((struct SE_option_type *t));
static int valid_option_type(t)
     struct SE_option_type *t;
{
    int i;
    
#ifdef USE_DLOPEN
    for (i = 0; i < shared_SE_option_type_count; i++)
	if (t == shared_SE_option_types[i].T)
	    return 1;
#endif

    return 0;
}

struct service_entry * service_list   = NULL;
int                    service_count  = 0;


/* Hard coded */
static struct service_type {
    char * name;
    int    flags;
    PORTS  defport;
} SERVICE_TYPES[] = {
    { "*",       
      STFLAG_browser|STFLAG_mbox|
      STFLAG_is_imap|STFLAG_is_pop,    
      PORT_end },
    { "imap",    STFLAG_browser|STFLAG_mbox|STFLAG_is_imap,    PORT_imap4 },
    { "pop",     STFLAG_mbox|STFLAG_is_pop,                    PORT_pop3 },
    { NULL,      0,                                            PORT_end },
};


const struct service_type *       IMAP_SERVICE     =  & SERVICE_TYPES[1];
const struct service_type *       POP_SERVICE      =  & SERVICE_TYPES[1];


static void zero_service_entry P_((struct service_entry *entry,
				   const char *name,
				   const struct service_type *st,
				   int flags));
static void zero_service_entry(entry,name,st,flags)
     struct service_entry *entry;
     CONST char *name;
     CONST struct service_type *st;
     int flags;
{
    bzero((void *)entry,sizeof (*entry));

    entry->flags          = flags;           
    entry->official_name  = safe_strdup(name);
    entry->aliases_list   = 0;
    entry->aliases_count  = 0;
    entry->addr_list      = NULL;
    entry->addr_count     = 0;

    entry->addr_name_list  = NULL;
    entry->addr_name_count = 0;

    entry->service        = st;
    /* ??? */
    entry->port_list      = NULL;
    entry->port_count     = 0;

    entry->option_list    = NULL;
    entry->option_count   = 0;    
}

static void add_alias_to_entry P_((struct service_entry *entry,
				   const char *name));
static void add_alias_to_entry(entry,name)
     struct service_entry *entry;
     const char *name;
{
    entry->aliases_list = 
	safe_realloc(entry->aliases_list,
		     (entry->aliases_count+1) * 
		     sizeof (entry->aliases_list[0]));
    entry->aliases_list[entry->aliases_count++]
	= safe_strdup(name);
}


static struct service_entry *malloc_service_entry P_((const char *name,
						      const struct 
						      service_type *st));
static struct service_entry *malloc_service_entry(name,st)
     CONST char * name;
     CONST struct service_type *st;
{
    struct service_entry *ret = safe_malloc (sizeof (struct service_entry));

    /* Free this */
    zero_service_entry(ret,name,st,SE_temporary);
   
    return ret;
}

static struct service_entry *scan_list P_((const char *hostname,
					   int flag));
static struct service_entry *scan_list(hostname,flag)
     CONST char *hostname;
     int flag;
{
    int i;
    
    for (i = 0; i < service_count; i++) {
	if (0 != (flag & service_list[i].service->flags)) {
	    int j;
	    if (0 == istrcmp(hostname,service_list[i].official_name)) {
		return &(service_list[i]);	    
	    }
	    
	    for (j = 0; j < service_list[i].aliases_count; j++) {
		if (0 == istrcmp(hostname,service_list[i].aliases_list[j])) {
		    return &(service_list[i]);
		}
	    }
	}	
    }
    return NULL;
}

static struct hostent * lookup_name P_((const char *hostname,int silent));
static struct hostent * lookup_name(hostname,silent)
     CONST char *hostname;
     int silent;
{
    struct hostent *he = NULL;

    lib_transient(CATGETS(elm_msg_cat, MeSet,MeLookingUp,
			  "Looking up %s ..."),
		  hostname);
	
    he = gethostbyname(hostname);
	
    if (!he) {
	DPRINT(Debug,9,(&Debug, 
			"lookup_name: %s: h_errno=%d\n",
			hostname,h_errno));
	    
	/* We are silent about errors if cached
	   data exists
	*/
	if (!silent) {
	    switch(h_errno) {
	    case HOST_NOT_FOUND:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeNoRemoteHost,
				  "No remote mailbox server host: %s"),
			  hostname);		
		break;
	    case NO_ADDRESS:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeNoAddress,
				  "Name %s have not IP-address"),
			  hostname);
		break;
	    case TRY_AGAIN:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeTemporary,
				  "Address for %s is not yet found..."),
			  hostname);
		break;			  
	    default:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeNameError,
				  "Failed to get address for %s"),
			  hostname);
		break;
	    }
	}
    } else {
	lib_transient(CATGETS(elm_msg_cat, MeSet,MeLookingUpOK,
			      "Looking up %s ... OK"),
		      hostname);

    }

    return he;
}

static void add_addr_to_entry P_((struct service_entry *entry,
				  SOCKADDR *result));
static void add_addr_to_entry(entry,result)
     struct service_entry *entry;
     SOCKADDR *result;
{
    entry->addr_list = 
	safe_realloc(entry->addr_list,
		     (entry->addr_count+1) * 
		     sizeof (entry->addr_list[0]));
    bzero((void *)&(entry->addr_list[entry->addr_count]),
	  sizeof (entry->addr_list[entry->addr_count]));

    entry->addr_list[entry->addr_count++] = *result;
}

static void fill_address P_((struct service_entry *ret,
			     struct hostent *he));
static void fill_address(ret,he)
     struct service_entry *ret;
     struct hostent *he;
{   
    if (he->h_addr_list) {
	int x;
	for (x = 0; he->h_addr_list[x]; x++) {
	    SOCKADDR X;

	    switch(he->h_addrtype) {
#ifdef I_NETINET_IN
	    case AF_INET:
		X.sin.sin_family = he->h_addrtype;
	        X.sin.sin_port = htons(PORT_end);    /* No port given */

		if (sizeof(X.sin.sin_addr)  != he->h_length) {
		    lib_error(FRM("%s: Bad IP-addr length %d (should be %d)"),
			      he->h_name,he->h_length,
			      sizeof(X.sin.sin_addr));
		    return;
		}
		memcpy(&(X.sin.sin_addr), he->h_addr_list[x], he->h_length);
		add_addr_to_entry(ret,&X);
		break;
#endif			    
	    default:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeUnsupportedAddrType,
				  "Name %s have odd type address"),
			  he->h_name);
		/* NOTE: We do not know on what compoents arbitary
		         sockect address is constructed so we can't
			 fill it
		*/
		return;
	    }
	}
    }
}


struct service_entry * give_service_entry(hostname,flag)
     CONST char *hostname;
     int flag;
{
    struct service_entry *ret;       
    int have_port = 0;   /* set if addresses have port set */

    DPRINT(Debug,9,(&Debug, 
		    "give_service_entry(name=\"%s\",flags=%d)\n",
		    hostname,flag));

    ret = scan_list(hostname,flag);
    /* If on service entry do not have ip addresses
       they needed to be retrieved. Notice that we
       retrieve new entries every time (alternatively
       we should keep track of expiry times).
       However all other data is cached (also addresses if
       them is generated from names given on configuration data)
    */
    if (ret)
	hostname = ret->official_name;

    if (ret && (0 != (ret->flags & SE_given_name_addr) &&
		ret->addr_count == 0 ||
		0 != (ret->flags & SE_rescan_name_addr))) {
	int x;

	/* Clear data on case of rescan */
	if (ret->addr_list) {
	    free(ret->addr_list);
	    ret->addr_list = NULL;
	    ret->addr_count = 0;
	}
	ret->flags   &= ~SE_rescan_name_addr;

	for (x = 0; x < ret->addr_name_count; x++) {
	    struct hostent *he  = 
		lookup_name(ret->addr_name_list[x],0);
	    
	    if (he)
		fill_address(ret,he);
	    else
		ret->flags |= SE_rescan_name_addr;
	}
    }

    if (!ret || 
	0 == (ret->flags & SE_given_addr) &&
	0 == (ret->flags & SE_given_name_addr)) {

	struct hostent *he    = lookup_name(hostname,
					    /* Silent: */
					    ret && ret->addr_count > 0);

	if (he) {  	  
	    /* Now try found entry again ... */
	    if (!ret) { 
		ret = scan_list(he->h_name,flag);
		
		if (!ret && he->h_aliases) {
		    int j;
		    for (j = 0; he->h_aliases[j] && !ret; j++)
			ret = scan_list(he->h_aliases[j],flag);
		}
	    
		/* we need to add hostname to aliases because
		   it was not on there (otherwise entry
		   should have found earlier)
		*/
		if (ret) 
		    add_alias_to_entry(ret,hostname);
	    }
	    
	    if (!ret) {
		struct service_type *st;

		for (st = &SERVICE_TYPES[0]; st->name; st++) {
		    if (flag == (flag &st->flags))
			break;
		}

		if (!st->name)
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "give_service_entry",
			  "Service type do not found", 0);

		ret = malloc_service_entry(hostname,st);

	    }
	
	    if (he->h_aliases) {  /* Add missing aliases */
		int x;
		for (x = 0; he->h_aliases[x]; x++) {
		    int j;
		    for (j=0; j < ret->aliases_count; j++)
			if (0 == istrcmp(he->h_aliases[x],
					 ret->aliases_list[j]))
			    break;
		    if (j >= ret->aliases_count) 
			add_alias_to_entry(ret,he->h_aliases[x]);
		}
	    }

	    if (he->h_addr_list) {
		/* Clear old data */
	    	if (ret->addr_list) {
		    free(ret->addr_list);
		    ret->addr_list = NULL;
		    ret->addr_count = 0;
		}

		fill_address(ret,he);
	    }		
	}    
    }

    if (ret && ret->addr_count > 0) {
	int x;

	for (x = 0; x < ret->addr_count; x++) {
	    switch(ret->addr_list[x].sa.sa_family) {
#ifdef I_NETINET_IN
	    case AF_INET:
		if (ret->addr_list[x].sin.sin_port == htons(PORT_end))
		    have_port = 0;
		break;
#endif
	    default:
		have_port = 0;
		break;
	    }
	}
    }


    if (ret && ret->port_count == 0 && !have_port) {
	PORTS r = PORT_end;
	
	/* builtin definations here */
	if ( PORT_end == ret->service->defport) {
	    
	    /* Caller will define portlist */
	    if (ret->port_list)
	 	free(ret->port_list);
	    ret->port_list = NULL;
	    
	} else 
	    r = ret->service->defport;
	
	if (r != PORT_end) {
	    ret->port_list = safe_realloc(ret->port_list,
					  2 * sizeof (ret->port_list[0]));
	    ret->port_list[0] = r;
	    ret->port_list[1] = PORT_end;
	    ret->port_count  = 1;
	}
    }

    if (ret) {
	DPRINT(Debug,9,(&Debug, 
			"give_service_entry=%p: type=%p (%s)%s\n",
			ret,ret->service,ret->service->name,
			ret->flags & SE_temporary ? ", temporary" : ""));
    } else {
	DPRINT(Debug,9,(&Debug, 
			"give_service_entry=NULL\n"));
    }

    return ret;
}

void free_temporary_service_entry(Y)
     struct service_entry **Y;
{
    if (!(*Y))
	return;

    if ((*Y)->flags  & SE_temporary) {
	int x;

	DPRINT(Debug,12,(&Debug, 
			 "free_temporary_service_entry: Freeing entry %p\n",
			 *Y));
	if ((*Y)->official_name)
	    free((*Y)->official_name);
	(*Y)->official_name = NULL;

	for (x = 0; x < (*Y)->aliases_count; x++) {
	    if ((*Y)->aliases_list[x])
		free((*Y)->aliases_list[x]);
	    (*Y)->aliases_list[x] = NULL;
	}
	if ((*Y)->aliases_list)
	    free((*Y)->aliases_list);
	(*Y)->aliases_list  = NULL;
	(*Y)->aliases_count = 0;

	if ((*Y)->addr_list)
	    free((*Y)->addr_list);
	(*Y)->addr_list = NULL;
	(*Y)->addr_count = 0;

	for (x = 0; x < (*Y)->addr_name_count; x++) {
	    if ((*Y)->addr_name_list[x])
		free((*Y)->addr_name_list[x]);
	    (*Y)->addr_name_list[x] = NULL;
	}
	if ((*Y)->addr_name_list)
	    free((*Y)->addr_name_list);
	(*Y)->addr_name_list = NULL;
	(*Y)->addr_name_count = 0;

	(*Y)->service = NULL;

	if ((*Y)->port_list)
	    free((*Y)->port_list);
	(*Y)->port_list  = NULL;
	(*Y)->port_count = 0;

	for (x = 0; x < (*Y)->option_count; x++) {
	    if (!valid_option_type((*Y)->option_list[x].type))
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "free_temporary_service_entry",
		      "Bad option type on service list",0);
	    (*Y)->option_list[x].type->
		free_options(& ((*Y)->option_list[x]));	    

	    if ((*Y)->option_list[x].prefix)
		free((*Y)->option_list[x].prefix);
	    (*Y)->option_list[x].prefix = NULL;
	}
	if ((*Y)->option_list)
	    free((*Y)->option_list);
	(*Y)->option_list = NULL;
	(*Y)->option_count = 0;
    }
    /* If not temporary, we assume that this entry is on list
       and therefore that does not produce dangling pointer
    */
    (*Y) = NULL;
}

int parse_service_entries (filename,system, errors)
     CONST char *filename; 
     int system;
     int *errors;
{
    /*   hostname   service        options               */
    int max_count = 0;
    int count = 0;
    FILE * f = fopen(filename,"r");
    char buf[LONG_STRING];
    int c, l1;

    if (!f) {
	int err = errno;
	DPRINT(Debug,2,(&Debug, 
			"parse_service_entries=0: %s: %s\n",
			filename,error_description(err)));
	return 0;
    }
    
    while(EOF != (c = fgetc(f)))
	if ('\n' == c)
	    max_count++;

    DPRINT(Debug,9,(&Debug, 
		    "parse_service_entries: %s, max_count=%d\n",
		    filename,max_count));
    
    if (!max_count) {
	fclose(f);
	return 0;
    }
    rewind(f);

    service_list = safe_realloc(service_list,
				(service_count + max_count) *
				sizeof (service_list[0]));

    while (count < max_count && 
	   (l1 = mail_gets(buf,sizeof buf, f)) > 0) {
	char *c,*c1, *options;
	struct service_type *st;

	if ('\n' == buf[l1 -1]) 
	    buf[l1 - 1] = '\0';
	else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLine,
			      "%30s: Too long line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    break;
	}

	while (l1-- > 0 && whitespace(buf[l1]))
	    buf[l1] = '\0';

	c = buf;
	while (*c && whitespace (*c)) /* skip leading whitespace */
	    c++;
	if ('#' == *c)
	    continue;
	if (!*c)
	    continue;

	c1 = strpbrk(c," \t");
	if (!c1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
			      "%30s: Bad line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    break;	    
	}
	*c1 = '\0';
	c1++;

	while (*c1 && whitespace (*c1)) /* skip leading whitespace */
	    c1++;
	if (!*c1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
			      "%30s: Bad line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    break;	    
	}
	
	options = strpbrk(c1," \t;");
	if (options) {
	    *options = '\0';
	    options++;
	}
	
	for (st = &SERVICE_TYPES[0]; st->name; st++) 
	    if (0 == istrcmp(c1,st->name))
		break;

	if (!st->name) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeServiceType,
			      "%30s: Bad service type %s"),
		      filename,c1);
	    (*errors) ++;
	    break;	    	    
	}

	zero_service_entry( & (service_list[service_count+count]),
			    c  /* hostname */,
			    st /* service */,
			    0  /* This is NOT temporary entry */);

	if (system)
	    service_list[service_count+count].flags |= SE_system;
		
	if (options) {
	    char *opt;

	    for (opt = mime_parse_content_opts(options); 
		 opt; 
		 opt = mime_parse_content_opts(NULL)) {
		char *val = NULL;          /* malloced by dequote_opt */ 
		char * q = strchr(opt,'=');
		char * zopt = NULL;

		DPRINT(Debug,65,(&Debug, 
				 "mime_parse_content_opts gives: %s\n",
				 opt));
		
		if (q) 
		    *q++ = '\0';		    

		if (0 == strcmp(opt,"alias")) {
		    if (!q || !q[0]) {
			lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionReqValue,
					  "%.30s: Option %s requires value"),
				  filename,opt);
			(*errors) ++;
			continue;
		    }

		    val = dequote_opt(q,strlen(q));		
		    add_alias_to_entry(& (service_list[service_count+count]),
				       val);
		    free(val);  /* add_alias_to_entry strdup's value */
		    service_list[service_count+count].flags |= SE_given_alias;

		} else if (0 == strcmp(opt,"addr")) {
		    char *q1 = NULL;
		    if (!q || !q[0]) {
			lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionReqValue,
					  "%.30s: Option %s requires value"),
				  filename,opt);
			(*errors) ++;
			continue;
		    }
		    if ('"' != q[0])
			q1 = strchr(q,':');

		    if (!q1) { /* name given */
			service_list[service_count+count].addr_name_list =
			    safe_realloc(service_list[service_count+count].
					 addr_name_list,
					 (service_list[service_count+count].
					  addr_name_count+1) *
					 sizeof (service_list[service_count+
							     count].
						 addr_name_list[0]));
			service_list[service_count+count].
			    addr_name_list[service_list[service_count+
						       count].
                                          addr_name_count++] =
			    dequote_opt(q,strlen(q));  /* Malloces result */
			service_list[service_count+count].flags |= 
			    SE_given_name_addr;
		    } else {  /* Addr? given */
			SOCKADDR result;
#ifndef USE_INET_ATON
			long tmp;
#endif
			char * port;

			*q1++ = '\0';
			port = qstrpbrk(q1,":/");
			if (port)
			    *port++ = '\0';
			val = dequote_opt(q1,strlen(q1));		

			/* inet_aton() is better but not available on
			   some systems so we need use inet_addr()
			*/

#ifdef I_NETINET_IN			
			if (0 == strcmp(q,"ip") &&
#ifdef USE_INET_ATON
			    inet_aton(q1,&result.sin.sin_addr)
#else
			    -1 != (tmp = inet_addr(q1)) 
#endif
			    ) {
			    int p = PORT_end;   /* No port given */

#ifndef USE_INET_ATON
			    memcpy(&(result.sin.sin_addr.s_addr),
				   &tmp,
				   sizeof (result.sin.sin_addr.s_addr));
#endif

			    result.sin.sin_family = AF_INET;
			    if (port)
				p = atoi(port);
			    result.sin.sin_port = htons(p);
			} else
#endif			
			    {
				lib_error(CATGETS(elm_msg_cat, MeSet,
						  MeBadAddrSpec,
						  "%.30s: Bad address specification %s:%s"),
					  filename,q,q1);	
				(*errors) ++;
				free(val); val= NULL;
				continue;			    
			    }	

			add_addr_to_entry(& (service_list[service_count+
							 count]),
					  &result);
			free(val); val = NULL;
			service_list[service_count+count].flags |= 
			    SE_given_addr;
		    }
		} else if (0 == strcmp(opt,"port")) {
		    if (!q || !q[0]) {
			lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionReqValue,
					  "%.30s: Option %s requires value"),
				  filename,opt);
			(*errors) ++;
			continue;
		    }

		    service_list[service_count+count].port_list = 
			safe_realloc(service_list[service_count+count].
				     port_list,
				     sizeof (service_list[service_count+count].
					     port_list[0]) *
				     (2 + service_list[service_count+count].
				      port_count));
		    service_list[service_count+count].
			port_list[service_list[service_count+count].
				 port_count] = atoi(q);
		    service_list[service_count+count].
			port_list[service_list[service_count+count].
				 port_count+1] = PORT_end;

		    if (service_list[service_count+count].
			port_list[service_list[service_count+count].
				 port_count] == PORT_end) {
			lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionPortBad,
					  "%.30s: Option port=%d unsupported"),
				  filename,
				  service_list[service_count+count].
				  port_list[service_list[service_count+count].
					   port_count]);
			(*errors) ++;
			continue;
		    }
		    service_list[service_count+count].flags |= 
			SE_given_port;

#ifdef USE_DLOPEN
		} else if (NULL != (zopt = strchr(opt,':'))) {
		    int idx = -1;
		    int x;

		    *zopt++ = '\0';

		    for (x = 0; x < service_list[service_count+count].
			     option_count; x++) {
			if (0 == strcmp(opt,service_list[service_count+count].
			    option_list[x].prefix))
			    idx = x;
		    }
		    if (-1 == idx) {
			struct SE_option_type * Y = get_option_type(opt);
			if (!Y)
			    continue;

			service_list[service_count+count].option_list =
			    safe_realloc(service_list[service_count+count].
					 option_list,
					 (service_list[service_count+count].
					  option_count+1) *
					 sizeof (service_list[service_count+
							     count].
						 option_list[0]));
			bzero( &(service_list[service_count+count].
				 option_list[service_list[service_count+count].
					    option_count]),
			       sizeof (service_list[service_count+
						   count].
				       option_list[0]));
			service_list[service_count+count].
			    option_list[service_list[service_count+count].
				       option_count].type = Y;
			service_list[service_count+count].
			    option_list[service_list[service_count+count].
				       option_count].prefix = 
			    safe_strdup(opt);
			service_list[service_count+count].
			    option_list[service_list[service_count+count].
				       option_count].value = NULL;

			Y->zero_options(& (service_list[service_count+count].
					   option_list[service_list
						      [service_count+count].
						      option_count]));
			idx = service_list[service_count+count].
			    option_count++;
		    }

		    if (!valid_option_type(service_list[service_count+count].
					   option_list[idx].type))
			panic("CONNECTION PANIC",__FILE__,__LINE__,
			      "parse_service_entries",
			      "Bad option type on service list",0);

		    if (!service_list[service_count+count].
			option_list[idx].type ->
			parse_on_option(& (service_list[service_count+count].
					   option_list[idx]),
					zopt,q)) {
			(*errors) ++;
			continue;
		    }					   
#endif
		} else {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionUnspported,
				      "%.30s: Option %s unsupported"),
			      filename,opt);
		    (*errors) ++;
		    continue;
		}
	    }

	}

	count++;	    
    }

    service_count += count;
    DPRINT(Debug,9,(&Debug, 		    
		    "parse_service_entries: %s, %d parsed (%d total)\n",
		    filename,count,service_count));
    fclose(f);
    return 1;
}

static int name_ok P_((CONST char *name));
static int name_ok(name)
     CONST char *name;
{
    if ('\0' == *name)
	return 0;

    return strlen(name) == strspn(name,"abcdefghijklmnopqrstuvxyz.-");
}


void dump_service_entries(f,system)
     FILE *f; 
     int system;
{
    int x;

    for (x = 0; x < service_count; x++) {
	int idx;
	char * sep = "\t";

	if (system &&
	    (service_list[x].flags & SE_system) == 0)
	    continue;
	if (!system &&
	    (service_list[x].flags & SE_system) != 0)
	    continue;

	fputs(service_list[x].official_name,f);
	putc('\t',f);
	fputs(service_list[x].service->name,f);
	
	if (service_list[x].flags & SE_given_alias) {
	    int y;
	    for (y = 0; y < service_list[x].aliases_count; y++) {

		if (name_ok(service_list[x].aliases_list[y]))
		    elm_fprintf(f,FRM("%salias=%s"),
				sep,service_list[x].aliases_list[y]);
		else
		    elm_fprintf(f,FRM("%salias=%Q"),
				sep,service_list[x].aliases_list[y]);
		sep = "; ";
	    }
	}

	if (service_list[x].flags & SE_given_name_addr) {
	    int y;
	    for (y = 0; y < service_list[x].addr_name_count; y++) {
		if (name_ok(service_list[x].addr_name_list[y]))
		    elm_fprintf(f,FRM("%saddr=%s"),
				sep,service_list[x].addr_name_list[y]);
		else
		    elm_fprintf(f,FRM("%salias=%Q"),
				sep,service_list[x].addr_name_list[y]);
		sep = "; ";
	    }
	}
	
	if (service_list[x].flags & SE_given_addr) {
	    int y;
	    for (y = 0; y < service_list[x].addr_count; y++) {
		SOCKADDR X = service_list[x].addr_list[y];
		switch (X.sa.sa_family) {
		    char * z;
#ifdef I_NETINET_IN
		case AF_INET:
		    z = inet_ntoa(X.sin.sin_addr);
		    elm_fprintf(f,FRM("%salias=ip:%Q"),
				sep,z);
		    if (ntohs(X.sin.sin_port) != PORT_end)
			elm_fprintf(f,FRM(":%d"),
				    ntohs(X.sin.sin_port));
		    
		    sep = "; ";
		    break;
#endif
		default:
		    /* none */
		    break;
		}		
	    }
	}

	if (service_list[x].flags & SE_given_port) {
	    int y;
	    for (y = 0; y < service_list[x].port_count; y++) {
		elm_fprintf(f,FRM("%sport=%d"),
			    sep,service_list[x].port_list[y]);
		sep = "; ";
	    }
	}

	for (idx = 0; idx < service_list[x].option_count; idx++) {
	    char * val;

	    if (!valid_option_type(service_list[x].option_list[idx].type))
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "dump_service_entries",
		      "Bad option type on service list",0);
	    
	    val = service_list[x].option_list[idx].type->
		give_values(& (service_list[x].option_list[idx]),
			    service_list[x].option_list[idx].prefix);

	    if (val) {
		fputs(sep,f);
		fputs(val,f);
		free(val); val = NULL;
	    }
	}
	putc('\n',f);
    }
    fflush(f);
}
#endif

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */
