static char rcsid[] = "@(#)$Id: curses.c,v 1.24 2001/06/17 14:01:16 hurtta Exp $";

/*****************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.24 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 ******************************************************************************
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *
 *****************************************************************************/

/**  This library gives programs the ability to easily access the
     termcap information and write screen oriented and raw input
     programs.  The routines can be called as needed, except that
     to use the cursor / screen routines there must be a call to
     InitScreen() first.  

**/

/** NOTE THE ADDITION OF: the #ifndef ELM stuff around routines that
    we don't use.  This is for code size and compile time speed...
**/

#include "headers.h"
#include <errno.h>
#include "me.h"

DEBUG_VAR(Debug,__FILE__,"ui");

#ifdef I_CURSES
#include <curses.h>             /* <term.h> requires ... */
#endif /* I_CURSES */
#ifdef I_TERM
#include <term.h>               /* Defination for tgetent.h and so on ... */
#undef lines                    /* ... But undefine lines macro */
#undef columns                  /* ... But undefine columns macro */
#endif /* I_TERM */

#ifdef TERMIOS
# include <termios.h>
# ifndef sun
#  include <sys/ioctl.h>	/* for TIOCGWINSZ */
# endif
#else
# ifdef TERMIO
#  include <termio.h>
# else
#  include <sgtty.h>
# endif
#endif

#include    <errno.h>

#ifdef PTEM
#  include <sys/stream.h>
#  include <sys/ptem.h>
#endif

#ifdef SHORTNAMES
# define _clearinverse	_clrinv
# define _cleartoeoln	_clrtoeoln
# define _cleartoeos	_clr2eos
# define _transmit_off	xmit_off
# define _transmit_on	xmit_on
#endif

#ifdef TERMIOS
struct termios _raw_tty,
	       _original_tty;
#define	ttgetattr(fd,where)	tcgetattr((fd),(where))
#define	ttsetattr(fd,where)	tcsetattr((fd),TCSADRAIN,(where))
#else	/*TERMIOS*/
# ifdef TERMIO
struct termio _raw_tty, 
              _original_tty;
#define	ttgetattr(fd,where)	ioctl((fd),TCGETA,(where))
#define	ttsetattr(fd,where)	ioctl((fd),TCSETAW,(where))
# else
struct tty_modes {
    struct sgttyb sgttyb;
    struct tchars tchars;
}  _raw_tty, _original_tty;
static int ttgetattr P_((int,struct tty_modes *));  /* Prototype */
static int ttsetattr P_((int,struct tty_modes *)); /* Prototype */
# endif	/*TERMIO*/
#endif	/*TERMIOS*/

static int _inraw = 0;                  /* are we IN rawmode?    */

#define DEFAULT_LINES_ON_TERMINAL	24
#define DEFAULT_COLUMNS_ON_TERMINAL	80

static int _memory_locked = 0;		/* are we IN memlock??   */

static VOLATILE int _line  = -1;       /* initialize to "trash" */
static VOLATILE int _col   = -1;

static int _intransmit = -1;	        /* are we transmitting keys? */

static
char *_clearscreen = NULL, *_moveto = NULL, *_up = NULL, *_down = NULL, 
  *_right = NULL, *_left = NULL, *_setbold = NULL, *_clearbold = NULL, 
  *_setunderline = NULL, *_clearunderline = NULL, *_sethalfbright = NULL, 
  *_clearhalfbright = NULL, *_setinverse = NULL, *_clearinverse = NULL,
  *_cleartoeoln = NULL, *_cleartoeos = NULL, *_transmit_on = NULL, 
  *_transmit_off = NULL, *_set_memlock = NULL, *_clear_memlock = NULL, 
  *_start_termcap = NULL, *_end_termcap = NULL, *_bell = NULL,

  *_key_up = NULL, *_key_down = NULL, *_key_left = NULL, *_key_right = NULL,
  *_key_pageup = NULL, *_key_pagedown = NULL, *_key_home = NULL, 
  *_key_help = NULL, *_key_find = NULL;

static int _lines, _columns, _automargin, _eatnewlineglitch;
int tabspacing;
static int tabexpand = 0;          /* Is terminal driver expanding tabs? */

static char _terminal[1024];              /* Storage for terminal entry */
static char _capabilities[1024];           /* String for cursor motion */

static char *ptr = _capabilities;	/* for buffering         */

static int  cursor_control    = 0;
static int  need_moveabsolute = 0;

char  *tgetstr(),     		       /* Get termcap capability */
      *tgoto();				/* and the goto stuff    */

static int outchar P_((int c));

static char terminal_type[40] = "";

static charset_t target_display_charset = NULL;
static struct charset_state * last_state = NULL;
static screen_info_p  last_display_state = NULL;

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}

static int terminal_fd = 1;   /* Standard output, later terminal ... */

int InitScreen()
{
	/* Set up all this fun stuff: returns zero if all okay, or;
        -1 indicating no terminal name associated with this shell,
        -2..-n  No termcap for this terminal type known
   */

	int  tgetent(),      /* get termcap entry */
	     err;
	char *termenv;
	int fd;

	if ((termenv = getenv("TERM")) == NULL) return(-1);

	strfcpy(terminal_type, termenv, sizeof terminal_type);

	if ((err = tgetent(_terminal, terminal_type)) != 1)
		return(err-2);
	
	_line  =  0;		/* where are we right now?? */
	_col   =  0;		/* assume zero, zero...     */

	fd = open("/dev/tty",O_RDWR);
	if (fd < 0) {
	    DPRINT(Debug,4, (&Debug, "Opening of /dev/tty failed\n"));
	} else {
	    FlushBuffer();
	    terminal_fd = fd;
	    DPRINT(Debug,4,(&Debug,
			    "/dev/tty is fd %d\n",terminal_fd));
	}
	    

	/* load in all those pesky values */
	_clearscreen       = tgetstr("cl", &ptr);
	_moveto            = tgetstr("cm", &ptr);
	_up                = tgetstr("up", &ptr);
	_down              = tgetstr("do", &ptr);
	_right             = tgetstr("nd", &ptr);
	_left              = tgetstr("bc", &ptr);
	_bell              = tgetstr("bl", &ptr);

	_setbold           = tgetstr("so", &ptr);
	_clearbold         = tgetstr("se", &ptr);
	_setunderline      = tgetstr("us", &ptr);
	_clearunderline    = tgetstr("ue", &ptr);
	_setinverse        = tgetstr("so", &ptr);
	_clearinverse      = tgetstr("se", &ptr);
	_sethalfbright     = tgetstr("hs", &ptr);
	_clearhalfbright   = tgetstr("he", &ptr);
	_cleartoeoln       = tgetstr("ce", &ptr);
	_cleartoeos        = tgetstr("cd", &ptr);
	_lines	      	   = tgetnum("li");
	_columns	   = tgetnum("co");
	tabspacing	   = ((tabspacing=tgetnum("it"))<= 0 ? 8 : tabspacing);
	_automargin	   = tgetflag("am");
	_eatnewlineglitch   = tgetflag("xn");
	_transmit_on	   = tgetstr("ks", &ptr);
	_transmit_off      = tgetstr("ke", &ptr);
	_set_memlock	   = tgetstr("ml", &ptr);
	_clear_memlock	   = tgetstr("mu", &ptr);
	_start_termcap	   = tgetstr("ti", &ptr);
	_end_termcap	   = tgetstr("te", &ptr);

	_key_up            = tgetstr("ku", &ptr);
	_key_down          = tgetstr("kd", &ptr);
	_key_left          = tgetstr("kl", &ptr);
	_key_right         = tgetstr("kr", &ptr);
	_key_pageup        = tgetstr("kP", &ptr);
	_key_pagedown      = tgetstr("kN", &ptr);
	_key_home          = tgetstr("kh", &ptr);
	_key_help          = tgetstr("%1", &ptr);
	_key_find          = tgetstr("@0", &ptr);

	if (_transmit_on && _transmit_off && _key_up && _key_down) {
	  cursor_control = TRUE;
	}

	if (_setinverse && _clearinverse) {
	  has_highlighting = TRUE;
	}

	if (!_left) {
		_left = "\b";
	}

	return 0;
}

charset_t * give_display_charsets(storage,max)
     charset_t *storage;
     int max;
{
    int n = 0,i;
    max--;

    storage[n++] = display_charset;

    if (system_charset != display_charset &&
	n < max)
	storage[n++] = system_charset;

    if (allow_charset_switching)
	terminal_can_switch(terminal_type,storage,&n,max);
    storage[n] = NULL;

    SIGDPRINT(Debug,4,(&Debug, 
		    "Possible sets:"));
    for (i = 0; i < n; i++) {
	SIGDPRINT(Debug,4,(&Debug, 
			" %s%c",
			storage[i]->MIME_name ? storage[i]->MIME_name : 
			"<no MIME name>",
			i < n-1 ? ',' : '\n'));
    }

    return storage;
}

int set_display_charset(set,silent)
     charset_t set; 
     int silent;
{   
    if (!terminal_can_switch_to(terminal_type,set,silent))
	return 0;
    
    target_display_charset = set;
    switch_display_charset(1);
    return 1;
}

void switch_display_charset(to_display)
     int to_display;
{
    charset_t t;
    char * str;
    
    if (!target_display_charset)
	target_display_charset = display_charset;
    

    t = to_display ? target_display_charset : system_charset;
    
    if (!t)
	return;

    str = terminal_switch_to(terminal_type,t,&last_display_state);

    if (str) {
	int i;

	SIGDPRINT(Debug,4,(&Debug, 
			"Setting display charset to %s (",
			t->MIME_name ? t->MIME_name : "<no MIME name>"));
	for (i = 0; str[i]; i++) {
	    SIGDPRINT(Debug,4,(&Debug, 
			    "%s%02X",
			    i ? " " : "", str[i]));
	    outchar(str[i]);
	}
	SIGDPRINT(Debug,10,(&Debug,
		  ") [terminal %s]\n",terminal_type));
	
	display_charset = t;
	free(str);

    } else {
	SIGDPRINT(Debug,10,(&Debug,
			    "No way to set terminal %s to charset %s\n",
			    terminal_type,
			    t->MIME_name ? t->MIME_name : "<no MIME name>"));
		  
    }
}

void transmit_functions(newstate)
     int newstate;
{
	/** turn function key transmission to ON | OFF **/

	if (newstate != _intransmit) {
		_intransmit = newstate;
		if (newstate == ON)
		  tputs(_transmit_on, 1, outchar);
		else 
		  tputs(_transmit_off, 1, outchar);
	       FlushBuffer();      /* clear the output buffer */
	}
}

/****** now into the 'meat' of the routines...the cursor stuff ******/

void ScreenSize(lines, columns)
     int *lines, *columns;
{
	/** returns the number of lines and columns on the display. **/

#ifdef TIOCGWINSZ
	struct winsize w;

	if (ioctl(1,TIOCGWINSZ,&w) != -1) {
		if (w.ws_row > 0)
			_lines = w.ws_row;
		if (w.ws_col > 0)
			_columns = w.ws_col;
	}
#endif

	if (_lines == 0) _lines = DEFAULT_LINES_ON_TERMINAL;
	if (_columns == 0) _columns = DEFAULT_COLUMNS_ON_TERMINAL;


	/* WARNING: elm_LINES and elm_COLUMNS are inconsistent!
	 *
	 * elm_LINES    == number of lines in screen -1
	 * elm_COLUMNS  == number of rows in screen
	 *
	 *
	 * Check code in MoveCursor!
	 *
	 * row          == 0 .. elm_LINES
	 * col          == 0 .. elm_COLUMNS-1
	 *
	 *
	 * Who was this smart programmer!!!!!!!!!!!!!!!!!!!!!
	 *
	 *                         - K E H <hurtta@ozone.FMI.FI>
	 */

	*lines = _lines - 1;	     
	*columns = _columns;
}

void SetXYLocation(x,y)
     int x,y;
{
	/* declare where the cursor is on the screen - useful after using
	 * a function that moves cursor in predictable fasion but doesn't
	 * set the static x and y variables used in this source file -
	 * e.g. getpass().
	 */

	_line = x;
	_col = y;
}

void GetXYLocation(x,y)
     int *x,*y;
{
	/* return the current cursor location on the screen */

	*x = _line;
	*y = _col;
}

int ClearScreen()
{
	/* clear the screen: returns -1 if not capable */

	_line = 0;	/* clear leaves us at top... */
	_col  = 0;

	redraw_screen = 0;
	_intransmit = -1;   /* Re-set state */

	if (!_clearscreen)
		return(-1);

	tputs(_clearscreen, 1, outchar);
	FlushBuffer();      /* clear the output buffer */
	return(0);
}

static void moveabsolute P_((int col, int row));

static int CursorUp(n)
     int n;
{
  /** move the cursor up 'n' lines **/
  /** Calling function must check that _up is not null before calling **/

  if (need_moveabsolute)
      moveabsolute(_col, _line);

  _line = (_line-n > 0? _line - n: 0);	/* up 'n' lines... */

  while (n-- > 0)
      tputs(_up, 1, outchar);

  return(0);
}

static int CursorDown(n)
     int n;
{
  /** move the cursor down 'n' lines **/
  /** Caller must check that _down is not null before calling **/

  if (need_moveabsolute)
      moveabsolute(_col, _line);;

  _line = (_line+n <= elm_LINES? _line + n: elm_LINES);    /* down 'n' lines... */

  while (n-- > 0)
      tputs(_down, 1, outchar);
  
  return(0);
}


static int CursorLeft(n)
     int n;
{
    /** move the cursor 'n' characters to the left **/
    /** Caller must check that _left is not null before calling **/
    
    if (need_moveabsolute)
	moveabsolute(_col, _line);
    
    _col = (_col - n> 0? _col - n: 0);	/* left 'n' chars... */
  
    while (n-- > 0)
	tputs(_left, 1, outchar);
        
    return(0);
}


static int CursorRight(n)
     int n;
{
    /** move the cursor 'n' characters to the right (nondestructive) **/
    /** Caller must check that _right is not null before calling **/
    
    if (need_moveabsolute)
	moveabsolute(_col, _line);
    
    _col = (_col+n < elm_COLUMNS? _col + n: elm_COLUMNS);	
    /* right 'n' chars... */
    
    while (n-- > 0)
	tputs(_right, 1, outchar);
    
    return(0);
}

static void moveabsolute(col, row)
     int col, row;
{

    char *stuff, *tgoto();

    if (need_moveabsolute) {
	SIGDPRINT(Debug,4,(&Debug, 			   
			   "Curses: moveabsolute: Syncronizing cursos position (col=%d,row=%d)\n",
			   col,row));
    }
    
    stuff = tgoto(_moveto, col, row);
    tputs(stuff, 1, outchar);
     
    if(need_moveabsolute)
       FlushBuffer();
       
    need_moveabsolute = 0;
}

int MoveCursor(row, col)
     int row, col;
{
    /** move cursor to the specified row column on the screen.
	0,0 is the top left! **/
    
    int scrollafter = 0;
    
    /* we don't want to change "rows" or we'll mangle scrolling... */
    
    if (need_moveabsolute)
	moveabsolute(_col, _line);;
    
    if (col < 0)
	col = 0;
    if (col >= elm_COLUMNS)
	col = elm_COLUMNS - 1;
    if (row < 0)
	row = 0;
    if (row > elm_LINES) {
	if (col == 0)
	    scrollafter = row - elm_LINES;
	row = elm_LINES;
    }
    
    if (!_moveto)
	return(-1);

    if (row == _line) {
	if (col == _col)
	    return(0);				/* already there! */
	
	else if (abs(col - _col) < 5) {	/* within 5 spaces... */
	    if (col > _col && _right)
		CursorRight(col - _col);
	    else if (col < _col &&  _left)
		CursorLeft(_col - col);
	    else
		moveabsolute(col, row);
	}
	else 		/* move along to the new x,y loc */
	    moveabsolute(col, row);
    }
    else if (_line == row-1 && col == 0) {
	if (_col != 0)
	    outchar('\r');
	outchar('\n');      
    }
    else if (col == _col && abs(row - _line) < 5) {
	if (row < _line && _up)
	    CursorUp(_line - row);
	else if (row > _line && _down)
	    CursorDown(row - _line);
	else
	    moveabsolute(col, row);
    }
    else 
	moveabsolute(col, row);
    
    _line = row;	/* to ensure we're really there... */
    _col  = col;
    
    if (scrollafter) {
	outchar('\r');
	while (scrollafter--) {
	    outchar('\n');
	}
    }

    FlushBuffer();	
    return(0);
}

void CarriageReturn()
{
    /** move the cursor to the beginning of the current line **/
    Writechar('\r');
}

void NewLine()
{
    /** move the cursor to the beginning of the next line **/

    Writechar('\r');
    Writechar('\n');
    FlushBuffer();	
}

int StartBold()
{
    /** start boldface/standout mode **/
    
    if (!_setbold)
	return(-1);
    
    tputs(_setbold, 1, outchar);
    
    return(0);
}

int EndBold()
{
    /** compliment of startbold **/
    
    if (!_clearbold)
	return(-1);
    
    tputs(_clearbold, 1, outchar);
    
    return(0);
}

#ifndef ELM

int StartUnderline()
{
    /** start underline mode **/
    
    if (!_setunderline)
	return(-1);
    
    tputs(_setunderline, 1, outchar);
       
    return(0);
}


int EndUnderline()
{
    /** the compliment of start underline mode **/
    
    if (!_clearunderline)
	return(-1);
    
    tputs(_clearunderline, 1, outchar);
    
    return(0);
}

int StartHalfbright()
{
    /** start half intensity mode **/
    
    if (!_sethalfbright)
	return(-1);

    tputs(_sethalfbright, 1, outchar);
    
    return(0);
}

int EndHalfbright()
{
    /** compliment of starthalfbright **/
    
    if (!_clearhalfbright)
	return(-1);

    tputs(_clearhalfbright, 1, outchar);
    
    return(0);
}

int StartInverse()
{
    /** set inverse video mode **/
    
    if (!_setinverse)
	return(-1);
    
    tputs(_setinverse, 1, outchar);
    
    return(0);
}

int EndInverse()
{
    /** compliment of startinverse **/

    if (!_clearinverse)
	return(-1);
    
    tputs(_clearinverse, 1, outchar);
    
    return(0);
}

int
HasMemlock()
{
    /** returns TRUE iff memory locking is available (a terminal
	feature that allows a specified portion of the screen to
	be "locked" & not cleared/scrolled... **/
    
    return ( _set_memlock && _clear_memlock );
}

static int _old_LINES;

int
StartMemlock()
{
    /** mark the current line as the "last" line of the portion to 
	be memory locked (always relative to the top line of the
	screen) Note that this will alter LINES so that it knows
	the top is locked.  This means that (plus) the program 
	will scroll nicely but (minus) End memlock MUST be called
	whenever we leave the locked-memory part of the program! **/
    
    if (! _set_memlock)
	return(-1);
    
    if (! _memory_locked) {
	
	_old_LINES = elm_LINES;
	elm_LINES -= _line;		/* we can't use this for scrolling */
	
	tputs(_set_memlock, 1, outchar);
	
	_memory_locked = TRUE;
    }

    return(0);
}

int
EndMemlock()
{
    /** Clear the locked memory condition...  **/
    
    if (! _set_memlock)
	return(-1);
    
    if (_memory_locked) {
	elm_LINES = _old_LINES;		/* back to old setting */
	
	tputs(_clear_memlock, 1, outchar);
       
	_memory_locked = FALSE;
    }
    return(0);
}

#endif /* ndef ELM */

VOLATILE static int wrappedlastchar = 0;

static char outchar_write_buffer[2048];
VOLATILE static char * buffer_ptr = &(outchar_write_buffer[0]);

/* Note that there is several race conditions -- losing of characters possible 
   (this may be called from signal handler)
*/
void FlushBuffer P_((void))
{
    int len; 

    while (0 < (len = buffer_ptr-outchar_write_buffer)) {
	int L = write(terminal_fd,outchar_write_buffer,len);
	int err = errno;

	if (L < 0 && 
	    ((err != EINTR)
#ifdef	EAGAIN
	     && (err != EAGAIN)
#endif
#ifdef	EWOULDBLOCK
	     && (err != EWOULDBLOCK)
#endif
	     )) {
	    SIGDPRINT(Debug,4,(&Debug, 			       
			       "FlushBuffer: Write failed errno=%d\n",err));
	    break;
	}	   

	if (L > 0) {
	    len = buffer_ptr-outchar_write_buffer;

	    len -= L;

	    DPRINT(Debug,9,(&Debug,
			    "FlushBuffer: %d bytes writted to %d, %d left\n",
			    L,terminal_fd,len));

	    if (len > 0)
		memmove(outchar_write_buffer,outchar_write_buffer+L,len);
	    if (len >= 0)
		buffer_ptr = outchar_write_buffer + len;
	    else
		buffer_ptr = outchar_write_buffer;

	} else {
	    DPRINT(Debug,2,(&Debug,
			    "FlushBuffer: Looping...\n"));
	}
    }   
}

/* Note that there is several race conditions -- losing of characters possible 
   (this may be called from signal handler)
*/
static int outchar(c)
     int c;
{

    /** output the given character.  From tputs... **/
    /** Note: this CANNOT be a macro!              **/

    if (buffer_ptr >= outchar_write_buffer + sizeof (outchar_write_buffer))
	FlushBuffer();
   
    if (buffer_ptr < outchar_write_buffer + sizeof (outchar_write_buffer)) {
	VOLATILE char *p = buffer_ptr++;
	
	*p = c;
	
	SIGDPRINT(Debug,40,(&Debug, 
			    "outchar [%d] = %02X\n",
			    p-outchar_write_buffer,*p));

	return (unsigned char)c;
    }
    SIGDPRINT(Debug,2,(&Debug,
		       "outchar: Buffer full!\n"));
    return EOF;
}

/* called with value of -1 if just preprosessing needed 
   (ie need_moveabsolute) 
*/
static int write_control P_((int ch));
static int write_control(ch)
     int ch;
{
    int justwrapped, nt;

    if (need_moveabsolute)
	moveabsolute(_col, _line);

    justwrapped = 0;

    switch (ch) {
    /* if return, just go to left column. */
    case 0x0D: /* CR */
	if (wrappedlastchar)
	    justwrapped = 1;                /* preserve wrap flag */
	else {
	    outchar('\r');
	    _col = 0;
	}
	break;
    
	/* if newline and terminal just did a newline without our asking,
	 * do nothing, else output a newline and increment the line count */
    case 0x0A: /* LF */
	if (!wrappedlastchar) {
	    outchar('\n');
	    if (_line < elm_LINES)
		++_line;
	}
	break;
    
    /* if backspace, move back  one space  if not already in column 0 */
    case 0x08:  /* BS */
	if (_col != 0) {
	    tputs(_left, 1, outchar);
	    _col--;
	}
	else if (_line > 0) {
	    _col = elm_COLUMNS - 1;
	    _line--;
	    moveabsolute (_col, _line);
	}
	/* else BACKSPACE does nothing */
	break;

	/* if bell, ring the bell but don't advance the column */
    case 0x07: /* BEL */
	if (_bell)
	    tputs(_bell,1,outchar);
	else
	    outchar(ch);
	break;

    /* if a tab, output it */
    case 0x09: /* HT */
	/* If terminal driver is expanding tabs, don't trust it ... */
	if (!tabexpand)
	    outchar(ch);
	if((nt=next_tab(_col+1)) > prev_tab(elm_COLUMNS))
	    _col = elm_COLUMNS-1;
	else
	    _col = nt-1;
	if (tabexpand)
	    moveabsolute (_col, _line);
	break;
    default:    /* Is 'printable' character (or -1) */
	return 1;
    }
    
    wrappedlastchar = justwrapped;
    return 0;   /* Need no printing */
}

static void increment_position P_((int len));
static void increment_position(len)
     int len;
{
    int justwrapped;

    justwrapped = 0;
    
    if (_col + len > elm_COLUMNS) {
	_col = 0;
	_line = 0;
	moveabsolute (_col, _line);
	panic("CURSES PANIC",__FILE__,__LINE__,"increment_position",
	      "column position overflow",0);
    }

    /* if we only had one column left, simulate automargins if
     * the terminal doesn't have them */
    if (_col + len == elm_COLUMNS) {
	if (!_automargin || _eatnewlineglitch) {
	    outchar('\r');
	    outchar('\n');
	}
	if (_line < elm_LINES)
	    ++_line;
	_col = 0;
	justwrapped = 1;
    }
    /* if we are here this means we have no interference from the
     * right margin - just increment the column position. */
    else {
	_col += len;
    }

    wrappedlastchar = justwrapped;
}

/* Called by signal handler ... */

void WriteRaw(str)
     CONST char *str;
{
     CONST char * p;

     for (p = str; *p; p++) {
	 int ch = *p;
	 
	 /** write a character to the current screen location. **/
	 int need_print = 1;

	 ch &= 0xFF;

	 need_print = write_control(ch);

	 if (need_print) {
	     outchar(ch);
	     increment_position(1);
	 }
     }

     FlushBuffer();
}



/* This assumes one byte charset */
void Writechar(ch)
     int ch;
{
    /** write a character to the current screen location. **/
    int need_print = 1;

    ch &= 0xFF;

    /* ASCII assumed */
    need_print = write_control(ch);

    if (need_print) {
#ifndef ASCII_CTYPE
	if (display_charset == system_charset) {
	    /* if some kind of non-printable character change to a '?' */

	    if(!isprint((unsigned char)ch))
		ch = '?';
	} else 
#endif
	    {
		/* If system (= locale) charset is not display charset
		 * need ask from charset system is that character
		 * printable -- complex and expensive!
		 */
		struct charset_state *st = new_state(display_charset);
		add_streambyte_to_state(st,ch);
		
		if (!state_ready(st) || !state_printable(st))
		    ch = '?';		
		free_state(&st);
	    }
	outchar(ch);
	increment_position(1);
    }
}

static void PutLineS P_((struct string *S));
static void PutLineS(S)
     struct string *S;
{
    int idx = 0;
    struct string *buffer = NULL;
    charset_t stored_display = target_display_charset;
    int sw = -1;

    if (S->string_type == display_charset)
	buffer = S;
    else if (target_display_charset == display_charset &&
	     S->string_type == system_charset) {
	sw = 1;  /* Arrange return to back to display charset */
	buffer = S;
	switch_display_charset(0); /* Print using system charset */
    } else if (allow_charset_switching && 
	       0 != (charset_properties(S->string_type) & CS_printable) &&
	       set_display_charset(S->string_type,1)) {
	sw = 2;  /* Arrange return to correct display charset */
	buffer = S;
    } else
	buffer = convert_string(display_charset,S,0);

    DPRINT(Debug,40,(&Debug, 
		    "PutLineS: S=%S cs=%s  -- using %s\n",
		     S,
		     S->string_type->MIME_name ? 
		     S->string_type->MIME_name : 
		     "<no MIME name>",
		     buffer->string_type->MIME_name ? 
		     buffer->string_type->MIME_name : "<no MIME name>"));
    
    while (idx < string_len(buffer)) {
	char * s, *s1;
	uint16 ch;
	int old_idx = idx;
	int need_print;
	int len = 1;

	ch = give_unicode_from_string(buffer,idx);

	/* ASCII assumed */
	if (ch <= 31) 
	    need_print = write_control(ch);
	else 
	    need_print = write_control(-1);  /* HACK for position of cursor */
	if (!need_print) {
	    DPRINT(Debug,40,(&Debug,
		       "PutLineS: [%d] unicode=%d (control)\n",
		       idx,ch));
	    idx++;
	    continue;   /* character 'printed' */
	}

	while (len < elm_COLUMNS-_col &&
	       len + idx < string_len(buffer) &&
	       give_unicode_from_string(buffer,idx + len) > 31)
	    len++;

	s = us2s(streamclip_from_string(buffer,&idx,len,last_display_state));

	DPRINT(Debug,10,(&Debug,
			 "PutLineS: [%d-%d] stream=%s\n",
			 old_idx,idx-1,s));

	for (s1 = s; *s1; s1++)
	    outchar(*s1);

	increment_position(idx-old_idx);
	free(s);
    }

    /* Free if converted */
    if (buffer != S)
	free_string(&buffer);
    
    if (sw == 2)
	set_display_charset(stored_display,0);
    else if (sw != -1) {
	switch_display_charset(sw);
    }
}

/*VARARGS2*/

void Write_to_screen (
#if ANSI_C
		      CONST char *format, CONST char *msg, ...
#else
		      format, msg, va_alist
#endif
		      )
#if !ANSI_C
     CONST char *format; 
     CONST char *msg;
     va_dcl
#endif
{
    va_list vl;
    struct string *text;

    Va_start(vl, msg);           /* defined in defs.h */
    
    /** This routine writes to the screen at the current location.
	when done, it increments lines & columns accordingly by
	looking for "\n" sequences... **/

    text = elm_smessage(0,format,msg,vl);
    PutLineS(text);
    free_string(&text);
    
    va_end(vl);
    
}

void PutLine0(x, y, line)
     int x,y;
     CONST char *line;
{
    /** Write a zero argument line at location x,y **/

    MoveCursor(x,y);
    while(*line)
	Writechar(*line++);
    
}

void PutLineX(
#if ANSI_C
     int x, int y, const char *format, const char *line, ...
#else
     x, y, format, line, va_alist
#endif
     )
#if !ANSI_C
     int x; 
     int y; 
     CONST char *format; 
     CONST char *line; 
     va_dcl
#endif
{
    va_list vl;
    struct string *text;

    Va_start(vl, line);           /* defined in defs.h */
    
    MoveCursor(x,y);
    
    text = elm_smessage(0,format,line,vl);
    PutLineS(text);
    free_string(&text);

    va_end(vl);
    
}

int CleartoEOLN()
{
    /** clear to end of line **/
    
    if (need_moveabsolute)
	moveabsolute(_col, _line);
    
    if (!_cleartoeoln)
	return(-1);
    
    tputs(_cleartoeoln, 1, outchar);
    
    return(0);
}

int CleartoEOS()
{
    /** clear to end of screen **/
    
    if (need_moveabsolute)
	moveabsolute(_col, _line);
    
    if (!_cleartoeos)
	return(-1);
    
    tputs(_cleartoeos, 1, outchar);
    
    return(0);
}


int RawState()
{
    /** returns either 1 or 0, for ON or OFF **/
    
    FlushBuffer();

    return( _inraw );
}

#ifdef ultrix
force_raw()
{
	(void) ttsetattr(terminal_fd,&_original_tty);
	_inraw = 0;
	Raw(ON);
}
#endif

#if POLL_METHOD	  
static void clear_input_buffer P_((void));
#endif

void Raw(state)
     int state;
{
	int do_tite     = (state & NO_TITE) == 0;
	int do_charset  = (state & NO_CHARSET) == 0;

	state = state & ~NO_TITE;

	SIGDPRINT(Debug,4,(&Debug, 
			   "curses: Raw: state=%d do_tite=%d do_charset=%d\n",
			   state,do_tite,do_charset));

	/** state is either ON or OFF, as indicated by call **/

	if (state == OFF && _inraw) {
	    SIGDPRINT(Debug,4,(&Debug,
				"curses: Raw: Setting Raw state OFF\n"));;

#if POLL_METHOD	  
	  clear_action(terminal_fd);
	  clear_input_buffer();
#endif

	  if (cursor_control)
	    transmit_functions(OFF);

	  if (do_charset)
	      switch_display_charset(0);
	
	  if (use_tite && _end_termcap && do_tite) {
	      tputs(_end_termcap, 1, outchar);	    
	  }
	  FlushBuffer();
	  (void) ttsetattr(terminal_fd,&_original_tty);
	  _inraw = 0;
	}
	else if (state == ON && ! _inraw) {

	    SIGDPRINT(Debug,4,(&Debug,
				"curses: Raw: Setting Raw state ON\n"));;
	    
	    (void) ttgetattr(terminal_fd, &_original_tty);
	    (void) ttgetattr(terminal_fd, &_raw_tty);    /** again! **/
	    
#if !defined(TERMIO) && !defined(TERMIOS)
	    _raw_tty.sg_flags &= ~(ECHO);	/* echo off */
	    _raw_tty.sg_flags |= CBREAK;	/* raw on    */

#else
	  _raw_tty.c_lflag &= ~(ICANON | ECHO);	/* noecho raw mode        */

	  _raw_tty.c_cc[VMIN] = '\01';	/* minimum # of chars to queue    */
	  _raw_tty.c_cc[VTIME] = '\0';	/* minimum time to wait for input */


#if defined(OXTABS)
	  /* Some systems use OXTABS bit */
	  if (_raw_tty.c_oflag & OXTABS) {
	      SIGDPRINT(Debug,4,(&Debug,
				 "curses: [OXTABS] Terminal driver is expanding tabs...\n"));
	      tabexpand = 1;
	  } else {
	      if (tabexpand) 	      
		  SIGDPRINT(Debug,4,(&Debug,
				     "curses: [OXTABS] Terminal driver isn't expanding tabs...\n"));
	      tabexpand = 0;
	  }
#endif /* defined(OXTABS) */

#if defined(TABDLY) && !defined(OXTABS)
	  /* Some systems expands tab when TABDLY is XTABS */
	  if ((_raw_tty.c_oflag & TABDLY) == 
#ifdef XTABS
	      XTABS
#else
	      TAB3
#endif
	      ) {
	      SIGDPRINT(Debug,4,(&Debug,
				 "curses: [TABDLY] Terminal driver is expanding tabs...\n"));
	      tabexpand = 1;
	  } else {
	      if (tabexpand) 
		  SIGDPRINT(Debug,4,(&Debug,
				     "curses: [TABDLY] Terminal driver isn't expanding tabs...\n"));
	      tabexpand = 0;
	  }
#endif /* defined(TABDLY) && !defined(OXTABS) */

#if !defined(TABDLY) && !defined(OXTABS)

	  /* If _POSIX_SOURCE is defined then OXTABS or TABDLY are not 
	   * defined -- so print warning anyway
	   */
	  SIGDPRINT(Debug,4,(&Debug,
			     "curses: No information is terminal driver expanding tabs!\n"));	  
#endif /* !defined(TABDLY) && !defined(OXTABS) */

#endif
	  (void) ttsetattr(terminal_fd, &_raw_tty);
	  if (use_tite && _start_termcap && do_tite) 
	      tputs(_start_termcap, 1, outchar);
	  _intransmit = -1; /* state unclear */
	  _inraw = 1;

	  if (do_charset)
	      switch_display_charset(1);

	  need_moveabsolute = 1;
	}
}

int raw_off_called() {
    return need_moveabsolute;
}

#if !defined(TERMIO) && !defined(TERMIOS)
static int ttgetattr(fd, where)
     int fd;
     struct tty_modes *where;
{
  if (ioctl(fd, TIOCGETP, &where->sgttyb) < 0)
    return(-1);
  if (ioctl(fd, TIOCGETC, &where->tchars) < 0)
    return(-1);
  return(0);
}

static int ttsetattr(fd, where)
     int fd;
     struct tty_modes *where;
{
  if (ioctl(fd, TIOCSETP, &where->sgttyb) < 0)
    return(-1);
  if (ioctl(fd, TIOCSETC, &where->tchars) < 0)
    return(-1);
  return(0);
}
#endif

#if POLL_METHOD
static unsigned char *read_buffer   = NULL;
static int read_buffer_len = 0;
static int last_read_error = 0;

static int background_read P_((int fd, void *data));
static int background_read(fd,data)
     int fd; 
     void *data;
{
    int    result;
    unsigned char   ch;
    
    result = read(fd, &ch, 1);

    if (result < 0) {
	int err = errno;

	SIGDPRINT(Debug,4,(&Debug, 
			   "background_read: errno = %d [%s]\n",err,
			   error_description(err)));

	if((err == EINTR)
#ifdef	EAGAIN
	   || (err == EAGAIN)
#endif
#ifdef	EWOULDBLOCK
	   || (err == EWOULDBLOCK)
#endif
	   ) {
	    return 1;  /* Retry */
	}
	
	last_read_error = err;
	return 0;
    }
    
    if (0 == result) {
	SIGDPRINT(Debug,4,(&Debug,
			   "background_read: Got zero bytes...\n"));
	last_read_error = -1;
	return 0;
    }

    read_buffer = safe_realloc(read_buffer,read_buffer_len+1);
    SIGDPRINT(Debug,20,(&Debug,
	       "background_read(fd=%d)=1 [%d]",fd,read_buffer_len));
    SIGDPRINT(Debug,50,(&Debug,"=%d",ch));
    SIGDPRINT(Debug,20,(&Debug,"\n"));
    
    read_buffer[read_buffer_len++] = ch; 
    last_read_error = 0;

    return 1;
}

static void clear_input_buffer() 
{
    SIGDPRINT(Debug,4,(&Debug, 
		       "Resetting background_read buffer (%d bytes)...\n",
		       read_buffer_len));
    read_buffer_len = 0;
    last_read_error = 0;
}

int error_sleep(seconds) 
     int seconds;
{
    int ret;
    if (read_buffer_len > 0) {
	SIGDPRINT(Debug,5,(&Debug, 
			   "error_sleep(%d)=1  -- sleep skipped\n",
			   seconds));
	return 1;
    }
    SIGDPRINT(Debug,5,(&Debug, 
		       "error_sleep(%d) ...\n",seconds));
    ret = wait_for_action_or_timeout(background_read,seconds);
    DPRINT(Debug,5,(&Debug, 
		    "error_sleep=%d\n",ret));
    return ret;
}
#endif


struct charset_state * ReadCh2(flags)
     int flags;
{
    int redraw      = (flags & READCH_MASK);
    int cursorkeys  = (flags & READCH_CURSOR)    != 0;
    int nocursor    = (flags & READCH_NOCURSOR)  != 0 && cursor_control;
    int term_char   = (flags & READCH_term_char) != 0;
    int x,y;

    unsigned char input_buffer[20];

    /*
     *	read a character with Raw mode set!
     *
     *	EAGAIN & EWOULDBLOCK are recognized just in case
     *	O_NONBLOCK happens to be in effect.
     */

    /* This is static array so we can initialize it in here ...
     */
    static struct {
	char ** CONST str;
	CONST int result;
	int maybe;
    } keytable[] =
      { { &_key_up,       UP_MARK,       0 },
	{ &_key_down,     DOWN_MARK,     0 },
	{ &_key_left,     LEFT_MARK,     0 },
	{ &_key_right,    RIGHT_MARK,    0 },

	{ &_key_pageup,   PAGEUP_MARK,   0 },
	{ &_key_pagedown, PAGEDOWN_MARK, 0 },

	{ &_key_home,     HOME_MARK,     0 },
	{ &_key_help,     HELP_MARK,     0 },
	{ &_key_find,     FIND_MARK,     0 },
	{ NULL,           0,             0 }
      };
    
    int read_p,found_key;

#if POLL_METHOD
    change_action(terminal_fd,0,background_read,no_action_routine,
		  no_action_routine,
		  NULL);
#endif    

    GetXYLocation(&x,&y); /* Background actions may print error messages .. */

    if (!last_state) {
	last_state = new_state(system_charset);
	DPRINT(Debug,4,(&Debug,
		       "Keyboard input charset is %s\n",
		       last_state->charset->MIME_name ?
		       last_state->charset->MIME_name :
		       "<no MIME name>"));
    }

    if (state_ready(last_state)) {
	DPRINT(Debug,4,(&Debug, 
			"ReadCh2: starting reading new character (set %s)\n",
			last_state->charset->MIME_name ? 
			last_state->charset->MIME_name :
			"<no MIME name>"));
	reset_state(last_state,0);
    }
    last_state->caller_flags = 0;

 reinit_ReadChar:
    read_p = 0;
    found_key = 0;
    
    FlushBuffer();

    if (redraw && !RawState()) { /* Check that we have in 'raw' mode */
	DPRINT(Debug,4,(&Debug,
			"ReadCh2: Going to Raw mode\n"));
	Raw(ON);
	ClearScreen();

	reset_state(last_state,1);
	last_state->caller_flags = redraw;

	return last_state;
    }

    if (redraw && redraw_screen) {
	DPRINT(Debug,4,(&Debug,
			"ReadCh2: Pending redraw...\n"));
	ClearScreen();

	reset_state(last_state,1);
	last_state->caller_flags = redraw;

	return last_state;
    }
  
    if ((_intransmit != ON || redraw_screen) &&
	cursorkeys && _transmit_on) {
	DPRINT(Debug,4,(&Debug,
			"ReadCh2: Enabling cursor keys...\n"));
	transmit_functions(ON);
    }

    if ((_intransmit != OFF || redraw_screen) &&
	nocursor && _transmit_off) {
	DPRINT(Debug,4,(&Debug,
			"ReadCh2: Disabling cursor keys...\n"));
	transmit_functions(OFF);
    }
    
    if (cursorkeys) {
	int i;
	DPRINT(Debug,8,(&Debug,
			"ReadCh2: Available function keys:"));

	for (i = 0; keytable[i].str != 0; i++) {
	    char * CONST str = *(keytable[i].str);
	    if(str && str[0] != '\0') {
		keytable[i].maybe = 1;  /* Initially every function key is possible */
		SIGDPRINT(Debug,8,(&Debug,
				   " [%d] %d",i,keytable[i].result));	     
	    }
	}      
	DPRINT(Debug,4,(&Debug,
			"\n"));
    }

    while (found_key == 0) {
	unsigned char   ch;
#if POLL_METHOD
	if (read_buffer_len < 1 && last_read_error == 0) {
	    errno= 0;
	    if(!wait_for_action(background_read)) {
		if (errno != 0) {
		    DPRINT(Debug,4,(&Debug,
				    "ReadCh2: wait_for_action got error %d (%s)\n",
				    errno, error_description(errno)));
		    /* Return error: */
		    if ((redraw && redraw_screen)
			|| InGetPrompt   /* GetPrompt wants to see errors! */
			) {
			DPRINT(Debug,4,(&Debug,
					"       ... return error to caller\n"));
			found_key = -1;
			continue;
		    }
		    if (errno == EINTR
#ifdef	EAGAIN
			|| (errno == EAGAIN)
#endif
#ifdef	EWOULDBLOCK
			|| (errno == EWOULDBLOCK)
#endif
			) {
			DPRINT(Debug,4,(&Debug,
					"       ... retry (ignored)\n"));
		    } else
			found_key = -1;
		    continue;		    
		}
		DPRINT(Debug,4,(&Debug,
				"ReadCh2: interrupted (ignored)\n"));
		continue;
	    }
	}
	if (read_buffer_len > 0) {
	    ch = read_buffer[0];
	    if (--read_buffer_len > 0)
		memmove(read_buffer,read_buffer+1,read_buffer_len);
	} else if (last_read_error != 0) {
	    if (-1 == last_read_error) {
		DPRINT(Debug,4,(&Debug,
				"ReadCh2: background_read got zero bytes...\n"));
		found_key = -1;
		continue;
	    }
	    /* Return error: */
	    if ((redraw && redraw_screen)
		|| InGetPrompt   /* GetPrompt wants to see errors! */
		) {
		found_key = -1;
		continue;
	    }
	    if (last_read_error != 0) {
		errno = last_read_error;
		DPRINT(Debug,4,(&Debug,
				"ReadCh2: background_read got error %d (%s)\n",
				last_read_error,
				error_description(last_read_error)));
		found_key = -1;
	    }
	    continue;         /* continue or quit */	    
	} else {
	    DPRINT(Debug,1,(&Debug,
			    "ReadCh2: SOFTWARE ERROR !!\n"));
	    break;
	}
#else
	int    result;

	result = read(terminal_fd, &ch, 1);
	
	if (result < 0) {
	    int err = errno;
	    DPRINT(Debug,4,(&Debug,
			    "ReadCh2: errno = %d [%s]\n",err,
			    error_description(err)));
	    
	     /* Return error: */
	    if (redraw && redraw_screen
		|| InGetPrompt   /* GetPrompt wants to see errors! */
		) {
		found_key = -1;
		continue;
	    }
	    if((errno == EINTR)
#ifdef	EAGAIN
	       || (errno == EAGAIN)
#endif
#ifdef	EWOULDBLOCK
	       || (errno == EWOULDBLOCK)
#endif
	       ) {
		continue;  /* Retry */
	    }
	    break;
	}
	if (0 == result) {
	    DPRINT(Debug,4,(&Debug,
			    "ReadCh2: Got zero bytes...\n"));
	    found_key = -1;
	    continue;
	}
#endif

#ifdef USE_DLOPEN
	{
	    union xxx_rand {
		int ch;
		char bytes[sizeof (int)];
	    } A;
	    A.ch = ch;

	    seed_rand_bits(A.bytes, sizeof A, 
			   3 /* Assume 3 bits per character ... */);
	}
#endif

	

	DPRINT(Debug,50,(&Debug,
			 "ReadCh2: Looking char %d (read_p=%d)\n",ch,read_p));

	if (term_char) {
	    if (backspace == ch) 
		found_key = TERMCH_backspace;
	    if (kill_line == ch)
		found_key = TERMCH_kill_line;
	    if (word_erase  == ch)
		found_key = TERMCH_word_erase;
	    if (interrupt_char  == ch)
		found_key = TERMCH_interrupt_char;
	    if (reprint_char    == ch)
		found_key = TERMCH_reprint_char;
	    if (eof_char        == ch)
		found_key = TERMCH_eof_char;
	    if (found_key) {
		DPRINT(Debug,4,(&Debug,
				"ReadCh2: found termchar = %d (char=%02X)\n",
				found_key,ch));
		/* Soft reset state */
		reset_state(last_state,0); 
		break;
	    }
	}

	if (cursorkeys) {
	    int match = 0;
	    int i;
	    for (i = 0; keytable[i].str != NULL; i++) {
		if (keytable[i].maybe) {
		    unsigned char * CONST str = 
			(unsigned char *) *(keytable[i].str);
		    if (str[read_p] == ch) {
			match++;
			if (str[read_p+1] == '\0') {
			    found_key = keytable[i].result;
			    DPRINT(Debug,4,(&Debug,
					    "ReadCh2: Found function key = %d (keytable = %d, read_p =%d)\n",
					    found_key,i,read_p));

			    /* Soft reset state */
			    reset_state(last_state,0); 
			}
		    } else {
			keytable[i].maybe = 0;
		    }
		}
	    }

	    if (read_p < sizeof input_buffer -1) {
		input_buffer[read_p] = ch;
		input_buffer[read_p+1] = '\0';
	    }

	    if (match == 0) {    /* Not in keytable */
		if (read_p == 0) 
		    found_key = ch;  /* Normal key */
		else {
		    int i;
		    
		    /* But maybe escape sequence is valid state change ? */
		    	
		    for (i = 0; 
			 i <= read_p && i < sizeof input_buffer -1; 
			 i++) {

			if (state_ready(last_state)) {
			    DPRINT(Debug,4,(&Debug,
					    "ReadCh2: Unsupporting -- function keys and state change sequences overlap? (OR BAD SEQUENCE) \n"));
			    goto BAD_sequence;
			}
			
			if (!add_streambyte_to_state(last_state,
						     input_buffer[i]))
			    goto BAD_sequence;
			
		    }
		    
		    if (state_ready(last_state))
			goto got_key;
		    DPRINT(Debug,4,(&Debug,
				    "ReadCh2: need more bytes for character...\n"));
		    goto reinit_ReadChar;	    


		BAD_sequence:

		    /* Soft reset state */
		    reset_state(last_state,0); 

		    DPRINT(Debug,4,(&Debug,
				    "ReadCh2: Bad escape sequence; ch = %d, read_p = %d\n",
				    ch,read_p));
#ifdef DEBUG
		    DPRINT(Debug,4,(&Debug,
				    "ReadCh2: Sequence was:"));
		    for (i = 0; i <= read_p && i < sizeof input_buffer -1; i++) {
			if (isascii(input_buffer[i]) && isprint(input_buffer[i])) {
			    DPRINT(Debug,4,(&Debug," %c (0x%02X)", 
					    input_buffer[i],input_buffer[i]));
			} else {
			    DPRINT(Debug,4,(&Debug," 0x%02X", input_buffer[i]));
			}
		    }
		    if (read_p > sizeof input_buffer -1)
			DPRINT(Debug,4,(&Debug," ..."));
		    DPRINT(Debug,4,(&Debug,"\n"));
#endif

		    /* Ring a bell */
		    Writechar('\007');
		    goto reinit_ReadChar;
		}
	    } else
		read_p++;
	} else
	    found_key = ch;
	
    }

    if (found_key <= 0 && redraw && redraw_screen) {
	DPRINT(Debug,4,(&Debug,
			"ReadCh2: Redraw...\n"));
	if(!RawState()) {  /* Check that we have in 'raw' mode */
	    DPRINT(Debug,4,(&Debug,
			    "ReadCh2: Going to Raw mode\n"));
	    Raw(ON);      
	}
	ClearScreen();

	reset_state(last_state,1);
	last_state->caller_flags = redraw;

	return last_state;
    }

    MoveCursor(x,y);  
    DPRINT(Debug,50,(&Debug,
		     "ReadCh2: found_key=%d\n",found_key));
    
    if (found_key < 0) {
	reset_state(last_state,1);
	DPRINT(Debug,4,(&Debug,
			"ReadCh2=NULL (state)\n")); 
	return NULL;
    } else {
	if (found_key >= 256) {
	    reset_state(last_state,1);
	    last_state->caller_flags = found_key;
	} else {
	    last_state->caller_flags = 0;
	    if (!add_streambyte_to_state(last_state,found_key)) {
		/* Ring a bell */
		Writechar('\007');
		DPRINT(Debug,4,(&Debug,
				"ReadCh2: bad sequence...\n"));
		reset_state(last_state,1);
	    }
	}
    }
    
    if (!state_ready(last_state) &&
	!last_state->caller_flags) {
	DPRINT(Debug,4,(&Debug,
			"ReadCh2: need more bytes for character...\n"));
	goto reinit_ReadChar;
    }

 got_key:
    DPRINT(Debug,4,(&Debug,
		    "ReadCh2=%p (state): caller_flags=%d, ready=%d\n",
		    last_state,last_state->caller_flags,
		    state_ready(last_state))); 
    return last_state;
}


int ReadCh(flags)
     int flags;
{
    int ret = 0;
    
    do {
	struct charset_state *ch = ReadCh2(flags);

	if (!ch) {
	    ret = EOF;
	    DPRINT(Debug,4,(&Debug, 
			    "ReadCh=EOF (errno=%d)\n",errno));
	} else if (ch -> caller_flags) {
	    ret = ch -> caller_flags;
	    DPRINT(Debug,4,(&Debug, 
			    "ReadCh=%d (flags)\n",ret));
	} else
	    ret =  state_is_onebyte(ch);
	if (!ret) {
	    DPRINT(Debug,4,(&Debug,
			    "ReadCh: Not a one byte character or zero byte!"));
	    /* Ring a bell */
	    Writechar('\007');
	} else {
	    DPRINT(Debug,4,(&Debug,
			    "ReadCh=%d (one byte character)\n",
			    ret));
	}
    } while (0 == ret);
    return ret;
}




/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */


