/*****************************************************************************
 *  ENTROPY - emerging network to reduce orwellian potency yield
 *
 *  Copyright (C) 2002 Juergen Buchmueller <pullmoll@stop1984.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *	$Id: crypt2.c,v 1.4 2005/07/12 23:12:28 pullmoll Exp $
 *****************************************************************************/
#include "osd.h"
#include "crypt2.h"
#if	CRYPT2_TEST
#define	xmalloc(size) malloc(size)
#define xcalloc(num,size) calloc(num,size)
#define xfree(ptr) if (NULL != ptr) { free(ptr); ptr = NULL; }
#else
#include "memalloc.h"
#endif
#include "logger.h"

#define	CRYPT2_MAGIC	(('P'<<0)|('R'<<8)|('I'<<16)|('M'<<24))

/* maximum buffer size for key data */
#define	MAXKEYSIZE	8192

/* maximum number of fingers per 'prime clock' */
#define	MAXFINGERS	64

typedef struct finger_s {
	size_t	base;				/* base offset into key[] */
	size_t	current;			/* current counter value */
	size_t	modulo;				/* modulo (a prime number) */
}	finger_t;

typedef struct prime_clock_s {
	uint32_t modulo;				/* modulo of fingers in this 'clock' */
	finger_t finger[MAXFINGERS];	/* the fingers of this clock */
	uint8_t data;					/* current data */
}	prime_clock_t;

typedef struct crypt2_s {
	uint8_t key[MAXKEYSIZE];		/* the encoded key material buffer */
	size_t size;					/* number of bytes (initial and key) */
	prime_clock_t inner;			/* the inner prime clock */
	prime_clock_t outer;			/* the outer prime clock */
	uint32_t magic;
}	crypt2_t;

static uint32_t *sieve_bits = NULL;
#define U32BITS   32
#define NOT_PRIME(n)    \
    sieve_bits[(n)/U32BITS] |= (1 << ((n) % U32BITS))
#define IS_NOT_PRIME(n)     \
    ((sieve_bits[(n)/U32BITS] >> ((n) % U32BITS)) & 1)

static void mark_sieve(size_t p, size_t max);
static void init_sieve(size_t max);
static uint32_t key_bits(crypt2_t *c, size_t offs, size_t count);
static uint8_t next_byte(crypt2_t *c);

/***********************************************************************
 *	mark_sieve()
 *	Mark all multiples of 'p' in the bit array 'prime'
 ***********************************************************************/
static void mark_sieve(size_t p, size_t max)
{
	size_t n;
	for (n = 2 * p; n < max; n += p)
		NOT_PRIME(n);
}

/***********************************************************************
 *	init_sieve()
 *	Initialize table of primes using the Sieve of Erathostenes
 ***********************************************************************/
static void init_sieve(size_t max)
{
	size_t words = (max + U32BITS - 1) / U32BITS;
	size_t p;

	sieve_bits = (uint32_t *)xcalloc(words, U32BITS/8);
	p = 2;
	while (p < max) {
		mark_sieve(p, max);
		while (++p < max && IS_NOT_PRIME(p))
			;
	}
}

/***********************************************************************
 * return 'count' bits (1 to 17) starting at bit offset 'offs'
 ***********************************************************************/
static uint32_t key_bits(crypt2_t *c, size_t offs, size_t count)
{
	size_t offs_bytes = offs / 8;
	uint32_t val;

	val = c->key[offs_bytes] |
		(c->key[offs_bytes+1] << 8) |
		(c->key[offs_bytes+2] << 16) |
		(c->key[offs_bytes+3] << 24);
	/* shift down to offs%8 bit position */
	val >>= (offs % 8);
	/* mask count bits */
	val &= (1 << count) - 1;

	return val;
}

/***********************************************************************
 * initialize one of the prime clocks.
 * base is the offset of key material in c->key for this clock.
 * max is the available number of bytes for this clock.
 * pc is a pointer to the prime_clock_t struct of this clock.
 ***********************************************************************/
static int prime_clock_init(crypt2_t *c, size_t base, size_t max,
	prime_clock_t *pc)
{
	size_t i, j, h, b;
	size_t offs, sum = 0;
	uint32_t bits;

	int rc = 0;
	FUN("prime_clock_init");

	/* select primes for modulo values of the clock fingers */
	for (h = max/2 - 1, i = 0; h > 7 && i < MAXFINGERS; h /= 2, i++) {
		for (j = h; IS_NOT_PRIME(j); j--)
			;
		pc->finger[i].base = base + sum;
		sum += j;
		pc->finger[i].modulo = j;
		LOGS(L_CRYPTO,L_MINOR,("clock #%d modulo %d (used %d of %d)\n",
			(int)i, (int)j, (int)sum, (int)max));
#if	CRYPT2_TEST
		printf("clock #%d modulo %d (used %d of %d)\n", i, j, sum, max);
#endif
	}
	pc->modulo = i;

	/* make 'offs' the bit offset of remaining data */
	offs = (base + sum) * 8;

	/* set the initial values of the counters for each finger */
	for (i = 0; i < pc->modulo; i++) {

		/* find the smallest 2^b that is just greater than finger.modulo */
		for (b = 1; (size_t)(1 << b) < pc->finger[i].modulo; b++)
			;

		/* fetch 'b' key bits starting at 'offs' */
		bits = key_bits(c, offs, b);
		LOGS(L_CRYPTO,L_MINOR,("clock #%d uses %d bits @ %d (%#x)\n",
			(int)i, (int)b, (int)offs, bits));
#if	CRYPT2_TEST
		printf("clock #%d uses %d bits @ %d (%#x)\n",
			(int)i, (int)b, (int)offs, bits);
#endif
		offs += b;

		/* sanity check: in no case use more than 'max' bytes */
		if ((offs / 8) >= base + max) {
			LOGS(L_CRYPTO,L_ERROR,("FATAL! Out of bounds (bit offs %d)\n",
				(int)offs));
#if	CRYPT2_TEST
			printf("FATAL! Out of bounds (bit offs %d)\n",
				(int)offs);
#endif
			return -1;
		}
	}

	return rc;
}

/***********************************************************************
 *  deliver next PRNG byte
 ***********************************************************************/
static uint8_t next_byte(crypt2_t *c)
{
	uint8_t i_data, o_data;
	size_t i, j;

	/* increment and wrap offset of outer clock finger */
	for (i = 0, o_data = 0; i < c->outer.modulo; i++) {
		finger_t *fo = &c->outer.finger[i];

		o_data += c->key[fo->base + fo->current];

		for (j = 0, i_data = 0; j < c->inner.modulo; j++) {
			finger_t *fi = &c->inner.finger[j];

			i_data += c->key[fi->base + fi->current];
			if (++fi->current == fi->modulo)
				fi->current = 0;
		}

		fo->current = (fo->current + i_data) % fo->modulo;
	}

	return o_data;
}

/***********************************************************************
 *  crypt2_crypto()
 ***********************************************************************/
int crypt2_crypto(void)
{
	FUN("crypt2_crypto");

	/* Initialize the prime number bit array */
	init_sieve(MAXKEYSIZE);

	return 0;
}

/***********************************************************************
 *  crypt2_init()
 ***********************************************************************/
int crypt2_init(void **pptr, int init, void *data, size_t size)
{
	crypt2_t *c;
	size_t part_i, part_o;
	FUN("crypt2_init");

	if (NULL == pptr) {
		LOGS(L_CRYPTO,L_ERROR,("pptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (0 != init) {
		LOGS(L_CRYPTO,L_ERROR,("init is not 0\n"));
		errno = EINVAL;
		return -1;
	}
	c = (crypt2_t *)xcalloc(1, sizeof(crypt2_t));
	*pptr = c;

	c->magic = CRYPT2_MAGIC;

	if (size > MAXKEYSIZE) {
		size = MAXKEYSIZE;
	}
	c->size = size;
	memcpy(c->key, data, size);

	part_i = size / 8;
	part_o = size - part_i;
#if	CRYPT2_TEST
	printf("part_i: %d\n", part_i);
	printf("part_o: %d\n", part_o);
#endif
	prime_clock_init(c, 0,      part_i, &c->inner);
	prime_clock_init(c, part_i, part_o, &c->outer);

#if	CRYPT2_TEST
	printf("c->inner.modulo: %d\n", c->inner.modulo);
	printf("c->outer.modulo: %d\n", c->outer.modulo);
#endif

	return 0;
}

/***********************************************************************
 *  crypt2_exit()
 ***********************************************************************/
int crypt2_exit(void *ptr)
{
	crypt2_t *c = ptr;
	FUN("crypt2_exit");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (CRYPT2_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("magic is %x (!= %x)\n",
			c->magic, CRYPT2_MAGIC));
		errno = EINVAL;
		return -1;
	}

	xfree(ptr);

	return 0;
}

/***********************************************************************
 *  crypt2_encrypt_msg()
 ***********************************************************************/
int crypt2_encrypt_msg(void *ptr, void *ciphertext, void *plaintext,
	size_t *outsize, size_t insize)
{
	crypt2_t *c = (crypt2_t *)ptr;
	uint8_t *ct = (uint8_t *)ciphertext;
	size_t i;
	FUN("crypt2_encrypt_msg");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (insize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %x, want %x)\n",
			(unsigned)*outsize, (unsigned)insize));
		errno = ENOMEM;
		return -1;
	}
	if (CRYPT2_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("magic is %x (!= %x)\n",
			c->magic, CRYPT2_MAGIC));
		errno = EINVAL;
		return -1;
	}
	if (ciphertext != plaintext) {
		memcpy(ciphertext, plaintext, insize);
	}
	for (i = 0; i < insize; i++)
		*ct++ ^= next_byte(c);

	*outsize = insize;
	return 0;
}

/***********************************************************************
 *  crypt2_decrypt_msg()
 ***********************************************************************/
int crypt2_decrypt_msg(void *ptr, void *plaintext, void *ciphertext,
	size_t *outsize, size_t insize)
{
	crypt2_t *c = (crypt2_t *)ptr;
	uint8_t *pt = (uint8_t *)plaintext;
	size_t i;
	FUN("crypt2_decrypt_msg");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (insize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %x, want %x)\n",
			(unsigned)*outsize, (unsigned)insize));
		errno = ENOMEM;
		return -1;
	}
	if (CRYPT2_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("magic is %x (!= %x)\n",
			c->magic, CRYPT2_MAGIC));
		errno = EINVAL;
		return -1;
	}
	if (plaintext != ciphertext) {
		memcpy(plaintext, ciphertext, insize);
	}
	for (i = 0; i < insize; i++)
		*pt++ ^= next_byte(c);

	*outsize = insize;
	return 0;
}

#if	CRYPT2_TEST
configuration_t *g_conf = NULL;
#define	SIZE	1024

int pm_vsnprintf(char *dst, size_t size, const char *fmt, va_list ap)
{
	return vsnprintf(dst, size, fmt, ap);
}

int pm_snprintf(char *dst, size_t size, const char *fmt, ...)
{
	va_list ap;
	int len;

	va_start(ap, fmt);
	len = vsnprintf(dst, size, fmt, ap);
	va_end(ap);
	return len;
}

int main(int argc, char **argv)
{
	char rndname[MAXPATHLEN] = "/dev/urandom";
	char outname[MAXPATHLEN] = "crypt2.bin";
	FILE *rnd, *out;
	void *ctx;
	int mode = 0;
	size_t outsize = SIZE;
	size_t insize = SIZE;
	size_t count = 1;
	size_t n;
	size_t initial_size;
	unsigned char *initial;
	unsigned char *plaintext;
	unsigned char *ciphertext;
	int i;
	int rc;

	for (i = 1; i < argc; i++) {
		if (0 == strcmp(argv[i], "--diehard")) {
			insize = outsize = 0x1000;
			count = 0x1000;
		} else if (0 == strncmp(argv[i], "-r", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-r")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-o", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-o")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-m", 2) && strlen(argv[i]) > 2) {
			mode = strtol(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-m")) {
			mode = strtoul(argv[++i], NULL, 0);
		} else if (0 == strncmp(argv[i], "-s", 2) && strlen(argv[i]) > 2) {
			insize = outsize = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-s")) {
			insize = outsize = strtoul(argv[++i], NULL, 0);
		} else if (0 == strncmp(argv[i], "-c", 2) && strlen(argv[i]) > 2) {
			count = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-c")) {
			count = strtoul(argv[++i], NULL, 0);
		}
	}

	initial_size = 256;

	if (0 != crypt2_crypto()) {
		fprintf(stderr, "crypt2_crypto() call failed\n");
		exit(1);
	}

	printf("open random source: %s\n", rndname);
	rnd = fopen(rndname, "rb");
	if (NULL == rnd) {
		perror(rndname);
		exit(1);
	}

	initial = xcalloc(initial_size, sizeof(uint8_t));
	if (initial_size != fread(initial, 1, initial_size, rnd)) {
		perror("PRNG data");
		exit(2);
	}
	fclose(rnd);

	printf("create output file: %s\n", outname);
	out = fopen(outname, "wb");
	if (NULL == out) {
		perror(outname);
		exit(3);
	}

	plaintext = xcalloc(insize, sizeof(uint8_t));
	ciphertext = xcalloc(outsize, sizeof(uint8_t));

	if (0 != (rc = crypt2_init(&ctx, mode, initial, initial_size))) {
		fprintf(stderr, "crypt2_init failed (%d)", rc);
		exit(4);
	}

	for (n = 0; n < count; n++) {
		outsize = insize;
		if (0 != crypt2_encrypt_msg(ctx, ciphertext, plaintext,
			&outsize, insize)) {
			fprintf(stderr, "crytp4_encrypt_msg failed (%d)", rc);
			exit(5);
		}
		if (outsize != fwrite(ciphertext, 1, outsize, out)) {
			perror("fwrite");
			exit(6);
		}
	}

	fclose(out);

	return 0;
}

#endif
