/*****************************************************************************
 *  ENTROPY - emerging network to reduce orwellian potency yield
 *
 *  Copyright (C) 2003 Juergen Buchmueller <pullmoll@stop1984.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *	$Id: crypt3.c,v 1.3 2005/07/12 23:12:28 pullmoll Exp $
 *****************************************************************************/

/* implementation einer bakannten ARC-stream-cipher methode. 
 *                                      Mr.Ikg. 10.Apr.2003	
 */

#include "osd.h"
#include "crypt.h"
#if CRYPT3_TEST
#define	xmalloc(size) malloc(size)
#define	xcalloc(num,size) calloc(num,size)
#define	xfree(ptr) if (NULL != ptr) { free(ptr); ptr = NULL; }
#else
#include "memalloc.h"
#endif
#include "logger.h"

#define	AR_SIZE			256
#define	CRYPT3_MAGIC	0x29ABD
#define	SKIPLENG		1042

typedef	struct	crypt3_s {
	uint32_t	totalsize;
	uint8_t	sbox[AR_SIZE];	/* S-Box */
	uint32_t	indx_a, indx_b;
	uint32_t	magic;
}	crypt3_t;

#define	Mod256(uval)	(uint8_t)(uval)
#define SWAP(a,b,type)	{type tmp; tmp=a; a=b; b=tmp;}

static uint8_t crypt3_next(crypt3_t *c)
{
	uint32_t i,j, t;

	i = Mod256(c->indx_a + 1);
	j = Mod256(c->indx_b + c->sbox[i]);
	SWAP(c->sbox[i], c->sbox[j], uint8_t);
	t = Mod256(c->sbox[i] + c->sbox[j]);
	c->indx_a = i;
	c->indx_b = j;
	c->totalsize++;
	return	c->sbox[t];
}

int crypt3_crypto(void)
{
	FUN("crypt3_crypto");

	/* This method needs no module initialization */

	return 0;
}

/*****************************************************************************
 *	crypt3_init()
 *	out-init: pptr		- pointer to crypt3_t struct
 *	in: data			- crypt key
 *	in: size			- crypt key size
 *****************************************************************************/
int crypt3_init(void **pptr, int init, void *data, size_t size)
{
	crypt3_t *c;
	uint32_t i, j;
	size_t klen = size > AR_SIZE ? AR_SIZE : size;
	uint8_t *keyp = data;
	FUN("crypt3_init");

	if (NULL == pptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (0 != init) {
		LOGS(L_CRYPTO,L_ERROR,("init is not 0\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == data) {
		LOGS(L_CRYPTO,L_ERROR,("need crypt key\n"));
		errno = EINVAL;
		return -1;
	}
	if (size < 8) {
		LOGS(L_CRYPTO,L_ERROR,("need min. 8 bytes key\n"));
		errno = EINVAL;
		return -1;
	}
	c = (crypt3_t *)xcalloc(1, sizeof(crypt3_t));
	*pptr = c;

	for (i = 0; i < AR_SIZE; i++) {
	      c->sbox[i] = (uint8_t)i;
	}

	/* random s-box */
	for ( j = i = 0; i < AR_SIZE; i++ ) {
	      j = Mod256(j + c->sbox[i] + keyp[i % klen]);
	      SWAP(c->sbox[i], c->sbox[j], uint8_t);
	}

	c->indx_a = 0;
	c->indx_b = 0;
	c->totalsize = 0;
	c->magic = CRYPT3_MAGIC;

	/* paranoid-part */
	for (i = 0; i < SKIPLENG; i++) {
	      (void)crypt3_next(c);
	}
	c->totalsize = 0;

	return 0;
}

/*****************************************************************************
 *	crypt3_exit()
 *	in: ptr		- pointer to crypt3_t struct
 *****************************************************************************/
int crypt3_exit(void *ptr) 
{
	crypt3_t *c = ptr;
	FUN("crypt3_exit");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (CRYPT3_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("magic is %x (!= %x)\n",
			c->magic, CRYPT3_MAGIC));
		errno = EINVAL;
		return -1;
	}
#if CRYPT3_TEST
	printf("AR_CRYPT: Total size = %u;\n",
		c->totalsize);
#endif
	memset(c, 0, sizeof(*c));
	xfree(ptr);

	return 0;
}

/*
 *	crypt3_encrypt_msg()
 *	pptr			in:out - pointer to crypt3_t struct
 *	ciphertext		out - ciphertext
 *	plaintext		in - plaintext
 *	outsize 		out - blocksize
 *	insize 			in - blocksize
 */
int	crypt3_encrypt_msg(void *ptr, void *ciphertext, void *plaintext,
	size_t *outsize, size_t insize)
{
	crypt3_t *c = (crypt3_t *)ptr;
	uint8_t *ct = (uint8_t *)ciphertext;
	uint8_t *pt = (uint8_t *)plaintext;
	uint32_t i;
	FUN("crypt3_encrypt");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (insize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %x, want %x)\n",
			(unsigned)*outsize, (unsigned)insize));
		errno = EINVAL;
		return -1;
	}
	if (CRYPT3_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("magic is %x (!= %x)\n",
			c->magic, CRYPT3_MAGIC));
		errno = EINVAL;
		return -1;
	}

	for (i = 0; i < insize; i++)
		*ct++ = *pt++ ^ crypt3_next(c);

	*outsize = insize;
	return 0;
}

/*
 *	crypt3_decrypt_msg()
 *	pptr			in:out - pointer to crypt3_t struct
 *	plaintext		out - plaintext
 *	ciphertext		in - ciphertext
 *	outsize 		out - blocksize
 *	insize 			in - blocksize
 */
int	crypt3_decrypt_msg(void	*ptr, void *plaintext, void *ciphertext,
	size_t *outsize, size_t insize)
{
	crypt3_t *c = (crypt3_t *)ptr;
	uint8_t *pt = (uint8_t *)plaintext;
	uint8_t *ct = (uint8_t *)ciphertext;
	uint32_t i;
	FUN("crypt3_decrypt");
 
	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (insize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %x, want %x)\n",
			(unsigned)*outsize, (unsigned)insize));
		errno = EINVAL;
		return -1;
	}
	if (CRYPT3_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("magic is %x (!= %x)\n",
			c->magic, CRYPT3_MAGIC));
		errno = EINVAL;
		return -1;
	}

	for (i = 0; i < insize; i++)
		*pt++ = *ct++ ^ crypt3_next(c);

	*outsize = insize;
	return 0;
}

#if CRYPT3_TEST
configuration_t *g_conf = NULL;
#define	SIZE	1024

int pm_vsnprintf(char *dst, size_t size, const char *fmt, va_list ap)
{
	return vsnprintf(dst, size, fmt, ap);
}

int pm_snprintf(char *dst, size_t size, const char *fmt, ...)
{
	va_list ap;
	int len;

	va_start(ap, fmt);
	len = vsnprintf(dst, size, fmt, ap);
	va_end(ap);
	return len;
}

int main(int argc, char **argv)
{
	char rndname[MAXPATHLEN] = "/dev/urandom";
	char outname[MAXPATHLEN] = "crypt3.bin";
	FILE *rnd, *out;
	void *ctx;
	int mode = 0;
	size_t outsize = SIZE;
	size_t insize = SIZE;
	size_t count = 1;
	size_t n;
	size_t initial_size;
	unsigned char *initial;
	unsigned char *plaintext;
	unsigned char *ciphertext;
	int i;
	int rc;

	for (i = 1; i < argc; i++) {
		if (0 == strcmp(argv[i], "--diehard")) {
			insize = outsize = 0x1000;
			count = 0x1000;
		} else if (0 == strncmp(argv[i], "-r", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-r")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-o", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-o")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-m", 2) && strlen(argv[i]) > 2) {
			mode = strtol(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-m")) {
			mode = strtoul(argv[++i], NULL, 0);
		} else if (0 == strncmp(argv[i], "-s", 2) && strlen(argv[i]) > 2) {
			insize = outsize = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-s")) {
			insize = outsize = strtoul(argv[++i], NULL, 0);
		} else if (0 == strncmp(argv[i], "-c", 2) && strlen(argv[i]) > 2) {
			count = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-c")) {
			count = strtoul(argv[++i], NULL, 0);
		}
	}

	initial_size = 256;

	printf("open random source: %s\n", rndname);
	rnd = fopen(rndname, "rb");
	if (NULL == rnd) {
		perror(rndname);
		exit(1);
	}

	initial = xcalloc(initial_size, sizeof(uint8_t));
	if (initial_size != fread(initial, 1, initial_size, rnd)) {
		perror("PRNG data");
		exit(2);
	}
	fclose(rnd);

	printf("create output file: %s\n", outname);
	out = fopen(outname, "wb");
	if (NULL == out) {
		perror(outname);
		exit(3);
	}

	plaintext = xcalloc(insize, sizeof(uint8_t));
	ciphertext = xcalloc(outsize, sizeof(uint8_t));

	if (0 != (rc = crypt3_init(&ctx, mode, initial, initial_size))) {
		fprintf(stderr, "crypt3_init failed (%d)", rc);
		exit(4);
	}

	for (n = 0; n < count; n++) {
		outsize = insize;
		if (0 != crypt3_encrypt_msg(ctx, ciphertext, plaintext,
			&outsize, insize)) {
			fprintf(stderr, "crypt3_encrypt_msg failed (%d)", rc);
			exit(5);
		}
		if (outsize != fwrite(ciphertext, 1, outsize, out)) {
			perror("fwrite");
			exit(6);
		}
	}

	fclose(out);

	return 0;
}

#endif
