/*****************************************************************************
 *  ENTROPY - emerging network to reduce orwellian potency yield
 *
 *  Copyright (C) 2002 Juergen Buchmueller <pullmoll@stop1984.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *	$Id: crypt4.c,v 1.3 2005/07/12 23:12:29 pullmoll Exp $
 *****************************************************************************/
#include "crypt.h"
#if	CRYPT4_TEST
#define	xmalloc(size) malloc(size)
#define	xcalloc(num,size) calloc(num,size)
#define	xfree(ptr) if (NULL != ptr) { free(ptr); ptr = NULL; }
#else
#include "memalloc.h"
#endif
#include "logger.h"

/*****************************************************************************
 * "Virtual Machine Crypto"
 * 
 * 16 opcodes				4 bit
 * 16 x 8-bit registers		4 bit
 * 8-bit operand			8 bit
 * -----------------------------
 * instruction size			16 bit
 * 
 * 16 * 256 = 4K bytes program memory
 * R15 is program counter, too
 * Page register with 4 (invisible) bits giving the MSB
 * 
 * opcode		mnemonic									flags
 *		does
 * --------------------------------------------------------------
 * 0x yy		LD  Rx,#yy									S-PZ-
 *		load Rx with #yy
 *
 * 1x yz		ST  (Ry*16+z),Rx							S-PZ-
 *		store Rx to [Ry*16+z]
 *
 * 2x yy		ADC Rx,#yy									SVPZC
 *		add with carry #yy to Rx
 *
 * 3x yz		ADC Rx,Ry~z (2)								SVPZC
 *		add with carry Ry rol/shr z to Rx
 *
 * 4x yy		SBB Rx,#yy									SVPZC
 *		subtract with borrow #yy from Rx
 *
 * 5x yz		SBB Rx,Ry~z (2)								SVPZC
 *		subtract with borrow Ry rol/shr z from Rx
 *
 * 6x yy		AND Rx,#yy									S0PZ0
 *		and Rx with #yy
 *
 * 7x yz		AND Rx,Ry~z (2)								S0PZ0
 *		and Rx with Ry rol/shr z
 *
 * 8x yy		XOR Rx,#yy									S0PZ0
 *		xor Rx with #yy
 *
 * 9x yz		XOR Rx,Ry~z (2)								S0PZ0
 *		xor Rx with Ry rol/shr z
 *
 * Ax yy		OR  Rx,#yy									S0PZ0
 *		or Rx with #yy
 *
 * Bx yy		OR  Rx,Ry~z (2)								S0PZ0
 *		or Rx with Ry rol/shr z
 *
 * Cx yy		CMP Rx,#yy									SVPZC
 *		compare Rx with #yy
 *
 * Dx yy		CMP Rx,Ry~z (2)								SVPZC
 *		compare Rx with Ry rol/shr z
 *
 * Ex yz		XCH (Ry*16+z),Rx							S-PZ-
 *		exchange Rx with [Ry*16+z]
 *
 * Fx yz		Jcc $+Ry+z									-----
 *		jump on condition 'x' (1) to $+Ry+z
 *
 * If Rx is R15, result is written into lower four bits of page register.
 * 
 * (1) condition codes
 * 
 * x	meaning
 * -------------------
 * 0	always
 * 1	zero
 * 2	negative
 * 3	less or equal zero
 * 4	carry set
 * 5	carry or zero
 * 6	overflow
 * 7	even parity
 * 8	odd parity
 * 9	not overflow
 * A	neither carry nor zero
 * B	carry clear
 * C	greater zero
 * D	positive
 * E	not zero
 * F	to yz
 * 
 * Flag bits are modified by all opcodes except fx (jump)
 * Overflow flag is modified by ADC, SBB and CMP opcodes
 * Parity flag is set for even parity, clear for odd parity
 * 
 * (2) rotate left, shift right, not
 * z	meaning
 * ----------------------------
 * 0	do nothing, return Ry
 * 1-7	rotate left by 1 to 7 bit positions
 * 8-E	shift right by 1 to 7 bit positions
 * F	return Ry's ones complement
 *****************************************************************************/

#if	CRYPT4_TEST
FILE *fpout = NULL;
#endif
 
#define	CRYPT4_MAGIC	(('V'<<0)|('M'<<8)|('E'<<16)|('8'<<24))
#define	CRYPT4_MEM	(16*256)

typedef struct crypt4_s {
	uint64_t encrypt_count;
	uint64_t decrypt_count;
	uint32_t magic;
	uint32_t cc;				/* condition codes */
	uint32_t page;				/* 4 bit msb of progam counter reg[15] */
	uint32_t cnt;
	uint8_t pad[16];			/* 16 scratch pad entries */
	uint8_t reg[16];			/* 16 registers */
	uint8_t mem[CRYPT4_MEM];	/* program memory */
}	crypt4_t;

#define	FLAG_C	0x01			/* carry flag */
#define	FLAG_Z	0x02			/* zero flag */
#define	FLAG_P	0x04			/* parity flag */
#define	FLAG_S	0x08			/* sign flag */
#define	FLAG_V	0x10			/* overflow flag */

#define	SET_C	c->cc |= FLAG_C
#define	CLR_C	c->cc &= ~FLAG_C
#define	SET_Z	c->cc |= FLAG_Z
#define	CLR_Z	c->cc &= ~FLAG_Z
#define	SET_P	c->cc |= FLAG_P
#define	CLR_P	c->cc &= ~FLAG_P
#define	SET_V	c->cc |= FLAG_V
#define	CLR_V	c->cc &= ~FLAG_V
#define	SET_S	c->cc |= FLAG_S
#define	CLR_S	c->cc &= ~FLAG_S

#define	PARITY(b7,b6,b5,b4,b3,b2,b1,b0)	\
	((b7^b6^b5^b4^b3^b2^b1^b0)&1)

static uint8_t parity[] = {
	PARITY(0,0,0,0,0,0,0,0), PARITY(0,0,0,0,0,0,0,1),
	PARITY(0,0,0,0,0,0,1,0), PARITY(0,0,0,0,0,0,1,1),
	PARITY(0,0,0,0,0,1,0,0), PARITY(0,0,0,0,0,1,0,1),
	PARITY(0,0,0,0,0,1,1,0), PARITY(0,0,0,0,0,1,1,1),
	PARITY(0,0,0,0,1,0,0,0), PARITY(0,0,0,0,1,0,0,1),
	PARITY(0,0,0,0,1,0,1,0), PARITY(0,0,0,0,1,0,1,1),
	PARITY(0,0,0,0,1,1,0,0), PARITY(0,0,0,0,1,1,0,1),
	PARITY(0,0,0,0,1,1,1,0), PARITY(0,0,0,0,1,1,1,1),
	PARITY(0,0,0,1,0,0,0,0), PARITY(0,0,0,1,0,0,0,1),
	PARITY(0,0,0,1,0,0,1,0), PARITY(0,0,0,1,0,0,1,1),
	PARITY(0,0,0,1,0,1,0,0), PARITY(0,0,0,1,0,1,0,1),
	PARITY(0,0,0,1,0,1,1,0), PARITY(0,0,0,1,0,1,1,1),
	PARITY(0,0,0,1,1,0,0,0), PARITY(0,0,0,1,1,0,0,1),
	PARITY(0,0,0,1,1,0,1,0), PARITY(0,0,0,1,1,0,1,1),
	PARITY(0,0,0,1,1,1,0,0), PARITY(0,0,0,1,1,1,0,1),
	PARITY(0,0,0,1,1,1,1,0), PARITY(0,0,0,1,1,1,1,1),
	PARITY(0,0,1,0,0,0,0,0), PARITY(0,0,1,0,0,0,0,1),
	PARITY(0,0,1,0,0,0,1,0), PARITY(0,0,1,0,0,0,1,1),
	PARITY(0,0,1,0,0,1,0,0), PARITY(0,0,1,0,0,1,0,1),
	PARITY(0,0,1,0,0,1,1,0), PARITY(0,0,1,0,0,1,1,1),
	PARITY(0,0,1,0,1,0,0,0), PARITY(0,0,1,0,1,0,0,1),
	PARITY(0,0,1,0,1,0,1,0), PARITY(0,0,1,0,1,0,1,1),
	PARITY(0,0,1,0,1,1,0,0), PARITY(0,0,1,0,1,1,0,1),
	PARITY(0,0,1,0,1,1,1,0), PARITY(0,0,1,0,1,1,1,1),
	PARITY(0,0,1,1,0,0,0,0), PARITY(0,0,1,1,0,0,0,1),
	PARITY(0,0,1,1,0,0,1,0), PARITY(0,0,1,1,0,0,1,1),
	PARITY(0,0,1,1,0,1,0,0), PARITY(0,0,1,1,0,1,0,1),
	PARITY(0,0,1,1,0,1,1,0), PARITY(0,0,1,1,0,1,1,1),
	PARITY(0,0,1,1,1,0,0,0), PARITY(0,0,1,1,1,0,0,1),
	PARITY(0,0,1,1,1,0,1,0), PARITY(0,0,1,1,1,0,1,1),
	PARITY(0,0,1,1,1,1,0,0), PARITY(0,0,1,1,1,1,0,1),
	PARITY(0,0,1,1,1,1,1,0), PARITY(0,0,1,1,1,1,1,1),
	PARITY(0,1,0,0,0,0,0,0), PARITY(0,1,0,0,0,0,0,1),
	PARITY(0,1,0,0,0,0,1,0), PARITY(0,1,0,0,0,0,1,1),
	PARITY(0,1,0,0,0,1,0,0), PARITY(0,1,0,0,0,1,0,1),
	PARITY(0,1,0,0,0,1,1,0), PARITY(0,1,0,0,0,1,1,1),
	PARITY(0,1,0,0,1,0,0,0), PARITY(0,1,0,0,1,0,0,1),
	PARITY(0,1,0,0,1,0,1,0), PARITY(0,1,0,0,1,0,1,1),
	PARITY(0,1,0,0,1,1,0,0), PARITY(0,1,0,0,1,1,0,1),
	PARITY(0,1,0,0,1,1,1,0), PARITY(0,1,0,0,1,1,1,1),
	PARITY(0,1,0,1,0,0,0,0), PARITY(0,1,0,1,0,0,0,1),
	PARITY(0,1,0,1,0,0,1,0), PARITY(0,1,0,1,0,0,1,1),
	PARITY(0,1,0,1,0,1,0,0), PARITY(0,1,0,1,0,1,0,1),
	PARITY(0,1,0,1,0,1,1,0), PARITY(0,1,0,1,0,1,1,1),
	PARITY(0,1,0,1,1,0,0,0), PARITY(0,1,0,1,1,0,0,1),
	PARITY(0,1,0,1,1,0,1,0), PARITY(0,1,0,1,1,0,1,1),
	PARITY(0,1,0,1,1,1,0,0), PARITY(0,1,0,1,1,1,0,1),
	PARITY(0,1,0,1,1,1,1,0), PARITY(0,1,0,1,1,1,1,1),
	PARITY(0,1,1,0,0,0,0,0), PARITY(0,1,1,0,0,0,0,1),
	PARITY(0,1,1,0,0,0,1,0), PARITY(0,1,1,0,0,0,1,1),
	PARITY(0,1,1,0,0,1,0,0), PARITY(0,1,1,0,0,1,0,1),
	PARITY(0,1,1,0,0,1,1,0), PARITY(0,1,1,0,0,1,1,1),
	PARITY(0,1,1,0,1,0,0,0), PARITY(0,1,1,0,1,0,0,1),
	PARITY(0,1,1,0,1,0,1,0), PARITY(0,1,1,0,1,0,1,1),
	PARITY(0,1,1,0,1,1,0,0), PARITY(0,1,1,0,1,1,0,1),
	PARITY(0,1,1,0,1,1,1,0), PARITY(0,1,1,0,1,1,1,1),
	PARITY(0,1,1,1,0,0,0,0), PARITY(0,1,1,1,0,0,0,1),
	PARITY(0,1,1,1,0,0,1,0), PARITY(0,1,1,1,0,0,1,1),
	PARITY(0,1,1,1,0,1,0,0), PARITY(0,1,1,1,0,1,0,1),
	PARITY(0,1,1,1,0,1,1,0), PARITY(0,1,1,1,0,1,1,1),
	PARITY(0,1,1,1,1,0,0,0), PARITY(0,1,1,1,1,0,0,1),
	PARITY(0,1,1,1,1,0,1,0), PARITY(0,1,1,1,1,0,1,1),
	PARITY(0,1,1,1,1,1,0,0), PARITY(0,1,1,1,1,1,0,1),
	PARITY(0,1,1,1,1,1,1,0), PARITY(0,1,1,1,1,1,1,1),
	PARITY(1,0,0,0,0,0,0,0), PARITY(1,0,0,0,0,0,0,1),
	PARITY(1,0,0,0,0,0,1,0), PARITY(1,0,0,0,0,0,1,1),
	PARITY(1,0,0,0,0,1,0,0), PARITY(1,0,0,0,0,1,0,1),
	PARITY(1,0,0,0,0,1,1,0), PARITY(1,0,0,0,0,1,1,1),
	PARITY(1,0,0,0,1,0,0,0), PARITY(1,0,0,0,1,0,0,1),
	PARITY(1,0,0,0,1,0,1,0), PARITY(1,0,0,0,1,0,1,1),
	PARITY(1,0,0,0,1,1,0,0), PARITY(1,0,0,0,1,1,0,1),
	PARITY(1,0,0,0,1,1,1,0), PARITY(1,0,0,0,1,1,1,1),
	PARITY(1,0,0,1,0,0,0,0), PARITY(1,0,0,1,0,0,0,1),
	PARITY(1,0,0,1,0,0,1,0), PARITY(1,0,0,1,0,0,1,1),
	PARITY(1,0,0,1,0,1,0,0), PARITY(1,0,0,1,0,1,0,1),
	PARITY(1,0,0,1,0,1,1,0), PARITY(1,0,0,1,0,1,1,1),
	PARITY(1,0,0,1,1,0,0,0), PARITY(1,0,0,1,1,0,0,1),
	PARITY(1,0,0,1,1,0,1,0), PARITY(1,0,0,1,1,0,1,1),
	PARITY(1,0,0,1,1,1,0,0), PARITY(1,0,0,1,1,1,0,1),
	PARITY(1,0,0,1,1,1,1,0), PARITY(1,0,0,1,1,1,1,1),
	PARITY(1,0,1,0,0,0,0,0), PARITY(1,0,1,0,0,0,0,1),
	PARITY(1,0,1,0,0,0,1,0), PARITY(1,0,1,0,0,0,1,1),
	PARITY(1,0,1,0,0,1,0,0), PARITY(1,0,1,0,0,1,0,1),
	PARITY(1,0,1,0,0,1,1,0), PARITY(1,0,1,0,0,1,1,1),
	PARITY(1,0,1,0,1,0,0,0), PARITY(1,0,1,0,1,0,0,1),
	PARITY(1,0,1,0,1,0,1,0), PARITY(1,0,1,0,1,0,1,1),
	PARITY(1,0,1,0,1,1,0,0), PARITY(1,0,1,0,1,1,0,1),
	PARITY(1,0,1,0,1,1,1,0), PARITY(1,0,1,0,1,1,1,1),
	PARITY(1,0,1,1,0,0,0,0), PARITY(1,0,1,1,0,0,0,1),
	PARITY(1,0,1,1,0,0,1,0), PARITY(1,0,1,1,0,0,1,1),
	PARITY(1,0,1,1,0,1,0,0), PARITY(1,0,1,1,0,1,0,1),
	PARITY(1,0,1,1,0,1,1,0), PARITY(1,0,1,1,0,1,1,1),
	PARITY(1,0,1,1,1,0,0,0), PARITY(1,0,1,1,1,0,0,1),
	PARITY(1,0,1,1,1,0,1,0), PARITY(1,0,1,1,1,0,1,1),
	PARITY(1,0,1,1,1,1,0,0), PARITY(1,0,1,1,1,1,0,1),
	PARITY(1,0,1,1,1,1,1,0), PARITY(1,0,1,1,1,1,1,1),
	PARITY(1,1,0,0,0,0,0,0), PARITY(1,1,0,0,0,0,0,1),
	PARITY(1,1,0,0,0,0,1,0), PARITY(1,1,0,0,0,0,1,1),
	PARITY(1,1,0,0,0,1,0,0), PARITY(1,1,0,0,0,1,0,1),
	PARITY(1,1,0,0,0,1,1,0), PARITY(1,1,0,0,0,1,1,1),
	PARITY(1,1,0,0,1,0,0,0), PARITY(1,1,0,0,1,0,0,1),
	PARITY(1,1,0,0,1,0,1,0), PARITY(1,1,0,0,1,0,1,1),
	PARITY(1,1,0,0,1,1,0,0), PARITY(1,1,0,0,1,1,0,1),
	PARITY(1,1,0,0,1,1,1,0), PARITY(1,1,0,0,1,1,1,1),
	PARITY(1,1,0,1,0,0,0,0), PARITY(1,1,0,1,0,0,0,1),
	PARITY(1,1,0,1,0,0,1,0), PARITY(1,1,0,1,0,0,1,1),
	PARITY(1,1,0,1,0,1,0,0), PARITY(1,1,0,1,0,1,0,1),
	PARITY(1,1,0,1,0,1,1,0), PARITY(1,1,0,1,0,1,1,1),
	PARITY(1,1,0,1,1,0,0,0), PARITY(1,1,0,1,1,0,0,1),
	PARITY(1,1,0,1,1,0,1,0), PARITY(1,1,0,1,1,0,1,1),
	PARITY(1,1,0,1,1,1,0,0), PARITY(1,1,0,1,1,1,0,1),
	PARITY(1,1,0,1,1,1,1,0), PARITY(1,1,0,1,1,1,1,1),
	PARITY(1,1,1,0,0,0,0,0), PARITY(1,1,1,0,0,0,0,1),
	PARITY(1,1,1,0,0,0,1,0), PARITY(1,1,1,0,0,0,1,1),
	PARITY(1,1,1,0,0,1,0,0), PARITY(1,1,1,0,0,1,0,1),
	PARITY(1,1,1,0,0,1,1,0), PARITY(1,1,1,0,0,1,1,1),
	PARITY(1,1,1,0,1,0,0,0), PARITY(1,1,1,0,1,0,0,1),
	PARITY(1,1,1,0,1,0,1,0), PARITY(1,1,1,0,1,0,1,1),
	PARITY(1,1,1,0,1,1,0,0), PARITY(1,1,1,0,1,1,0,1),
	PARITY(1,1,1,0,1,1,1,0), PARITY(1,1,1,0,1,1,1,1),
	PARITY(1,1,1,1,0,0,0,0), PARITY(1,1,1,1,0,0,0,1),
	PARITY(1,1,1,1,0,0,1,0), PARITY(1,1,1,1,0,0,1,1),
	PARITY(1,1,1,1,0,1,0,0), PARITY(1,1,1,1,0,1,0,1),
	PARITY(1,1,1,1,0,1,1,0), PARITY(1,1,1,1,0,1,1,1),
	PARITY(1,1,1,1,1,0,0,0), PARITY(1,1,1,1,1,0,0,1),
	PARITY(1,1,1,1,1,0,1,0), PARITY(1,1,1,1,1,0,1,1),
	PARITY(1,1,1,1,1,1,0,0), PARITY(1,1,1,1,1,1,0,1),
	PARITY(1,1,1,1,1,1,1,0), PARITY(1,1,1,1,1,1,1,1)
};

/* modify sign, parity and zero flags depending on 'val' */
static void SPZ(crypt4_t *c, uint8_t val)
{
	if (0 == val) {
		/* clear sign, set zero */
		c->cc = (c->cc & ~FLAG_S) | FLAG_Z;
	} else if (0x80 == (val & 0x80)) {
		/* set sign, clear zero */
		c->cc = (c->cc & ~FLAG_Z) | FLAG_S;
	} else {
		/* clear sign and zero */
		c->cc &= ~(FLAG_S | FLAG_Z);
	}
	if (parity[val]) {
		c->cc |= FLAG_P;
	} else {
		c->cc &= ~FLAG_P;
	}
}

/* modify sign, parity and zero flags depending on 'val'
 * always clear overflow and carry flags
 */
static void S0PZ0(crypt4_t *c, uint8_t val)
{
	c->cc = 0;
	if (0 == val) {
		/* set zero */
		c->cc |= FLAG_Z;
	} else if (0x80 == (val & 0x80)) {
		/* set sign */
		c->cc |= FLAG_S;
	} else {
		/* sign and zero are clear */
	}
	if (parity[val]) {
		c->cc |= FLAG_P;
	}
}

/* modify all flags after addition, depending on 'now' vs. 'old' */
static void SVPZC_ADC(crypt4_t *c, uint8_t now, uint8_t old)
{
	c->cc = 0;
	if (0 == now) {
		c->cc |= FLAG_Z;
	} else if (0x80 == (now & 0x80)) {
		c->cc |= FLAG_S;
	}
	if (now < old) {
		c->cc |= FLAG_C;
	}
	if (0x80 & (now ^ old)) {
		c->cc |= FLAG_V;
	}
	if (parity[now]) {
		c->cc |= FLAG_P;
	}
}

/* modify all flags after subtract, depending on 'now' vs. 'old' */
static void SVPZC_SBB(crypt4_t *c, uint8_t now, uint8_t old)
{
	c->cc = 0;
	if (0 == now) {
		c->cc |= FLAG_Z;
	} else if (0x80 == (now & 0x80)) {
		c->cc |= FLAG_S;
	}
	if (now > old) {
		c->cc |= FLAG_C;
	}
	if (0x80 & (now ^ old)) {
		c->cc |= FLAG_V;
	}
	if (parity[now]) {
		c->cc |= FLAG_P;
	}
}

/* return (modified) register Ry */
static uint8_t Ry(crypt4_t *c, uint8_t yz)
{
	uint8_t ry = c->reg[yz / 16];

	switch (yz % 16) {
	case 0:	/* unmodified */
		return ry;
	case 1:	/* rotate left 1 */
		return (ry << 1) | (ry >> 7);
	case 2:	/* rotate left 2 */
		return (ry << 2) | (ry >> 6);
	case 3:	/* rotate left 3 */
		return (ry << 3) | (ry >> 5);
	case 4:	/* rotate left 4 (ie. nibble swap) */
		return (ry << 4) | (ry >> 4);
	case 5:	/* rotate left 5 */
		return (ry << 5) | (ry >> 3);
	case 6:	/* rotate left 6 */
		return (ry << 6) | (ry >> 2);
	case 7:	/* rotate left 7 */
		return (ry << 7) | (ry >> 1);
	case 8: /* shift right 1 */
		return ry >> 1;
	case 9: /* shift right 2 */
		return ry >> 2;
	case 10:/* shift right 3 */
		return ry >> 3;
	case 11:/* shift right 4 */
		return ry >> 4;
	case 12:/* shift right 5 */
		return ry >> 5;
	case 13:/* shift right 6 */
		return ry >> 6;
	case 14:/* shift right 7 */
		return ry >> 7;
	/* default case follows */
	}
	/* ones complement */
	return ~ry;
}

#if	CRYPT4_TEST
#define	C4LOG0(fp,fmt)	\
	if (NULL != fp) width += fprintf(fp,fmt)
#define	C4LOG1(fp,fmt,arg)	\
	if (NULL != fp) width += fprintf(fp,fmt,arg)
#define	C4LOG2(fp,fmt,arg1,arg2)	\
	if (NULL != fp) width += fprintf(fp,fmt,arg1,arg2)
#define	C4LOG3(fp,fmt,arg1,arg2,arg3)	\
	if (NULL != fp) width += fprintf(fp,fmt,arg1,arg2,arg3)
#define	C4LOG4(fp,fmt,arg1,arg2,arg3,arg4)	\
	if (NULL != fp) width += fprintf(fp,fmt,arg1,arg2,arg3,arg4)
#define	C4LOG5(fp,fmt,arg1,arg2,arg3,arg4,arg5)	\
	if (NULL != fp) width += fprintf(fp,fmt,arg1,arg2,arg3,arg4,arg5)


const char *Ry_str(uint8_t arg)
{
	static char buff[4][32];
	static int which = 0;
	uint8_t ah = arg / 16, al = arg % 16;
	char *dst;

	which = (which + 1) % 4;
	dst = buff[which];
	switch (arg % 16) {
	case 0:
		sprintf(dst, "R%x", ah);
		break;
	case 1: case 2: case 3: case 4: case 5: case 6: case 7:
		sprintf(dst, "R%x~rol%d", ah, al);
		break;
	case 8: case 9: case 10: case 11: case 12: case 13: case 14:
		sprintf(dst, "R%x~shr%d", ah, al - 7);
		break;
	default:
		sprintf(dst, "~R%x", ah);
	}

	return buff[which];
}
#else
#define	C4LOG0(fp,fmt)
#define	C4LOG1(fp,fmt,arg1)
#define	C4LOG2(fp,fmt,arg1,arg2)
#define	C4LOG3(fp,fmt,arg1,arg2,arg3)
#define	C4LOG4(fp,fmt,arg1,arg2,arg3,arg4)
#define	C4LOG5(fp,fmt,arg1,arg2,arg3,arg4,arg5)
#endif

static uint8_t crypt4_next(crypt4_t *c)
{
	uint32_t adr;
	uint8_t op, ri, ah, al, res, reg, arg;
#if	CRYPT4_TEST
	int width = 0;
#endif

	adr = c->page * 256 + c->reg[15];
	C4LOG1(fpout,"%03x:", adr);
	op = c->mem[adr];
	adr = (adr + 1) % CRYPT4_MEM;

	arg = c->mem[adr];
	adr = (adr + 1) % CRYPT4_MEM;

	C4LOG2(fpout,"%02x %02x  ", op, arg);
	ri = op % 16;	/* register index */
	op = op / 16;	/* opcode type */

	al = arg % 16;	/* argument lower nibble */
	ah = arg / 16;	/* argument upper nibble */

	c->page = (uint8_t)(adr / 256);
	c->reg[15] = (uint8_t)(adr);

	switch (op) {

	/*	LD	Rx,#yy */
	case 0:
		C4LOG2(fpout,"LD  R%x,#%02x", ri, arg);
		res = arg;
		SPZ(c, arg);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		} else {
			c->reg[ri] = arg;
		}
		break;

	/*	ST	(Ry*16+z),Rx */
	case 1:
		C4LOG3(fpout,"ST  (R%x*16+%x),R%x", arg / 16, arg % 16, ri);
		reg = c->reg[ri];
		adr = c->reg[ah] * 16 + al;
		SPZ(c, reg);
		c->mem[adr] = reg;
		res = reg;
		break;

	/*	ADC	Rx,#yy */
	case 2:
		C4LOG2(fpout,"ADC R%x,#%02x", ri, arg);
		reg = c->reg[ri];
		res = reg + arg + (c->cc & FLAG_C);
		SVPZC_ADC(c, res, reg);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		} else {
			c->reg[ri] = res;
		}
		break;

	/*	ADC	Rx,Ry~z */
	case 3:
		C4LOG2(fpout,"ADC R%x,%s", ri, Ry_str(arg));
		reg = c->reg[ri];
		res = reg + Ry(c, arg) + (c->cc & FLAG_C);
		SVPZC_ADC(c, res, reg);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		} else {
			c->reg[ri] = res;
		}
		break;

	/*	SBB	Rx,#yy */
	case 4:
		C4LOG2(fpout,"SBB R%x,#%02x", ri, arg);
		reg = c->reg[ri];
		res = reg - arg - (c->cc & FLAG_C);
		SVPZC_SBB(c, res, reg);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		} else {
			c->reg[ri] = res;
		}
		break;

	/*	SBB	Rx,Ry~z */
	case 5:
		C4LOG2(fpout,"SBB R%x,%s", ri, Ry_str(arg));
		reg = c->reg[ri];
		res = reg - Ry(c, arg) - (c->cc & FLAG_C);
		SVPZC_SBB(c, res, reg);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		} else {
			c->reg[ri] = res;
		}
		break;

	/*	AND Rx,#yy */
	case 6:
		C4LOG2(fpout,"AND R%x,#%02x", ri, arg);
		reg = c->reg[ri];
		res = reg & arg;
		S0PZ0(c, res);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		} else {
			c->reg[ri] = res;
		}
		break;

	/*	AND Rx,Ry~z */
	case 7:
		C4LOG2(fpout,"AND R%x,%s", ri, Ry_str(arg));
		reg = c->reg[ri];
		res = reg & Ry(c, arg);
		S0PZ0(c, res);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		} else {
			c->reg[ri] = res;
		}
		break;

	/*	XOR Rx,#yy */
	case 8:
		C4LOG2(fpout,"XOR R%x,#%02x", ri, arg);
		reg = c->reg[ri];
		res = reg ^ arg;
		S0PZ0(c, res);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		} else {
			c->reg[ri] = res;
		}
		break;

	/*	XOR Rx,Ry~z */
	case 9:
		C4LOG2(fpout,"XOR R%x,%s", ri, Ry_str(arg));
		reg = c->reg[ri];
		res = reg ^ Ry(c, arg);
		S0PZ0(c, res);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		} else {
			c->reg[ri] = res;
		}
		break;

	/*	OR  Rx,#yy */
	case 10:
		C4LOG2(fpout,"OR  R%x,#%02x", ri, arg);
		reg = c->reg[ri];
		res = reg | arg;
		S0PZ0(c, res);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		} else {
			c->reg[ri] = res;
		}
		break;

	/*	OR  Rx,Ry~z */
	case 11:
		C4LOG2(fpout,"OR  R%x,%s", ri, Ry_str(arg));
		reg = c->reg[ri];
		res = reg | Ry(c, arg);
		S0PZ0(c, res);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		} else {
			c->reg[ri] = res;
		}
		break;

	/*	CMP Rx,#yy */
	case 12:
		C4LOG2(fpout,"CMP R%x,#%02x", ri, arg);
		reg = c->reg[ri];
		res = reg - arg;
		SVPZC_SBB(c, res, reg);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		}
		break;

	/*	CMP Rx,Ry~z */
	case 13:
		C4LOG2(fpout,"CMP R%x,%s", ri, Ry_str(arg));
		reg = c->reg[ri];
		res = reg - arg;
		SVPZC_SBB(c, res, reg);
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		}
		break;

	/*	XCH (Ry*16+z),Rx */
	case 14:
		C4LOG3(fpout,"XCH (R%x*16+%x),R%x", ah, al, ri);
		reg = c->reg[ri];
		adr = c->reg[ah] * 16 + al;
		SPZ(c, reg);
		res = c->mem[adr];
		/* R15 = PC special case: swap registers of result nibbles */
		if (15 == ri) {
			ah = res / 16;
			al = res % 16;
			reg = c->reg[al]; c->reg[al] = c->reg[ah]; c->reg[ah] = reg;
		} else {
			c->reg[ri] = res;
		}
		c->mem[adr] = reg;
		break;

	/*	Jcc $+yy */
	default:
		reg = c->reg[ah];
		switch (ri) {
		case 0:	/* always */
			C4LOG3(fpout,"JMP $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			adr = (adr + reg + al) % CRYPT4_MEM;
			break;
		case 1:	/* if zero flag is set */
			C4LOG3(fpout,"JZ  $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 != (c->cc & FLAG_Z)) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 2:	/* if sign flag is set */
			C4LOG3(fpout,"JS  $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 != (c->cc & FLAG_S)) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 3:	/* if zero or sign flags are set */
			C4LOG3(fpout,"JLE $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 != (c->cc & (FLAG_S | FLAG_Z))) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 4:	/* if carry flag is set */
			C4LOG3(fpout,"JC  $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 != (c->cc & FLAG_C)) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 5:	/* if carry or zero flags are set */
			C4LOG3(fpout,"JBE $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 != (c->cc & (FLAG_Z | FLAG_C))) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 6:	/* if overflow flag is set */
			C4LOG3(fpout,"JV  $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 != (c->cc & FLAG_V)) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 7:	/* if parity flag is set */
			C4LOG3(fpout,"JPE $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 != (c->cc & FLAG_P)) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 8:	/* if parity flag is clear */
			C4LOG3(fpout,"JPO $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 == (c->cc & FLAG_P)) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 9:	/* if overflow flag is clear */
			C4LOG3(fpout,"JNV $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 == (c->cc & FLAG_V)) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 10:/* if carry and zero flags are clear */
			C4LOG3(fpout,"JA  $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 == (c->cc & (FLAG_Z | FLAG_C))) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 11:/* if carry flag is clear */
			C4LOG3(fpout,"JNC $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 == (c->cc & FLAG_C)) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 12:/* if sign and zero flags are clear */
			C4LOG3(fpout,"JG  $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 == (c->cc & (FLAG_S | FLAG_Z))) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 13:/* if sign flag is clear */
			C4LOG3(fpout,"JP  $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 == (c->cc & FLAG_S)) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		case 14:/* if zero flag is clear */
			C4LOG3(fpout,"JNZ $+R%x+%x (#%03x)",
				ah, al, (adr + reg + al) % CRYPT4_MEM);
			if (0 == (c->cc & FLAG_Z)) {
				adr = (adr + reg + al) % CRYPT4_MEM;
			}
			break;
		default:/* to yz */
			C4LOG2(fpout,"JR  $+%02x (#%03x)",
				arg, (adr + arg) % CRYPT4_MEM);
			adr = (adr + arg) % CRYPT4_MEM;
		}
		c->page = (uint8_t)(adr / 256);
		c->reg[15] = (uint8_t)adr;
		res = adr;
		break;
	}

	c->pad[c->cnt % 16] += res;
	c->cnt++;

#if	CRYPT4_TEST
	if (NULL != fpout) {
		while (width < 32) {
			C4LOG0(fpout," ");
		}
		C4LOG5(fpout,"%c%c%c%c%c  ",
			(c->cc & FLAG_C) ? 'C' : '-',
			(c->cc & FLAG_Z) ? 'Z' : '-',
			(c->cc & FLAG_P) ? 'P' : '-',
			(c->cc & FLAG_S) ? 'S' : '-',
			(c->cc & FLAG_V) ? 'V' : '-');
		C4LOG2(fpout,"%02x%02x", c->cc, c->page);
		C4LOG2(fpout,"%02x%02x", c->reg[ 0], c->reg[ 1]);
		C4LOG2(fpout,"%02x%02x", c->reg[ 2], c->reg[ 3]);
		C4LOG2(fpout,"%02x%02x", c->reg[ 4], c->reg[ 5]);
		C4LOG2(fpout,"%02x%02x", c->reg[ 6], c->reg[ 7]);
		C4LOG2(fpout,"%02x%02x", c->reg[ 8], c->reg[ 9]);
		C4LOG2(fpout,"%02x%02x", c->reg[10], c->reg[11]);
		C4LOG2(fpout,"%02x%02x", c->reg[12], c->reg[13]);
		C4LOG2(fpout,"%02x%02x", c->reg[14], c->reg[15]);
		C4LOG0(fpout,"\n");
	}
#endif

	res = c->cc;
	al = c->pad[15] % 16;
	for (adr = 0; adr < 15; adr++) {
		al = (al + c->pad[adr]) % 16;
		if (c->cnt & (1 << adr)) {
			c->pad[adr] += res;
		} else {
			res += c->pad[al] ^ c->reg[adr];
		}
	}

	C4LOG1(fpout,"%02x\n", res);

	return res;
}

/***********************************************************************
 *  crypto()
 ***********************************************************************/
int crypt4_crypto(void)
{
	FUN("crypt4_crypto");

	/* no initialization required */

	return 0;
}

/***********************************************************************
 *  init()
 ***********************************************************************/
int crypt4_init(void **pptr, int init, void *data, size_t size)
{
	crypt4_t *c;
	uint8_t *bytes = data;
	size_t i, j;
	FUN("crypt4_init");

	if (NULL == pptr) {
		LOGS(L_CRYPTO,L_ERROR,("pptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (0 != init) {
		LOGS(L_CRYPTO,L_ERROR,("init is not 0\n"));
		errno = EINVAL;
		return -1;
	}
	c = (crypt4_t *)xcalloc(1, sizeof(crypt4_t));
	*pptr = c;

	c->magic = CRYPT4_MAGIC;
	for (i = 0; i < sizeof(c->mem); i++) {
		c->mem[i] = bytes[i % size];
	}

	S0PZ0(c,0);

	for (i = 0; i < 16; i++) {
		c->pad[i] = 0;
		for (j = 0; j < 16; j++)
			c->pad[i] = c->pad[i] + c->mem[i*16+j];
		c->reg[i] = 0;
		for (j = 0; j < 16; j++)
			c->reg[i] = c->reg[i] + c->mem[256+i*16+j];
	}

	return 0;
}

/***********************************************************************
 *  exit()
 ***********************************************************************/
int crypt4_exit(void *ptr)
{
	crypt4_t *c = ptr;
	FUN("crypt4_exit");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (CRYPT4_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("magic is %x (!= %x)\n",
			c->magic, CRYPT4_MAGIC));
		errno = EINVAL;
		return -1;
	}
	memset(c, 0, sizeof(*c));
	xfree(ptr);

	return 0;
}

/***********************************************************************
 *  encrypt_msg()
 ***********************************************************************/
int crypt4_encrypt_msg(void *ptr, void *ciphertext, void *plaintext,
	size_t *outsize, size_t insize)
{
	crypt4_t *c = (crypt4_t *)ptr;
	uint8_t *ct = (uint8_t *)ciphertext;
	size_t i;
	FUN("crypt4_encrypt_msg");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (insize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %x, want %x)\n",
			(unsigned)*outsize, (unsigned)insize));
		errno = ENOMEM;
		return -1;
	}
	if (CRYPT4_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("magic is not 0x%x\n",
			CRYPT4_MAGIC));
		errno = EINVAL;
		return -1;
	}
	if (ciphertext != plaintext) {
		memcpy(ciphertext, plaintext, insize);
	}
	for (i = 0; i < insize; i++)
		*ct++ ^= crypt4_next(c);

	c->encrypt_count += insize;

	*outsize = insize;
	return 0;
}

/***********************************************************************
 *  decrypt_msg()
 ***********************************************************************/
int crypt4_decrypt_msg(void *ptr, void *plaintext, void *ciphertext,
	size_t *outsize, size_t insize)
{
	crypt4_t *c = (crypt4_t *)ptr;
	uint8_t *pt = (uint8_t *)plaintext;
	size_t i;
	FUN("crypt4_decrypt_msg");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (insize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %x, want %x)\n",
			(unsigned)*outsize, (unsigned)insize));
		errno = ENOMEM;
		return -1;
	}
	if (CRYPT4_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("magic is not 0x%x\n",
			CRYPT4_MAGIC));
		errno = EINVAL;
		return -1;
	}
	if (plaintext != ciphertext) {
		memcpy(plaintext, ciphertext, insize);
	}
	for (i = 0; i < insize; i++)
		*pt++ ^= crypt4_next(c);

	c->decrypt_count += insize;

	*outsize = insize;
	return 0;
}

#if	CRYPT4_TEST
configuration_t *g_conf = NULL;
#define	SIZE	1024

int pm_vsnprintf(char *dst, size_t size, const char *fmt, va_list ap)
{
	return vsnprintf(dst, size, fmt, ap);
}

int pm_snprintf(char *dst, size_t size, const char *fmt, ...)
{
	va_list ap;
	int len;

	va_start(ap, fmt);
	len = vsnprintf(dst, size, fmt, ap);
	va_end(ap);
	return len;
}

int main(int argc, char **argv)
{
	char rndname[MAXPATHLEN] = "/dev/urandom";
	char outname[MAXPATHLEN] = "crypt4.bin";
	FILE *rnd, *out;
	void *ctx;
	int mode = 0;
	size_t outsize = SIZE;
	size_t insize = SIZE;
	size_t count = 1;
	size_t n;
	size_t initial_size;
	unsigned char *initial;
	unsigned char *plaintext;
	unsigned char *ciphertext;
	int i;
	int rc;

	for (i = 1; i < argc; i++) {
		if (0 == strcmp(argv[i], "--diehard")) {
			insize = outsize = 0x1000;
			count = 0x1000;
		} else if (0 == strncmp(argv[i], "-r", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-r")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-o", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-o")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-m", 2) && strlen(argv[i]) > 2) {
			mode = strtol(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-m")) {
			mode = strtoul(argv[++i], NULL, 0);
		} else if (0 == strncmp(argv[i], "-s", 2) && strlen(argv[i]) > 2) {
			insize = outsize = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-s")) {
			insize = outsize = strtoul(argv[++i], NULL, 0);
		} else if (0 == strncmp(argv[i], "-c", 2) && strlen(argv[i]) > 2) {
			count = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-c")) {
			count = strtoul(argv[++i], NULL, 0);
		}
	}

	fpout = NULL;
	if (1 == mode) {
		fpout = fopen("crypt4.log", "w");
		mode = 0;
	}

	initial_size = 256;

	printf("open random source: %s\n", rndname);
	rnd = fopen(rndname, "rb");
	if (NULL == rnd) {
		perror(rndname);
		exit(1);
	}

	initial = xcalloc(initial_size, sizeof(uint8_t));
	if (initial_size != fread(initial, 1, initial_size, rnd)) {
		perror("PRNG data");
		exit(2);
	}
	fclose(rnd);

	printf("create output file: %s\n", outname);
	out = fopen(outname, "wb");
	if (NULL == out) {
		perror(outname);
		exit(3);
	}

	plaintext = xcalloc(insize, sizeof(uint8_t));
	ciphertext = xcalloc(outsize, sizeof(uint8_t));

	if (0 != (rc = crypt4_init(&ctx, mode, initial, initial_size))) {
		fprintf(stderr, "crytp4_init failed (%d)", rc);
		exit(4);
	}

	for (n = 0; n < count; n++) {
		outsize = insize;
		if (0 != crypt4_encrypt_msg(ctx, ciphertext, plaintext,
			&outsize, insize)) {
			fprintf(stderr, "crytp4_encrypt_msg failed (%d)", rc);
			exit(5);
		}
		if (outsize != fwrite(ciphertext, 1, outsize, out)) {
			perror("fwrite");
			exit(6);
		}
	}

	fclose(out);

	return 0;
}

#endif
