/* Notes *//*{{{C}}}*//*{{{*/
/*

The functions in this module implement common editor primitives.  They
respect the fold structure and reposition the screen cursor position.
If needed, they cause a full screen redraw.  There is only one condition
under which the editor may bypass this module and directly call buffer
gap functions for buffers currently associated with a display: If they
do not modify the buffer contents and if they do not move the cursor
further than the current line.

*/
/*}}}*/

/* #includes *//*{{{*/
#include "config.h"

#ifdef DMALLOC
#include "dmalloc.h"
#endif

#include <sys/types.h>
#include <sys/wait.h>
#include <assert.h>
#include <ctype.h>
#include <stdarg.h> /* required by some curses.h implementations */
#ifdef NEED_NCURSES
#include <ncurses.h>
#else
#include <curses.h>
#endif
#include <errno.h>
#include <fcntl.h>
#ifdef HAVE_GETTEXT
#include <libintl.h>
#define _(String) gettext(String)
#else
#define _(String) String
#endif
#include <limits.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef BROKEN_REALLOC
#define realloc(s,l) myrealloc(s,l)
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "buffer.h"
#include "display.h"
#include "macro.h"
#include "misc.h"
#include "msgline.h"
/*}}}*/
/* #defines */ /*{{{*/
#define CO_L "co -l"
#define CI_U "ci -u"
/*}}}*/

/* variables */ /*{{{*/
static struct Display displays[MAXDISPLAYS];
static struct Display *current;
/* All buffers used for displays or attachable to displays. */
struct Buffer *buffers[MAXDISPLAYS];
/*}}}*/

/* ds_chstr      -- convert current character to displayed string */ /*{{{*/
static const char *ds_chstr(unsigned char c, char mark, int opened, unsigned int x, unsigned int tabsize, int *highlight)
{
  static char control[]="^?";
  static char metadash[]="M-?";
  static char character[]="?";
  static char tab[MAXTAB+1]="";

  if (tab[0]!=' ') { memset(tab,' ',MAXTAB); tab[MAXTAB]='\0'; }
  if (c=='\0')
  {
    switch (mark)
    {
      case NUL: return "^@";
      case FOLDSTART: if (highlight) *highlight=1; return (opened ? "{{{" : "...");
      case FOLDEND:   if (highlight) *highlight=1; return "}}}";
      default: assert(0); return (const char*)0;
    }
  }
  else if (tabsize && c=='\t')
  {
    if (highlight) *highlight=0;
    return (tab+(MAXTAB-tabsize)+(x%tabsize));
  }
  else if (c>='\1' && c<='\37')
  {
    if (highlight) *highlight=0;
    control[1]=c+'A'-1;
    return control;
  }
  else if (c=='\177')
  {
    if (highlight) *highlight=0;
    control[1]='?';
    return control;
  }
  else if (c>=(unsigned char)'\200' && c<=(unsigned char)'\237')
  {
    if (highlight) *highlight=0;
    metadash[2]=c-'\200'+'A'-1;
    return metadash;
  }
  else
  {
    if (highlight) *highlight=0;
    character[0]=c;
    return character;
  }  
}
/*}}}*/
/* ds_drawline   -- draw current line */ /*{{{*/
static void ds_drawline(struct Display *d, int note_x, unsigned int y, int opened)
{
  int cur_gapstart,cur_linestart;
  char c,mark;
  unsigned int curx;
  unsigned int offx;

  move(d->oriy+y,d->orix);
  cur_gapstart=d->buffer->curch;
  bf_linestart(d->buffer,1);
  cur_linestart=d->buffer->curch;
  if (note_x) /* note x position and determine display offset *//*{{{*/
  {
    curx=0;
    while (bf_rchar(d->buffer,&c,&mark) && c!='\n')
    {
      if (d->buffer->curch==cur_gapstart) d->x=curx;
      curx+=strlen(ds_chstr(c,mark, opened, curx,d->tabsize,(int*)0));
      bf_forward(d->buffer);
    }
    if (d->buffer->curch==cur_gapstart) d->x=curx;
    while (d->buffer->curch>cur_linestart) bf_backward(d->buffer);
    while (d->buffer->curch<cur_linestart) bf_forward(d->buffer);
    if (d->x>d->offx+d->maxx-1-(d->mode&LINENO ? LINENOWIDTH : 0))
    {
      d->offx=d->x-d->maxx+1+(d->mode&LINENO ? LINENOWIDTH : 0);
      if (d->mode&SIDESCROLL) d->update=ALL;
    }
    else if (d->x<d->offx)
    {
      d->offx=d->x;
      if (d->mode&SIDESCROLL) d->update=ALL;
    }
  }
  /*}}}*/
  if ((d->mode&SIDESCROLL)==0 && y!=d->y) offx=0;
  else offx=d->offx;
  /* draw line *//*{{{*/
  if (d->mode&LINENO)
  {
    char s[LINENOWIDTH+1];

    sprintf(s,"%0*u:",LINENOWIDTH-1,d->buffer->cury+1);
    addstr(s);
  }
  for (curx=0; curx<d->maxx+offx && bf_rchar(d->buffer,&c,&mark) && c!='\n'; )
  {
    const char *s;
    int highlight;
      
    s=ds_chstr(c,mark, opened,curx,d->tabsize,&highlight);
    if (highlight && d->mode&BOLDMARKS) wattron(stdscr,A_BOLD);
    while (*s && curx<d->maxx+offx-(d->mode&LINENO ? LINENOWIDTH : 0))
    {
      if (curx>=offx) addch((chtype)(unsigned char)*s);
      ++s;
      ++curx;
    }
    if (highlight) wattroff(stdscr,A_BOLD);
    if (bf_forward(d->buffer)==0) break;
  }
  if (curx>=d->maxx+offx)
  {
    move(d->oriy+y,d->orix+d->maxx-1);
    addch((chtype)'$');
  }
  else
  {
    int clr;

    for (clr=curx; clr<d->maxx+offx-(d->mode&LINENO ? LINENOWIDTH : 0); ++clr)
    {
      if (clr>=offx) addch(' ');
    }
  }
  /*}}}*/
  while (d->buffer->curch>cur_gapstart) bf_backward(d->buffer);
  while (d->buffer->curch<cur_gapstart) bf_forward(d->buffer);
}
/*}}}*/
/* hangup        -- save everything after a signal arrived */ /*{{{*/
static void hangup(int sig)
{
  int i;
  const struct passwd *pw;
  int fd[2];

  for (i=0; i<MAXDISPLAYS; ++i)
  {
    if (displays[i].buffer)
    {
      if (displays[i].buffer->mode&CHANGED)
      {
        char name[_POSIX_PATH_MAX];
        size_t len=strlen(displays[i].buffer->name);

        if (len>(_POSIX_PATH_MAX-5)) len=_POSIX_PATH_MAX-5;
        strncpy(name,displays[i].buffer->name,len);
        name[len]='\0';
        strcat(name+len,".hup");
        bf_save(displays[i].buffer,name,"{{{","}}}",(unsigned int*)0);
      }
      else ds_free(&displays[i]);
    }
  }
  ds_exit(0);
#ifdef HAVE_SENDMAIL
  if ((pw=getpwuid(getuid()))!=(struct passwd*)0 && pipe(fd)!=-1)
  {
    switch (fork())
    {
      case 0:
      {
        close(fd[1]);
        dup2(fd[0],0);
        close(fd[0]);
        execlp(SENDMAIL,"sendmail","-t",(const char*)0);
        exit(127);
      }
      case -1: break;
      default:
      {
        FILE *fp;

        close(fd[0]);
        fp=fdopen(fd[1],"w");
        fprintf(fp,"To: %s\n",pw->pw_name);
        fprintf(fp,_("Subject: Abnormal termination of fe caused by signal %d.\n"),sig);
        fprintf(fp,"\n");
        fprintf(fp,_("The following files contain the contents of unsaved buffers:\n"));
        for (i=0; i<MAXDISPLAYS; ++i)
        {
          if (displays[i].buffer)
          {
            char name[_POSIX_PATH_MAX];
            size_t len=strlen(displays[i].buffer->name);

            assert(displays[i].buffer->refcnt==2);
            if (len>(_POSIX_PATH_MAX-5)) len=_POSIX_PATH_MAX-5;
            strncpy(name,displays[i].buffer->name,len);
            name[len]='\0';
            strcat(name+len,".hup");
            fprintf(fp,"%s\n",name);
            ds_free(&displays[i]);
          }
        }
        fclose(fp);
      }
    }
  }
#endif
  exit(2);
}
/*}}}*/

/* ds_init       -- init editor */ /*{{{*/
int ds_init(void)
{
  int fd;
  struct sigaction hupact;

  if (isatty(0)==0)
  {
    fd=dup(0); assert(fd!=-1);
    close(0);
    if (open("/dev/tty",O_RDWR)!=0)
    {
      fputs("Initialising terminal failed (no controlling tty).\n",stderr);
      exit(1);
    }
  }
  else fd=-1;
  if (getenv("TERM")==(const char*)0)
  {
    fputs("Initialising terminal failed (TERM variable not set).\n",stderr);
    exit(1);
  }
  initscr();
  noecho();
  raw();
  nonl();
#ifndef BROKEN_CURSES_IDLOK
  idlok(stdscr,TRUE);
#endif
#ifdef HAVE_IDCOK
  idcok(stdscr,TRUE);
#endif
  scrollok(stdscr,TRUE);
  keypad(stdscr,TRUE);
  clear();
  hupact.sa_handler=hangup;
  sigemptyset(&hupact.sa_mask);
  sigaddset(&hupact.sa_mask,SIGHUP);
  sigaddset(&hupact.sa_mask,SIGTERM);
  hupact.sa_flags=0;
  sigaction(SIGHUP,&hupact,(struct sigaction *)0);
  sigaction(SIGTERM,&hupact,(struct sigaction *)0);
  return fd;
}
/*}}}*/
/* ds_exit       -- exit editor */ /*{{{*/
void ds_exit(int regular)
{
  showmsg(regular ? _("[Bye]") : _("[Signal arrived, bye]"));
  scrl(1);
  refresh();
  echo();
  noraw();
  endwin();
}
/*}}}*/
/* ds_winch      -- window changed */ /*{{{*/
void ds_winch(void)
{
  struct Display *i,*bottom=(struct Display*)0;

  for (i=displays; i<displays+sizeof(displays)/sizeof(displays[0]); ++i)
  {
    if (i->buffer)
    {
      i->maxx=COLS;
      i->update=ALL;
      if (bottom==(struct Display*)0 || bottom->oriy+bottom->maxy<i->oriy+i->maxy) bottom=i;
    }
  }
  if (bottom->oriy<LINES-1) bottom->maxy=LINES-1-bottom->oriy;
}
/*}}}*/
/* ds_alloc      -- allocate display */ /*{{{*/
struct Display *ds_alloc(struct Buffer *b, unsigned int orix, unsigned int oriy, unsigned int maxx, unsigned int maxy)
{
  struct Display *d;
  int i;

  for (i=0; i<sizeof(displays)/sizeof(displays[0]) && displays[i].buffer; ++i);
  if (i==sizeof(displays)/sizeof(displays[0])) return (struct Display*)0;
  d=&(displays[i]);
  d->buffer=b;
  ++d->buffer->refcnt;
  d->x=0;
  d->offx=0;
  d->y=0;
  d->maxx=maxx;
  d->maxy=maxy;
  d->orix=orix;
  d->oriy=oriy;
  d->update=ALL;
  d->mode=0;
  d->tabsize=8;
  d->cursor=bf_mark(b,-1);
  d->mark=bf_mark(b,-1);
  if (d->buffer->refcnt==1)
  {
    ++d->buffer->refcnt;
    for (i=0; i<MAXDISPLAYS; ++i) if (buffers[i]==(struct Buffer*)0)
    {
      buffers[i]=b;
      break;
    }
  }
  return d;
}
/*}}}*/
/* ds_realloc    -- allocate new buffer to existing display */ /*{{{*/
void ds_realloc(struct Display *d, struct Buffer *b)
{
  int i;

  if (d->buffer) --d->buffer->refcnt;
  d->buffer=b;
  ++d->buffer->refcnt;
  if (d->buffer->refcnt==1)
  {
    ++d->buffer->refcnt;
    for (i=0; i<MAXDISPLAYS; ++i) if (buffers[i]==(struct Buffer*)0)
    {
      buffers[i]=b;
      break;
    }
  }
  d->update=ALL;
  d->cursor=bf_mark(d->buffer,-1);
  d->mark=bf_mark(d->buffer,-1);
}
/*}}}*/
/* ds_free       -- free display */ /*{{{*/
void ds_free(struct Display *d)
{
  int i;

  bf_unmark(d->buffer,d->cursor);
  bf_unmark(d->buffer,d->mark);
  if (--d->buffer->refcnt==1)
  {
    for (i=0; i<MAXDISPLAYS; ++i) if (buffers[i]==d->buffer)
    {
      buffers[i]=(struct Buffer*)0;
      break;
    }
    bf_free(d->buffer);
  }
  d->buffer=(struct Buffer*)0;
  if (d==current) current=(struct Display*)0;
}
/*}}}*/
/* ds_detach     -- detach display */ /*{{{*/
void ds_detach(struct Display *d)
{
  bf_unmark(d->buffer,d->cursor);
  bf_unmark(d->buffer,d->mark);
  --d->buffer->refcnt;
  d->buffer=(struct Buffer*)0;
  if (d==current) current=(struct Display*)0;
}
/*}}}*/
/* ds_redraw     -- redraw screen */ /*{{{*/
void ds_redraw(void)
{
  touchwin(curscr);
  wrefresh(curscr);
}
/*}}}*/
/* ds_current    -- select current display */ /*{{{*/
void ds_current(struct Display *d)
{
  if (current) bf_mark(current->buffer,current->cursor);
  current=d;
  bf_tomark(current->buffer,current->cursor);
}
/*}}}*/
/* ds_tocol      -- position cursor to a specific column */ /*{{{*/
void ds_tocol(unsigned int tox)
{
  char c,mark;
  int x,dx;

  c='\0';
  x=0;
  while (bf_lchar(current->buffer,&c,&mark) && c!='\n' && bf_backward(current->buffer));
  if (c=='\n' || current->buffer->gapstart==current->buffer->buffer)
  {
    while (tox)
    {
      if (bf_rchar(current->buffer,&c,&mark)==0) break;
      if (c=='\n') break;
      if ((dx=strlen(ds_chstr(c,mark,1,x,current->tabsize,(int*)0)))<=tox)
      {
        tox-=dx;
        x+=dx;
      }
      else tox=0;
      bf_forward(current->buffer);
    }
  }
}
/*}}}*/
/* ds_torow      -- position cursor to a specific row */ /*{{{*/
void ds_torow(struct Display *d, unsigned int row)
{
  d->y=row;
  d->update=(d->update&~UPDATEMASK)|ALL;
}
/*}}}*/
/* ds_topos      -- go to buffer position */ /*{{{*/
void ds_topos(struct Display *d, const char *start)
{
  while (d->buffer->gapstart<start) bf_forward(d->buffer);
  while (d->buffer->gapstart>start) bf_backward(d->buffer);
  d->update=(d->update&~UPDATEMASK)|ALL;
}
/*}}}*/
/* ds_refresh    -- refresh displays */ /*{{{*/
void ds_refresh(void)
{
  /* variables *//*{{{*/
  struct Display *d;
  char *cur_gapstart;
  int i,j;
  /*}}}*/

  bf_mark(current->buffer,current->cursor);
  for (i=0; i<sizeof(displays)/sizeof(displays[0]); ++i) if (displays[i].buffer && displays[i].update&MODIFIED)
  {
    for (j=0; j<sizeof(displays)/sizeof(displays[0]); ++j) if (displays[i].buffer && displays[j].buffer==displays[i].buffer && i!=j) displays[j].update=ALL;
  }
  for (i=0; i<sizeof(displays)/sizeof(displays[0]); ++i) if (displays[i].buffer)
  {
    d=&(displays[i]);
    do
    {
      bf_tomark(d->buffer,d->cursor);

      /* note where start of gap is *//*{{{*/
      cur_gapstart=d->buffer->gapstart;
      /*}}}*/
      if (d->update&LINE) ds_drawline(d,1,d->y,0);
      if (d->update&ALL)
      {
        /* variables */ /*{{{*/
        int opened;
        unsigned int cury;
        char c,m;
        /*}}}*/

        d->update=NONE;
        /* draw current line and all previous lines until top of screen */ /*{{{*/
        cury=d->y+1;
        opened=0;
        do
        {
          --cury;
          /* draw current line *//*{{{*/
/* 0 means lines are not scrolled sideways when moving up from a scrolled line */
#if 0
          ds_drawline(d,cury==d->y,cury,opened);
#else
          ds_drawline(d,0,cury,opened);
#endif
          /*}}}*/
          /* go to beginning of it *//*{{{*/
          bf_linestart(d->buffer,1);
          /*}}}*/
          /* go one more back to get to end of previous line *//*{{{*/
          if (bf_backward(d->buffer)==0) break;
          /*}}}*/
          /* if this is an endfold, go to its matching startfold *//*{{{*/
          opened=1;
          while (bf_lchar(d->buffer,&c,&m))
          {
            if (c=='\0' && m==FOLDEND)
            {
              unsigned int level;

              level=0;
              do
              {
                if (bf_backward(d->buffer)==0) break;
                if (bf_rchar(d->buffer,&c,&m)==0) break;
                if (c=='\0' && m==FOLDSTART) --level;
                if (c=='\0' && m==FOLDEND) ++level;
              } while (!(c=='\0' && m==FOLDSTART && level==0));
              opened=0;
              break;
            }
            else if (c=='\n') break;
            else bf_backward(d->buffer);
          }
          /*}}}*/
        } while (cury);
        /*}}}*/
        /* clear all lines above start of text */ /*{{{*/
        if (cury && d->mode&SHOWBEGEND)
        {
          const char *msg;
          size_t len;
          int clr;

          msg=_("{{{ Beginning of file");
          len=strlen(msg);
          --cury;
          move(d->oriy+cury,d->orix);
          for (clr=0; clr<d->maxx; ++clr) addch(clr<len ? msg[clr] : ' ');
        }
        if (cury) { d->y-=cury; d->update=ALL; }
        else while (cury)
        {
          int clr;      

          --cury;
          move(d->oriy+cury,d->orix);
          for (clr=0; clr<d->maxx; ++clr) addch(' ');
        }
        /*}}}*/
        /* move behind current line */ /*{{{*/
        while (d->buffer->gapstart>cur_gapstart) bf_backward(d->buffer);
        while (d->buffer->gapstart<cur_gapstart) bf_forward(d->buffer);
        if (bf_isfold(d->buffer,FOLDSTART,LEFT|RIGHT))
        {
          int r;
        
          bf_lineend(d->buffer,1);
          r=bf_foldend(d->buffer,1,(unsigned int*)0);
          assert(r);
        }
        /*}}}*/
        /* draw all lines below current line */ /*{{{*/
        bf_lineend(d->buffer,1);
        cury=d->y;
        for (; bf_forward(d->buffer) && cury<d->maxy-1; ++cury)
        {
          /* draw current line *//*{{{*/
          ds_drawline(d,0,cury+1,0);
          /*}}}*/
          /* if this is an startfold, go to its matching endfold *//*{{{*/
          if (bf_isfold(d->buffer,FOLDSTART,LEFT|RIGHT))
          {
            int r;

            bf_lineend(d->buffer,1);
            r=bf_foldend(d->buffer,1,(unsigned int*)0);
            assert(r);
          }
          /*}}}*/
          /* go to end of it *//*{{{*/
          bf_lineend(d->buffer,1);
          /*}}}*/
        }
        if (bf_lchar(d->buffer, &c, &m) && c!='\n') ++cury;
        /*}}}*/
        /* clear all lines below end of text */ /*{{{*/
        if (cury<d->maxy-1 && d->mode&SHOWBEGEND)
        {
          const char *msg;
          size_t len;
          int clr;

          msg=_("}}} End of file");
          len=strlen(msg);
          move(d->oriy+cury,d->orix);
          for (clr=0; clr<d->maxx; ++clr) addch(clr<len ? msg[clr] : ' ');
          ++cury;
        }
        for (; cury<d->maxy-1; ++cury)
        {
          int clr;

          move(d->oriy+cury,d->orix);
          for (clr=0; clr<d->maxx; ++clr) addch(' ');
        }
        /*}}}*/
      }
      else d->update=NONE;
      /* move back to old start of gap *//*{{{*/
      while (d->buffer->gapstart>cur_gapstart) bf_backward(d->buffer);
      while (d->buffer->gapstart<cur_gapstart) bf_forward(d->buffer);
      /*}}}*/
    } while (d->update&ALL);
    /* draw status line *//*{{{*/
    {
      int x;
      time_t t;
      static time_t ot=(time_t)0;
      static char timebuf[40];
      static char *titlebuf=(char*)0;
      static size_t titlebuflen=0;

      if (titlebuflen<80) titlebuf=malloc(titlebuflen=80);
      if (d->buffer->mode&CHANGED) strcpy(titlebuf," * ");
      else strcpy(titlebuf,"   ");
      strcat(titlebuf,"fe " VERSION " (");
      strcat(titlebuf,d->buffer->lang && d->buffer->lang->name[0] ? d->buffer->lang->name : _("No language"));
      if (d->buffer->mode&READONLY)
      {
        strcat(titlebuf,",");
        strcat(titlebuf,_("Read-only"));
      }
      if (d->mode&OVERWRITE)
      {
        strcat(titlebuf,",");
        strcat(titlebuf,_("Overwrite"));
      }
      if (d->mode&MAGIC)
      {
        strcat(titlebuf,",");
        strcat(titlebuf,_("Magic"));
      }
      strcat(titlebuf,") ");
      if (d->buffer->name)
      {
        size_t namelen;
        
        strcat(titlebuf,"\"");
        if (titlebuflen<(namelen=(80+strlen(d->buffer->name)))) titlebuf=realloc(titlebuf,titlebuflen=namelen);
        strcat(titlebuf,d->buffer->name);
        strcat(titlebuf,"\"");
      }
      else strcat(titlebuf,_("'unnamed'"));
      strcat(titlebuf," ");
      if (d->mode&POSITION)
      {
        sprintf(titlebuf+strlen(titlebuf),_("Line %d Column %d"),d->buffer->cury+1,d->x);
        strcat(titlebuf," ");
      }
      if (d->mode&TIME)
      {
        t=time((time_t*)0);
        if (t>ot)
        {
          struct tm *tm;

          tm=localtime(&t);
          assert(tm!=(struct tm*)0);
          ot=t+59-tm->tm_sec;
          strftime(timebuf,sizeof(timebuf),_("(%H:%M)"),tm);
        }
        strcat(titlebuf,timebuf);
      }
      if (titlebuflen>d->maxx) titlebuf[d->maxx]='\0';
      move(d->oriy+d->maxy-1,d->orix);
      wattron(stdscr,A_REVERSE);
      addstr(titlebuf);
      for (x=strlen(titlebuf); x<d->maxx; ++x) addch(' ');
      wattroff(stdscr,A_REVERSE);
    }
    /*}}}*/
  }
  move(current->oriy+current->y,current->orix+current->x-current->offx+(current->mode&LINENO ? LINENOWIDTH : 0));
  bf_tomark(current->buffer,current->cursor);
}
/*}}}*/
/* ds_forward    -- move one character forward, skipping folds */ /*{{{*/
int ds_forward(void)
{
  char c,mark;
  int res;
        
  if (bf_rchar(current->buffer,&c,&mark)==0) return 0;
  if (c=='\n') 
  {
    int olddy;

    olddy=current->y;
    if (bf_isfold(current->buffer,FOLDEND,LEFT|RIGHT)) current->update=(current->update&~UPDATEMASK)|ALL;
    if (current->y<current->maxy-2) ++current->y; else current->update=(current->update&~UPDATEMASK)|ALL;
    if (bf_isfold(current->buffer,FOLDSTART,LEFT|RIGHT))
    {
      char *gapstart;

      gapstart=current->buffer->gapstart;
      if (bf_foldend(current->buffer,1,(unsigned int*)0)==0 || bf_lineend(current->buffer,1)==0 || (res=bf_forward(current->buffer))==0)
      {
        while (current->buffer->gapstart>gapstart) bf_backward(current->buffer);
        current->y=olddy;
        return 0;
      }
    }
    else res=bf_forward(current->buffer);
    if ((current->mode&SIDESCROLL)==0 && current->offx)
    {
      current->offx=0;
      current->update=(current->update&~UPDATEMASK)|ALL;
    }
  }
  else res=bf_forward(current->buffer);
  if ((current->update&UPDATEMASK)==NONE) current->update|=LINE;
  return (res);
}
/*}}}*/
/* ds_backward   -- move one character backward, skipping folds *//*{{{*/
int ds_backward(void)
{
  char c,mark;
      
  if (bf_lchar(current->buffer,&c,&mark)==0) return 0;
  bf_backward(current->buffer);
  if (c=='\n')
  {
    if (bf_isfold(current->buffer,FOLDSTART,LEFT|RIGHT)) current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    if (current->y>0) --current->y; else current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    if (bf_isfold(current->buffer,FOLDEND,LEFT|RIGHT))
    {
      bf_foldstart(current->buffer,0,(unsigned int*)0);
      bf_lineend(current->buffer,1);
    }
    if ((current->mode&SIDESCROLL)==0 && current->offx)
    {
      current->offx=0;
      current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    }
  }
  if ((current->update&UPDATEMASK)==NONE) current->update|=LINE;
  return 1;
}
/*}}}*/
/* ds_prevline   -- previous line, skipping folds *//*{{{*/
int ds_prevline(void) /* 0=failed, 1=success, 2=success & hit start */
{
  char *gapstart;
  int hitstart=0;

  gapstart=current->buffer->gapstart;
  if (current->buffer->cury==0 || bf_linestart(current->buffer,1)==0 || bf_backward(current->buffer)==0)
  {
    while (current->buffer->gapstart<gapstart) bf_forward(current->buffer);
    if (current->y==0 && current->mode&SHOWBEGEND)
    {
      current->y=1;
      current->update=(current->update&~UPDATEMASK)|ALL;
      return 1;
    }
    else return 0;
  }
  if (bf_isfold(current->buffer,FOLDEND,LEFT|RIGHT))
  {
    if (bf_foldstart(current->buffer,0,(unsigned int*)0)==0)
    {
      while (current->buffer->gapstart<gapstart) bf_forward(current->buffer);
      if (current->y==0 && current->mode&SHOWBEGEND)
      {
        current->y=1;
        current->update=(current->update&~UPDATEMASK)|ALL;
        return 1;
      }
      else return 0;
    }
  }
  else if (bf_isfold(current->buffer,FOLDSTART,LEFT|RIGHT))
  {
    current->update=(current->update&~UPDATEMASK)|ALL;
    hitstart=1;
  }
  if ((current->mode&SIDESCROLL)==0 && current->offx)
  {
#if 0
    current->offx=0;
#endif
    current->update=(current->update&~UPDATEMASK)|ALL;
  }
  if (current->y>0) --current->y;
  else current->update=(current->update&~UPDATEMASK)|ALL;
  if (current->y==0 && current->buffer->cury>0)
  {
    current->y=(current->maxy-1)/2;
    current->update=(current->update&~UPDATEMASK)|ALL;
  }
  ds_tocol(current->x);
  return 1+hitstart;
}
/*}}}*/
/* ds_nextline   -- next line, skipping folds *//*{{{*/
int ds_nextline(void)
{
  char *gapstart;

  gapstart=current->buffer->gapstart;
  if (bf_isfold(current->buffer,FOLDSTART,LEFT|RIGHT))
  {
    if (bf_lineend(current->buffer,1)==0 || bf_foldend(current->buffer,1,(unsigned int*)0)==0)
    {
      while (current->buffer->gapstart>gapstart) bf_backward(current->buffer);
      return 0;
    }
  }
  else if (bf_isfold(current->buffer,FOLDEND,LEFT|RIGHT)) current->update=(current->update&~UPDATEMASK)|ALL;
  if (bf_lineend(current->buffer,1))
  { 
    if (bf_forward(current->buffer))
    {
      if (current->y<current->maxy-2) ++current->y;
      if (current->y==(current->maxy-2))
      {
        if (bf_lineend(current->buffer,1) && bf_forward(current->buffer))
        {
          current->update=(current->update&~UPDATEMASK)|ALL;
          current->y=(current->maxy-1)/2;
          bf_backward(current->buffer);
        }
      }
    }
    else
    {
      while (current->buffer->gapstart>gapstart) bf_backward(current->buffer);
      return 0;
    }
  }
  else
  {
    while (current->buffer->gapstart>gapstart) bf_backward(current->buffer);
    return 0;
  }
  if ((current->mode&SIDESCROLL)==0 && current->offx)
  {
#if 0
    current->offx=0;
#endif
    current->update=(current->update&~UPDATEMASK)|ALL;
  }
  ds_tocol(current->x);
  return 1;
}
/*}}}*/
/* ds_closefold  -- close fold *//*{{{*/
int ds_closefold(void)
{
  unsigned int back;

  if (bf_foldstart(current->buffer,1,&back))
  {
    if (back>current->y) current->y=0; else current->y-=back;
    ds_tocol(current->x);
    current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    return 1;
  }
  else return 0;
}
/*}}}*/
/* ds_openfold   -- open fold *//*{{{*/
int ds_openfold(void)
{
  if (bf_isfold(current->buffer,FOLDSTART,LEFT|RIGHT))
  {
    bf_lineend(current->buffer,1);
    if (bf_forward(current->buffer)==1) if (current->y<current->maxy-2) ++current->y;
    ds_tocol(current->x);
    current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    return 1;
  }
  else return 0;
}
/*}}}*/
/* ds_begin      -- go to begin of buffer *//*{{{*/
void ds_begin(void)
{
  bf_begin(current->buffer);
  current->y=(current->mode&SHOWBEGEND)!=0;
  current->update=(current->update&~UPDATEMASK)|ALL|LINE;
}
/*}}}*/
/* ds_end        -- go to end of buffer *//*{{{*/
void ds_end(void)
{
  bf_end(current->buffer);
  current->y=current->maxy-2;
  current->update=(current->update&~UPDATEMASK)|ALL|LINE;
}
/*}}}*/
/* ds_beginfold  -- go to beginning of current fold *//*{{{*/
int ds_beginfold(void)
{
  int x;
  unsigned int back;

  x=current->x;
  if (bf_foldstart(current->buffer,1,&back)==1)
  {
    if (current->y>(back-1)) current->y-=(back-1);
    else
    {
      current->y=0;
      current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    }
    bf_lineend(current->buffer,1);
    bf_forward(current->buffer);
    ds_tocol(current->x);
    return 1;
  }
  else return 0;
}
/*}}}*/
/* ds_endfold    -- go to end of current fold *//*{{{*/
int ds_endfold(void)
{
  int x;
  unsigned int forward;

  x=current->x;
  if (bf_foldend(current->buffer,1,&forward)==1)
  {
    if ((current->y+forward)<(current->maxy-1)) current->y+=forward;
    else
    {
      current->y=current->maxy-1;
      current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    }
    ds_prevline();
    ds_tocol(x);
    return 1;
  }
  else return 0;
}
/*}}}*/
/* ds_beginline  -- go to beginning of line */ /*{{{*/
void ds_beginline(void)
{
  bf_linestart(current->buffer,0);
  if ((current->update&UPDATEMASK)==NONE) current->update|=LINE;
}
/*}}}*/
/* ds_endline    -- go to end of line */ /*{{{*/
void ds_endline(void)
{
  bf_lineend(current->buffer,0);
  if ((current->update&UPDATEMASK)==NONE) current->update|=LINE;
}
/*}}}*/
/* ds_delmode    -- delete mode *//*{{{*/
void ds_delmode(enum Displaymode mode)
{
  current->mode&=~mode;
  current->update=(current->update&~UPDATEMASK)|ALL;
}
/*}}}*/
/* ds_addmode    -- add mode *//*{{{*/
void ds_addmode(enum Displaymode mode)
{
  current->mode|=mode;
  current->update=(current->update&~UPDATEMASK)|ALL;
}
/*}}}*/
/* ds_tabsize    -- get/set tab size *//*{{{*/
unsigned int ds_tabsize(int tabsize)
{
  if (tabsize>=0)
  {
    current->tabsize=tabsize;
    current->update=(current->update&~UPDATEMASK)|ALL;
  }
  return current->tabsize;
}
/*}}}*/
/* ds_insertch   -- insert a character */ /*{{{*/
int ds_insertch(char c)
{
  if (current->mode&OVERWRITE)
  {
    char ch,mark;

    if (bf_rchar(current->buffer,&ch,&mark)==0 || (ch=='\0' && mark!=NUL) || ch=='\n')
    {
      showmsg(_("[You can not overwrite this position]"));
      return 0;
    }
    bf_forward(current->buffer);
    bf_delete(current->buffer);
  }
  if (c=='\0')
  {
    int s;

    if ((s=bf_insert(current->buffer,'\0',NUL)))
    {
      current->update|=MODIFIED;
      if ((current->update&UPDATEMASK)==NONE) current->update|=LINE;
      return s;
    }
    else
    {
      showmsg(current->buffer->mode&READONLY ? _("[Buffer is read-only]") : _("[Out of memory]"));
      return 0;
    }
  }
  else if (bf_insert(current->buffer,c,0))
  {
    if (c=='\n')
    {
      if (current->y<current->maxy-2) ++current->y;
      current->update=ALL|MODIFIED|LINE;
    }
    if ((current->update&UPDATEMASK)==NONE) current->update|=LINE|MODIFIED;
    return 1;
  }
  else
  {
    showmsg(current->buffer->mode&READONLY ? _("[Buffer is read-only]") : _("[Out of memory]"));
    return 0;
  }
}        
/*}}}*/
/* ds_deletech   -- delete a character *//*{{{*/
int ds_deletech(int previous)
{
  char c,mark;

  if (current->buffer->mode&READONLY) showmsg(_("[Buffer is read-only]"));
  if (previous) /* delete previous character */ /*{{{*/
  {
    if (bf_lchar(current->buffer,&c,&mark))
    {
      if (c=='\n')
      {
        int isfold;

        isfold=bf_isfold(current->buffer,FOLDSTART|FOLDEND,LEFT|RIGHT);
        bf_backward(current->buffer);
        if (isfold && bf_isfold(current->buffer,FOLDSTART|FOLDEND,LEFT|RIGHT))
        {
          bf_forward(current->buffer);
          return 0;
        }
        bf_forward(current->buffer);
        if (current->y>0) --current->y;
        current->update=ALL|MODIFIED|LINE;
      }
      else if (c=='\0' && mark!=NUL) return 0;
      if (bf_delete(current->buffer))
      {
        current->update|=MODIFIED;
        if ((current->update&UPDATEMASK)==NONE) current->update|=LINE;
        return 1;
      }
      else return 0;
    }
    else return 0;
  }
  /*}}}*/
  else
  /* delete character under cursor *//*{{{*/
  {
    int cy;

    cy=current->y;
    if (bf_forward(current->buffer)==0) return 0;
    if (ds_deletech(1)==0)
    {
      bf_backward(current->buffer);
      return 0;
    }
    current->y=cy;
    return 1;
  }
  /*}}}*/
}
/*}}}*/
/* ds_mark       -- set a mark *//*{{{*/
void ds_mark(void)
{
  bf_mark(current->buffer,current->mark);
}
/*}}}*/
/* ds_swapmark   -- swap cursor and mark */ /*{{{*/
void ds_swapmark(void)
{
  int mark;

  bf_mark(current->buffer,current->cursor);
  bf_tomark(current->buffer,current->mark);
  mark=current->mark;
  current->mark=current->cursor;
  current->cursor=mark;
  current->update|=ALL|LINE;
}
/*}}}*/
/* ds_fold       -- fold region *//*{{{*/
int ds_fold(void)
{
  int crossedLine,level;
  char c,mark;
  unsigned int distance,moved;

  if (current->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }
  if (bf_isfold(current->buffer,FOLDSTART|FOLDEND,LEFT|RIGHT)) return 0;
  moved=crossedLine=level=0;
  bf_mark(current->buffer,current->cursor);
  if (current->buffer->mark[current->cursor]>current->buffer->mark[current->mark])
  {
    unsigned int x;

    x=current->cursor;
    current->cursor=current->mark;
    current->mark=x;
    bf_tomark(current->buffer,current->cursor);
  }
  distance=current->buffer->mark[current->mark]-current->buffer->mark[current->cursor];
  for (moved=0; moved<distance; ++moved) /* scan if region crosses folds and contains at least one newline *//*{{{*/
  {
    bf_forward(current->buffer);
    if (bf_lchar(current->buffer,&c,&mark))
    {
      if (c=='\n') crossedLine=1;
      else if (c=='\0') switch (mark)
      {
        case FOLDSTART: ++level; break;
        case FOLDEND: --level; break;
      }
    }
  }
  /*}}}*/
  if (bf_isfold(current->buffer,FOLDSTART|FOLDEND,LEFT|RIGHT))
  {
    for (moved=0; moved<distance; ++moved) bf_backward(current->buffer);
    return 0;
  }
  current->update=ALL|MODIFIED|LINE;
  if (crossedLine && level==0)
  {
    bf_insert(current->buffer,'\0',FOLDEND);
    ++distance;
    for (moved=0; moved<distance; ++moved) bf_backward(current->buffer);
    bf_insert(current->buffer,'\0',FOLDSTART);
    return 1;
  }
  else
  {
    for (moved=0; moved<distance; ++moved) bf_backward(current->buffer);
    return 0;
  }
}
/*}}}*/
/* ds_unfold     -- unfold *//*{{{*/
int ds_unfold(void)
{
  char c,mark;

  if (current->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }
  bf_mark(current->buffer,current->cursor);
  bf_linestart(current->buffer,1);
  while (bf_rchar(current->buffer,&c,&mark))
  {
    if (c=='\0' && mark==FOLDSTART)
    {
      bf_forward(current->buffer);
      bf_delete(current->buffer);
      bf_foldend(current->buffer,1,(unsigned int*)0);
      bf_delete(current->buffer);
      bf_tomark(current->buffer,current->cursor);
      current->update=ALL|MODIFIED|LINE;
      return 1;
    }
    else if (c=='\0' && mark==FOLDEND)
    {
      bf_forward(current->buffer);
      bf_delete(current->buffer);
      bf_foldstart(current->buffer,1,(unsigned int*)0);
      bf_forward(current->buffer);
      bf_delete(current->buffer);
      bf_tomark(current->buffer,current->cursor);
      current->update=ALL|MODIFIED|LINE;
      return 1;
    }
    else if (c=='\n') break;
    if (bf_forward(current->buffer)==0) break;
  }
  bf_tomark(current->buffer,current->cursor);
  return 0;
}
/*}}}*/
/* ds_forword    -- forward word *//*{{{*/
int ds_forword(enum Wordaction action)
{
  char c,mark;
            
  if (bf_rchar(current->buffer,&c,&mark)==0) return 0;
  if (isalnum((int)c))
  do
  {
    if (ds_forward()==0) break;
    switch (action)
    {
      case NOTHING: break;
      case TOUPPER: bf_delete(current->buffer); bf_insert(current->buffer,toupper(c),'\0'); break;
      case TOLOWER: bf_delete(current->buffer); bf_insert(current->buffer,tolower(c),'\0'); break;
      case CAPITALISE: bf_delete(current->buffer); bf_insert(current->buffer,toupper(c),'\0'); action=TOLOWER; break;
      case DELETE: bf_delete(current->buffer); break;
    }
    if (bf_rchar(current->buffer,&c,&mark)==0) break;
  } while (isalnum((int)c));
  else if (c==' ' || c=='\t' || c=='\n')
  do
  {
    if (ds_forward()==0) break;
    if (action==DELETE) 
    {
      bf_delete(current->buffer);
      if (c=='\n') current->update|=ALL|LINE;
    }
    if (bf_rchar(current->buffer,&c,&mark)==0) break;
  } while (c==' ' || c=='\t' || c=='\n');
  else
  {
    ds_forward();
    if (action==DELETE) bf_delete(current->buffer);
  }
  if ((current->update&UPDATEMASK)==NONE) current->update|=LINE;
  return 1;
}
/*}}}*/
/* ds_backword   -- backward word *//*{{{*/
int ds_backword(enum Wordaction action)
{
  char c,mark;
            
  assert(action==NOTHING || action==DELETE);
  if (bf_lchar(current->buffer,&c,&mark)==0) return 0;
  if (isalnum((int)c))
  do
  {
    if (action==NOTHING) ds_backward();
    else bf_delete(current->buffer);
    if (bf_lchar(current->buffer,&c,&mark)==0) break;
  } while (isalnum((int)c));
  else if (c==' ' || c=='\t' || c=='\n')
  do
  {
    if (action==NOTHING) ds_backward();
    else
    {
      bf_delete(current->buffer);
      if (c=='\n') current->update|=ALL|LINE;
    }
    if (bf_lchar(current->buffer,&c,&mark)==0) break;
  } while (c==' ' || c=='\t' || c=='\n');
  else
  {
    if (action==NOTHING) ds_backward();
    else bf_delete(current->buffer);
  }
  if ((current->update&UPDATEMASK)==NONE) current->update|=LINE;
  return 1;
}
/*}}}*/
/* ds_gotoline   -- go to line *//*{{{*/
int ds_gotoline(int line)
{
  unsigned int dx;

  dx=current->x;
  if (bf_gotoline(current->buffer,line)==0) return 0;
  current->update=(current->update&~UPDATEMASK)|ALL|LINE;
  current->y=(current->maxy-1)/2;
  ds_tocol(dx);
  return 1;
}
/*}}}*/
/* ds_transpose  -- transpose characters *//*{{{*/
int ds_transpose(void)
{
  char lc,lmark,rc,rmark;

  if (current->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }
  if (bf_lchar(current->buffer,&lc,&lmark)==0 || bf_rchar(current->buffer,&rc,&rmark)==0)
  {
    showmsg(_("[These characters can not be transposed]"));
    return 0;
  }
  if (lc=='\n')
  {
    if (bf_isfold(current->buffer,FOLDSTART|FOLDEND,LEFT|RIGHT) && rc=='\0' && (rmark==FOLDSTART || rmark==FOLDEND))
    {
      showmsg(_("[These characters can not be transposed]"));
      return 0;
    }
    current->update=ALL|MODIFIED|LINE;
  }
  if (rc=='\n')
  {
    bf_forward(current->buffer);
    if (bf_isfold(current->buffer,FOLDSTART|FOLDEND,LEFT|RIGHT) && lc=='\0' && (lmark==FOLDSTART || lmark==FOLDEND))
    {
      bf_backward(current->buffer);
      showmsg(_("[These characters can not be transposed]"));
      return 0;
    }
    bf_backward(current->buffer);
    if (current->y<current->maxy-2) ++current->y;
    current->update=ALL|MODIFIED|LINE;
  }
  bf_delete(current->buffer);
  bf_forward(current->buffer);
  bf_insert(current->buffer,lc,lmark);
  if ((current->update&UPDATEMASK)==NONE) current->update|=LINE;
  return 1;
}
/*}}}*/
/* ds_strfsearch -- search string forward *//*{{{*/
int ds_strfsearch(const char *needle, size_t needlesize)
{
  if (bf_strfsearch(current->buffer,needle,needlesize))
  {
    current->y=(current->maxy-1)/2;
    current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    return 1;
  }
  else
  {
    return 0;
  }
}
/*}}}*/
/* ds_regfsearch -- search regexp forward *//*{{{*/
int ds_regfsearch(const regex_t *needle)
{
  if (bf_regfsearch(current->buffer,needle))
  {
    current->y=(current->maxy-1)/2;
    current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    return 1;
  }
  else
  {
    return 0;
  }
}
/*}}}*/
/* ds_strbsearch -- search string backward *//*{{{*/
int ds_strbsearch(const char *needle, size_t needlesize)
{
  if (bf_strbsearch(current->buffer,needle,needlesize))
  {
    current->y=(current->maxy-1)/2;
    current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    return 1;
  }
  else
  {
    return 0;
  }
}
/*}}}*/
/* ds_regbsearch -- search regexp backward *//*{{{*/
int ds_regbsearch(const regex_t *needle)
{
  if (bf_regbsearch(current->buffer,needle))
  {
    current->y=(current->maxy-1)/2;
    current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    return 1;
  }
  else
  {
    return 0;
  }
}
/*}}}*/
/* ds_insertbuf  -- insert buffer at cursor position *//*{{{*/
int ds_insertbuf(struct Buffer *b, int markstart)
{
  int count,i;
  char c,mark;

  assert(b!=(struct Buffer*)0);
  if (current->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }
  if (bf_isfold(current->buffer,FOLDSTART|FOLDEND,LEFT))
  {
    bf_begin(b);
    if (bf_isfold(b,FOLDSTART|FOLDEND,LEFT|RIGHT))
    {
      showmsg(_("[Insertion failed at this cursor position]"));
      return 0;
    }
  }
  else if (bf_isfold(current->buffer,FOLDSTART|FOLDEND,RIGHT))
  {
    bf_end(b);
    if (bf_isfold(b,FOLDSTART|FOLDEND,LEFT|RIGHT))
    {
      showmsg(_("[Insertion failed at this cursor position]"));
      return 0;
    }
  }
  bf_begin(b);
  for (count=0; bf_rchar(b,&c,&mark); ++count)
  {
    if (c=='\n') current->update=ALL|MODIFIED|LINE;
    if (bf_insert(current->buffer,c,mark)==0)
    {
      while (count>0) bf_delete(current->buffer);
      showmsg(_("[Out of memory]"));
      return 0;
    }
    bf_forward(b);
  }
  if (markstart)
  {
    for (i=0; i<count; ++i) bf_backward(current->buffer);
    bf_mark(current->buffer,current->mark);
    for (i=0; i<count; ++i) bf_forward(current->buffer);
  }
  if ((current->update&UPDATEMASK)==NONE) current->update|=(LINE|MODIFIED);
  return 1;
}
/*}}}*/
/* ds_covebuf    -- copy or move region into new buffer *//*{{{*/
struct Buffer *ds_covebuf(struct Display *d, int delete, int copy)
{
  /* variables *//*{{{*/
  struct Buffer *b;
  int level;
  unsigned int distance,moved;
  int lf;
  int joinedLine;
  char c,mark;
  /*}}}*/

  assert(d!=(struct Display*)0);
  if (delete && current->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }
  if ((b=bf_alloc())==(struct Buffer*)0) return (struct Buffer*)0;
  bf_mark(d->buffer,d->cursor);
  if (d->buffer->mark[d->cursor]>d->buffer->mark[d->mark])
  {
    unsigned int x;

    x=d->cursor;
    d->cursor=d->mark;
    d->mark=x;
    bf_tomark(d->buffer,d->cursor);
  }
  distance=d->buffer->mark[d->mark]-d->buffer->mark[d->cursor];
  lf=bf_isfold(d->buffer,FOLDSTART|FOLDEND,LEFT);
  joinedLine=level=0;
  for (moved=0; moved<distance; ++moved) /* scan if region crosses folds, copy region if needed *//*{{{*/
  {
    bf_forward(d->buffer);
    if (bf_lchar(d->buffer,&c,&mark))
    {
      if (c=='\0') switch (mark)
      {
        case FOLDSTART: ++level; break;
        case FOLDEND: --level; break;
      }
      if (level==0 && c=='\n' && delete) joinedLine=1;
      if (copy && bf_insert(b,c,mark)==0)
      {
        bf_free(b);
        bf_tomark(d->buffer,d->cursor);
        return (struct Buffer*)0;
      }
    }
    else assert(0);
  }
  /*}}}*/
  if (level || (delete && lf && bf_isfold(d->buffer,FOLDSTART|FOLDEND,RIGHT))) /* cross fold or deleting would violate fold structure *//*{{{*/
  {
    bf_free(b);
    bf_tomark(d->buffer,d->cursor);
    return (struct Buffer*)0;
  }
  /*}}}*/
  if (delete)
  {
    for (moved=0; moved<distance; ++moved) bf_delete(d->buffer);
  }
  if (delete)
  {
    if (joinedLine) d->update=ALL|MODIFIED|LINE;
    else if ((d->update&UPDATEMASK)==NONE) d->update|=(LINE|MODIFIED);
  }
  b->lang=d->buffer->lang;
  return b;
}
/*}}}*/
/* ds_moveline   -- move line beginning with cursor into new buffer *//*{{{*/
struct Buffer *ds_moveline(void)
{
  char c,mark;

  if (current->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }
  if (bf_rchar(current->buffer,&c,&mark)==0 || bf_isfold(current->buffer,FOLDEND,RIGHT))
  {
    showmsg(_("[You can not kill this line]"));
    return (struct Buffer*)0;
  }
  bf_mark(current->buffer,current->mark);
  if (bf_isfold(current->buffer,FOLDSTART,RIGHT))
  {
    int r;

    r=bf_lineend(current->buffer,1);
    assert(r);
    r=bf_foldend(current->buffer,1,(unsigned int*)0);
    assert(r);
    r=bf_lineend(current->buffer,1);
    assert(r);
  }
  else
  {
    if (c=='\n') bf_forward(current->buffer);
    else bf_lineend(current->buffer,1);
  }
  return (ds_covebuf(current,1,1));
}
/*}}}*/
/* ds_indent     -- copy current indentation into new buffer *//*{{{*/
struct Buffer *ds_indent(void)
{
  struct Buffer *b;
  char mark,ch;

  if ((b=bf_alloc())==(struct Buffer*)0) return (struct Buffer*)0;
  bf_mark(current->buffer,current->cursor);
  bf_linestart(current->buffer,1);
  while (bf_rchar(current->buffer,&ch,&mark) && (ch=='\t' || ch==' ') && (current->buffer->curch<current->buffer->mark[current->cursor]))
  {
    if (bf_insert(b,ch,'\0')==0 || bf_forward(current->buffer)==0)
    {
      bf_free(b);
      return (struct Buffer*)0;
    }
  }
  bf_tomark(current->buffer,current->cursor);
  return b;
}
/*}}}*/
/* ds_reshape    -- reshape display *//*{{{*/
void ds_reshape(int orix, int oriy, int maxx, int maxy)
{
  current->orix=orix;
  current->oriy=oriy;
  current->maxx=maxx;
  current->maxy=maxy;
  if (current->y>=current->maxy-1) current->y=current->maxy-2;
  current->update=ALL;
}
/*}}}*/
/* ds_prevpage   -- move to previous page but stay inside fold */ /*{{{*/
int ds_prevpage(void)
{
  int i,hitstart;

  for (i=0; i<current->maxy-1; ++i)
  {
    if ((hitstart=ds_prevline())==0) return 0;
    else if (hitstart==2) return 2;
  }
  return 1;
}
/*}}}*/
/* ds_nextpage   -- move to next page but stay inside fold */ /*{{{*/
int ds_nextpage(void)
{
  int i;

  for (i=0; i<current->maxy-1; ++i)
  {
    if (ds_nextline()==0) return 0;
    else if (bf_isfold(current->buffer,FOLDEND,LEFT|RIGHT)) return 2;
  }
  return 1;
}
/*}}}*/
/* ds_gotofence  -- go to matching fence */ /*{{{*/
int ds_gotofence(void)
{
  char c,mark;

  if (bf_rchar(current->buffer,&c,&mark))
  {
    char l,r;
    int level,forw;

    l=c;
    switch (c)
    {
      case '(': r=')'; forw=1; goto join;
      case '{': r='}'; forw=1; goto join;
      case '[': r=']'; forw=1; goto join;
      case ')': r='('; forw=0; goto join;
      case '}': r='{'; forw=0; goto join;
      case ']': r='['; forw=0;
      join:
      {
        level=1;
        /*bf_mark(current->buffer,current->mark);*/
        do
        {
          if ((forw ? bf_forward(current->buffer) : bf_backward(current->buffer))==0 || bf_rchar(current->buffer,&c,&mark)==0)
          {
            bf_tomark(current->buffer,current->cursor);
            return 0;
          }
          if (c==r) --level;
          else if (c==l) ++level;
        }
        while (level);
        current->update=(current->update&~UPDATEMASK)|ALL|LINE;
        return 1;
      }
      default: return 0;
    }
  }
  else return 0;
}
/*}}}*/
/* ds_shell      -- start a sub shell */ /*{{{*/
const char *ds_shell(const char *msg, const char *cmd)
{
  pid_t pid;
  struct sigaction interrupt;

  refresh();
  interrupt.sa_flags=0;
  sigemptyset(&interrupt.sa_mask);
  interrupt.sa_handler=SIG_IGN;
  sigaction(SIGINT,&interrupt,(struct sigaction *)0);
  sigaction(SIGQUIT,&interrupt,(struct sigaction *)0);
  switch (pid=fork())
  {
    /*      -1 */ /*{{{*/
    case -1: return strerror(errno);
    /*}}}*/
    /*       0 */ /*{{{*/
    case 0:
    {
      if (msg) showmsg(msg);
      move(LINES-1,0);
      refresh();
      reset_shell_mode();
      write(1,"\n",1);
      interrupt.sa_handler=SIG_DFL;
      sigaction(SIGINT,&interrupt,(struct sigaction *)0);
      sigaction(SIGQUIT,&interrupt,(struct sigaction *)0);
      if (cmd) execlp(getshell(),getshell(),"-c",cmd,(const char*)0);
      else execlp(getshell(),getshell(),(const char*)0);
      exit(127);
      break;
    }
    /*}}}*/
    /* default */ /*{{{*/
    default:
    {
      pid_t r;
      int status;
      static char buf[20];
      const char *cont;

      while ((r=wait(&status))!=-1 && r!=pid);
      reset_prog_mode();
      interrupt.sa_handler=SIG_DFL;
      sigaction(SIGINT,&interrupt,(struct sigaction *)0);
      sigaction(SIGQUIT,&interrupt,(struct sigaction *)0);
      if (msg==(const char*)0)
      {
        cont=_("[Press any key to continue]");
        write(1,cont,strlen(cont));
        mc_get();
      }
      touchwin(curscr);
      wrefresh(curscr);
      sprintf(buf,"%d",status);
      return (status ? buf : (const char*)0);
    }
    /*}}}*/
  }
}
/*}}}*/
/* ds_suspend    -- suspend fe */ /*{{{*/
void ds_suspend(const char *msg)
{
  assert(msg!=(const char*)0);
  showmsg(msg);
  move(LINES-1,0);
  refresh();
  reset_shell_mode();
  write(1,"\n",1);
  kill(getpid(),SIGSTOP);
  alarm(0); /* and hope for the best */
  reset_prog_mode();
  touchwin(curscr);
  wrefresh(curscr);
  showmsg((const char*)0);
}
/*}}}*/
/* ds_compose    -- compose character from two characters */ /*{{{*/
char ds_compose(chtype ch1, chtype ch2)
{
  /* variables */ /*{{{*/
  static struct { char a,b,c; } compose [] =
  {
    { '+',  '+', '#'    }, /* #  number sign            */
    { 'a',  'a', '@'    }, /* @  commercial at          */
    { '(',  '(', '['    }, /* [  opening bracket        */
    { ')',  ')', ']'    }, /* ]  closing bracket        */
    { '(',  '-', '{'    }, /* {  opening brace          */
    { ')',  '-', '}'    }, /* }  closing brace          */
    { '/',  '/', '\\'   }, /* \  backslash              */
    { '/',  '^', '|'    }, /* |  vertical bar           */
    { ' ',  ' ', '\240' }, /*   no-break space         */
    { '|',  '|', '\246' }, /*   broken bar             */
    { '-',  '-', '\255' }, /*   soft hyphen            */
    { 'u',  '/', '\265' }, /*   micro sign             */
    { '!',  '!', '\241' }, /*   inverted !             */
    { '?',  '?', '\277' }, /*   inverted ?             */
    { 'c',  '/', '\242' }, /*   cent sign              */
    { 'c',  '|', '\242' }, /*   cent sign              */
    { 'l',  '-', '\243' }, /*   pound sign             */
    { 'l',  '=', '\243' }, /*   pound sign             */
    { 'x',  'o', '\244' }, /*   currency sign          */
    { 'x',  '0', '\244' }, /*   currency sign          */
    { 'e',  'u', '\244' }, /*   euro sign              */
    { 'y',  '-', '\245' }, /*   yen sign               */
    { 'y',  '=', '\245' }, /*   yen sign               */
    { 's',  'o', '\247' }, /*   section sign           */
    { 's',  '!', '\247' }, /*   section sign           */
    { 's',  '0', '\247' }, /*   section sign           */
    { 'p',  '!', '\266' }, /*   pilcrow sign           */
    { '"',  '"', '\250' }, /*   diaeresis              */
    { ':',  ':', '\250' }, /*   diaeresis              */
    { '^',  '-', '\257' }, /*   macron                 */
    { '^',  '_', '\257' }, /*   macron                 */
    { '\'', '\'','\264' }, /*   acute accent           */
    { ',',  ',', '\270' }, /*   cedilla                */
    { 'c',  'o', '\251' }, /*   copyright sign         */
    { 'c',  '0', '\251' }, /*   copyright sign         */
    { 'r',  'o', '\256' }, /*   registered sign        */
    { 'a',  '_', '\252' }, /*   feminine ordinal       */
    { 'o',  '_', '\272' }, /*   masculine ordinal      */
    { '<',  '<', '\253' }, /*   opening angle brackets */
    { '>',  '>', '\273' }, /*   closing angle brakets  */
    { '0',  '^', '\260' }, /*   degree sign            */
    { '1',  '^', '\271' }, /*   superscript 1          */
    { '2',  '^', '\262' }, /*   superscript 2          */
    { '3',  '^', '\263' }, /*   superscript 3          */
    { '+',  '-', '\261' }, /*   plus or minus sign     */
    { '1',  '4', '\274' }, /*   fraction one-quarter   */
    { '1',  '2', '\275' }, /*   fraction one-half      */
    { '3',  '4', '\276' }, /*   fraction three quarter */
    { '.',  '^', '\267' }, /*   middle dot             */
    { '-',  '|', '\254' }, /*   not sign               */
    { '-',  ',', '\254' }, /*   not sign               */
    { 'x',  'x', '\327' }, /*   multiplication sign    */
    { ':',  '-', '\367' }, /*   division sign          */
    { '`',  'A', '\300' }, /*   A grave                */
    { '`',  'a', '\340' }, /*   a grave                */
    { '\'', 'A', '\301' }, /*   A acute                */
    { '\'', 'a', '\341' }, /*   a acute                */
    { '^',  'A', '\302' }, /*   A circumflex           */
    { '^',  'a', '\342' }, /*   a circumflex           */
    { '~',  'A', '\303' }, /*   A tilde                */
    { '~',  'a', '\343' }, /*   a tilde                */
    { ':',  'A', '\304' }, /*   A diaeresis            */
    { '"',  'A', '\304' }, /*   A diaeresis            */
    { ':',  'a', '\344' }, /*   a diaeresis            */
    { '"',  'a', '\344' }, /*   a diaeresis            */
    { 'A',  'O', '\305' }, /*   A ring                 */
    { 'A',  '0', '\305' }, /*   A ring                 */
    { 'A',  '*', '\305' }, /*   A ring                 */
    { 'a',  'o', '\345' }, /*   a ring                 */
    { 'a',  '0', '\345' }, /*   a ring                 */
    { 'a',  '*', '\345' }, /*   a ring                 */
    { 'A',  'E', '\306' }, /*   AE ligature            */
    { 'a',  'e', '\346' }, /*   ae ligature            */
    { ',',  'C', '\307' }, /*   C cedilla              */
    { ',',  'c', '\347' }, /*   c cedilla              */
    { '`',  'E', '\310' }, /*   E grave                */
    { '`',  'e', '\350' }, /*   e grave                */
    { '\'', 'E', '\311' }, /*   E acute                */
    { '\'', 'e', '\351' }, /*   e acute                */
    { '^',  'E', '\312' }, /*   E circumflex           */
    { '^',  'e', '\352' }, /*   e circumflex           */
    { ':',  'E', '\313' }, /*   E diaeresis            */
    { '"',  'E', '\313' }, /*   E diaeresis            */
    { ':',  'e', '\353' }, /*   e diaeresis            */
    { '"',  'e', '\353' }, /*   e diaeresis            */
    { '`',  'I', '\314' }, /*   I grave                */
    { '`',  'i', '\354' }, /*   i grave                */
    { '\'', 'I', '\315' }, /*   I acute                */
    { '\'', 'i', '\355' }, /*   i acute                */
    { '^',  'I', '\316' }, /*   I circumflex           */
    { '^',  'i', '\356' }, /*   i circumflex           */
    { ':',  'I', '\317' }, /*   I diaeresis            */
    { '"',  'I', '\317' }, /*   I diaeresis            */
    { ':',  'i', '\357' }, /*   i diaeresis            */
    { '"',  'i', '\357' }, /*   i diaeresis            */
    { '-',  'D', '\320' }, /*   capital eth            */
    { '-',  'd', '\360' }, /*   small eth              */
    { '~',  'N', '\321' }, /*   N tilde                */
    { '~',  'n', '\361' }, /*   n tilde                */
    { '`',  'O', '\322' }, /*   O grave                */
    { '`',  'o', '\362' }, /*   o grave                */
    { '\'', 'O', '\323' }, /*   O acute                */
    { '\'', 'o', '\363' }, /*   o acute                */
    { '^',  'O', '\324' }, /*   O circumflex           */
    { '^',  'o', '\364' }, /*   o circumflex           */
    { '~',  'O', '\325' }, /*   O tilde                */
    { '~',  'o', '\365' }, /*   o tilde                */
    { ':',  'O', '\326' }, /*   O diaeresis            */
    { '"',  'O', '\326' }, /*   O diaeresis            */
    { ':',  'o', '\366' }, /*   o diaeresis            */
    { '"',  'o', '\366' }, /*   o diaeresis            */
    { '/',  'O', '\330' }, /*   O slash                */
    { '/',  'o', '\370' }, /*   o slash                */
    { '`',  'U', '\331' }, /*   U grave                */
    { '`',  'u', '\371' }, /*   u grave                */
    { '\'', 'U', '\332' }, /*   U acute                */
    { '\'', 'u', '\372' }, /*   u acute                */
    { '^',  'U', '\333' }, /*   U circumflex           */
    { '^',  'u', '\373' }, /*   u circumflex           */
    { ':',  'U', '\334' }, /*   U diaeresis            */
    { '"',  'U', '\334' }, /*   U diaeresis            */
    { ':',  'u', '\374' }, /*   u diaeresis            */
    { '"',  'u', '\374' }, /*   u diaeresis            */
    { '\'', 'Y', '\335' }, /*   Y acute                */
    { '\'', 'y', '\375' }, /*   y acute                */
    { 'T',  'H', '\336' }, /*   capital thorn          */
    { 't',  'h', '\376' }, /*   small thorn            */
    { 's',  's', '\337' }, /*   German small sharp s   */
    { 's',  'z', '\337' }, /*   German small sharp s   */
    { '"',  'y', '\377' }, /*   y diaeresis            */
    { 'i',  'j', '\377' }, /*   y diaeresis            */
    { '\0', '\0', '\0' }
  };

  int i;
  /*}}}*/  

  if (ch1!=(ch1&0xff) || ch2!=(ch2&0xff)) return '\0';
  if (ch2==0)
  {
    for (i=0; compose[i].a && ch1!=compose[i].a && ch1!=compose[i].b; ++i);
    return (compose[i].a);
  }
  for (i=0; compose[i].a && (ch1!=compose[i].a || ch2!=compose[i].b) && (ch1!=compose[i].b || ch2!=compose[i].a); ++i);
  if (compose[i].a) return compose[i].c;
  else return '\0';
}
/*}}}*/
/* ds_checkout   -- check out file and reload it */ /*{{{*/
int ds_checkout(void)
{
  char cmd[255+sizeof(CO_L)+1];
  const char *msg;
  int y=current->buffer->cury;

  strcpy(cmd,CO_L);
  strcat(cmd," ");
  strncat(cmd,current->buffer->name,255);
  cmd[255+sizeof(CO_L)]='\0';
  if ((msg=ds_shell((const char*)0,cmd)))
  {
    showmsg(_("[Check-out failed: %s]"),msg);
    return 0;
  }
  else
  {
    current->buffer->mode&=~READONLY;
    bf_empty(current->buffer);
    if (bf_load(current->buffer,(const char*)0,current->buffer->name,-1,"{{{","}}}"))
    {
      current->buffer->mode&=~CHANGED;
      current->update=(current->update&~UPDATEMASK)|ALL|LINE;
      bf_gotoline(current->buffer,y);
      ds_tocol(current->x);
      return 1;
    }
    else
    {
      showmsg(_("[Loading failed: %s]"),strerror(errno));
      return 0;
    }
  }
}
/*}}}*/
/* ds_checkin    -- check in file and set buffer read-only */ /*{{{*/
int ds_checkin(void)
{
  char cmd[255+sizeof(CI_U)+1];
  const char *msg;

  if (bf_save(current->buffer,current->buffer->name,"{{{","}}}",(unsigned int *)0))
  {
    strcpy(cmd,CI_U);
    strcat(cmd," ");
    strncat(cmd,current->buffer->name,255);
    cmd[255+sizeof(CI_U)]='\0';
    if ((msg=ds_shell((const char*)0,cmd)))
    {
      showmsg(_("[Check-in failed: %s]"),msg);
      return 0;
    }
    else
    {
      current->buffer->mode|=READONLY;
      return 1;
    }
  }
  else
  {
    showmsg(_("[Check-in failed: %s]"),strerror(errno));
    return 0;
  }
}
/*}}}*/
/* ds_describe   -- describe line */ /*{{{*/
void ds_describe(void)
{
  char c,mark;

  if (bf_rchar(current->buffer,&c,&mark))
  {
    if (c=='\0') switch (mark)
    {
      case NUL: showmsg(_("[Line %d Column %d %s]"),current->buffer->cury+1,current->x,"^@"); break;
      case FOLDSTART: showmsg(_("[Line %d Column %d %s]"),current->buffer->cury+1,current->x,_("Opening fold mark")); break;
      case FOLDEND: showmsg(_("[Line %d Column %d %s]"),current->buffer->cury+1,current->x,_("Closing fold mark")); break;
      default: assert(0);
    }
    else
    {
      if (c>=' ' && (c&0x80)=='\0') showmsg(_("[Line %d Column %d '%c']"),current->buffer->cury+1,current->x,c);
      else showmsg(_("[Line %d Column %d '%d'='%o']"),current->buffer->cury+1,current->x,(unsigned char)c,(unsigned char)c);
    }
  }
  else showmsg(_("[Line %d Column %d]"),current->buffer->cury+1,current->x);
}
/*}}}*/
/* ds_setlang    -- set language */ /*{{{*/
void ds_setlang(void)
{
  MenuChoice menu[LANGUAGES];
  int i;

  for (i=0; i<LANGUAGES-1; ++i)
  {
    const char *lang;

    if (*(lang=language[i].name)=='\0') lang=_("No language");
    menu[i].str=malloc(strlen(lang)+2);
    strcpy(menu[i].str+1,lang);
    menu[i].str[0]=tolower(lang[0]);
    menu[i].c='\0';
  }
  menu[LANGUAGES-1].str=(char*)0;
  i=menuline(_("Language:"),menu,current->buffer->lang-language);
  if (i>=0) bf_lang(current->buffer,&(language[i]));
  for (i=0; i<LANGUAGES-1; ++i) free(menu[i].str);
}
/*}}}*/
