#ifndef DISPLAY_H
#define DISPLAY_H

#include <sys/types.h>
#include <regex.h>

#include "buffer.h"
#include "msgline.h"

#define MAXDISPLAYS 16
#define MAXTAB 64

#define UPDATEMASK 3

enum Update { NONE=0, LINE=1, ALL=2, MODIFIED=4 };

#define LINENOWIDTH (6+1)

enum Displaymode
{
  SIDESCROLL=1, TIME=2, POSITION=4, OVERWRITE=8, SHOWBEGEND=16, LINENO=32,
  MAGIC=64, BOLDMARKS=128
};

enum Wordaction { NOTHING, TOUPPER, TOLOWER, CAPITALISE, DELETE };

struct Display
{
  struct Buffer *buffer;
  int x;
  int offx;
  int y;
  int maxx;
  int maxy;
  int orix;
  int oriy;
  int cursor;
  int mark;
  enum Displaymode mode;
  enum Update update;
  unsigned char tabsize;
};

extern struct Buffer *buffers[MAXDISPLAYS];

int ds_init(void);
void ds_exit(int regular);
void ds_winch(void);
struct Display *ds_alloc(struct Buffer *b, unsigned int orix, unsigned int oriy, unsigned int maxx, unsigned int maxy);
void ds_realloc(struct Display *d, struct Buffer *b);
void ds_free(struct Display *d);
void ds_detach(struct Display *d);
void ds_redraw(void);
void ds_current(struct Display *d);
void ds_tocol(unsigned int tox);
void ds_torow(struct Display *d, unsigned int row);
void ds_topos(struct Display *d, const char *gapstart);
void ds_refresh(void);
int ds_backward(void);
int ds_forward(void);
int ds_prevline(void);
int ds_nextline(void);
int ds_openfold(void);
int ds_closefold(void);
void ds_begin(void);
void ds_end(void);
int ds_beginfold(void);
int ds_endfold(void);
void ds_beginline(void);
void ds_endline(void);
void ds_delmode(enum Displaymode mode);
void ds_addmode(enum Displaymode mode);
unsigned int ds_tabsize(int tabsize);
int ds_insertch(char c);
int ds_deletech(int previous);
void ds_mark(void);
void ds_swapmark(void);
int ds_fold(void);
int ds_unfold(void);
int ds_forword(enum Wordaction action);
int ds_backword(enum Wordaction action);
int ds_gotoline(int line);
int ds_transpose(void);
int ds_strfsearch(const char *needle, size_t needlesize);
int ds_regfsearch(const regex_t *needle);
int ds_strbsearch(const char *needle, size_t needlesize);
int ds_regbsearch(const regex_t *needle);
int ds_insertbuf(struct Buffer *b, int markstart);
struct Buffer *ds_covebuf(struct Display *d, int delete, int copy);
struct Buffer *ds_moveline(void);
struct Buffer *ds_indent(void);
void ds_reshape(int orix, int oriy, int maxx, int maxy);
int ds_prevpage(void);
int ds_nextpage(void);
int ds_gotofence(void);
const char *ds_shell(const char *msg, const char *cmd);
void ds_suspend(const char *msg);
char ds_compose(chtype ch1, chtype ch2);
int ds_checkout(void);
int ds_checkin(void);
void ds_describe(void);
void ds_setlang(void);

#endif
